package clinical.tools.rls;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.Console;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.InputStreamReader;
import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.globus.myproxy.MyProxy;
import org.globus.replica.rls.CatalogQuery;
import org.globus.replica.rls.LocalReplicaCatalog;
import org.globus.replica.rls.Mapping;
import org.globus.replica.rls.MappingResult;
import org.globus.replica.rls.RLSConnection;
import org.globus.replica.rls.RLSConnectionSource;
import org.globus.replica.rls.RLSException;
import org.globus.replica.rls.RLSStatusCode;
import org.globus.replica.rls.Results;
import org.globus.replica.rls.SimpleCatalogQuery;
import org.globus.replica.rls.impl.SimpleRLSConnectionSource;
import org.globus.util.GlobusURL;
import org.ietf.jgss.GSSCredential;

import clinical.storage.plugins.gridftp.GridFtpClient;
import clinical.utils.FileUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: RLSTool.java 98 2009-10-13 23:14:21Z bozyurt $
 */
public class RLSTool {
	GSSCredential credential;
	String myProxyHost = "certs.nbirn.org";
	String rlsURL = "rls://rls1.nbirn.org:39281"; // test RLS
	// "rls://chi-vm-4.isi.edu:39281";
	RLSConnectionSource source = null;
	RLSConnection con = null;
	boolean verbose = false;
	public final static int FILE = 1;
	public final static int DIR = 2;
	public final static int UNKNOWN = 3;

	public RLSTool(String myProxyHost, String rlsURL) {
		this.myProxyHost = myProxyHost;
		this.rlsURL = rlsURL;
	}

	public void connect(String user, String passPhrase) throws Exception {
		MyProxy proxy = new MyProxy(myProxyHost, 7512);
		credential = proxy.get(user, passPhrase, 0);
		source = new SimpleRLSConnectionSource();
		GlobusURL url = new GlobusURL(rlsURL);
		con = source.connect(url, credential);
	}

	public void shutdown() {
		if (con != null) {
			try {
				con.close();
			} catch (RLSException e) {
				e.printStackTrace();
			}
		}
	}

	public void createMappings(String fileListFile, String localCommonPath,
			String gridFtpHostname, boolean testMode) throws Exception {
		final int BUFFER_SIZE = 100;
		List<Mapping> mpList = new ArrayList<Mapping>(BUFFER_SIZE);
		BufferedReader in = null;
		int count = 0;
		int totCount = 0;
		if (localCommonPath != null) {
			localCommonPath = localCommonPath.replaceAll("\\/+$", "");
		}
		StringBuilder sb = new StringBuilder(100);
		sb.append("gsiftp://").append(gridFtpHostname);
		String targetLocPrefix = sb.toString();
		int mappingCount = 0;
		try {
			in = new BufferedReader(new FileReader(fileListFile), 4096);
			LocalReplicaCatalog lrc = con.catalog();
			String line = null;
			while ((line = in.readLine()) != null) {
				line = line.trim();
				if (line.indexOf(',') != -1) {
					String[] toks = line.split("\\s*,\\s*");
					line = toks[0];
				}
				if (localCommonPath != null) {
					line = applyCommonPath(line, localCommonPath);
				}
				if (count < BUFFER_SIZE) {
					Mapping mapping = new Mapping(line, targetLocPrefix + line);
					// System.out.println("target: " + (targetLocPrefix + line));
					mpList.add(mapping);
				} else {
					count = 0;
					if (verbose) {
						System.out.println("target: " + (targetLocPrefix + line)
								+ "\nlogical:" + line);
					}
					long start = System.currentTimeMillis();
					mappingCount = createMappings(mpList, lrc, mappingCount,
							testMode);
					if (verbose) {
						System.out.println("Elapsed time (msecs):"
								+ (System.currentTimeMillis() - start));
					}
					mpList.clear();

					Mapping mapping = new Mapping(line, targetLocPrefix + line);

					mpList.add(mapping);
				}

				count++;
				totCount++;
				// System.out.println(totCount);
			}
			if (!mpList.isEmpty()) {
				mappingCount = createMappings(mpList, lrc, mappingCount, testMode);
			}

			System.out.println("Total number of logical names to be registered:"
					+ totCount);
			System.out
					.println("Total number of logical names succesfully registered:"
							+ mappingCount);
		} finally {
			FileUtils.close(in);
		}
	}

	public List<RLSMapping> queryForPhysicalNames(String pattern)
			throws RLSException {
		if (pattern.endsWith("*")) {
			pattern = pattern.replaceAll("\\*", "%");
		}
		System.out.println("pattern:" + pattern);

		CatalogQuery cq = new SimpleCatalogQuery(
				SimpleCatalogQuery.queryMappingsByLogicalNamePattern, pattern, null);
		LocalReplicaCatalog lrc = null;
		Map<String, RLSMapping> map = new LinkedHashMap<String, RLSMapping>();
		lrc = this.con.catalog();
		Results results = lrc.query(cq);
		if (results.getRC() == RLSStatusCode.RLS_SUCCESS) {
			List<?> batch = results.getBatch();
			Iterator<?> it = batch.iterator();
			while (it.hasNext()) {
				MappingResult mr = (MappingResult) it.next();
				if (mr.getRC() != RLSStatusCode.RLS_SUCCESS) {
					continue;
				}
				String ln = mr.getLogical();
				String pn = mr.getTarget();
				RLSMapping rm = map.get(ln);
				if (rm == null) {
					rm = new RLSMapping(ln, pn);
					map.put(ln, rm);
				} else {
					rm.addReplica(pn);
				}
			}
		}
		return new ArrayList<RLSMapping>(map.values());
	}

	public static class RLSMapping {
		String logicalName;
		String physicalName;
		List<String> otherPNList;

		public RLSMapping(String logicalName, String physicalName) {
			super();
			this.logicalName = logicalName;
			this.physicalName = physicalName;
		}

		public void addReplica(String physicalName) {
			if (otherPNList == null) {
				otherPNList = new ArrayList<String>(1);
			}
			otherPNList.add(physicalName);
		}

		public String getSummary() {
			StringBuilder sb = new StringBuilder();
			sb.append(logicalName).append(" - ").append(physicalName);

			return sb.toString();
		}
	}// ;

	public static void showQueryResultSummary(List<RLSMapping> rmList) {
		Map<String, Integer> siteMap = new HashMap<String, Integer>(17);
		for (RLSMapping rm : rmList) {
			String gftpServer = extractGridFTPServer(rm.physicalName);
			if (gftpServer != null) {
				Integer count = siteMap.get(gftpServer);
				if (count == null) {
					count = new Integer(1);
				} else {
					count = new Integer(count.intValue() + 1);
				}
				siteMap.put(gftpServer, count);
			}
		}
		System.out.println("# results returned:" + rmList.size());
		int numGFTPServers = siteMap.size() > 0 ? siteMap.size() : 1;
		System.out.println("# of GridFTPServers:" + numGFTPServers);
		List<String> gftpNames = new ArrayList<String>(siteMap.keySet());
		Collections.sort(gftpNames);
		System.out.println("GridFTP Server versus # of Query Matches");
		System.out.println("----------------------------------------");
		for (String gftpName : gftpNames) {
			System.out.println(gftpName + " : " + siteMap.get(gftpName));
		}
	}

	public static String extractGridFTPServer(String physicalName) {
		if (!physicalName.startsWith("gsiftp://"))
			return null;
		String s = physicalName.replaceFirst("^gsiftp://", "");
		int idx = s.indexOf('/');
		if (idx == -1)
			return null;
		return s.substring(0, idx);
	}

	protected int createMappings(List<Mapping> mpList, LocalReplicaCatalog lrc,
			int mappingCount, boolean testMode) throws RLSException {
		if (testMode) {
			mappingCount += mpList.size();
			return mappingCount;
		}
		List<?> failedMappings = lrc.createMappings(mpList);
		mappingCount += mpList.size();
		if (failedMappings != null && !failedMappings.isEmpty()) {
			for (Object o : failedMappings) {
				MappingResult mr = (MappingResult) o;
				System.err.println("Failed mapping:" + mr);
			}
			mappingCount -= failedMappings.size();
		}
		return mappingCount;
	}

	protected String applyCommonPath(String path, String commonPathPrefix) {
		if (path.startsWith(commonPathPrefix)) {
			// already has the prefix
			return path;
		} else {
			if (path.startsWith("/")) {
				StringBuilder sb = new StringBuilder(path.length()
						+ commonPathPrefix.length());
				sb.append(commonPathPrefix).append(path);
				return sb.toString();
			} else {
				StringBuilder sb = new StringBuilder(path.length()
						+ commonPathPrefix.length() + 1);
				sb.append(commonPathPrefix).append('/').append(path);
				return sb.toString();
			}
		}
	}

	public void extractLogicalNamesFromHID(String dbUser, String pwd,
			String dbURL, String outFile) throws Exception {
		Class.forName("org.postgresql.Driver").newInstance();
		Statement st = null;
		Connection conn = null;
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(outFile), 4096);
			conn = DriverManager.getConnection(dbURL, dbUser, pwd);
			st = conn.createStatement();

			ResultSet rs = st.executeQuery("select datauri from nc_rawdata");
			int count = 0;
			while (rs.next()) {
				out.write(rs.getString(1));
				out.newLine();
				count++;
			}
			rs.close();
			System.out.println("Wrote " + count + " entries to file:" + outFile);
		} finally {
			close(conn);
			FileUtils.close(out);

		}
	}

	private void close(Connection con) {
		if (con != null) {
			try {
				con.close();
			} catch (SQLException se) {}
		}
	}

	public String getPhysicalLocation(String logicalName) throws RLSException {
		LocalReplicaCatalog lrc = con.catalog();
		Results results = lrc.query(new SimpleCatalogQuery(
				SimpleCatalogQuery.queryMappingsByLogicalName, logicalName, null));
		if (results.getRC() == RLSStatusCode.RLS_SUCCESS) {
			List<?> batch = results.getBatch();
			for (Object o : batch) {
				MappingResult mr = (MappingResult) o;
				if (mr.getRC() == RLSStatusCode.RLS_SUCCESS) {
					return mr.getTarget();
				}
			}
		}
		return null;
	}

	public void getFile(String logicalName, String localDir, int fileType)
			throws Exception {
		GridFtpClient client = null;
		String physicalLocation = getPhysicalLocation(logicalName);
		System.out.println(physicalLocation);
		if (physicalLocation != null) {
			physicalLocation = physicalLocation.replaceFirst("^gsiftp:\\/\\/", "");
			int idx = physicalLocation.indexOf('/');
			String gridFtpHost = physicalLocation.substring(0, idx);
			String location = physicalLocation.substring(idx);

			try {
				System.out.println("gridFtpHost:" + gridFtpHost);
				client = new GridFtpClient(gridFtpHost, myProxyHost, rlsURL);

				client.connect(credential);
				if (fileType == UNKNOWN) {
					// guess by file extension (i.e. if there is a file extension
					// than a file otherwise a directory
					String basename = new File(logicalName).getName();
					if (basename.indexOf('.') != -1) {
						fileType = FILE;
					} else {
						fileType = DIR;
					}
				}

				if (fileType == FILE) {
					String filename = new File(logicalName).getName();
					String remoteFile = location;
					File localFile = new File(localDir, filename);
					client.downloadFileNoRLSLookup(remoteFile, localFile, false);
					System.out.println("Downloaded file to " + localFile);
				} else if (fileType == DIR) {
					String basename = new File(logicalName).getName();
					File localFolder = new File(localDir, basename);
					String destFolder = location;
					client.downloadFolderNoRLSLookup(destFolder, localFolder, false);
					System.out.println("Downloaded directory contents to "
							+ localFolder);
				}
			} finally {
				if (client != null)
					client.shutdown();
			}
		}

		// client = new GridFtpClient();
	}

	static String usageTxt = "Usage: RLSTool -c <command> OPTIONS\n\n"
			+ "\t-c <command> [register-db-content|register-from-filelist|query]\n"
			+ "Common OPTIONS\n"
			+ "\t-g <gridftp-host> The the gridftp hostname for physical locations where the logical names are mapped to\n"
			+ "\t-u <user-name> Your myproxy user name\n"
			+ "\t-r <RLS-server-url> (default: rls://rls1.nbirn.org:39281)\n"
			+ "\t-m <myproxy-server-host> (default: certs.nbirn.org)\n"
			+ "\nOPTIONS for register-db-content command\n"
			+ "\t-dbhost <db-host> the Postgres database server host name (default: localhost)\n"
			+ "\t-schema <schema> the Postgres database schema name (default: hidprdp)\n"
			+ "\t-dbport <port> the Postgres database connection port (default: 5432)\n"
			+ "\t-dbuser <dbuser> the Postgres database user for connection (default: postgres)\n"
			+ "\t-dbpwd <pwd> the Postgres database user's password\n"
			+ "\nOPTIONS for register-from-filelist\n"
			+ "\t-f <logical-file-name-list>\n"
			+ "\nOPTIONS for query\n"
			+ "\t-q <logical-name or pattern> The wildcard '*' can be used after the end of the common prefix of a logical name\n"
			+ "\t-summary if specified show summary statistics about the query results.\n";

	public static void usage() {
		System.err.println(usageTxt);
		System.exit(1);
	}

	public static String getArgValue(int idx, String[] args) {
		if (idx >= args.length) {
			usage();
		}
		String val = args[idx];
		if (val.startsWith("-")) {
			usage();
		}
		return val;
	}

	public static int toInt(String value, int defaultVal) {
		if (value == null) {
			return defaultVal;
		}
		try {
			return Integer.parseInt(value);
		} catch (NumberFormatException nfe) {
			return defaultVal;
		}
	}

	public static void main(String[] args) throws Exception {
		String user = null;
		String gridFtpHost = null;
		String cmd = null;
		String dbHost = "localhost";
		String schema = "hidprdp";
		String dbUser = "postgres";
		String dbPwd = null;
		int dbPort = 5432;
		String logicalNameListFile = null;
		String myProxyHost = "certs.nbirn.org";
		String rlsURL = "rls://chi-vm-4.isi.edu:39281";
		rlsURL = "rls://rls1.nbirn.org:39281";
		final String FROM_DB = "register-db-content";
		final String FROM_LIST = "register-from-filelist";
		final String QUERY = "query";
		String queryStr = null;
		boolean showSummary = false;

		int i = 0;
		while (i < args.length) {
			if (args[i].equals("-c")) {
				cmd = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-g")) {
				gridFtpHost = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-u")) {
				user = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-schema")) {
				schema = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-dbhost")) {
				dbHost = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-dbuser")) {
				dbUser = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-dbpwd")) {
				dbPwd = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-dbport")) {
				dbPort = toInt(getArgValue(i + 1, args), 5432);
				i += 2;
			} else if (args[i].equals("-f")) {
				logicalNameListFile = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-m")) {
				myProxyHost = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-r")) {
				rlsURL = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-q")) {
				queryStr = getArgValue(i + 1, args);
				i += 2;
			} else if (args[i].equals("-summary")) {
				showSummary = true;
				i++;
			} else {
				i++;
			}
		}

		if (cmd == null || (!cmd.equalsIgnoreCase(QUERY) && gridFtpHost == null)
				|| user == null) {
			usage();
		}
		if (!cmd.equalsIgnoreCase(FROM_DB) && !cmd.equalsIgnoreCase(FROM_LIST)
				&& !cmd.equalsIgnoreCase(QUERY)) {
			usage();
		}
		if (cmd.equalsIgnoreCase(QUERY)
				&& (queryStr == null || queryStr.trim().length() == 0)) {
			usage();
		}

		RLSTool tool = null;
		try {
			tool = new RLSTool(myProxyHost, rlsURL);

			if (cmd.equalsIgnoreCase(FROM_DB)) {
				logicalNameListFile = File.createTempFile("logical_names", "txt")
						.getAbsolutePath();
				String dbURL = "jdbc:postgresql://" + dbHost + ":" + dbPort + "/"
						+ schema;
				System.out.println("dbURL:" + dbURL);
				tool.extractLogicalNamesFromHID(dbUser, dbPwd, dbURL,
						logicalNameListFile);
			}

			System.out.println("Enter passphrase:");
			Console console = System.console();
			String pwd = null;
			if (console != null) {
				pwd = new String(console.readPassword());
			} else {
				BufferedReader reader = new BufferedReader(new InputStreamReader(
						System.in));
				pwd = reader.readLine();
			}
			tool.connect(user, pwd);
			System.out.println("connected.");
			if (cmd.equalsIgnoreCase(QUERY)) {
				List<RLSMapping> rmList = tool.queryForPhysicalNames(queryStr);
				if (rmList.isEmpty()) {
					System.out.println("No results for query:" + queryStr);
				} else {
					System.out.println("Results for query:" + queryStr);
					System.out.println("---------------------------");
					for (RLSMapping rm : rmList) {
						System.out.println(rm.getSummary());
					}
					System.out.println();
					if (showSummary) {
						showQueryResultSummary(rmList);
					}
				}
			} else {
				// register mappings
				System.out
						.println("creating mappings on RLS server:" + tool.rlsURL);
				tool.createMappings(logicalNameListFile, null, gridFtpHost, false);
			}
		} finally {
			if (tool != null)
				tool.shutdown();
		}

	}

}
