package clinical.upload;

import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.Timestamp;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.StringTokenizer;

import org.apache.log4j.Logger;
import org.jdom.Document;
import org.jdom.Element;
import org.jdom.JDOMException;
import org.jdom.input.SAXBuilder;
import org.xml.sax.SAXException;

import clinical.server.AssessmentManager;
import clinical.server.ExperimentManager;
import clinical.server.ScoreInfo;
import clinical.server.dao.oracle.AssessmentDAO;
import clinical.server.dao.oracle.AssessmentscoreDAO;
import clinical.server.dao.oracle.ExpcomponentDAO;
import clinical.server.dao.oracle.SubjexperimentDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentscore;
import clinical.server.vo.Expcomponent;
import clinical.server.vo.Protocol;
import clinical.server.vo.Subjexperiment;
import clinical.utils.GenUtils;
import clinical.utils.NamedUserPool;
import clinical.web.Constants;

/**
 * 
 * @version $Id: AssessmentLoader.java 91 2009-08-17 23:38:26Z bozyurt $
 * @author I. Burak Ozyurt
 */

public class AssessmentLoader implements ILog {
   protected String metaFilename;
   protected String dataFilename;
   protected MetaDataLoader metaLoader;
   protected DataLoader dataLoader;
   protected Properties props;
   protected String owner;
   protected String modUser;
   protected String message;
   protected boolean testMode = true;
   protected PropertyChangeSupport pcs = new PropertyChangeSupport(this);

   protected String birnIDColName = "birnID";
   protected static Logger logger = Logger.getLogger(AssessmentLoader.class);
   public static final int UPLOAD_VISITS = 1;
   public static final int UPLOAD_ASSESSMENTS = 2;
   public static final int UPLOAD_AS_SCORE_VALUES = 4;
   public static final int REMOVE_VISITS = 8;
   public static final int REMOVE_AS_SCORE_VALUES = 16;
   // TODO fix dbID
   String dbID = null;

   public AssessmentLoader(Properties props) throws JDOMException, IOException {
      this.props = props;
      this.metaFilename = props.getProperty("metafile");
      this.dataFilename = props.getProperty("assessment.datafile");

      this.owner = props.getProperty("owner");
      this.modUser = props.getProperty("moduser");

      metaLoader = new MetaDataLoader(metaFilename);
   }

   public void setTestMode(boolean tmode) {
      this.testMode = tmode;
   }

   public boolean isTestMode() {
      return testMode;
   }

   public void setBirnIDColName(String newBirnIDColName) {
      this.birnIDColName = newBirnIDColName;
   }

   public String getBirnIDColName() {
      return this.birnIDColName;
   }

   // bounded property
   public String getMessage() {
      return message;
   }

   public void setMessage(String msg) {
      String oldMsg = message;
      message = msg;
      pcs.firePropertyChange("message", oldMsg, msg);
   }

   public void loadData() throws IOException, SAXException {
      metaLoader.parse();
      logger.info("loaded assesment metadata.");
      dataLoader = new DataLoader(dataFilename, metaLoader, 15);
      dataLoader.loadData();
      logger.info("loaded assesment data.");
      dataLoader.dumpData();
   }

   public List<String[]> loadCSV(String filename) throws IOException {
      BufferedReader in = null;
      List<String[]> rows = new LinkedList<String[]>();
      try {
         in = new BufferedReader(new FileReader(filename), 4096);
         String line = null;
         while ((line = in.readLine()) != null) {
            StringTokenizer stok = new StringTokenizer(line, ",");
            String[] row = new String[stok.countTokens()];
            int idx = 0;
            while (stok.hasMoreTokens()) {
               row[idx++] = stok.nextToken();
            }
            rows.add(row);
         }
      } finally {
         if (in != null)
            try {
               in.close();
            } catch (Exception x) {}
      }

      return rows;
   }

   public Map<String, String> prepareResearchGroupNameMap(String filename)
         throws IOException {
      Map<String, String> resGroupMap = new HashMap<String, String>();
      List<String[]> rows = loadCSV(filename);
      for (Iterator<String[]> iter = rows.iterator(); iter.hasNext();) {
         String[] row = iter.next();
         resGroupMap.put(row[0], row[1]); // birnID, researchGroupName
      }
      return resGroupMap;
   }

   public Map<String, String> getAllSubjectIdsFromDB(Connection con)
         throws Exception {
      Map<String, String> sidMap = new HashMap<String, String>();
      SubjexperimentDAO dao = new SubjexperimentDAO();
      List<?> subjexps = dao.find(con, new Subjexperiment());
      for (Iterator<?> iter = subjexps.iterator(); iter.hasNext();) {
         Subjexperiment item = (Subjexperiment) iter.next();
         String sid = item.getSubjectid();
         if (sidMap.get(sid) == null) {
            sidMap.put(sid, sid);
         }
      }
      return sidMap;
   }

   public void uploadVisits(Connection con, String experimentName,
         Map<String, String> resGroupMap, ExperimentStaticData expData,
         BigDecimal secLabel, String visitSerFile) throws Exception {
      List<Protocol> protocols = UploadHelper.getProtocols(dbID, con);

      ExpcomponentDAO dao = new ExpcomponentDAO();
      Expcomponent criteria = new Expcomponent();
      List<Expcomponent> allVisits = dao.find(con, criteria);

      // group by subjectids
      Map<String, List<Expcomponent>> subjectVisitsMap = new HashMap<String, List<Expcomponent>>();
      for (Iterator<?> iter = allVisits.iterator(); iter.hasNext();) {
         Expcomponent visit = (Expcomponent) iter.next();
         List<Expcomponent> visits = null;
         if ((visits = subjectVisitsMap.get(visit.getSubjectid())) == null) {
            subjectVisitsMap.put(visit.getSubjectid(),
                  visits = new LinkedList<Expcomponent>());
         }
         visits.add(visit);
      }

      // find out the missing visits for the assessment data
      Map<String, ColumnInfo> colInfoMap = metaLoader.getColInfoMap();

      ExperimentManager em = new ExperimentManager();

      // get all subjectids in the database to skip over non existing subjects
      Map<String, String> allSubjIDsMap = getAllSubjectIdsFromDB(con);

      Map<String, List<ExperimentInfo>> missingVisitsMap = new HashMap<String, List<ExperimentInfo>>();

      for (List<String> row : dataLoader.getRowDataList()) {

         ColumnInfo birnIDCol = colInfoMap.get(birnIDColName);

         String birnID = birnIDCol.getValue(row);
         if (birnID == null || birnID.length() == 0)
            continue;
         if (allSubjIDsMap.get(birnID) == null) {
            System.out.println("Subject " + birnID
                  + " is not a in the database skipping.");
            setMessage("Subject " + birnID
                  + " is not a in the database skipping.");
            continue;
         }

         List<Expcomponent> subjectVisits = subjectVisitsMap.get(birnID);
         for (Iterator<String> it2 = colInfoMap.keySet().iterator(); it2
               .hasNext();) {
            String colName = it2.next();
            if (colName.equals("birnID"))
               continue;
            ColumnInfo ci = colInfoMap.get(colName);
            if (ci.getStored() == false)
               continue; // skip non persistent columns
            // get the date at which the assessment is done
            ColumnInfo dateColumn = colInfoMap.get(ci.getDateCol());
            String dateValue = dateColumn.getValue(row);
            java.util.Date theDate = (java.util.Date) dateColumn.getType()
                  .convertToSQL(dateValue, "date");
            // check if there is any visit in db for that day
            Expcomponent theVisit = getVisit(ci.getVisitType(), subjectVisits,
                  theDate);
            if (theVisit == null) {
               // we need to add the visit info to the database
               List<ExperimentInfo> visitList = missingVisitsMap.get(birnID);
               if (visitList == null) {
                  visitList = new LinkedList<ExperimentInfo>();
                  missingVisitsMap.put(birnID, visitList);
               }
               // only unique visits
               if (hasSeenVisit(ci.getVisitType(), visitList, theDate))
                  continue;

               ExperimentInfo ei = new ExperimentInfo();
               ei.setBirnID(birnID);
               ei.setResearchGroupName(resGroupMap.get(birnID));
               ei.setExpName(experimentName);
               ei.setVisitDescp(expData.visitDescr);
               ei.setSegmentDescr(expData.segmentDescr);
               ei.setVisitType(ci.getVisitType());
               ei.setVisitDate(theDate);
               Protocol protocol = UploadHelper.findProtocol(ci.getVisitType(),
                     protocols);
               ei.setProtocolID(protocol.getProtocolid());
               ei.setProtocolVersion(protocol.getProtocolversion());

               visitList.add(ei);
            }

         } // it2
      }// it

      // set the visit ids
      for (Map.Entry<String, List<ExperimentInfo>> entry : missingVisitsMap
            .entrySet()) {
         String birnID = (String) entry.getKey();
         List<ExperimentInfo> visitList = (List<ExperimentInfo>) entry
               .getValue();

         List<Expcomponent> subjectVisits = subjectVisitsMap.get(birnID);
         assignVisitIds(subjectVisits, visitList);

      }
      // show the missing visits
      dumpMissingVisits(missingVisitsMap);

      // now add the missing visits to the database

      for (Map.Entry<String, List<ExperimentInfo>> entry : missingVisitsMap
            .entrySet()) {
         String birnID = (String) entry.getKey();
         List<ExperimentInfo> visitList = entry.getValue();
         System.out.println("BirnID=" + birnID);
         setMessage("BirnID=" + birnID);
         for (Iterator<ExperimentInfo> it = visitList.iterator(); it.hasNext();) {
            ExperimentInfo ei = it.next();

            if (testMode) {
               System.out.println("[TEST] Adding exprimentinfo 2 " + ei);
               setMessage("[TEST] Adding exprimentinfo 2 " + ei);
            } else {
               System.out.println(ei.toString());
               setMessage(ei.toString());
               em.addExperimentInfo2(con, birnID, ei.getExpName(), ei
                     .getResearchGroupName(), new BigDecimal(""
                     + ei.getVisitID()), ei.getVisitDate(), ei.getVisitDescp(),
                     ei.getVisitType(), new BigDecimal("1"), ei
                           .getSegmentDescr(), ei.getProtocolVersion(), ei
                           .getProtocolID(), owner, modUser, secLabel);
            }

         }
      }

      // serialize the missingVisitsMap to be used later to remove what is
      // stored
      if (!testMode && visitSerFile != null)
         GenUtils.serialize(missingVisitsMap, visitSerFile);

   }

   protected void assignVisitIds(List<Expcomponent> subjectVisits,
         List<ExperimentInfo> newVisits) {
      int curId = 1;
      for (Object element : subjectVisits) {
         Expcomponent visit = (Expcomponent) element;
         int val = visit.getComponentid().intValue();
         if (val > curId)
            curId = val;
      }

      Collections.sort(newVisits, new Comparator<ExperimentInfo>() {
         public int compare(ExperimentInfo o1, ExperimentInfo o2) {
            return (o1.getVisitDate().compareTo(o2.getVisitDate()));
         }
      });

      for (Object element : newVisits) {
         ExperimentInfo ei = (ExperimentInfo) element;
         ei.setVisitID(++curId);
      }
   }

   protected boolean hasSeenVisit(String visitType,
         List<ExperimentInfo> visits, java.util.Date theDate) {
      for (Iterator<ExperimentInfo> iter = visits.iterator(); iter.hasNext();) {
         ExperimentInfo ei = iter.next();
         if (ei.getVisitType().equals(visitType)
               && ei.getVisitDate().compareTo(theDate) == 0)
            return true;
      }
      return false;
   }

   protected void dumpMissingVisits(
         Map<String, List<ExperimentInfo>> missingVisitMap) {
      for (Map.Entry<String, List<ExperimentInfo>> entry : missingVisitMap
            .entrySet()) {
         String birnID = entry.getKey();
         List<ExperimentInfo> visitList = entry.getValue();
         System.out.println("BirnID=" + birnID);
         for (Object element : visitList) {
            ExperimentInfo ei = (ExperimentInfo) element;
            System.out.println(ei.toString());
         }
      }
   }

   public void uploadScoreValues(Connection con, String serFilename,
         BigDecimal secLabel) throws Exception {

      // get all assessments
      List<?> asList = null;
      Assessment ac = new Assessment();
      AssessmentDAO dao = new AssessmentDAO();
      asList = dao.find(con, ac);

      Map<String, ScoreInfo> scoreInfoMap = new HashMap<String, ScoreInfo>();
      // prepare the ScoreInfo objects
      Map<String, AssessmentWrapper> scoreMap = new HashMap<String, AssessmentWrapper>();
      AssessmentscoreDAO scoreDAO = new AssessmentscoreDAO();
      for (Iterator<?> it = asList.iterator(); it.hasNext();) {
         Assessment as = (Assessment) it.next();
         Assessmentscore criteria = new Assessmentscore();
         criteria.setAssessmentid(as.getAssessmentid());
         List<?> scoreList = scoreDAO.find(con, criteria);
         List<Assessmentscore> usedScoresList = new LinkedList<Assessmentscore>();
         AssessmentWrapper aw = null;
         for (Iterator<?> it2 = scoreList.iterator(); it2.hasNext();) {
            Assessmentscore score = (Assessmentscore) it2.next();
            if (metaLoader.colInfoMap.get(score.getScorename()) != null) {
               // found a matching column
               usedScoresList.add(score);
               ScoreInfo si = new ScoreInfo(as, score);
               /** @todo check if score name is unique enough */
               scoreInfoMap.put(score.getScorename(), si);

               if (aw == null) {
                  aw = new AssessmentWrapper(as, usedScoresList);
               }
            }
         }
         if (aw != null)
            scoreMap.put(as.getName(), aw);
      }

      // get visit info for each subject
      Expcomponent criteria = new Expcomponent();
      ExpcomponentDAO visitDAO = new ExpcomponentDAO();
      List<?> visits = visitDAO.find(con, criteria);
      Map<String, List<Expcomponent>> visitMap = new HashMap<String, List<Expcomponent>>();
      for (Object item : visits) {
         Expcomponent visit = (Expcomponent) item;
         List<Expcomponent> visitsPerSubject = null;
         if ((visitsPerSubject = visitMap.get(visit.getSubjectid())) == null) {
            visitsPerSubject = new LinkedList<Expcomponent>();
            visitMap.put(visit.getSubjectid(), visitsPerSubject);
         }
         visitsPerSubject.add(visit);
      }

      Map<String, ColumnInfo> colInfoMap = metaLoader.getColInfoMap();

      // now it is time to prepare score values for upload
      for (Iterator<List<String>> it = dataLoader.getRowDataList().iterator(); it
            .hasNext();) {
         List<String> row = it.next();
         ColumnInfo birnIDCol = colInfoMap.get(birnIDColName);

         String birnID = birnIDCol.getValue(row);
         // skip subjects without a BIRN ID
         if (birnID == null || birnID.length() == 0)
            continue;

         List<Expcomponent> visitsPerSubject = visitMap.get(birnID);
         if (visitsPerSubject == null) {
            System.out.println("*** Skipping subject " + birnID
                  + " since no visits exists!");
            setMessage("*** Skipping subject " + birnID
                  + " since no visits exists!");
            continue;
         }
         Expcomponent scanVisit = getVisit(Constants.VISIT_TYPE_SCAN,
               visitsPerSubject);
         Expcomponent clinVisit = getVisit(Constants.VISIT_TYPE_CLINICAL,
               visitsPerSubject);

         for (Iterator<String> it2 = colInfoMap.keySet().iterator(); it2
               .hasNext();) {
            String colName = it2.next();
            if (colName.equals(this.birnIDColName))
               continue;
            ColumnInfo ci = colInfoMap.get(colName);
            if (ci.getStored() == false)
               continue; // skip non persistent columns
            /** @todo needs to be more generalized */
            Expcomponent theVisit = (ci.getVisitType()
                  .equals(Constants.VISIT_TYPE_SCAN)) ? scanVisit : clinVisit;

            ScoreInfo si = scoreInfoMap.get(ci.getNewName());

            ColumnInfo dateColumn = colInfoMap.get(ci.getDateCol());
            String dateValue = dateColumn.getValue(row);
            java.util.Date theDate = (java.util.Date) dateColumn.getType()
                  .convertToSQL(dateValue, "date");

            // find the clinical visit matching the date of the assessment
            if (ci.getVisitType().equals(Constants.VISIT_TYPE_CLINICAL)) {
               theVisit = getVisit(ci.getVisitType(), visitsPerSubject, theDate);
            }

            ConditionerInfo condInfo = ci.getConditioner();
            String value = ci.getValue(row);
            if (condInfo != null) {
               // condition the value before type conversion
               value = (String) condInfo.conditionValue(value, ci);
            }

            if (value == null) {
               System.out.println("Skipping column " + colName + " with value "
                     + ci.getValue(row));
               setMessage("Skipping column " + colName + " with value "
                     + ci.getValue(row));
               continue;
            }
            Object transformedValue = null;
            // normalized value is null currently
            Object transformedNormValue = null;
            if (ci.hasValueMap()) {
               value = ci.getValueMapValue(value);
            }

            transformedValue = ci.getType()
                  .convertToSQL(value, ci.getSqlType());
            /** @todo check segmentID */
            si.addValue(transformedValue, transformedNormValue, theVisit
                  .getComponentid().intValue(),
                  1, // segmentID
                  theVisit.getNcExperimentUniqueid().intValue(), birnID,
                  theDate, ci.getComments()); // comments
         }

      }
      AssessmentManager am = new AssessmentManager();
      for (Iterator<ScoreInfo> it = scoreInfoMap.values().iterator(); it
            .hasNext();) {
         ScoreInfo si = it.next();
         System.out.println("score -" + si.getAsScore().getScorename());
         setMessage("score -" + si.getAsScore().getScorename());
         for (ScoreInfo.AssessmentValue asv : si.getValues()) {
            System.out.println(asv.toString());
            setMessage(asv.toString());
         }
      }

      for (Iterator<ScoreInfo> it = scoreInfoMap.values().iterator(); it
            .hasNext();) {
         ScoreInfo si = it.next();
         System.out.println("score -" + si.getAsScore().getScorename());
         setMessage("score -" + si.getAsScore().getScorename());
         am.addScoreValues(con, si, owner, modUser, secLabel, isTestMode(),
               this);
      }

      // serialize the scoreInfoMap to be used later to remove what is stored
      if (!testMode && serFilename != null)
         GenUtils.serialize(scoreInfoMap, serFilename);
   }

   @SuppressWarnings("unchecked")
   public void removeScoreValues(Connection con, String valueSerFile)
         throws Exception {
      Map<String, ScoreInfo> scoreInfoMap = (Map<String, ScoreInfo>) GenUtils
            .deserialize(valueSerFile);
      if (scoreInfoMap == null) {
         throw new RuntimeException("Cannot deserialize scoreInfoMap!");
      }

      for (Iterator<ScoreInfo> it = scoreInfoMap.values().iterator(); it
            .hasNext();) {
         ScoreInfo si = it.next();
         System.out.println("score -" + si.getAsScore().getScorename());
         for (Object element : si.getValues()) {
            ScoreInfo.AssessmentValue asv = (ScoreInfo.AssessmentValue) element;
            // remove score value from the database
            // am.removeScoreValue2(con, si, asv);
            System.out.println("removed " + asv.toString());
         }
      }
   }

   @SuppressWarnings("unchecked")
   public void removeVisits(Connection con, String visitSerFile)
         throws Exception {
      Map<String, List<ExperimentInfo>> missingVisitsMap = (Map<String, List<ExperimentInfo>>) GenUtils
            .deserialize(visitSerFile);
      ExperimentManager expMan = new ExperimentManager();

      for (Map.Entry<String, List<ExperimentInfo>> entry : missingVisitsMap
            .entrySet()) {
         String birnID = entry.getKey();
         List<ExperimentInfo> visitList = entry.getValue();
         System.out.println("BirnID=" + birnID);
         for (Iterator<ExperimentInfo> it = visitList.iterator(); it.hasNext();) {
            ExperimentInfo ei = it.next();
            System.out.println("Removing " + ei);

            expMan.removeVisitsForSubject(con, birnID, ei.getExpName(), ei
                  .getResearchGroupName(), ei.getVisitID(), ei.getVisitID(),
                  owner, modUser);
         }
      }
   }

   public void uploadAssessments(Connection con,
         Map<String, AssessmentInfo> asiMap, BigDecimal secLabel)
         throws Exception {
      List<Assessment> asList = null;
      Assessment ac = new Assessment();
      AssessmentDAO dao = new AssessmentDAO();
      asList = dao.find(con, ac);

      AssessmentscoreDAO asDAO = new AssessmentscoreDAO();
      List<Assessmentscore> scoreList = asDAO.find(con, new Assessmentscore());
      // find the missing subset and create them
      Map<String, AssessmentInfo> missingAsMap = new HashMap<String, AssessmentInfo>();
      Map<String, Assessment> asMap = new HashMap<String, Assessment>();
      for (Object element : asList) {
         Assessment a = (Assessment) element;
         asMap.put(a.getName(), a);
      }

      List<Object> missingAsList = new LinkedList<Object>();
      for (Iterator<AssessmentInfo> it = asiMap.values().iterator(); it
            .hasNext();) {
         AssessmentInfo ai = it.next();
         Assessment a = getAssessment(ai.getName(), asList);
         if (a == null) {
            missingAsList.add(ai);
            missingAsMap.put(ai.getName(), ai);
         }
      }

      // group the scores by the assessmentid
      Map<BigDecimal, List<Assessmentscore>> scoresByAsIDMap = new HashMap<BigDecimal, List<Assessmentscore>>();
      for (Object element : scoreList) {
         Assessmentscore as = (Assessmentscore) element;
         List<Assessmentscore> asScoreList = null;
         if ((asScoreList = scoresByAsIDMap.get(as.getAssessmentid())) == null) {
            asScoreList = new LinkedList<Assessmentscore>();
            scoresByAsIDMap.put(as.getAssessmentid(), asScoreList);
         }
         asScoreList.add(as);
      }

      List<Object> missingScoreList = new LinkedList<Object>();
      for (Iterator<AssessmentInfo> it = asiMap.values().iterator(); it
            .hasNext();) {
         AssessmentInfo ai = it.next();
         AssessmentInfo a = missingAsMap.get(ai.getName());
         if (a != null) {
            // the assessment is not in the database. add all of its scores
            // as missing
            missingScoreList.addAll(ai.getScores());
            continue;
         }
         Assessment a2 = asMap.get(ai.getName());
         if (a2 != null) {
            List<Assessmentscore> asScoreList = scoresByAsIDMap.get(a2
                  .getAssessmentid());
            if (asScoreList == null) {
               // assesment has no scores. add all of its scores as missing
               missingScoreList.addAll(ai.getScores());
            } else {
               for (Object element : ai.getScores()) {
                  AssessmentScoreInfo asi = (AssessmentScoreInfo) element;
                  if (!isScoreInDatabase(asi, asScoreList)) {
                     missingScoreList.add(asi);
                  }
               }
            }
         }
      }

      showListContents(missingAsList, "");
      showListContents(missingScoreList, "Scores");

      // now create the missing assessments and scores
      AssessmentManager asMan = new AssessmentManager();
      for (Iterator<Object> it = missingAsList.iterator(); it.hasNext();) {
         AssessmentInfo ai = (AssessmentInfo) it.next();
         if (testMode) {
            System.out.println("[TEST] adding assessment " + ai.getName()
                  + " with description " + ai.getDescription());
            setMessage("[TEST] adding assessment " + ai.getName()
                  + " with description " + ai.getDescription());
         } else {
            asMan.addAssessment(con, ai.getName(), ai.getDescription(), owner,
                  modUser, secLabel);
         }
      }

      // now add the scores also
      for (Iterator<Object> it = missingScoreList.iterator(); it.hasNext();) {
         AssessmentScoreInfo asi = (AssessmentScoreInfo) it.next();

         if (testMode) {
            System.out.println("[TEST] adding assessment score "
                  + asi.toString());
            setMessage("[TEST] adding assessment score " + asi.toString());
         } else {
            asMan.addAssessmentScore2(con, asi.getAssessment().getName(),
                  owner, modUser, secLabel, asi.getName(), asi.getType(),
                  new BigDecimal("0"), null, null, // scoreLevel, parentAsID,
                  // parentScore
                  asi.getSecClass(), asi.getDefValue(), asi.getNullable(), asi
                        .getOntology(), asi.getOntologyConcept());
         }
      }

   }

   public void showListContents(List<Object> list, String msg) {
      if (msg != null) {
         System.out.println(msg);
         setMessage(msg);
      }
      for (Iterator<Object> it = list.iterator(); it.hasNext();) {
         Object o = it.next();
         System.out.println(o);
         setMessage(o.toString());
      }
      System.out.println("----");
      setMessage("----");
   }

   protected static Assessmentscore findScore(String name,
         List<Assessmentscore> scores) {
      for (Object element : scores) {
         Assessmentscore as = (Assessmentscore) element;
         if (as.getScorename().equals(name))
            return as;

      }
      return null;
   }

   static boolean isScoreInDatabase(AssessmentScoreInfo asi,
         List<Assessmentscore> scores) {
      for (Iterator<Assessmentscore> it = scores.iterator(); it.hasNext();) {
         Assessmentscore score = it.next();
         if (score.getScorename().equals(asi.getName()))
            return true;
      }
      return false;
   }

   protected static Assessment getAssessment(String name,
         List<Assessment> asList) {
      // linear search (change if performance issues)
      for (Iterator<Assessment> it = asList.iterator(); it.hasNext();) {
         Assessment a = it.next();
         if (a.getName().equals(name))
            return a;
      }
      return null;
   }

   public static void dumpAssessmentMap(Map<String, AssessmentInfo> asMap,
         Writer w, AssessmentLoader al) throws IOException {
      String eol = System.getProperty("line.separator");
      w.write("AssessmentMap" + eol);
      al.setMessage("AssessmentMap");
      for (Iterator<AssessmentInfo> it = asMap.values().iterator(); it
            .hasNext();) {
         AssessmentInfo ai = it.next();
         w.write(ai.toString());
         al.setMessage(ai.toString());
         w.write(eol);
         w.flush();
      }
      w.write("-----" + eol);
      al.setMessage("-----");
      w.flush();
   }

   public void dumpAssessmentMap(Map<String, AssessmentInfo> asMap)
         throws IOException {
      dumpAssessmentMap(asMap, new OutputStreamWriter(System.out), this);
   }

   public Map<String, AssessmentInfo> loadAssessments(String filename)
         throws JDOMException, IOException {
      logger.info("loading the assessment description file contents.");
      setMessage("loading the assessment description file contents.");

      SAXBuilder builder = null;
      builder = new SAXBuilder(false); // no validation
      Document doc = builder.build(new File(filename));

      Map<String, AssessmentInfo> asMap = new LinkedHashMap<String, AssessmentInfo>();
      Element root = doc.getRootElement();
      Element asRoot = root.getChild("assessments");
      if (asRoot != null) {
         List<?> asList = asRoot.getChildren("assessment");
         for (Iterator<?> it = asList.iterator(); it.hasNext();) {
            Element e = (Element) it.next();
            AssessmentInfo ai = loadAssessment(e);
            asMap.put(ai.getName(), ai);
         }
      }

      return asMap;
   }

   protected AssessmentInfo loadAssessment(Element asElem) {
      AssessmentInfo ai = new AssessmentInfo();
      ai.setName(asElem.getAttributeValue("name"));
      ai.setDescription(asElem.getAttributeValue("description"));
      List<?> scoreElems = asElem.getChildren("as-score");
      for (Iterator<?> it = scoreElems.iterator(); it.hasNext();) {
         Element e = (Element) it.next();
         AssessmentScoreInfo asi = new AssessmentScoreInfo(ai);
         asi.setName(e.getAttributeValue("name"));
         asi.setType(e.getAttributeValue("type"));
         asi.setSecClass(e.getAttributeValue("secClass").toUpperCase());
         asi.setOntology(e.getAttributeValue("ontology"));
         asi.setOntologyConcept(e.getAttributeValue("ontology-concept"));
         String nullable = e.getAttributeValue("nullable").toLowerCase();
         asi.setNullable(nullable.equals("true"));
         String defValue = e.getAttributeValue("defValue");
         if (defValue != null && !defValue.equalsIgnoreCase("null"))
            asi.setDefValue(defValue);

         ai.addScore(asi);
      }

      return ai;
   }

   protected Expcomponent getVisit(String visitType, List<Expcomponent> visits) {
      // linear search O(n)
      for (Object element : visits) {
         Expcomponent visit = (Expcomponent) element;

         if (visit.getVisittype().equalsIgnoreCase(visitType)) {
            return visit;
         }
      }
      return null;
   }

   protected Expcomponent getVisit(String visitType, List<Expcomponent> visits,
         java.util.Date visitDate) {
      // linear search O(n)
      if (visits == null)
         return null;
      Timestamp visitTstamp = new Timestamp(visitDate.getTime());
      for (Object element : visits) {
         Expcomponent visit = (Expcomponent) element;
         if (visit.getVisittype().equalsIgnoreCase(visitType)) {
            if (visit.getTimeStamp().equals(visitTstamp)) {
               return visit;
            }
         }
      }
      return null;
   }

   static class AssessmentWrapper {
      Assessment assessment;
      List<Assessmentscore> scores;

      public AssessmentWrapper(Assessment assessment,
            List<Assessmentscore> scores) {
         this.assessment = assessment;
         this.scores = scores;
      }

      public Assessmentscore getScore(String scoreName) {
         for (Iterator<Assessmentscore> it = scores.iterator(); it.hasNext();) {
            Assessmentscore score = it.next();
            if (score.getScorename().equals(scoreName))
               return score;
         }
         return null;
      }

   }

   static class ExperimentStaticData {
      String visitDescr;
      String segmentDescr;

      public ExperimentStaticData(String visitDescr, String segmentDescr) {
         this.visitDescr = visitDescr;
         this.segmentDescr = segmentDescr;
      }
   }

   public void testVisitUpload(Connection con, String resGroupFilename,
         BigDecimal secLabel, String visitSerFile) throws Exception {
      Map<String, String> resGrpMap = prepareResearchGroupNameMap(resGroupFilename);
      ExperimentStaticData expData = new ExperimentStaticData("", "");
      String expName = "Alzheimer Testbed";
      uploadVisits(con, expName, resGrpMap, expData, secLabel, visitSerFile);
   }

   public void execute(Connection con, int cmd, BigDecimal secLabel)
         throws Exception {
      String asDefFile = props.getProperty("assessment.definition_file");
      String valueSerFile = props.getProperty("assessment.value.serfile");
      String researchGrpFile = props.getProperty("research.group.file");
      String visitSerFile = props.getProperty("visit.serfile");

      loadData();

      Map<String, AssessmentInfo> asMap = loadAssessments(asDefFile);
      dumpAssessmentMap(asMap);

      if ((cmd & UPLOAD_ASSESSMENTS) > 0) {
         uploadAssessments(con, asMap, secLabel);
      }
      if ((cmd & UPLOAD_VISITS) > 0) {
         testVisitUpload(con, researchGrpFile, secLabel, visitSerFile);
      }

      if ((cmd & UPLOAD_AS_SCORE_VALUES) > 0) {
         uploadScoreValues(con, valueSerFile, secLabel);
      }

      if ((cmd & REMOVE_AS_SCORE_VALUES) > 0) {
         removeScoreValues(con, valueSerFile);
      }

      if ((cmd & REMOVE_VISITS) > 0) {
         ExpcomponentDAO dao = new ExpcomponentDAO();
         Expcomponent criteria = new Expcomponent();
         List<Expcomponent> allVisits = dao.find(con, criteria);
         /** @todo generalize for other experiments */
         String expName = "Alzheimer Testbed";
         Map<String, String> resGrpMap = prepareResearchGroupNameMap(researchGrpFile);

         // remove all visits with visit ids between 3 and 5
         UploadHelper.removeVisits(con, allVisits, 3, 5, expName, resGrpMap,
               this.owner, this.modUser);
      }
   }

   public void addPropertyChangeListener(PropertyChangeListener pcl) {
      pcs.addPropertyChangeListener(pcl);
   }

   public void removePropertyChangeListener(PropertyChangeListener pcl) {
      pcs.removePropertyChangeListener(pcl);
   }

   public static void usage() {
      System.err
            .println("java clinical.upload.AssessmentLoader <dbURL> <usr> <pwd>");
      System.err
            .println("\twhere dbURL has the following syntax jdbc:oracle:thin:@<host>:<port>:<sid>");
      System.exit(1);
   }

   public static void main(String[] args) {
      AssessmentLoader al = null;
      Properties props = null;
      NamedUserPool dbPool = null;
      if (args.length != 3)
         usage();

      String dbURL = args[0];
      String user = args[1];
      String pwd = args[2];
      try {
         props = GenUtils.loadProperties("upload_wiscr.properties");

         dbPool = NamedUserPool.getInstance("oracle.jdbc.driver.OracleDriver",
               dbURL);

         al = new AssessmentLoader(props);
         al.setBirnIDColName("JERNBLIND");
         Connection con = dbPool.getConnection(user, pwd);

         // just test upload
         al.setTestMode(false);
         // using oracle label security label tag for public:loc,brn
         al.execute(con, AssessmentLoader.UPLOAD_AS_SCORE_VALUES,
               new BigDecimal("11000"));

      } catch (Exception x) {
         x.printStackTrace();
      } finally {
         if (dbPool != null)
            dbPool.shutdown();
      }
   }
}