package clinical.utils;

import java.io.File;
import java.io.IOException;
import java.text.SimpleDateFormat;

import clinical.web.vo.HeaderFieldType;

import com.pixelmed.dicom.Attribute;
import com.pixelmed.dicom.AttributeList;
import com.pixelmed.dicom.AttributeTag;
import com.pixelmed.dicom.DicomException;
import com.pixelmed.dicom.DicomFileUtilities;
import com.pixelmed.dicom.TagFromName;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */

public class DICOMHeaderReader {
	String dicomSeriesDir;
	AttributeList atList = new AttributeList();
	public final static SimpleDateFormat tsFormat = new SimpleDateFormat(
	"yyyyMMdd hhmmss");

	public DICOMHeaderReader(String dicomSeriesDir) throws IOException,
			DicomException {
		this.dicomSeriesDir = dicomSeriesDir;
		File[] files = new File(dicomSeriesDir).listFiles();
		for (File f : files) {
			if (DicomFileUtilities.isDicomOrAcrNemaFile(f)) {
				atList.read(f.getAbsolutePath());
				break;
			}
		}
	}

	public HeaderFields extractFields() {
		HeaderFields hf = new HeaderFields();
		// extract series description,
		String seriesDescr = Attribute.getSingleStringValueOrNull(atList,
				TagFromName.SeriesDescription);
		Field<String> field = new Field<String>("SeriesDescription",
				seriesDescr);
		hf.addStringField(field, new HeaderFieldType("SeriesDescription",
				HeaderFieldType.STRING));

		String seriesTime = Attribute.getSingleStringValueOrNull(atList,
				TagFromName.SeriesTime);
		String seriesDate = Attribute.getSingleStringValueOrNull(atList,
				TagFromName.SeriesDate);

		if (seriesDate != null && seriesTime != null) {
			String timestamp = seriesDate + " " + seriesTime;
			field = new Field<String>("timestamp", timestamp);
			hf.addStringField(field, new HeaderFieldType("timestamp",
					HeaderFieldType.STRING));
		}
		// for DTI
		
		Float value = getFloatFieldValue(atList, 25, 4283);
		Field<Float> numberField = new Field<Float>("0019,10bb", value);
		hf.addFloatField(numberField, new HeaderFieldType("0019,10bb", HeaderFieldType.FLOAT));
		
		value = getFloatFieldValue(atList, 25, 4284);
		numberField = new Field<Float>("0019,10bc", value);
		hf.addFloatField(numberField, new HeaderFieldType("0019,10bc", HeaderFieldType.FLOAT));
		
		value = getFloatFieldValue(atList, 25, 4285);
		numberField = new Field<Float>("0019,10bd",value);
		hf.addFloatField(numberField, new HeaderFieldType("0019,10bd", HeaderFieldType.FLOAT));
		
		return hf;
	}
	
	
	public static boolean isDTI(HeaderFields hf) {
		Field<Float> dtiField1 = hf.getFloatField("0019,10bb");
		Field<Float> dtiField2 = hf.getFloatField("0019,10bc");
		Field<Float> dtiField3 = hf.getFloatField("0019,10bd");
		
		if (dtiField1.getValue() == null || dtiField1.getValue().floatValue() == 0) {
			return false;
		}
		if (dtiField2.getValue() == null || dtiField2.getValue().floatValue() == 0) {
			return false;
		}
		if (dtiField3.getValue() == null || dtiField3.getValue().floatValue() == 0) {
			return false;
		}
		return true;
	}
	
	public static Float getFloatFieldValue(AttributeList atList, int group, int element) {
		AttributeTag at = new AttributeTag(group, element); // (0019,10bb)
		
		String valueStr = Attribute.getSingleStringValueOrNull(atList, at);
		if (valueStr == null) {
			return null;
		} else {
			return new Float( GenUtils.toDouble(valueStr, 0));
		}
	}
	
	public static String getScannerInfo(File dcmFile) {
		try {
			AttributeList atList = new AttributeList();
			atList.read(dcmFile.getAbsolutePath());
			StringBuilder sb = new StringBuilder();
			String make = Attribute.getSingleStringValueOrNull(atList,
					TagFromName.Manufacturer);
			sb.append(make).append(' ');
			String model = Attribute.getSingleStringValueOrNull(atList,
					TagFromName.ManufacturerModelName);
			sb.append(model);
			String institutionName = Attribute.getSingleStringValueOrNull(
					atList, TagFromName.InstitutionName);
			if (institutionName != null) {
				sb.append(' ').append(institutionName);
			}
			return sb.toString();
		} catch (Exception x) {
			// no op
		}
		return null;
	}
}
