package clinical.utils;

import java.sql.Connection;
import java.sql.DriverManager;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.log4j.Logger;

/**
 * A named user database connection pool for maintaining named user identity but
 * providing connection pooling advantages.
 * <p>
 * Since creating a database connection is time consuming and using one or more
 * database connections for each web user is usually economically not feasable,
 * a pool of preinitialized database connections can be shared between users for
 * getting the best bang for the buck from a database connection. However, in a
 * connection pool a database connection looses its identity.
 * 
 * @version $Id: NamedUserPool.java 692 2012-10-04 01:23:42Z bozyurt $
 * @author I. Burak Ozyurt
 */

public class NamedUserPool {
	protected Map<String, NamedUser> namedUserMap = Collections
			.synchronizedMap(new HashMap<String, NamedUser>());
	protected String dbURL;
	/** test query used to check the health of the database connection */
	protected String testQuery = "SELECT 1 FROM DUAL";
	/**
	 * maximum number of connections after which the total connection pool cannot
	 * grow (default 30)
	 */
	protected int maxConnections;
	/** the initial number of connections in each sub-connection pool (default 2) */
	protected int subPoolSize = 2;
	/**
	 * the maximum idle time for a database connection before it is closed and
	 * removed from the connection pool. (default 30 minutes)
	 */
	protected long maxIdleTime = 30 * 60 * 1000; // 4*3600 * 1000;
	/** the interval for periodic pool cleanup checks (default 5 minutes) */
	protected long checkInterval = 300000; // 600000;
	protected boolean showStats = false;
	/** Pool idle connection cleanup thread */
	protected Thread cleanupThread;
	protected static Map<String, NamedUserPool> instanceMap = new LinkedHashMap<String, NamedUserPool>(
			7);
	protected static NamedUserPool instance = null;
	/** maximum number of database connections (30) */
	protected final static int MAX_CONNECTIONS = 30;
	protected static Logger log = Logger.getLogger(NamedUserPool.class);
	/**
	 * @todo make it more general than this (maybe read the test query, driver
	 *       class etc from a file?
	 */
	protected static Map<String, String> testQueryMap = new HashMap<String, String>();

	static {
		testQueryMap.put("oracle.jdbc.driver.OracleDriver", "SELECT 1 FROM DUAL");
		testQueryMap.put("org.postgresql.Driver", "SELECT 1");
	}

	/**
	 * Constructor for named user database connection pool for maintaining named
	 * user identity but providing connection pooling advantages. This also
	 * starts the cleanup process thread.
	 * 
	 * 
	 * @param driverClass
	 * @param dbURL
	 *           database URL for the JDBC driver
	 * @param maxConnections
	 *           maximum number of connections after which the total connection
	 *           pool cannot grow
	 * @param testQuery
	 *           test query used to check the health of the database connection
	 * @throws java.lang.ClassNotFoundException
	 * @throws java.lang.IllegalAccessException
	 * @throws java.lang.InstantiationException
	 */
	protected NamedUserPool(String driverClass, String dbURL,
			int maxConnections, String testQuery) throws ClassNotFoundException,
			IllegalAccessException, InstantiationException {
		Class.forName(driverClass).newInstance();
		this.dbURL = dbURL;
		this.maxConnections = maxConnections;

		this.testQuery = testQuery;

		// start the cleanup thread
		cleanupThread = new Thread(new PoolCleanup(this, maxIdleTime,
				checkInterval));
		cleanupThread.setDaemon(true);
		cleanupThread.setPriority(Thread.NORM_PRIORITY - 1);
		cleanupThread.start();
		if (log.isDebugEnabled())
			log.debug("registered driver " + driverClass
					+ " and started cleanupThread.");
	}

	/**
	 * Constructor for named user database connection pool for maintaining named
	 * user identity but providing connection pooling advantages. This also
	 * starts the cleanup process thread.
	 * 
	 * @param driverClass
	 *           JDBC Driver class name
	 * @param dbURL
	 *           database URL for the JDBC driver
	 * @throws java.lang.ClassNotFoundException
	 * @throws java.lang.IllegalAccessException
	 * @throws java.lang.InstantiationException
	 */
	protected NamedUserPool(String driverClass, String dbURL)
			throws ClassNotFoundException, IllegalAccessException,
			InstantiationException {
		this(driverClass, dbURL, MAX_CONNECTIONS, NamedUserPool
				.selectTestQuery(driverClass));
	}

	protected static String selectTestQuery(String driverClass) {
		String aTestQuery = testQueryMap.get(driverClass);
		if (aTestQuery == null) {
			throw new RuntimeException("JDBC Driver is not supported!:"
					+ driverClass);
		}
		return aTestQuery;
	}

	/**
	 * Returns the NamedUserPool singleton, creating it if necessary.
	 * 
	 * @param driverClass
	 *           JDBC Driver class name
	 * @param dbURL
	 *           database URL for the JDBC driver
	 * @return the NamedUserPool singleton
	 * 
	 * @throws java.lang.ClassNotFoundException
	 * @throws java.lang.IllegalAccessException
	 * @throws java.lang.InstantiationException
	 */
	public synchronized static NamedUserPool getInstance(String driverClass,
			String dbURL) throws ClassNotFoundException, IllegalAccessException,
			InstantiationException {
		NamedUserPool nup = instanceMap.get(dbURL);

		if (nup == null) {
			log.debug("*** creating a new NamedUserPool object");

			nup = new NamedUserPool(driverClass, dbURL);
			instanceMap.put(dbURL, nup);
		}
		return nup;
	}

	/**
	 * Returns the NamedUserPool singleton.
	 * 
	 * @return the NamedUserPool singleton
	 * @throws NamedUserPoolException
	 *            if the singleton is not created before
	 */
	public synchronized static NamedUserPool getInstance(String dbURL)
			throws NamedUserPoolException {
		NamedUserPool nup = instanceMap.get(dbURL);
		if (nup == null)
			throw new NamedUserPoolException("NamedUserPool is not initialized");
		return nup;
	}

	/**
	 * Returns the total number of connections in all of the named user subpools.
	 * 
	 * @return the total number of connections in all of the named user subpools
	 */
	public synchronized int getTotPoolSize() {
		int count = 0;
		for (Iterator<NamedUser> it = namedUserMap.values().iterator(); it
				.hasNext();) {
			NamedUser nu = it.next();
			count += nu.getPoolSize();
		}
		return count;
	}

	/**
	 * Given a named user username and a password returns a pooled database
	 * connection. If there is no subpool for the named user and if a password is
	 * specified, it tries to create the corresponding named user subpool. If the
	 * total pool size has reached the maximum pool size, it first tries to
	 * remove any idle connections and named user subpools. If this does not
	 * succeeds throws a <code>MaxConnectionsExceededException</code>. Otherwise
	 * it returns a pooled connection.
	 * 
	 * @param user
	 *           named database user's username
	 * @param pwd
	 *           named database user's password
	 * @return a pooled database connection for the named user
	 * 
	 * @throws MaxConnectionsExceededException
	 * @throws NamedUserPoolException
	 * @see NamedUser#getConnection
	 */

	public synchronized Connection getConnection(String user)
			throws SQLException, MaxConnectionsExceededException,
			NamedUserPoolException {
		return getConnection(user, null);
	}

	/**
	 * Given a named user username and a password returns a pooled database
	 * connection. If there is no subpool for the named user and if a password is
	 * specified, it tries to create the corresponding named user subpool. If the
	 * total pool size has reached the maximum pool size, it first tries to
	 * remove any idle connections and named user subpools. If this does not
	 * succeeds throws a <code>MaxConnectionsExceededException</code>. Otherwise
	 * it returns a pooled connection.
	 * 
	 * @param user
	 *           named database user's username
	 * @param pwd
	 *           named database user's password
	 * @return a pooled database connection for the named user
	 * 
	 * @throws MaxConnectionsExceededException
	 * @throws NamedUserPoolException
	 * @see NamedUser#getConnection
	 */
	public synchronized Connection getConnection(String user, String pwd)
			throws MaxConnectionsExceededException, NamedUserPoolException {

		NamedUser nu = namedUserMap.get(user);
		if (nu == null) {
			if (pwd == null) {
				throw new NamedUserPoolException(
						"Cannot resize subpool without password!");
			}
			int totPoolSize = getTotPoolSize();
			if (totPoolSize >= maxConnections) {
				if (log.isDebugEnabled())
					log.debug("number of database connections pooled has reached:"
							+ maxConnections);
				// first try to remove any idle connections
				for (Iterator<NamedUser> it = namedUserMap.values().iterator(); it
						.hasNext();) {
					NamedUser n = it.next();
					n.cleanupIdleConnections(-1);
					if (n.getPoolSize() == 0) {
						it.remove(); // remove the named user subpool
						if (log.isDebugEnabled())
							log.debug("removed subpool for " + n.user);
					}
				}

				// if still max number connections is exceeded, bail out!
				if (getTotPoolSize() <= maxConnections)
					throw new MaxConnectionsExceededException();
			}

			namedUserMap.put(user, nu = new NamedUser(user, pwd, dbURL, testQuery,
					subPoolSize));
			if (log.isDebugEnabled())
				log.debug("added a new named user <" + user + "> subpool!");
		}
		return nu.getConnection(pwd);
	}

	/**
	 * Given a named user username and a database connection, it returns the
	 * connection to the corresponding named user's subpool.
	 * 
	 * @param user
	 *           named database user's username
	 * @param con
	 *           a database connection
	 * @throws NamedUserPoolException
	 * @see NamedUser#releaseConnection
	 */
	public synchronized void releaseConnection(String user, Connection con)
			throws NamedUserPoolException {
		if (con == null)
			return;
		NamedUser nu = namedUserMap.get(user);
		if (nu != null) {

			nu.releaseConnection(con);
			if (log.isDebugEnabled())
				log.debug("released connection " + con);
		}
	}

	/**
	 * Shuts down the connection pool by closing all the database connections
	 * belonging to each named user's subpool.
	 * 
	 * @see NamedUser#shutdown
	 */
	public synchronized void shutdown() {
		for (Iterator<NamedUser> it = namedUserMap.values().iterator(); it
				.hasNext();) {
			NamedUser nu = it.next();
			nu.shutdown();
		}
		if (log.isDebugEnabled())
			log.debug("Shutdown all subpools.");
		instance = null;
	}

	public synchronized void setShowStats(boolean newShowStats) {
		this.showStats = newShowStats;
		for (Iterator<NamedUser> it = namedUserMap.values().iterator(); it
				.hasNext();) {
			NamedUser nu = it.next();
			nu.setDumpStats(this.showStats);
		}
	}

	public synchronized boolean getShowStats() {
		return this.showStats;
	}

	public static class NamedUser {
		private String user;
		private String dbURL;
		private boolean dumpStats = false;

		private int minPoolSize = 1;
		private int maxPoolSize = 25;

		private String testQuery;
		private Map<Connection, ConnectionInfo> pool = Collections
				.synchronizedMap(new HashMap<Connection, ConnectionInfo>(7));
		private List<ConnectionInfo> availList = new LinkedList<ConnectionInfo>();

		/**
		 * Constructor for NamedUser objects which maintains a database connection
		 * pool for the named user specified by the name and password
		 * 
		 * @param user
		 *           named database user's username
		 * @param pwd
		 *           named database user's password
		 * @param dbURL
		 *           database URL for the JDBC driver
		 * @param testQuery
		 *           test query used to check the health of the database
		 *           connection
		 * @param poolSize
		 *           the size of the connection pool for this named user
		 * @throws NamedUserPoolException
		 */
		public NamedUser(String user, String pwd, String dbURL, String testQuery,
				int poolSize) throws NamedUserPoolException {
			this.user = user;
			this.testQuery = testQuery;
			this.dbURL = dbURL;

			DriverManager.setLoginTimeout(5);
			System.out.println("DB connection timeout (secs):"
					+ DriverManager.getLoginTimeout());
			for (int i = 0; i < poolSize; ++i) {
				try {
					System.out.println("trying to get a connection:" + dbURL);
					Connection con = DriverManager.getConnection(dbURL, user, pwd);
					System.out.println("got connection:" + dbURL);
					
					ConnectionInfo ci = new ConnectionInfo(con, false);
					pool.put(con, ci);
					availList.add(ci);
				} catch (SQLException se) {
					log.error("NamedUser", se);
				}
				if (i > 1 && pool.isEmpty()) {
					break;
				}
			}
			if (pool.size() == 0)
				throw new NamedUserPoolException(
						"Error in named user subpool creation!");
			if (log.isDebugEnabled())
				log.debug("Created a subpool for user " + user + " of size "
						+ pool.size());
		}

		public int getPoolSize() {
			return pool.size();
		}

		public synchronized Connection getConnection()
				throws NamedUserPoolException {
			return getConnection(null);
		}

		public void setDumpStats(boolean newDumpStats) {
			this.dumpStats = newDumpStats;
		}

		public boolean getDumpStats() {
			return this.dumpStats;
		}

		/**
		 * Returns a database connection from the named user pool if one is
		 * available, otherwise it grows the connection pool and returns the
		 * connection. The maximum connection pool size is currently set to 25 per
		 * named user. The connection is tested for staleness before returning it
		 * back to the user.
		 * 
		 * @param pwd
		 *           named database user's password
		 * @return a pooled connection
		 * @throws NamedUserPoolException
		 */
		public synchronized Connection getConnection(String pwd)
				throws NamedUserPoolException {
			// check for an available connection
			if (availList.isEmpty()) {
				if (pwd != null && pool.size() < minPoolSize) {
					addConnections(this.user, pwd, this.dbURL, minPoolSize
							- pool.size());
					if (log.isDebugEnabled())
						log.debug("Subpool " + user
								+ " was below minPoolSize. New #pooled connections:"
								+ pool.size());
				} else if (pwd != null && pool.size() < maxPoolSize) {
					// grow the connection pool 25% upto the max pool size
					int numNewConnections = Math.min(Math.max(
							(int) (pool.size() * .25), 1), maxPoolSize - pool.size());
					addConnections(this.user, pwd, this.dbURL, numNewConnections);
					if (log.isDebugEnabled())
						log.debug("Subpool " + user
								+ " was not large enough. New #pooled connections:"
								+ pool.size());
				} else if (pwd == null) {
					throw new NamedUserPoolException("Cannot grow subpool for user "
							+ user + " without proper password!");
				} else {
					throw new NamedUserPoolException("Cannot grow subpool for user "
							+ user + " beyond the maximum subpool size :"
							+ pool.size());
				}
			}
			ConnectionInfo ci = ((LinkedList<ConnectionInfo>) availList)
					.removeFirst();

			try {
				ci.setInUse(true);
				ci.setLastAccessed(System.currentTimeMillis());
				ci.getConnection().setAutoCommit(true);
				testConnection(ci.getConnection());

				// log.info("Returning connection <"+ ci.getConnection()+ "," +
				// this.user+">");
				// logConnectionRequester();

				return ci.getConnection();
			} catch (SQLException se) {
				// log.debug("", se);
				pool.remove(ci.getConnection());
				log.debug("Connection was reset by the database server");
				return getConnection(pwd);
			}
		}

		@SuppressWarnings("unused")
		private void logConnectionRequester() {
			log.info("user=" + this.user + " time="
					+ DateTimeUtils.formatDate(System.currentTimeMillis()));
			try {
				throw new Exception();
			} catch (Exception x) {
				StackTraceElement[] stElems = GenUtils.prepareStackTrace(x,
						"clinical");
				for (int i = 0; i < stElems.length; i++) {
					log.info(stElems[i].toString());
				}
			}
			log.info("--- end of logConnectionRequester");
		}

		/**
		 * Returns a connection back to the connection pool to be used by other
		 * clients. Checks if the connection is still OK by issuing a test query.
		 * If the connection is bad, removes it from the connection pool.
		 * 
		 * @param con
		 *           the database connection to be returned back to the connection
		 *           pool.
		 * @throws NamedUserPoolException
		 */
		public synchronized void releaseConnection(Connection con)
				throws NamedUserPoolException {
			if (con == null)
				throw new NamedUserPoolException("Connection passed is null!");

			ConnectionInfo ci = pool.get(con);

			if (ci == null) {
				// occurs in some postgres environments
				log.error("connection passed was not in the pool: con=" + con);
				return;
			}

			// check if the connection is still OK
			Statement st = null;
			try {
				Connection c = ci.getConnection();
				if (c == null)
					throw new SQLException();
				st = c.createStatement();
				st.executeQuery(testQuery);
				// make it available for reuse
				ci.setInUse(false);
				availList.add(ci);
				ci.setLastAccessed(System.currentTimeMillis());
				if (log.isDebugEnabled()) {
					log.debug("Connection released back to subpool <" + user + ","
							+ ci.getConnection() + ">.");
				}

			} catch (SQLException se) {
				// the connection is bad - so remove it from the pool
				pool.remove(ci.getConnection());
				if (log.isDebugEnabled())
					log.debug("Bad connection (removed) :" + ci.getConnection());
				log.info("Bad connection (removed) :" + ci.toString());
				// logConnectionRequester();
			} finally {
				if (st != null)
					try {
						st.close();
					} catch (Exception x) {}
			}
		}

		private void testConnection(Connection con) throws SQLException {
			if (con == null) {
				throw new SQLException();
			}
			Statement st = null;
			try {
				st = con.createStatement();
				st.executeQuery(testQuery);
			} finally {
				if (st != null) {
					try {
						st.close();
					} catch (Exception x) {}
				}
			}
		}

		/**
		 * Given a database user name, password and the JDBC database URL, tries
		 * to start <code>count</code> number of connections to the database and
		 * pool (cache) them.
		 * 
		 * @param user
		 *           named database user's username
		 * @param pwd
		 *           named database user's password
		 * @param dbURL
		 *           database URL for the JDBC driver
		 * @param count
		 *           the number of database connections to add to the connection
		 *           pool of this named user
		 * 
		 * @throws NamedUserPoolException
		 * @see #addConnection
		 */
		protected synchronized void addConnections(String user, String pwd,
				String dbURL, int count) throws NamedUserPoolException {
			for (int i = 0; i < count; i++) {
				addConnection(user, pwd, dbURL);
			}
		}

		/**
		 * Given a database user name, password and the JDBC database URL, tries
		 * to start a new connection to the database and cache it in the
		 * connection pool. If not successful in the first time, waits a second
		 * and tries it again two more times. If unsuccessful, throws
		 * NamedUserPoolException.
		 * 
		 * @param user
		 *           named database user's username
		 * @param pwd
		 *           named database user's password
		 * @param dbURL
		 *           database URL for the JDBC driver
		 * 
		 * @throws NamedUserPoolException
		 *            if can not pool the connection after 3 trials
		 */
		protected void addConnection(String user, String pwd, String dbURL)
				throws NamedUserPoolException {
			boolean success = false;
			String errMsg = null;
			int count = 0;
			while (!success && count < 1) {
				long start = 0;
				try {
					DriverManager.setLoginTimeout(5);
					System.out.println("connection timeout:"
							+ DriverManager.getLoginTimeout());
					start = System.currentTimeMillis();
					Connection con = DriverManager.getConnection(dbURL, user, pwd);
					System.out.println("Elapsed time:"
							+ (System.currentTimeMillis() - start));
					ConnectionInfo ci = new ConnectionInfo(con, false);
					pool.put(con, ci);
					availList.add(ci);
					success = true;
					break;
				} catch (SQLException se) {
					System.out.println("Elapsed time:"
							+ (System.currentTimeMillis() - start));
					errMsg = se.getMessage();
				}
				++count;
				if (count >= 1)
					break;
				// sleep a second and try again
				try {
					Thread.sleep(1000);
				} catch (InterruptedException ie) {}
			}
			if (!success)
				throw new NamedUserPoolException(
						"Cannot create new database connection:" + errMsg);
		}

		public synchronized void dumpPoolStatus() {
			log.info("pool status at "
					+ DateTimeUtils.formatDate(System.currentTimeMillis()));
			for (Iterator<ConnectionInfo> it = pool.values().iterator(); it
					.hasNext();) {
				ConnectionInfo ci = it.next();
				log.info("connection " + ci.getConnection().toString() + " inUse="
						+ ci.inUse + " last accessed="
						+ DateTimeUtils.formatDate(ci.lastAccessed));
			}
			log.info("---- end of dump ---");
		}

		/**
		 * Checks for idle connections in the pool and closes the ones which are
		 * more idle than maxIdleTime.
		 * 
		 * @param maxIdleTime
		 *           the maximum allowed idle time for a pooled connection before
		 *           it is closed.
		 * 
		 */
		protected synchronized void cleanupIdleConnections(long maxIdleTime) {
			if (log.isDebugEnabled())
				log.debug("cleaning up idle database connections");
			for (Iterator<ConnectionInfo> it = pool.values().iterator(); it
					.hasNext();) {
				ConnectionInfo ci = it.next();
				if (!ci.inUse
						&& (maxIdleTime > 0 && (System.currentTimeMillis() - ci.lastAccessed) >= maxIdleTime)) {
					try {
						ci.getConnection().close();
					} catch (Exception x) {}
					it.remove();
					log.info("removing idle DB connection " + ci);
				}
			}
			if (dumpStats) {
				dumpPoolStatus();
			}
		}

		/**
		 * Closes all the database connections belonging to this particular named
		 * user's pool
		 */
		public synchronized void shutdown() {
			for (Iterator<ConnectionInfo> it = pool.values().iterator(); it
					.hasNext();) {
				ConnectionInfo ci = it.next();
				try {
					ci.getConnection().close();
				} catch (Exception x) {}
			}
		}
	}

	public static class ConnectionInfo {
		private boolean inUse;
		private long lastAccessed;
		private Connection con;

		public ConnectionInfo(Connection con, boolean inUse) {
			this.con = con;
			this.inUse = inUse;
		}

		public ConnectionInfo(Connection con) {
			this(con, true);
		}

		public Connection getConnection() {
			return con;
		}

		void setInUse(boolean value) {
			inUse = value;
		}

		void setLastAccessed(long lastAccessed) {
			this.lastAccessed = lastAccessed;
		}

		public String toString() {
			StringBuffer sb = new StringBuffer(128);
			sb.append("ConnectionInfo::[");
			sb.append("con=").append(con).append(" lastAccessed=").append(
					lastAccessed).append(" inUse=");
			sb.append(inUse);
			sb.append(']');
			return sb.toString();
		}

	}

	public static class PoolCleanup implements Runnable {
		protected NamedUserPool pool;
		protected long maxIdleTime;
		protected long checkInterval;

		public PoolCleanup(NamedUserPool pool, long maxIdleTime,
				long checkInterval) {
			this.maxIdleTime = maxIdleTime;
			this.checkInterval = checkInterval;
			this.pool = pool;
		}

		public void run() {

			while (true) {
				synchronized (this) {
					try {
						wait(checkInterval);
					} catch (InterruptedException ie) {}
				}
				// clean all idle connections which are older than max idle time
				synchronized (pool) {
					if (log.isDebugEnabled())
						log.debug("Doing scheduled named user pool cleanup!");
					for (Iterator<NamedUser> it = pool.namedUserMap.values()
							.iterator(); it.hasNext();) {
						NamedUser nu = it.next();
						nu.cleanupIdleConnections(maxIdleTime);
					}
				}
			}

		}
	}
}
