package clinical.utils;

import java.io.File;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: PFileHeaderUtils.java 757 2012-12-12 01:12:45Z bozyurt $
 */
public class PFileHeaderUtils {
	Map<String, String> headerPOIMap = new HashMap<String, String>();
	String pfilePath;
	int geHeaderVersion = -1;
	public final static SimpleDateFormat tsFormat = new SimpleDateFormat(
			"MM/dd/yyyy hh:mm");
	Map<String, Field<Integer>> integerFieldMap = new HashMap<String, Field<Integer>>(
			37);
	Map<String, Field<String>> stringFieldMap = new HashMap<String, Field<String>>(
			37);

	final static int tagBITS = 4;
	final static int qBITS = 3;
	final static int nprepBITS = 3;
	final static int caslBITS = 2;
	final static int bgsnBITS = 3;
	final static int tagaxisBITS = 2;
	final static int vtiBITS = 2;
	final static int nlBITS = 6;

	final static int tagPOSN = 0;
	final static int qPOSN = (tagPOSN + tagBITS);
	final static int nprepPOSN = (qPOSN + qBITS);
	final static int caslPOSN = (nprepPOSN + nprepBITS);
	final static int  nlPOSN = 0;

	public final static int GE_11x = 1;
	public final static int GE_12x = 2;
	public final static int GE_14x = 3;
	public final static int GE_20x = 10;
	public final static int GE_22x = 11;
	
	public PFileHeaderUtils(String pfilePath) {
		super();
		this.pfilePath = pfilePath;
	}

	
	protected void getCBFBIRNUserParams() throws Exception {
		String geHeaderExec = "gr14x_header_dump";
		switch (this.geHeaderVersion) {
		case GE_11x:
			geHeaderExec = "gr11x_header_dump";
			break;
		case GE_12x:
			geHeaderExec = "gr12x_header_dump";
			break;
		case GE_14x:
			geHeaderExec = "gr14x_header_dump";
			break;
		case GE_20x:
			// geHeaderExec = "gr20_header_dump";
			geHeaderExec = "gr22_header_dump";
			break;
		}
		Executor executor = new Executor(geHeaderExec, false);
		executor.execute(pfilePath + " x");
		String output = executor.getOutput();
		String[] lines = output.split("\n");
		for(String line : lines) {
			if (line.startsWith("tag:")) {
				String fieldValue = line.substring(line.indexOf(':') + 1).trim();
				int value = getAsInt(fieldValue);
				Field<Integer> tagField = new Field<Integer>("tag",
						value);
				integerFieldMap.put("tag", tagField);
				System.out.println("tag=" + value);
			} else if (line.startsWith("nprep:")) {
				String fieldValue = line.substring(line.indexOf(':') + 1).trim();
				int value = getAsInt(fieldValue);
				Field<Integer> nprepField = new Field<Integer>("nprep",
						value);
				integerFieldMap.put("nprep", nprepField);
				System.out.println("nprep=" + value);

			} else if (line.startsWith("reps:")) {
				String fieldValue = line.substring(line.indexOf(':') + 1).trim();
				int value = getAsInt(fieldValue);
				Field<Integer> repsField = new Field<Integer>("reps", value);
				integerFieldMap.put("reps", repsField);
				System.out.println("reps=" + value);
			}  else if (line.startsWith("dda:")) {
				String fieldValue = line.substring(line.indexOf(':') + 1).trim();
				int value = getAsInt(fieldValue);
				Field<Integer> ddaField = new Field<Integer>("dda", value);
				integerFieldMap.put("dda", ddaField);
				System.out.println("dda:" + value);
			} else if (line.startsWith("nIntlv:")) {
				String fieldValue = line.substring(line.indexOf(':') + 1).trim();
				int value = getAsInt(fieldValue);
				Field<Integer> nIntlvField = new Field<Integer>("nIntlv",
						value);
				integerFieldMap.put("nIntlv", nIntlvField);
				System.out.println("nIntlv=" + value);				
			}	else if (line.startsWith("speprev")) {
				String fieldValue = line.substring(line.indexOf(':') + 1).trim();
				Field<String> speprevField = new Field<String>("speprev", fieldValue);
				stringFieldMap.put("speprev", speprevField);
				System.out.println("speprev:" + fieldValue);
			} else if (line.startsWith("mpphase")){
				String fieldValue = line.substring(line.indexOf(':') + 1).trim();
				int value = getAsInt(fieldValue);
				Field<Integer> mpphaseField = new Field<Integer>("mpphase", value);
				integerFieldMap.put("mpphase", mpphaseField);
				System.out.println("mpphase:" + value);
			} else if (line.startsWith("mppcasl")){
				String fieldValue = line.substring(line.indexOf(':') + 1).trim();
				int value = getAsInt(fieldValue);
				Field<Integer> mppcaslField = new Field<Integer>("mppcasl", value);
				integerFieldMap.put("mppcasl", mppcaslField);
				System.out.println("mppcasl:" + value);
			} else if (line.startsWith("pcaslopt")) {
				extractIntField(line,"pcaslopt");
			} else if (line.startsWith("docsf")) {
				extractIntField(line,"docsf");
			} else if (line.startsWith("optpcasl")) {
				extractIntField(line, "optpcasl");
			}
		}
	}


	private void extractIntField(String line, String tagName) {
		String fieldValue = line.substring(line.indexOf(':') + 1).trim();
		int value = getAsInt(fieldValue);
		Field<Integer> field = new Field<Integer>(tagName, value);
		integerFieldMap.put(tagName, field);
		System.out.println(tagName + ":" + value);
	}
	
	
	public void readHeader() throws Exception {
		Executor executor = new Executor("rdgehdr", false);
		executor.execute(pfilePath);
		String output = executor.getOutput();
		int idx = output.indexOf('\n');
		if (idx != -1) {
			String line = output.substring(0, idx);
			Pattern p = Pattern.compile("\\/(\\d\\dx)\\/");
			Matcher matcher = p.matcher(line);
			if (matcher.find()) {
				String ver = matcher.group(1);
				if (ver.equals("20x")) {
					this.geHeaderVersion = GE_20x;
				} else if (ver.equals("14x")) {
					this.geHeaderVersion = GE_14x;
				} else if (ver.equals("12x")) {
					this.geHeaderVersion = GE_12x;
				} else if (ver.equals("11x")) {
					this.geHeaderVersion = GE_11x;
				} else {
					this.geHeaderVersion = GE_14x;
				}
			}
			System.out.println(line);
		}
		String[] fields = output.split("\\.\\.\\.");
		for (String field : fields) {
			if (field.startsWith("rhuser25")) {
				System.out.println(field.trim());
				String fieldValue = extractField(field);
				int value = getAsInt(fieldValue);
				Field<Integer> tagField = new Field<Integer>("tag",
						extractTagValue(value));
				integerFieldMap.put("tag", tagField);
				Field<Integer> nprepField = new Field<Integer>("nprep",
						extractNprepValue(value));
				integerFieldMap.put("nprep", nprepField);
				System.out.println("tag=" + extractTagValue(value));
				System.out.println("nprep=" + extractNprepValue(value));
			} else if (field.startsWith("Scan time")) {
				String fieldValue = extractField(field);
				Field<String> tsField = new Field<String>("scanTime",
						fieldValue);
				stringFieldMap.put("scanTime", tsField);
			} else if (field.startsWith("Scan date")) {
				String fieldValue = extractField(field);
				Field<String> tsField = new Field<String>("scanDate",
						fieldValue);
				stringFieldMap.put("scanDate", tsField);
			} else if (field.startsWith("rhuser1:")) {
				String fieldValue = extractField(field);
				int value = getAsInt(fieldValue);
				Field<Integer> repsField = new Field<Integer>("reps", value);
				integerFieldMap.put("reps", repsField);
			} else if (field.startsWith("rhuser4:")) {
				String fieldValue = extractField(field);
				int value = getAsInt(fieldValue);
				Field<Integer> ddaField = new Field<Integer>("dda", value);
				integerFieldMap.put("dda", ddaField);
			} else if (field.startsWith("rhuser8:")) {
				String fieldValue = extractField(field);
				int value = getAsInt(fieldValue);
				Field<Integer> nIntlvField = new Field<Integer>("nIntlv",
						extractNIntLv(value));
				System.out.println("nIntlv=" + extractNIntLv(value));
				integerFieldMap.put("nIntlv", nIntlvField);
			}
		}
		if (stringFieldMap.containsKey("scanDate")
				&& stringFieldMap.containsKey("scanTime")) {
			Field<String> scanDateField = stringFieldMap.get("scanDate");
			Field<String> scanTimeField = stringFieldMap.get("scanTime");
			String tsStr = correctYear(scanDateField.getValue().trim()) + " "
					+ scanTimeField.getValue().trim();
			Field<String> tsField = new Field<String>("timestamp", tsStr);
			stringFieldMap.put("timestamp", tsField);
			System.out.println("exam ts:" + extractTimestamp(tsStr));
		}
		// get reliable values for tag,nprep, reps,dda, nIntlv
		System.out.println("getCBFBIRNUserParams\n==============");
		getCBFBIRNUserParams();
	}

	public static String correctYear(String value) {
		String[] toks = value.split("\\/");
		int year = Integer.parseInt(toks[2]) + 1900;
		StringBuilder sb = new StringBuilder();
		sb.append(toks[0]).append('/').append(toks[1]).append('/').append(year);
		return sb.toString();
	}

	public Field<Integer> getIntField(String name) {
		return integerFieldMap.get(name);
	}

	public Field<String> getStringField(String name) {
		return stringFieldMap.get(name);
	}

	public List<String> getStringFieldNames() {
		List<String> names = new ArrayList<String>(stringFieldMap.keySet());
		Collections.sort(names);
		return names;
	}

	public List<String> getIntFieldNames() {
		List<String> names = new ArrayList<String>(integerFieldMap.keySet());
		Collections.sort(names);
		return names;
	}

	protected Date extractTimestamp(String value) throws ParseException {
		return tsFormat.parse(value);
	}

	protected String extractField(String field) {
		field = field.trim();
		int idx = field.indexOf(':');
		if (idx == -1) {
			return null;
		}
		String name = field.substring(0, idx);
		String value = field.substring(idx + 1).trim();
		headerPOIMap.put(name, value);
		return value;
	}

	static int getAsInt(String fieldValue) {
		int value = -1;
		try {
			value = Integer.parseInt(fieldValue);
		} catch (NumberFormatException nfe) {
			value = (int) Float.parseFloat(fieldValue);
		}
		return value;
	}

	public int getTag() {
		String fieldValue = headerPOIMap.get("rhuser25");
		int value = getAsInt(fieldValue);
		return extractTagValue(value);
	}

	public int getNprep() {
		String fieldValue = headerPOIMap.get("rhuser25");
		int value = getAsInt(fieldValue);
		return extractNprepValue(value);
	}

	public Date getTimestamp() throws ParseException {
		String fieldValue = headerPOIMap.get("Exam date/time stamp");
		return extractTimestamp(fieldValue);
	}

	public int extractTagValue(int value) {
		return value & 15;
	}

	public int extractNprepValue(int value) {
		return (value >> nprepPOSN) & ~(~0 << nprepBITS);
	}
	
	public int extractNIntLv(int value) {
		return (value >> nlPOSN) & ~(~0 <<nlBITS) ;
	}

	@SuppressWarnings("unused")
	private static void test2() throws Exception {
		String rootDir = System.getProperty("user.home") + "/cbfbirn_data/birnphase1";
		File[] files = new File(rootDir).listFiles();
		for(File f : files) {
			if (f.getName().endsWith(".7")) {
				System.out.println(f);
				PFileHeaderUtils phu = new PFileHeaderUtils(f.getAbsolutePath());
				phu.readHeader();
			}
		}
	}
	public static void main(String[] args) throws Exception {
		test3();
	}

	private static void test3() throws Exception {
		String rootDir = "/data/bozyurt/22x_dev_tests/111223_est_fair_3te";
		String[] pfiles = {"P26624.7","P27136.7","P27648.7"};
		for (String apf : pfiles) {
			String pfile = rootDir + "/" + apf;
			System.out.println(pfile);
			PFileHeaderUtils phu = new PFileHeaderUtils(pfile);
			phu.readHeader();
		}
	}
	
	@SuppressWarnings("unused")
	private static void test1() throws Exception {
		String rootDir = "/data/cbfbirn_data/sampledata/090713_ds_stern";
		rootDir = "/data/cbfbirn_data/sampledata/090924_aw_stern";
		rootDir = "/data/cbfbirn_data/sampledata/090819_vg_visual";

		// String[] pfiles = { "P34304.7", "P36352.7", "P31744.7", "P35840.7",
		// "P35328.7", "P34816.7", "P36864.7" };

		// String[] pfiles = { "P21504.7", "P22528.7", "P23552.7", "P27648.7",
		// "P20992.7", "P22016.7", "P23040.7", "P26112.7", "P28672.7" };
		String[] pfiles = { "P11776.7", "P14848.7", "P16384.7", "P17408.7",
				"P18944.7", "P19968.7", "P12800.7", "P15872.7", "P16896.7",
				"P18432.7", "P19456.7", "P21504.7" };
		for (String apf : pfiles) {
			String pfile = rootDir + "/" + apf;
			System.out.println(pfile);
			PFileHeaderUtils phu = new PFileHeaderUtils(pfile);
			phu.readHeader();
		}
	}
}
