package clinical.web.actions;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.cache.CacheUtils;
import clinical.server.vo.Experiment;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.IRemoteDBServices;
import clinical.web.ISubjectVisitManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.AssessmentMapping;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.AssessmentMapping.ScoreMap;
import clinical.web.common.query.BinaryOperator;
import clinical.web.common.query.Operator;
import clinical.web.common.query.QueryPartInfo;
import clinical.web.common.query.QueryUtils;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.UnaryOperator;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.common.vo.VisitSegmentScoreValues;
import clinical.web.exception.BaseException;
import clinical.web.forms.AsQueryBuilderForm;
import clinical.web.helpers.HeaderInfo;
import clinical.web.helpers.MultiValuedScoreGroupLayout;
import clinical.web.helpers.ScoreValueSummary;
import clinical.web.helpers.ScoreValueSummaryHeader;
import clinical.web.helpers.SubCorticalVarInfo;
import clinical.web.helpers.ScoreValueSummary.ScoreValueIdxInfo;
import clinical.web.services.SecurityService;
import clinical.web.vo.StudySegmentInfo;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.vo.SubjectDerivedDataValueSummary;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: AsQueryHelper.java 366 2011-05-05 20:06:27Z bozyurt $
 */

public class AsQueryHelper {
	protected static Log log = LogFactory.getLog(AsQueryHelper.class);

	protected AsQueryHelper() {
	}

	/**
	 *
	 * processes the query results to aid its display in the screen, it also
	 * combines the results from the assessment query and derived data query.
	 *
	 * @param summaryList
	 *            a list of <code>SubjectAsScoreValueSummary</code> objects
	 * @param asiMap
	 *            keyed by the assessment id and contains
	 *            <code>AssessmentSelectionInfo</code> objects
	 * @param svHeader
	 *            a new <code>ScoreValueSummaryHeader</code> object to be
	 *            populated by this method
	 * @param derivedDataSummaryList
	 *            null if no derived data is queried, otherwise a list of
	 *            <code>SubjectDerivedDataValueSummary</code> objects
	 * @param siteAsMap
	 *            for mediated queries, for other queries it must be null
	 * @param dbID
	 *            the database ID for the primary database
	 * @param ui
	 *            UserInfo
	 * @param allowImageDownload
	 * @param dbID2SiteIDMap
	 * @param expMap
	 *            a map keyed by experiment id containing
	 *            {@link clinical.server.vo.Experiment} objects
	 *
	 * @return a list of <code>ScoreValueSummary</code> objects
	 *
	 * @see clinical.web.helpers.ScoreValueSummaryHeader
	 * @see clinical.web.helpers.ScoreValueSummary
	 * @see clinical.web.common.vo.AssessmentSelectionInfo
	 * @see clinical.web.vo.SubjectDerivedDataValueSummary
	 * @see clinical.web.vo.SubjectAsScoreValueSummary
	 */
	public static List<ScoreValueSummary> processQueryResults(
			List<SubjectAsScoreValueSummary> summaryList,
			Map<String, AssessmentSelectionInfo> asiMap,
			ScoreValueSummaryHeader svHeader,
			List<SubjectDerivedDataValueSummary> derivedDataSummaryList,
			Map<Integer, List<AssessmentMapping>> siteAsMap, String dbID,
			UserInfo ui, boolean allowImageDownload,
			Map<String, String> dbID2SiteIDMap, Map<String, Experiment> expMap)
			throws BaseException {
		List<ScoreValueSummary> results = new LinkedList<ScoreValueSummary>();
		int locIdx = 0;
		Map<String, ScoreValueSummary> map = new TreeMap<String, ScoreValueSummary>();
		Map<String, HeaderInfo> headerMap = new HashMap<String, HeaderInfo>(11);
		ScoreValueSummary summary = null;
		HeaderInfo hi = null;

		Map<String, Integer> idxMap = null;
		AssessmentLocator aloc = null;
		GlobalAsIDRepository gaiRep = new GlobalAsIDRepository();

		try {
			aloc = prepareAssessmentLocator(dbID2SiteIDMap, gaiRep);
			idxMap = prepareScoreValueIdxMap2(summaryList,
					derivedDataSummaryList, siteAsMap, asiMap, dbID, aloc,
					dbID2SiteIDMap, gaiRep);
		} catch (Exception x) {
			throw new BaseException(x);
		}

		Map<Integer, String> expBaseURIMap = new HashMap<Integer, String>(7);

		for (Object element : summaryList) {
			SubjectAsScoreValueSummary sasv = (SubjectAsScoreValueSummary) element;
			summary = map.get(sasv.getSubjectID());

			if (summary == null) {
				summary = new ScoreValueSummary();
				summary.setSubjectID(sasv.getSubjectID());
				summary.setAssessmentID(sasv.getAssessmentID());
				// set the site ID also to differentiate records for same
				// subject ID
				// from multiple databases
				summary.setSiteID(sasv.getSiteID());
				String expIDStr = String.valueOf(sasv.getExperimentID());
				summary.setExperimentID(expIDStr);
				Experiment exp = expMap.get(expIDStr);
				if (exp != null) {
					summary.setExperimentName(exp.getName());
				} else {
					summary.setExperimentName(expIDStr);
				}

				map.put(sasv.getSubjectID(), summary);

				if (allowImageDownload) {
					String expName = getExpName(ui, dbID, sasv
							.getExperimentID(), expBaseURIMap);
					String sidURL = prepareSubjectImageDownloadURL(sasv
							.getSubjectID(), expName);
					summary.setSubjectImageDownloadURL(sidURL);
				}
			}

			VisitSegmentScoreValues vssv = summary.getVisitSegmentScoreValues(
					sasv.getVisitID(), sasv.getSegmentID(), sasv
							.getExperimentID());

			String vidURL = null;

			if (allowImageDownload) {
				String expName = getExpName(ui, dbID, sasv.getExperimentID(),
						expBaseURIMap);
				vidURL = prepareVisitImageDownloadURL(sasv.getVisitID(), sasv
						.getSubjectID(), sasv.getSiteID(), expName);
			}

			if (vssv == null) {
				int numVars = getNumOfVars(idxMap);
				if (numVars == 0) {
					// in case no single valued questions
					numVars = 1;
				}
				vssv = summary.addVisitSegmentScoreValues(sasv.getVisitID(),
						sasv.getSegmentID(), sasv.getExperimentID(), sasv
								.getSiteID(), numVars);

				vssv.setTimeStamp(sasv.getTimeStamp());
				vssv.setVisitDownloadURL(vidURL);
			} else {
				vssv.setVisitDownloadURL(vidURL);
			}

			int globalAsID = gaiRep.getGlobalAsID(sasv.getSiteID()
					.toLowerCase(), sasv.getAssessmentID());

			String svIdxKey = prepareScoreValueIdxKey(globalAsID, sasv
					.getScoreName());
			if (log.isDebugEnabled()) {
				log.debug("key= " + svIdxKey);
			}

			if (sasv.isMultiValued()) {
				// handle multi valued data for presentation
				// FIXME how is the siteID obtained in sasv?
				AssessmentSelectionInfo asi = (AssessmentSelectionInfo) getEquivalentASI(
						aloc, sasv.getAssessmentID(), sasv.getSiteID()
								.toLowerCase(), asiMap);

				String asName = asi.getName();
				MultiValuedScoreGroupLayout mvsgl = vssv
						.getMultiValuedScoreGroupLayout(asName);
				if (mvsgl == null) {
					mvsgl = new MultiValuedScoreGroupLayout(asName);
					vssv.addMultiValuedScoreLayoutGroup(mvsgl);
				}
				mvsgl.addMultiValuedScoreLayout(getPrimaryDBScoreName(sasv
						.getScoreName(), asName, siteAsMap, dbID), sasv
						.getValues());
				// since multivalued scores have their own header and layout,
				// skip the header creation
				continue;
			} else {
				vssv.addValue(idxMap.get(svIdxKey).intValue(),
						sasv.getValue() == null ? "" : sasv.getValue());
			}
			if (log.isDebugEnabled()) {
				log.debug("asiMap.get(sasv.getAssessmentID()="
						+ asiMap.get(sasv.getSiteID().toLowerCase() + '_'
								+ sasv.getAssessmentID()) + " assessmentID = "
						+ sasv.getAssessmentID());
			}

			// TODO assumption: assessment names and score names are identical
			// between sites!!
			// Needs to be relaxed!!

			AssessmentSelectionInfo asi = getEquivalentASI(aloc, sasv
					.getAssessmentID(), sasv.getSiteID().toLowerCase(), asiMap);
			if (asi != null) {
				hi = headerMap.get(asi.getName());
				if (hi == null) {
					hi = new HeaderInfo(asi.getName(), locIdx);
					headerMap.put(hi.getTitle(), hi);
					++locIdx;
				}
				if (!hi.contains(sasv.getScoreName())) {

					hi.addSubHeader(sasv.getScoreName(), idxMap.get(svIdxKey)
							.intValue());

					AsScoreInfo si = asi.findScoreInfo(sasv.getScoreName());
					if (si == null) {
						log
								.error("si == null scoreName "
										+ sasv.getScoreName());
					}

					hi.addSubHeaderMetaData(si);

					if (si.getType().equals("integer")
							|| si.getType().equals("float")) {
						hi.addSubHeaderDataType("CF");
					} else {
						hi.addSubHeaderDataType("DF");
					}
				}
			}// if (asi != null) {
		}// for summaryList

		Map<String, String> ddMap = new HashMap<String, String>();
		// SubjectDerivedDataValueSummary
		if (derivedDataSummaryList != null) {
			summary = prepareDerivedDataResults(derivedDataSummaryList, locIdx,
					map, headerMap, summary, hi, idxMap, ddMap);
		}

		List<HeaderInfo> headers = new ArrayList<HeaderInfo>(headerMap.size());
		for (Iterator<HeaderInfo> it = headerMap.values().iterator(); it
				.hasNext();) {
			headers.add(it.next());
		}

		headerMap = null;
		// sort headers by assessment name
		Collections.sort(headers, new Comparator<HeaderInfo>() {
			public int compare(HeaderInfo h1, HeaderInfo h2) {
				return h1.getTitle().compareTo(h2.getTitle());
			}
		});

		for (Iterator<HeaderInfo> it = headers.iterator(); it.hasNext();) {
			HeaderInfo h = it.next();
			svHeader.addHeader(h);
		}
		// TODO multi-valued values affected?
		List<ScoreValueIdxInfo> svIdxInfoList = prepareScoreValueIdxInfoList(
				idxMap.size(), headers);

		// filter the results based on the derived data results
		if (derivedDataSummaryList != null) {
			for (Iterator<String> it = map.keySet().iterator(); it.hasNext();) {
				String subjectID = it.next();
				if (ddMap.get(subjectID) == null)
					it.remove();
			}
		}
		Map<String, String> subjectIDMap = new LinkedHashMap<String, String>();
		for (Iterator<ScoreValueSummary> it = map.values().iterator(); it
				.hasNext();) {
			summary = it.next();
			summary.setSvIdxInfoList(svIdxInfoList);
			results.add(summary);
			if (subjectIDMap.get(summary.getSubjectID()) != null) {
				subjectIDMap
						.put(summary.getSubjectID(), summary.getSubjectID());
			}
		}
		if (!results.isEmpty()) {
			// set study information if any
			List<String> subjectIDList = new ArrayList<String>(subjectIDMap
					.values());
			subjectIDMap = null;
			setStudyInfos(dbID, subjectIDList, results);
		}

		return results;
	}

	static List<ScoreValueIdxInfo> prepareScoreValueIdxInfoList(int listSize,
			List<HeaderInfo> headers) {
		// FIXME
		List<ScoreValueIdxInfo> svIdxInfoList = new ArrayList<ScoreValueIdxInfo>(
				listSize);
		int i = 0;
		for (Iterator<HeaderInfo> iter = headers.iterator(); iter.hasNext();) {
			HeaderInfo hi = iter.next();
			for (Iterator<String> it = hi.getSubHeaders().iterator(); it
					.hasNext();) {
				String subHeader = it.next();
				int svIdx = hi.getSubHeaderIndex(subHeader);
				ScoreValueSummary.ScoreValueIdxInfo svIdxInfo = new ScoreValueSummary.ScoreValueIdxInfo(
						svIdx, i);
				svIdxInfoList.add(svIdxInfo);
				++i;
			}
		}
		Collections.sort(svIdxInfoList, new Comparator<ScoreValueIdxInfo>() {
			public int compare(ScoreValueIdxInfo s1, ScoreValueIdxInfo s2) {
				return s1.getHeadLocIdx() - s2.getHeadLocIdx();
			}
		});

		return svIdxInfoList;
	}

	public static String getExpName(UserInfo ui, String dbID, int expID,
			Map<Integer, String> expBaseURIMap) throws BaseException {
		String baseURI = expBaseURIMap.get(new Integer(expID));
		if (baseURI == null) {
			baseURI = getExpName(ui, dbID, expID);
			expBaseURIMap.put(new Integer(expID), baseURI);
		}
		return baseURI;
	}

	public static String getExpName(UserInfo ui, String dbID, int expID)
			throws BaseException {
		IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
		String siteID = CacheUtils.getDBID2SiteIDMap().get(dbID);
		try {
        Map<String, Experiment> expMap = rds.getAllExperiments(siteID);

		Experiment exp = expMap.get( String.valueOf(expID));
		if (exp != null) {
			return exp.getName();
		} else {
			return null;
		}
		} catch(Exception x) {
			throw new BaseException("getExpName",x);
		}
	}

	public static String prepareVisitImageDownloadURL(int visitID,
			String subjectID, String siteName, String expName) {
		StringBuffer buf = new StringBuffer(128);
		StringBuffer visitBuf = new StringBuffer();

		buf.append("https://portal.nbirn.net/Apps/get.cgi");
		// buf.append("/home/Public/fBIRNPhaseI__p0009/Data/").append(subjectID).append('/');
		buf.append("/home/Public/").append(expName).append("/Data/").append(
				subjectID).append('/');
		String siteID = GenUtils.findSiteIDFromSiteName(siteName);
		visitBuf.append("scanVisit__").append(siteID).append("__");
		String visit = GenUtils.leftPad(String.valueOf(visitID), '0',
				4 - String.valueOf(visitID).length());
		visitBuf.append(visit);
		buf.append(visitBuf.toString()).append("/archive/");
		buf.append(subjectID).append('-').append(visitBuf.toString()).append(
				".tgz");
		return buf.toString();
	}

	public static String prepareSubjectImageDownloadURL(String subjectID,
			String expName) {
		StringBuffer buf = new StringBuffer(128);
		// buf.append("https://portal.nbirn.net/Apps/get.cgi/home/Public/fBIRNPhaseI__p0009/Data/");
		buf.append("https://portal.nbirn.net/Apps/get.cgi/home/Public/");
		buf.append(expName).append("/Data/");
		buf.append(subjectID).append("/archive/");
		buf.append(subjectID).append('-').append(subjectID).append(".tgz");
		return buf.toString();
	}

	public static void setStudyInfos(String dbID, List<String> subjectIDList,
			List<ScoreValueSummary> results) throws BaseException {
		ISecurityService isec = ServiceFactory.getSecurityService();
		if (dbID == null) {
			dbID = isec.getDefaultDBID();
		}
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(dbID);
		UserInfo ui = new UserInfo(Constants.ADMIN_USER, null, null);
		List<StudySegmentInfo> studySegmentInfos = isvm.findStudyInfosForSubjects(ui,
				subjectIDList);
		Map<String, StudySegmentInfo> studySegmentMap = new LinkedHashMap<String, StudySegmentInfo>();
		for(StudySegmentInfo ssi : studySegmentInfos) {
			String key = createStudySegmentInfoKey(ssi.getSubjectID(), ssi
					.getExperimentID(), ssi.getVisitID(), ssi.getSegmentID());
			studySegmentMap.put(key, ssi);
		}
		studySegmentInfos = null;
		for (Iterator<ScoreValueSummary> iter = results.iterator(); iter
				.hasNext();) {
			ScoreValueSummary svs = iter.next();
			for (VisitSegmentScoreValues vssv : svs.getVisitSegmentScoreValues()) {

				String key = createStudySegmentInfoKey(svs.getSubjectID(), vssv
						.getExperimentID(), vssv.getVisitID(), vssv
						.getSegmentID());
				StudySegmentInfo ssi = studySegmentMap.get(key);
				if (ssi != null) {
					vssv.setStudyID(ssi.getStudyID());
					vssv.setStudyName(ssi.getStudyName());
				}
			} // it
		}
	}

	public static String createStudySegmentInfoKey(String subjectID,
			int experimentID, int visitID, int segmentID) {
		StringBuffer buf = new StringBuffer(100);
		buf.append(subjectID).append('_').append(experimentID).append('_');
		buf.append(visitID).append('_').append(segmentID);
		return buf.toString();
	}

	private static ScoreValueSummary prepareDerivedDataResults(
			List<SubjectDerivedDataValueSummary> derivedDataSummaryList,
			int locIdx, Map<String, ScoreValueSummary> map,
			Map<String, HeaderInfo> headerMap, ScoreValueSummary summary,
			HeaderInfo hi, Map<String, Integer> idxMap,
			Map<String, String> ddMap) {
		for (Iterator<SubjectDerivedDataValueSummary> iter = derivedDataSummaryList
				.iterator(); iter.hasNext();) {
			SubjectDerivedDataValueSummary sddv = iter.next();

			summary = map.get(sddv.getSubjectID());
			// if not in assessment scores ignore;
			if (summary == null)
				continue;

			if (ddMap.get(sddv.getSubjectID()) == null) {
				ddMap.put(sddv.getSubjectID(), sddv.getSubjectID());
			}

			VisitSegmentScoreValues vssv = summary.getVisitSegmentScoreValues(
					sddv.getVisitID(), sddv.getSegmentID(), sddv
							.getExperimentID());

			String aKey = prepareDeriveDataValueIdxKey(sddv.getScVarInfo()
					.getBrainRegionName(), sddv.getScVarInfo().getLaterality());
			if (vssv == null) {
				log.debug(">> " + sddv.getSubjectID() + " visitID="
						+ sddv.getVisitID() + ", segmentID="
						+ sddv.getSegmentID());
				vssv = summary.addVisitSegmentScoreValues(sddv.getVisitID(),
						sddv.getSegmentID(), sddv.getExperimentID(), null,
						getNumOfVars(idxMap));
			}

			vssv.addValue(idxMap.get(aKey).intValue(), new Float(sddv
					.getValue()));

			String key = sddv.getScVarInfo().getBrainRegionName()
					+ sddv.getScVarInfo().getLaterality();
			hi = headerMap.get(key);
			if (hi == null) {
				String laterality = sddv.getScVarInfo().getLaterality();
				if (laterality.equals("n/a")) {
					laterality = "";
				} else {
					laterality = GenUtils.toTitleCase(laterality);
				}
				hi = new HeaderInfo(laterality + " "
						+ sddv.getScVarInfo().getBrainRegionName(),
						HeaderInfo.DERIVED_DATA);
				headerMap.put(key, hi);
				++locIdx;
			}

			String measurementType = GenUtils.toTitleCase(sddv.getScVarInfo()
					.getMeasurementType());
			if (!hi.contains(measurementType)) {
				int svIdx = idxMap.get(aKey).intValue();
				hi.addSubHeader(measurementType, svIdx);
				// set data type to brain structure
				hi.addSubHeaderDataType("BS");
				hi.addSubHeaderMetaData(sddv.getScVarInfo());
			}
		}
		return summary;
	}

	static Operator prepareOpTree(AsQueryBuilderForm queryForm,
			boolean forDerivedData) {
		boolean first = true;
		int idx = 0;
		Operator root = null;
		BinaryOperator last = null;
		int noQParts;
		List<QueryPartInfo> queryParts = null;

		if (forDerivedData)
			queryParts = getDerivedDataQueryPart(queryForm.getQueryParts());
		else
			queryParts = getAssessmentQueryPart(queryForm.getQueryParts());

		noQParts = queryParts.size();

		for (Iterator<QueryPartInfo> it = queryForm.getQueryParts().iterator(); it.hasNext();) {
			QueryPartInfo qp = it.next();

			if (forDerivedData && qp.getSubCorticalVarInfo() == null)
				continue;

			if (!forDerivedData && qp.getScoreInfo() == null)
				continue;

			SearchPredicate.Range range = null;
			SearchPredicate.DateRange dateRange = null;

			SearchPredicate sp = null;

			int logOp = qp.getOperator();
			int spType;
			if (forDerivedData) {
				spType = SearchPredicate.FLOAT;
			} else {
				spType = QueryUtils.getSearchPredicateType(qp.getScoreInfo()
						.getType());
			}

			Object value = null;
			if (qp.getRhs().trim().equals("*")) {
				value = qp.getRhs().trim();
			} else {
				value = QueryUtils.convertToType(qp.getRhs(), spType);
			}

			if (qp.getLowBound() != null && qp.getLowBound().length() > 0
					&& qp.getUppBound() != null
					&& qp.getUppBound().length() > 0) {

				/** @todo */
				Object v = QueryUtils.convertToType(qp.getLowBound(), spType);
				if (v instanceof Date) {
					Date lowBound = (Date) v;
					Date uppBound = (Date) QueryUtils.convertToType(qp
							.getUppBound(), spType);
					dateRange = new SearchPredicate.DateRange(lowBound,
							uppBound);
				} else {
					Number lowBound = (Number) QueryUtils.convertToType(qp
							.getLowBound(), spType);
					Number uppBound = (Number) QueryUtils.convertToType(qp
							.getUppBound(), spType);

					range = new SearchPredicate.Range(lowBound, uppBound);
				}
			}

			Object attribute = null;
			if (forDerivedData) {
				String laterality = qp.getSubCorticalVarInfo().getLaterality();
				boolean hasLaterality = true;
				if (laterality == null || laterality.trim().length() == 0) {
					laterality = "n/a";
					hasLaterality = false;
				}
				String brname = qp.getSubCorticalVarInfo().getBrainRegionName();
				if (hasLaterality && brname.indexOf(' ') != -1) {
					brname = brname.substring(brname.lastIndexOf(' ') + 1);
				}
				attribute = new SubCorticalVarInfo(brname, laterality, qp
						.getSubCorticalVarInfo().getMeasurementType(), qp
						.getSubCorticalVarInfo().getUnit());

			} else {
				attribute = qp.getScoreInfo();
			}

			if (range != null) {
				sp = new SearchPredicate(attribute, range, logOp, spType);
			} else if (dateRange != null) {
				sp = new SearchPredicate(attribute, dateRange, logOp, spType);
			} else {
				sp = new SearchPredicate(attribute, value, logOp, spType);
			}

			if (noQParts == 1) {
				root = new UnaryOperator(sp, Operator.NONE);
			} else {
				if (first) {
					UnaryOperator uo = new UnaryOperator(sp, Operator.NONE);
					root = new BinaryOperator();
					((BinaryOperator) root).setLhs(uo);
					last = (BinaryOperator) root;
					first = false;
				} else {
					// do we have more query parts?
					if (it.hasNext()) {
						last
								.setLogicalOp(getSelectedCombinator(idx,
										queryForm));
						BinaryOperator bop = new BinaryOperator();
						UnaryOperator lhs = new UnaryOperator(sp, Operator.NONE);
						bop.setLhs(lhs);
						last.setRhs(bop);
						last = bop;
					} else {
						// the last one
						UnaryOperator uo = new UnaryOperator(sp, Operator.NONE);
						last
								.setLogicalOp(getSelectedCombinator(idx,
										queryForm));
						last.setRhs(uo);
					}
				}
			}

			++idx;
			if (forDerivedData)
				log.debug("BrainRegionName= "
						+ qp.getSubCorticalVarInfo().getBrainRegionName());
			else
				log.debug("Score= " + qp.getScoreInfo().getName());
			log.debug("operator= " + qp.getLogicalOps(qp.getOperator())
					+ " op=" + qp.getOperator());
			log.debug("rhs= " + qp.getRhs());

		}// for it

		return root;
	}

	protected static int getSelectedCombinator(int idx,
			AsQueryBuilderForm queryForm) {
		String combStr = queryForm.getCombinator(idx - 1);
		return (combStr.equals("AND")) ? Operator.AND : Operator.OR;
	}

	/**
	 *
	 * @param scoreName
	 *            the (potentially) secondary db score name
	 * @param asName
	 *            the primary db assessment name
	 * @param siteAsMap
	 * @return
	 */
	protected static String getPrimaryDBScoreName(String scoreName,
			String asName, Map<Integer, List<AssessmentMapping>> siteAsMap,
			String dbID) {
		if (siteAsMap == null)
			return scoreName;

		// for (Iterator iter = siteAsMap.values().iterator(); iter.hasNext();)
		// {
		for (List<AssessmentMapping> amList : siteAsMap.values()) {
			for (Iterator<AssessmentMapping> it1 = amList.iterator(); it1
					.hasNext();) {
				AssessmentMapping am = it1.next();
				if (am.getFirstDB().equals(dbID) && am.getName().equals(asName)) {
					for (Iterator<ScoreMap> it2 = am.getScoreMaps().iterator(); it2
							.hasNext();) {
						ScoreMap sm = it2.next();
						if (sm.getMappedName().equals(scoreName)) {
							return sm.getName();
						}
					}// it2
				}// if
			}// it1
		}
		// at this point as a last resort return the passed score name
		return scoreName;
	}

	/**
	 * @deprecated Returns a hash table, keyed by the assessment ID, score name
	 *             combination, of the location of a score within a single
	 *             assessment score values result record. This is used
	 *             internally to find the location of a score for any assessment
	 *             and score, even for multiple databases. Mapped assessment
	 *             scores always return the same score location.
	 *
	 * @param summaryList
	 *            a list of <code>SubjectAsScoreValueSummary</code> objects
	 * @param derivedDataSummaryList
	 *            null if no derived data is queried, otherwise a list of
	 *            <code>SubjectDerivedDataValueSummary</code> objects
	 * @param siteAsMap
	 *            for mediated queries, for other queries it must be null
	 * @param asiMap
	 *            keyed by the assessment id and contains
	 *            <code>AssessmentSelectionInfo</code> objects
	 * @param dbID
	 *            the database ID for the primary database
	 * @return a hash table, keyed by the assessment ID, score name combination,
	 *         of the location of a score within a single assessment score
	 *         values result record. This is used internally to find the
	 *         location of a score for any assessment and score even for
	 *         multiple databases.
	 * @throws BaseException
	 */
	public static Map<String, Integer> prepareScoreValueIdxMap(
			List<SubjectAsScoreValueSummary> summaryList,
			List<SubjectDerivedDataValueSummary> derivedDataSummaryList,
			Map<String, List<AssessmentMapping>> siteAsMap,
			Map<String, AssessmentSelectionInfo> asiMap, String dbID,
			Map<String, String> dbID2SiteIDMap, GlobalAsIDRepository gaiRep) throws BaseException {
		Map<String, Integer> idxMap = new LinkedHashMap<String, Integer>();
		int idx = 0;
		SecurityService ss = (SecurityService) ServiceFactory.getSecurityService();
		String theSiteID = ss.getSiteID(dbID);

		// String theSiteID = GenUtils.extractSiteID(dbID);

		for (SubjectAsScoreValueSummary sasv : summaryList) {

			// skip different variable names from the other sites
			if (!sasv.getSiteID().equalsIgnoreCase(theSiteID)) {
				continue;
			}

			// skip multi-valued scores
			if (sasv.isMultiValued()) {
				continue;
			}

			int globalAsID = gaiRep.getGlobalAsID(theSiteID.toLowerCase(), sasv
					.getAssessmentID());
			String key = prepareScoreValueIdxKey(globalAsID, sasv
					.getScoreName());
			Integer index = idxMap.get(key);
			if (index == null) {
				// log.debug("adding index for " + key + " idx="+ idx);
				index = new Integer(idx);
				idxMap.put(key, index);
				++idx;
			}
		} // for

		if (siteAsMap != null) {
			// now use the same index for the mapped variables
			for (Map.Entry<String, List<AssessmentMapping>> entry : siteAsMap
					.entrySet()) {
				List<AssessmentMapping> amList = entry.getValue();
				String siteID = entry.getKey();

				for (AssessmentMapping am : amList) {

					for (Iterator<ScoreMap> it2 = am.getScoreMaps().iterator(); it2
							.hasNext();) {
						ScoreMap sm = it2.next();

						int globalAsID = gaiRep.getGlobalAsID(siteID, String
								.valueOf(am.getAssessmentID()));
						String key = prepareScoreValueIdxKey(globalAsID, sm
								.getName());

						Integer index = idxMap.get(key);
						if (index != null) {
							globalAsID = gaiRep.getGlobalAsID(siteID, String
									.valueOf(sm.getMappedAsID()));
							String mappedScoreKey = prepareScoreValueIdxKey(
									globalAsID, sm.getMappedName());
							idxMap.put(mappedScoreKey, index);
						}
					} // it2
				} // it1
			}// iter

		} else {
			Map<String, Map<String, Integer>> retrievedAsNameIDMap = prepareRetrievedAsNameIDMap(dbID);
			if (!retrievedAsNameIDMap.isEmpty()) {
				String siteID = dbID2SiteIDMap.get(dbID);

				Map<String, Map<String, Integer>> mainSiteIDxMap = prepareMainSiteIdxMap(
						asiMap, idxMap);

				for (Iterator<Map<String, Integer>> iter = retrievedAsNameIDMap
						.values().iterator(); iter.hasNext();) {
					Map<String, Integer> asNameIDMap = iter.next();

					for (Map.Entry<String, Integer> entry : asNameIDMap
							.entrySet()) {
						String asName = (String) entry.getKey();
						Integer asID = (Integer) entry.getValue();
						Map<String, Integer> scoreIdxMap = mainSiteIDxMap
								.get(asName);
						if (scoreIdxMap == null)
							continue;
						for (Map.Entry<String, Integer> siEntry : scoreIdxMap
								.entrySet()) {
							String scoreName = siEntry.getKey();
							Integer index = siEntry.getValue();
							int globalAsID = gaiRep.getGlobalAsID(siteID, asID
									.toString());
							String key = prepareScoreValueIdxKey(globalAsID,
									scoreName);
							idxMap.put(key, index);
						}
					}
				}
			}

		}

		if (derivedDataSummaryList != null) {
			for (Object element : derivedDataSummaryList) {
				SubjectDerivedDataValueSummary sddv = (SubjectDerivedDataValueSummary) element;
				String regionName = sddv.getScVarInfo().getBrainRegionName();
				String laterality = sddv.getScVarInfo().getLaterality();
				String key = prepareDeriveDataValueIdxKey(regionName,
						laterality);

				Integer index = idxMap.get(key);
				if (index == null) {
					index = new Integer(idx);
					idxMap.put(key, index);
					++idx;
				}
			}
		}
		if (log.isDebugEnabled()) {
			for (Map.Entry<String, Integer> item : idxMap.entrySet()) {
				log.debug("idxMap key=" + item.getKey() + " value="
						+ item.getValue());
			}
		}
		return idxMap;
	}

	public static class IndexInfo {
		String siteID;
		String asName;
		String scoreName;
		int scoreOrder;
		int idx;

		public IndexInfo(String siteID, String asName, String scoreName,
				int scoreOrder, int idx) {
			this.siteID = siteID;
			this.asName = asName;
			this.scoreName = scoreName;
			this.scoreOrder = scoreOrder;
			this.idx = idx;
		}
	}

	protected static AssessmentSelectionInfo getEquivalentASI(
			AssessmentLocator aloc, String realASID, String siteID,
			Map<String, AssessmentSelectionInfo> asiMap) {
		String saKey = siteID + "_" + realASID;

		if (asiMap.get(saKey) != null) {
			return asiMap.get(saKey);
		}

		SAInfo sai = aloc.siteIDAsID2SAInfoMap.get(saKey);
		for (Iterator<AssessmentSelectionInfo> iter = asiMap.values()
				.iterator(); iter.hasNext();) {
			AssessmentSelectionInfo asi = iter.next();
			if (asi.getName().equals(sai.asName)) {
				return asi;
			}
		}

		return null;
	}

	protected static Map<String, IndexInfo> prepSiteID2AsListMap(
			AssessmentLocator aloc,
			List<SubjectAsScoreValueSummary> summaryList,
			Map<String, AssessmentSelectionInfo> asiMap, String theSiteID,
			GlobalAsIDRepository gaiRep) {
		// first determine the union of the assessments retrieved from each site
		// since each site can contribute a different disjoint subset to the
		// whole

		Map<String, Map<String, AssessmentSelectionInfo>> siteID2AsListMap = new HashMap<String, Map<String, AssessmentSelectionInfo>>(
				13);

		for (Object element : summaryList) {
			SubjectAsScoreValueSummary sasv = (SubjectAsScoreValueSummary) element;

			AssessmentSelectionInfo asi = getEquivalentASI(aloc, String
					.valueOf(sasv.getAssessmentID()), sasv.getSiteID()
					.toLowerCase(), asiMap);
			Map<String, AssessmentSelectionInfo> asMap = siteID2AsListMap
					.get(sasv.getSiteID().toLowerCase());
			if (asMap == null) {
				asMap = new LinkedHashMap<String, AssessmentSelectionInfo>(23);
				siteID2AsListMap.put(sasv.getSiteID().toLowerCase(), asMap);
			}

			if (asMap.get(asi.getName()) == null) {
				AssessmentSelectionInfo asiCopy = new AssessmentSelectionInfo(
						asi.getName(), asi.getAssessmentID());
				asMap.put(asiCopy.getName(), asiCopy);
				AsScoreInfo si = asi.findScoreInfo(sasv.getScoreName());
				asiCopy.addScore(si);
			} else {
				AssessmentSelectionInfo asiCopy = asMap.get(asi.getName());
				AsScoreInfo si = asiCopy.findScoreInfo(sasv.getScoreName());
				if (si == null) {
					si = asi.findScoreInfo(sasv.getScoreName());
					asiCopy.addScore(si);
				}
			}
		}

		Set<AssessmentSelectionInfo> seenAsSet = new HashSet<AssessmentSelectionInfo>();
		Map<String, Map<String, AssessmentSelectionInfo>> referenceMap = new LinkedHashMap<String, Map<String, AssessmentSelectionInfo>>(
				siteID2AsListMap);
		referenceMap.remove(theSiteID);
		Map<String, AssessmentSelectionInfo> theAsMap = siteID2AsListMap
				.get(theSiteID);
		if (theAsMap != null) {
			for (Iterator<AssessmentSelectionInfo> iter = theAsMap.values()
					.iterator(); iter.hasNext();) {
				seenAsSet.add(iter.next());
			}

			for (Map.Entry<String, Map<String, AssessmentSelectionInfo>> entry : referenceMap
					.entrySet()) {
				Map<String, AssessmentSelectionInfo> asMap = entry.getValue();
				// check if any of the assessments are in the reference
				// assessment set. if so remove
				for (Iterator<AssessmentSelectionInfo> it = asMap.values()
						.iterator(); it.hasNext();) {
					AssessmentSelectionInfo asi = it.next();
					if (theAsMap.get(asi.getName()) != null) {
						it.remove();
					}
				}
			}
		}

		for (Map.Entry<String, Map<String, AssessmentSelectionInfo>> entry : referenceMap
				.entrySet()) {
			Map<String, AssessmentSelectionInfo> asMap = entry.getValue();
			for (Iterator<AssessmentSelectionInfo> it2 = asMap.values()
					.iterator(); it2.hasNext();) {
				AssessmentSelectionInfo asi = it2.next();
				if (seenAsSet.contains(asi)) {
					it2.remove();
				} else {
					seenAsSet.add(asi);
				}
			}
		}

		// prepare master index sorted by assessment name and score order under
		// the
		// assessment name
		Map<String, IndexInfo> indexMap = new HashMap<String, IndexInfo>();

		Map<String, AssessmentSelectionInfo> nameSortedAsiMap = new TreeMap<String, AssessmentSelectionInfo>();
		for (Iterator<AssessmentSelectionInfo> iter = seenAsSet.iterator(); iter
				.hasNext();) {
			AssessmentSelectionInfo asi = iter.next();
			nameSortedAsiMap.put(asi.getName(), asi);
		}

		Map<AssessmentSelectionInfo, String> asi2SiteIDMap = new HashMap<AssessmentSelectionInfo, String>();
		for (Map.Entry<String, Map<String, AssessmentSelectionInfo>> entry : siteID2AsListMap
				.entrySet()) {
			String siteID = entry.getKey();
			Map<String, AssessmentSelectionInfo> asMap = entry.getValue();
			for (AssessmentSelectionInfo asi : asMap.values()) {
				asi2SiteIDMap.put(asi, siteID);
			}
		}

		// TODO assumption: score names for multi-valued scores are unique
		// within a site
		Set<String> multiValuedScoreSet = new HashSet<String>(11);
		for (SubjectAsScoreValueSummary sasv : summaryList) {
			if (sasv.isMultiValued()) {
				String key = sasv.getSiteID().toLowerCase() + "_"
						+ sasv.getScoreName();
				multiValuedScoreSet.add(key);
			}
		}

		int idx = 0;
		for (Iterator<AssessmentSelectionInfo> iter = nameSortedAsiMap.values()
				.iterator(); iter.hasNext();) {
			AssessmentSelectionInfo asi = iter.next();
			String siteID = asi2SiteIDMap.get(asi);
			asi.sortScoresBySequence();
			for (AsScoreInfo si : asi.getScores()) {
				// skip multi-valued scores
				String mvsKey = siteID.toLowerCase() + "_" + si.getName();
				if (!multiValuedScoreSet.contains(mvsKey)) {
					IndexInfo idxInfo = new IndexInfo(siteID.toLowerCase(), asi
							.getName(), si.getName(), si.getSequence(), idx);

					int globalAsID = gaiRep.getGlobalAsID(asi.getName());
					String key = prepareScoreValueIdxKey(globalAsID, si
							.getName());
					indexMap.put(key, idxInfo);
					++idx;
				}
			}
		}

		return indexMap;
	}

	public static Map<String, Integer> prepareScoreValueIdxMap2(
			List<SubjectAsScoreValueSummary> summaryList,
			List<SubjectDerivedDataValueSummary> derivedDataSummaryList,
			Map<Integer, List<AssessmentMapping>> siteAsMap,
			Map<String, AssessmentSelectionInfo> asiMap, String dbID,
			AssessmentLocator aloc, Map<String, String> dbID2SiteIDMap,
			GlobalAsIDRepository gaiRep) throws Exception {
		if (siteAsMap != null) {
			throw new UnsupportedOperationException(
					"assessment mapping file for multisite queries is not supported for this version yet");
		}
		// String theSiteID = GenUtils.extractSiteID(dbID);
		String theSiteID = dbID2SiteIDMap.get(dbID);
		Map<String, IndexInfo> idxInfoMap = prepSiteID2AsListMap(aloc,
				summaryList, asiMap, theSiteID, gaiRep);
		Map<String, Integer> idxMap = new HashMap<String, Integer>();

		for (Iterator<SubjectAsScoreValueSummary> it = summaryList.iterator(); it
				.hasNext();) {
			SubjectAsScoreValueSummary sasv = it.next();
			// skip multi-valued ones
			if (!sasv.isMultiValued()) {
				IndexInfo idxInfo = findIndexInfo(sasv.getAssessmentID(), sasv
						.getSiteID().toLowerCase(), sasv.getScoreName(),
						idxInfoMap, gaiRep);
				assert (idxInfo != null);

				int globalAsID = gaiRep.getGlobalAsID(sasv.getSiteID()
						.toLowerCase(), sasv.getAssessmentID());

				String key = prepareScoreValueIdxKey(globalAsID, sasv
						.getScoreName());
				Integer index = idxMap.get(key);
				if (index == null) {
					idxMap.put(key, new Integer(idxInfo.idx));
				}
			}
		}

		int idx = idxInfoMap.size();

		if (derivedDataSummaryList != null) {
			for (SubjectDerivedDataValueSummary sddv : derivedDataSummaryList) {
				String regionName = sddv.getScVarInfo().getBrainRegionName();
				String laterality = sddv.getScVarInfo().getLaterality();
				// TODO no siteID for derived data key
				String key = prepareDeriveDataValueIdxKey(regionName,
						laterality);

				Integer index = idxMap.get(key);
				if (index == null) {
					index = new Integer(idx);
					idxMap.put(key, index);
					++idx;
				}
			}
		}
		return idxMap;
	}

	protected static IndexInfo findIndexInfo(String siteAsID, String siteID,
			String scoreName, Map<String, IndexInfo> idxInfoMap,
			GlobalAsIDRepository gaiRep) {
		int globalAsID = gaiRep.getGlobalAsID(siteID, siteAsID);
		String key = prepareScoreValueIdxKey(globalAsID, scoreName);
		IndexInfo ii = idxInfoMap.get(key);
		return ii;
	}

	/**
	 * @deprecated
	 * @param asID
	 * @param siteID
	 * @param scoreName
	 * @param asLoc
	 * @param idxInfoMap
	 * @param gaiRep
	 * @return
	 */
	protected static IndexInfo findIndexInfoOld(String asID, String siteID,
			String scoreName, AssessmentLocator asLoc, Map<?, ?> idxInfoMap,
			GlobalAsIDRepository gaiRep) {
		String saKey = siteID + "_" + asID;
		SAInfo saInfo = asLoc.siteIDAsID2SAInfoMap.get(saKey);
		List<?> saInfos = asLoc.asName2SAInfoListMap.get(saInfo.asName);
		for (Iterator<?> iter = saInfos.iterator(); iter.hasNext();) {
			SAInfo sai = (SAInfo) iter.next();
			int globalAsID = gaiRep.getGlobalAsID(siteID, sai.asID);
			String key = prepareScoreValueIdxKey(globalAsID, scoreName);
			IndexInfo ii = (IndexInfo) idxInfoMap.get(key);
			if (ii != null) {
				return ii;
			}
		}
		return null;
	}

	public static class AssessmentLocator {
		Map<String, SAInfo> siteIDAsID2SAInfoMap = new HashMap<String, SAInfo>();
		Map<String, List<SAInfo>> asName2SAInfoListMap = new HashMap<String, List<SAInfo>>();
	}

	protected static Map<String, Map<String, Integer>> prepareMainSiteIdxMap(
			Map<String, AssessmentSelectionInfo> asiMap,
			Map<String, Integer> idxMap) {

		// keyed by assessment name
		// each entry contains a map of score name - index pair
		Map<String, Map<String, Integer>> mainIDxMap = new HashMap<String, Map<String, Integer>>();

		for (Map.Entry<String, Integer> entry : idxMap.entrySet()) {
			String key = (String) entry.getKey();
			Integer index = entry.getValue();
			int underscoreIDx = key.lastIndexOf("_");
			String scoreName = key.substring(0, underscoreIDx);
			int mainAsID = Integer.parseInt(key.substring(underscoreIDx + 1));
			AssessmentSelectionInfo asi = asiMap.get(String.valueOf(mainAsID));
			assert (asi != null);
			Map<String, Integer> scoreIDxMap = mainIDxMap.get(asi.getName());
			if (scoreIDxMap == null) {
				scoreIDxMap = new HashMap<String, Integer>(17);
				mainIDxMap.put(asi.getName(), scoreIDxMap);
			}
			scoreIDxMap.put(scoreName, index);
		}

		return mainIDxMap;
	}

	protected static Map<String, Map<String, Integer>> prepareRetrievedAsNameIDMap(
			String dbID) {
		Map<String, Map<String, Integer>> retrievedAsNameIDMap = new HashMap<String, Map<String, Integer>>(
				7);
		try {
			ISecurityService isec = ServiceFactory.getSecurityService();
			String[] allDBIDs = isec.getAllDBIDs();
			for (int i = 0; i < allDBIDs.length; i++) {
				if (allDBIDs[i].equals(dbID)) {
					continue;
				}
				prepareAsNameIDMapForDB(allDBIDs[i], retrievedAsNameIDMap);
			}
		} catch (Exception x) {
			log.error("prepareRetrievedAsNameIDMap", x);
		}
		return retrievedAsNameIDMap;
	}

	protected static void prepareAsNameIDMapForDB(String aDbID,
			Map<String, Map<String, Integer>> retrievedAsNameIDMap) {
		IDBPoolService pool = ServiceFactory.getPoolService(aDbID);
		Connection con = null;
		Statement st = null;
		try {
			con = pool.getConnection(Constants.ADMIN_USER);
			st = con.createStatement();
			ResultSet rs = st
					.executeQuery("select assessmentid, name from nc_assessment");
			Map<String, Integer> asNameIDMap = new HashMap<String, Integer>(17);
			while (rs.next()) {
				asNameIDMap.put(rs.getString(2), new Integer(rs.getInt(1)));
			}
			rs.close();

			retrievedAsNameIDMap.put(aDbID, asNameIDMap);
		} catch (Exception e) {
			log.error("prepareAsNameIDMapForDB", e);
		} finally {
			if (st != null)
				try {
					st.close();
				} catch (Exception x) { /* ignore */
				}
			if (con != null) {
				try {
					pool.releaseConnection(Constants.ADMIN_USER, con);
				} catch (Exception x) {
				}
			}
		}
	}

	public static class SAInfo {
		String dbID;
		String asName;
		String asID;
		String siteID;
		int hcode;

		public SAInfo(String dbID, String asName, String asID, String siteID) {
			this.dbID = dbID;
			this.asName = asName;
			this.asID = asID;
			this.siteID = siteID;
			StringBuffer buf = new StringBuffer();
			buf.append(dbID).append('_').append(asName);
			hcode = buf.toString().hashCode();
		}

		public boolean equals(Object other) {
			if (other == null) {
				return false;
			}
			if (other == this) {
				return true;
			}
			if (!(other instanceof SAInfo)) {
				return false;
			}

			SAInfo that = (SAInfo) other;
			if ((dbID != null && that.dbID != null && dbID.equals(that.dbID))
					&& (asName != null && that.asName != null && asName
							.equals(that.asName))) {
				return true;
			}
			return false;
		}

		public int hashCode() {
			return hcode;
		}
	}// ;

	protected static AssessmentLocator prepareAssessmentLocator(
			Map<String, String> dbID2SiteIDMap, GlobalAsIDRepository gaiRep)
			throws Exception {
		AssessmentLocator aloc = new AssessmentLocator();

		ISecurityService isec = ServiceFactory.getSecurityService();
		String[] allDBIDs = isec.getAllDBIDs();
		for (int i = 0; i < allDBIDs.length; i++) {
			prepareAsID2SAInfoMapForDB(allDBIDs[i], aloc.siteIDAsID2SAInfoMap,
					dbID2SiteIDMap, gaiRep);
		}
		for (Iterator<SAInfo> iter = aloc.siteIDAsID2SAInfoMap.values()
				.iterator(); iter.hasNext();) {
			SAInfo sai = iter.next();
			List<SAInfo> saiList = aloc.asName2SAInfoListMap.get(sai.asName);
			if (saiList == null) {
				saiList = new ArrayList<SAInfo>();
				aloc.asName2SAInfoListMap.put(sai.asName, saiList);
			}
			if (!saiList.contains(sai)) {
				saiList.add(sai);
			}
		}
		return aloc;
	}

	protected static void prepareAsID2SAInfoMapForDB(String aDbID,
			Map<String, SAInfo> asID2SAInfoMap,
			Map<String, String> dbID2SiteIDMap, GlobalAsIDRepository gaiRep)
			throws Exception {
		IDBPoolService pool = ServiceFactory.getPoolService(aDbID);
		Connection con = null;
		Statement st = null;
		try {
			con = pool.getConnection(Constants.ADMIN_USER);
			st = con.createStatement();
			ResultSet rs = st
					.executeQuery("select assessmentid, name from nc_assessment");
			while (rs.next()) {
				String siteID = dbID2SiteIDMap.get(aDbID);
				String asID = rs.getString(1);
				String asName = rs.getString(2);

				AssessmentSelectionInfo asi = new AssessmentSelectionInfo();
				asi.setAssessmentID(new BigDecimal(asID));
				asi.setName(asName);
				gaiRep.addASI(asi, siteID);

				String key = siteID + "_" + asID;
				if (asID2SAInfoMap.get(key) != null) {
					throw new Exception(
							"Duplicate assessmentID is found in multi-site query!:"
									+ key);
				}
				asID2SAInfoMap
						.put(key, new SAInfo(aDbID, asName, asID, siteID));
			}
			rs.close();
		} catch (Exception e) {
			log.error("prepareAsNameIDMapForDB", e);
		} finally {
			if (st != null)
				try {
					st.close();
				} catch (Exception x) { /* ignore */
				}
			if (con != null) {
				try {
					pool.releaseConnection(Constants.ADMIN_USER, con);
				} catch (Exception x) {
				}
			}
		}
	}

	protected static String prepareScoreValueIdxKey(int globalAsID,
			String scoreName) {
		StringBuffer buf = new StringBuffer();
		buf.append(scoreName).append('_').append(globalAsID);
		return buf.toString();
	}

	protected static String prepareDeriveDataValueIdxKey(String regionName,
			String laterality) {
		StringBuffer buf = new StringBuffer();
		buf.append(regionName).append('_').append(laterality);
		return buf.toString();
	}

	protected static int getNumOfVars(Map<String, Integer> idxMap) {
		Map<Integer, Integer> uniqueIdxMap = new HashMap<Integer, Integer>(
				idxMap.size());
		for (Iterator<Integer> iter = idxMap.values().iterator(); iter
				.hasNext();) {
			Integer idx = iter.next();
			uniqueIdxMap.put(idx, idx);
		}
		return uniqueIdxMap.size();
	}

	protected static List<QueryPartInfo> getAssessmentQueryPart(
			List<QueryPartInfo> queryParts) {
		List<QueryPartInfo> aqps = new LinkedList<QueryPartInfo>();
		for (QueryPartInfo qp : queryParts) {
			if (qp.getSubCorticalVarInfo() == null) {
				aqps.add(qp);
			}
		}
		return aqps;
	}

	protected static List<QueryPartInfo> getDerivedDataQueryPart(
			List<QueryPartInfo> queryParts) {
		List<QueryPartInfo> dqps = new LinkedList<QueryPartInfo>();
		for (QueryPartInfo qp : queryParts) {
			if (qp.getSubCorticalVarInfo() != null)
				dqps.add(qp);
		}
		return dqps;
	}

	protected static int getAssessmentQueryPartSize(
			List<QueryPartInfo> queryParts) {
		int count = 0;
		for (QueryPartInfo qp : queryParts) {
			if (qp.getSubCorticalVarInfo() == null)
				count++;
		}
		return count;
	}

	protected static int getDerivedDataQueryPartSize(
			List<QueryPartInfo> queryParts) {
		int count = 0;
		for (QueryPartInfo qp : queryParts) {
			if (qp.getSubCorticalVarInfo() != null)
				count++;
		}
		return count;
	}

	public static Map<String, AssessmentSelectionInfo> toAssessmentMap(
			List<AssessmentSelectionInfo> assessments, String siteID) {
		Map<String, AssessmentSelectionInfo> map = new HashMap<String, AssessmentSelectionInfo>();
		for (AssessmentSelectionInfo asi : assessments) {
			String key = siteID + "_" + asi.getAssessmentID().toString();
			map.put(key, asi);
		}
		return map;
	}

	public static class GlobalAsIDRepository {
		private Map<String, Map<String, AssessmentSelectionInfo>> siteID2ASIMapMap =
			new HashMap<String, Map<String, AssessmentSelectionInfo>>(17);
		/**
		 * each global assessment is keyed by a uniqueid. A global assessment is
		 * defined as a clinical assessment with unique semantic meaning
		 * regardless how it is defined between sites. This unique semantic
		 * meaning is assumed to be identifiable by the features (implicit or
		 * explicit) provided by each site. At minimum, the same assessment name ,
		 * will be interpreted as the same semantic meaning.
		 */
		private Map<String, Integer> globalAsMap = new HashMap<String, Integer>();
		private int idx = 0;

		public GlobalAsIDRepository() {
		}

		public void addASI(AssessmentSelectionInfo asi, String siteID) {
			Map<String, AssessmentSelectionInfo> asiMap = siteID2ASIMapMap
					.get(siteID);
			// asiMap is keyed by the assessmentID assigned to the assessment
			// by the particular site with siteID
			if (asiMap == null) {
				asiMap = new HashMap<String, AssessmentSelectionInfo>(17);
				siteID2ASIMapMap.put(siteID, asiMap);
			}
			assert (asiMap.get(asi.getAssessmentID().toString()) == null);
			asiMap.put(asi.getAssessmentID().toString(), asi);
			if (globalAsMap.get(asi.getName()) == null) {
				globalAsMap.put(asi.getName(), new Integer(++idx));
			}
		}

		public int getGlobalAsID(String assessmentName) {
			Integer globalIdxWrapper = globalAsMap.get(assessmentName);
			return globalIdxWrapper.intValue();
		}

		public int getGlobalAsID(String siteID, String siteAsID) {
			Map<String, AssessmentSelectionInfo> asiMap = siteID2ASIMapMap.get(siteID);
			if (asiMap == null) {
				throw new RuntimeException(
						"GlobalAsIDRepository is not properly initialized");
			}
			AssessmentSelectionInfo asi = asiMap
					.get(siteAsID);
			if (asi == null) {
				throw new RuntimeException(
						"GlobalAsIDRepository is not properly initialized! siteAsID:"
								+ siteAsID + " siteID:" + siteID);
			}
			Integer globalIdxWrapper = globalAsMap.get(asi.getName());
			return globalIdxWrapper.intValue();
		}
	}// ;

}
