package clinical.web.actions;

import java.awt.image.BufferedImage;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.imageio.ImageIO;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import clinical.web.exception.ValidationException;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.action.ActionForm;
import org.apache.struts.action.ActionForward;
import org.apache.struts.action.ActionMapping;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.common.UserInfo;
import clinical.web.forms.CBFROIForm;
import clinical.web.forms.JobManForm;
import clinical.web.scheduler.JobRecord;
import clinical.web.scheduler.JobScheduler;
import clinical.web.workflow.cbf.CBFWFContext;

/**
 * serves full javascript/SVG ROI annotation functionality.
 *
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class CBFROIAnnotateAction extends BaseLookupDispatchAction {
    private Map<String, String> map = new HashMap<String, String>(11);

    static Pattern fileNumberPattern = Pattern.compile("(\\d+)$");
    private Log log = LogFactory.getLog(CBFROIAnnotateAction.class);

    protected Map<String, String> getKeyMethodMap() {
        map.put("action.cbfroi.view", "viewMain");
        map.put("action.cbfroi.send.masks", "sendMasks");
        return map;
    }


    public ActionForward viewMain(ActionMapping mapping, ActionForm form,
                                  HttpServletRequest request, HttpServletResponse response)
            throws Exception {
        log.info(">> viewMain");
        try {
            UserInfo ui = getUserInfo(request);
            CBFROIForm roiForm = (CBFROIForm) form;
            HttpSession session = request.getSession(false);
            String dbID = (String) session
                    .getAttribute(Constants.SESSION_DBID_KEY);
            String jobID = request.getParameter("jobID");
            System.out.println("jobID:" + jobID);

            roiForm.setJobID(jobID);

            JobScheduler scheduler = JobScheduler.getInstance();

            JobRecord jr = scheduler.getJobRecord(ui, dbID, ui
                    .getPerceivedName(), jobID);

            String ctxAsJSON = jr.getContext();
            System.out.println("ctxAsJSON:" + ctxAsJSON);

            JSONObject json = new JSONObject(ctxAsJSON);
            CBFWFContext ctx = CBFWFContext.fromJSON(json);

            roiForm.setContext(ctx);

            String cacheDir = roiForm.getContext().getCacheDir();
            File matlabOutDir = new File(cacheDir, "matlab_out");

            String webRootPath = session.getServletContext().getRealPath("/");
            System.out.println("webRootPath:" + webRootPath);
            String imageDirName = jobID + "_" + ui.getPerceivedName();
            File imgDir = new File(webRootPath, imageDirName);
            if (!imgDir.exists()) {
                boolean ok = imgDir.mkdir();
                if (!ok) {
                    log.error("Cannot create image directory:" + imgDir);
                    throw new ValidationException(
                            "System Problem: Cannot create an image directory for processing");
                }
            }
            List<String> relImgPaths = new ArrayList<String>();

            if (matlabOutDir.isDirectory()) {
                File[] files = matlabOutDir.listFiles();
                List<File> imgFiles = new ArrayList<File>();
                assert files != null;
                for (File f : files) {
                    String fname = f.getName();
                    if (fname.toLowerCase().endsWith(".tif")
                            || fname.endsWith(".png")
                            || fname.endsWith(".jpeg")) {

                        File destImg = new File(imgDir, fname);
                        if (fname.toLowerCase().endsWith(".tif")
                                && ImageIO.getImageReadersBySuffix("tif")
                                .hasNext()) {
                            fname = fname.replaceFirst("\\.\\w+$", ".png");
                            destImg = new File(imgDir, fname);
                            BufferedImage tif = ImageIO.read(f);
                            System.out.println("reading from :" + f);
                            System.out.println("tif:" + tif);
                            System.out.println("writing to " + destImg);
                            ImageIO.write(tif, "png", destImg);
                        } else {
                            FileUtils.copyFile(f.getAbsolutePath(), destImg
                                    .getAbsolutePath());
                        }
                        imgFiles.add(destImg);
                    }
                }
                Collections.sort(imgFiles, new Comparator<File>() {
                    public int compare(File o1, File o2) {
                        int fn1 = getFileNumber(o1.getName());
                        int fn2 = getFileNumber(o2.getName());

                        return fn1 - fn2;
                    }
                });
                for (File f : imgFiles) {
                    relImgPaths.add(imageDirName + "/" + f.getName());
                }

            } else {
                // TEST
                File srcImg = new File(webRootPath, "brain.png");
                FileUtils.copyFile(srcImg.getAbsolutePath(), new File(imgDir,
                        srcImg.getName()).getAbsolutePath());

                for (int i = 0; i < 23; i++) {
                    String relPath = imageDirName + "/brain.png";
                    relImgPaths.add(relPath);
                }
            }

            JSONArray jsArr = new JSONArray();
            for (String relImgPath : relImgPaths) {
                jsArr.put(relImgPath);
            }
            log.info("relImgPaths:" + jsArr.toString(2));

            roiForm.setImagePathsJSON(jsArr.toString());


            return mapping.findForward(Constants.SUCCESS);
        } catch (Exception x) {
            log.error("viewMain", x);
            return processExceptions(request, response, mapping, form, x);
        }
    }


    public ActionForward sendMasks(ActionMapping mapping, ActionForm form,
                                   HttpServletRequest request, HttpServletResponse response)
            throws Exception {
        log.info(">> sendMask");
        try {
            UserInfo ui = getUserInfo(request);
            CBFROIForm roiForm = (CBFROIForm) form;
            HttpSession session = request.getSession(false);
            String dbID = (String) session
                    .getAttribute(Constants.SESSION_DBID_KEY);
            String payloadJSON = request.getParameter("payload");

            JSONObject json = new JSONObject(payloadJSON);

            Payload payload = Payload.fromJSON(json);
            StringBuilder snBuf = new StringBuilder(100);
            for (Integer sliceNo : payload.sliceNos) {
                snBuf.append(sliceNo).append("\n");
            }

            String cacheDir = roiForm.getContext().getCacheDir();

            File sliceNoFile = new File(cacheDir, "slice_no.txt");
            FileUtils
                    .save2File(snBuf.toString(), sliceNoFile.getAbsolutePath());

            for (int i = 0; i < payload.sliceNos.size(); i++) {
                String maskFilename = "mask" + payload.sliceNos.get(i) + ".txt";
                File maskFile = new File(cacheDir, maskFilename);
                saveMask(maskFile, payload.maskList.get(i));
                log.info("saved mask to :" + maskFile);
            }

            String jobID = roiForm.getJobID();
            System.out.println("jobID:" + jobID);

            JobScheduler scheduler = JobScheduler.getInstance();

            scheduler.resumeJob(ui, dbID, ui.getPerceivedName(), jobID);
            List<JobRecord> jrList = scheduler.getJobsForUser(ui, dbID, ui
                    .getPerceivedName());
            JobManForm jmForm = (JobManForm) session
                    .getAttribute(Constants.JOB_MAN_FORM_KEY);
            jmForm.setJrList(jrList);
            jmForm.setUser(ui.getPerceivedName());

            // cleanup state to release memory
            roiForm.clearState();

            return mapping.findForward(Constants.SHOW_JOBS);
        } catch (Exception x) {
            return processExceptions(request, response, mapping, form, x);
        }
    }

    static void saveMask(File path, byte[][] mask) throws IOException {
        BufferedWriter out = null;
        try {
            out = new BufferedWriter(new FileWriter(path));

            for (byte[] aMask : mask) {
                StringBuilder sb = new StringBuilder(mask[0].length * 2);
                for (int j = 0; j < aMask.length; j++) {
                    sb.append(aMask[j]);
                    if ((j + 1) < aMask.length) {
                        sb.append(' ');
                    }
                }
                out.write(sb.toString());
                out.newLine();
            }
        } finally {
            FileUtils.close(out);
        }
    }

    public static int getFileNumber(String filename) {
        filename = filename.replaceFirst("\\.\\w+$", "");
        Matcher m = fileNumberPattern.matcher(filename);
        if (m.find()) {
            return GenUtils.toInt(m.group(1), -1);
        }
        return -1;
    }

    public static class Payload {
        List<Integer> sliceNos = new ArrayList<Integer>(3);
        List<byte[][]> maskList = new ArrayList<byte[][]>(3);


        public static Payload fromJSON(JSONObject json) throws JSONException {
            Payload payload = new Payload();
            JSONArray jsArr = json.getJSONArray("sliceNos");
            for (int i = 0; i < jsArr.length(); i++) {
                payload.sliceNos.add(jsArr.getInt(i));
            }

            jsArr = json.getJSONArray("masks");
            for (int i = 0; i < jsArr.length(); i++) {
                JSONArray rows = jsArr.getJSONArray(i);
                byte[][] mask = new byte[rows.length()][rows.length()];
                payload.maskList.add(mask);
                for (int r = 0; r < rows.length(); r++) {
                    JSONArray cols = rows.getJSONArray(r);
                    for (int c = 0; c < cols.length(); c++) {
                        mask[r][c] = (byte) cols.getInt(c);
                    }
                }
            }

            return payload;
        }

        public List<Integer> getSliceNos() {
            return sliceNos;
        }

        public List<byte[][]> getMaskList() {
            return maskList;
        }
    }//;

}
