package clinical.web.common.query;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Assessment;
import clinical.server.vo.Experiment;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.utils.ThreadPoolMan;
import clinical.web.Constants;
import clinical.web.IAssessmentService;
import clinical.web.ServiceFactory;
import clinical.web.common.IAuthorizationService;
import clinical.web.common.IAuthorizationService.PrivilegeLabel;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.services.IJobProvenanceService;
import clinical.web.services.IJobResultService;
import clinical.web.vo.ASValueKey;
import clinical.web.vo.AssessmentQueryInfo;
import clinical.web.vo.AssessmentResultSummary;
import clinical.web.vo.CBFQueryResultSummary;
import clinical.web.vo.JobProvenanceQueryInfo;
import clinical.web.vo.JobProvenanceValueSummary;
import clinical.web.vo.JobResultQueryInfo;
import clinical.web.vo.JobResultResultSummary;
import clinical.web.vo.JobResultResultSummary.VisitJobResultValues;
import clinical.web.vo.JobResultValueSummary;
import clinical.web.vo.ProvenanceResultSummary;
import clinical.web.vo.ProvenanceResultSummary.VisitProvResultValues;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.vo.SynonymInfo;
import clinical.web.vo.VisitSegAsResultValues;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */

public class CBFQueryAggregator {
	protected String dbID;
	protected UserInfo ui;
	protected IDBCache dbCache = null;
	protected ExecutorService executor;
	protected Map<String, Experiment> expMap;
	protected Map<Integer, Assessment> asIDMap;
	protected List<SynonymInfo> synonymInfoList;

	private Log log = LogFactory.getLog(CBFQueryAggregator.class);

	public CBFQueryAggregator(String dbID, UserInfo ui,
			List<Integer> selectedExperiments, List<SynonymInfo> siList) throws BaseException {
		super();
		this.dbID = dbID;
		this.ui = ui;
		this.synonymInfoList = siList;
		dbCache = ServiceFactory.getDBCache(dbID);
		executor = ThreadPoolMan.getInstance().getExecutorService();
		List<Experiment> experiments = dbCache.getExperiments(ui, false);
		IAuthorizationService authService = ServiceFactory
				.getAuthorizationService();
		List<Experiment> filteredExps = new ArrayList<Experiment>();
		for (Experiment exp : experiments) {
			if (authService.isAuthorized(ui, dbID, PrivilegeLabel.READ, exp
					.getUniqueid().intValue())) {
				filteredExps.add(exp);
			}
		}
		if (selectedExperiments != null && !selectedExperiments.isEmpty()) {
			Set<Integer> expIdSet = new HashSet<Integer>(selectedExperiments);
			for (Iterator<Experiment> it = filteredExps.iterator(); it
					.hasNext();) {
				Experiment exp = it.next();
				if (!expIdSet.contains(exp.getUniqueid().intValue())) {
					it.remove();
				}
			}
		}
		experiments = filteredExps;
		this.expMap = toMap(experiments);
	}

	public List<CBFQueryResultSummary> handleQuery(AssessmentQueryInfo aqi,
			JobProvenanceQueryInfo pqi, JobResultQueryInfo jrqi,
			String primarySiteID) throws Exception {
		CAQueryWorker caqTask = null;
		ProvenanceQueryWorker pqTask = null;
		JobResultsQueryWorker jrqTask = null;
		// TODO connective splitting handling
		int totQueryPartSize = 0;
		int workerCount = 0;
		if (aqi != null) {
			AssessmentQueryInfo naqi = new AssessmentQueryInfo(aqi);
			for (QueryPartInfo qp : naqi.getQpiList()) {
				qp.setConnector(QueryPartInfo.OR);
			}
			caqTask = new CAQueryWorker(naqi, dbID, ui, primarySiteID);
			workerCount++;
		}
		if (pqi != null) {
			JobProvenanceQueryInfo npqi = new JobProvenanceQueryInfo(pqi);
			for (JobProvQueryPartInfo qp : npqi.getQpiList()) {
				qp.setConnector(QueryPartInfo.OR);
			}
			pqTask = new ProvenanceQueryWorker(npqi, dbID, ui);
			workerCount++;
		}
		if (jrqi != null) {
			JobResultQueryInfo njrqi = new JobResultQueryInfo(jrqi);
			for (JobResultQueryPartInfo qp : njrqi.getQpiList()) {
				qp.setConnector(QueryPartInfo.OR);
			}
			jrqTask = new JobResultsQueryWorker(njrqi, dbID, ui);
			workerCount++;
		}
		List<SubjectAsScoreValueSummary> sasvList = null;
		List<JobProvenanceValueSummary> pvsList = null;
		List<JobResultValueSummary> jrvsList = null;
		if (workerCount > 1) {
			Future<List<SubjectAsScoreValueSummary>> caFuture = null;
			Future<List<JobProvenanceValueSummary>> provFuture = null;
			Future<List<JobResultValueSummary>> jrFuture = null;
			if (caqTask != null) {
				caFuture = executor.submit(caqTask);
			}
			if (pqTask != null) {
				provFuture = executor.submit(pqTask);
			}
			if (jrqTask != null) {
				jrFuture = executor.submit(jrqTask);
			}
			if (caFuture != null) {
				sasvList = caFuture.get();
			}
			if (provFuture != null) {
				pvsList = provFuture.get();
			}
			if (jrFuture != null) {
				jrvsList = jrFuture.get();
			}
		} else {
			if (caqTask != null) {
				sasvList = caqTask.call();
			}
			if (pqTask != null) {
				pvsList = pqTask.call();
			}
			if (jrqTask != null) {
				jrvsList = jrqTask.call();
			}
		}

		if (aqi != null) {
			totQueryPartSize += aqi.getQueryPartSize();
		}
		if (pqi != null) {
			totQueryPartSize += pqi.getQueryPartSize();
		}
		if (jrqi != null) {
			totQueryPartSize += jrqi.getQueryPartSize();
		}

		// aggregate results
		/*
		 * if (pvsList != null) {
		 * System.out.println("Provenance Search Results"); for
		 * (JobProvenanceValueSummary jpvs : pvsList) {
		 * System.out.println(jpvs); }
		 * System.out.println("======================\n"); }
		 */

		List<CBFQueryResultSummary> qsList = null;

		if (sasvList != null) {
			qsList = prepareQSList(sasvList);
		}
		if (pvsList != null) {
			qsList = prepQsList4Provenance(pvsList, qsList);
		}
		if (jrvsList != null) {
			qsList = prepQsList4JobResults(jrvsList, qsList);
		}
		if (qsList == null) {
			qsList = new ArrayList<CBFQueryResultSummary>(0);
		}

		// second pass
		if (!qsList.isEmpty() && totQueryPartSize > 1) {
			CBFSecondaryFilter secFilter = new CBFSecondaryFilter(aqi, pqi,
					jrqi, this.synonymInfoList);
			secFilter.buildFilter();

			for (Iterator<CBFQueryResultSummary> it = qsList.iterator(); it
					.hasNext();) {
				CBFQueryResultSummary qs = it.next();
				//FIXME 
				if (!secFilter.isSatisfied2(qs)) {
					it.remove();
				}
			}
		}

		return qsList;
	}

	List<CBFQueryResultSummary> prepareQSList(
			List<SubjectAsScoreValueSummary> summaryList) throws Exception {
		Map<String, AssessmentResultSummary> arsMap = new LinkedHashMap<String, AssessmentResultSummary>(
				17);
		Map<Integer, Experiment> expIDMap = new HashMap<Integer, Experiment>();
		for (Experiment exp : expMap.values()) {
			expIDMap.put(exp.getUniqueid().intValue(), exp);
		}
		if (asIDMap == null) {
			asIDMap = new HashMap<Integer, Assessment>();
			List<Assessment> assessments = dbCache.getAssessments(ui, false);
			for (Assessment as : assessments) {
				asIDMap.put(as.getUniqueid().intValue(), as);
			}
		}
		for (SubjectAsScoreValueSummary sasvs : summaryList) {
			String key = AssessmentResultSummary.getKey(sasvs.getSubjectID(),
					sasvs.getExperimentID(), sasvs.getSiteID());
			AssessmentResultSummary ars = arsMap.get(key);
			int expID = sasvs.getExperimentID();
			int visitID = sasvs.getVisitID();
			int segmentID = sasvs.getSegmentID();
			String siteID = sasvs.getSiteID();
			int asID = GenUtils.toInt(sasvs.getAssessmentID(), -1);
			Assertion.assertTrue(asID != -1);
			if (ars == null) {
				Experiment experiment = expIDMap.get(sasvs.getExperimentID());
				if (experiment == null) {
					System.out
							.println("no experiment with the sasvs.getExperimentID:"
									+ sasvs.getExperimentID());
					continue;
				}
				String expName = expIDMap.get(sasvs.getExperimentID())
						.getName();
				ars = new AssessmentResultSummary(sasvs.getSubjectID(),
						String.valueOf(sasvs.getExperimentID()), expName,
						sasvs.getSiteID(), sasvs.getTimeStamp());
				arsMap.put(key, ars);
			}
			VisitSegAsResultValues vsarv = ars.getOrAdd(visitID, segmentID,
					expID, siteID);
			vsarv.setTimeStamp(sasvs.getTimeStamp());
			String asName = asIDMap.get(asID).getName();
			ASValueKey avKey = new ASValueKey(asID, asName,
					sasvs.getScoreName());

			vsarv.add(avKey, sasvs);
		}
		List<CBFQueryResultSummary> qsList = new ArrayList<CBFQueryResultSummary>(
				arsMap.size());
		for (AssessmentResultSummary ars : arsMap.values()) {
			CBFQueryResultSummary qs = new CBFQueryResultSummary(
					ars.getSubjectID(), ars.getExpID());
			qs.setArs(ars);
			qsList.add(qs);
		}
		return qsList;
	}

	List<CBFQueryResultSummary> prepQsList4JobResults(
			List<JobResultValueSummary> jrvsList,
			List<CBFQueryResultSummary> qsList) throws Exception {
		Map<String, JobResultResultSummary> jrsMap = new LinkedHashMap<String, JobResultResultSummary>();
		Map<Integer, Experiment> expIDMap = new HashMap<Integer, Experiment>();
		for (Experiment exp : expMap.values()) {
			expIDMap.put(exp.getUniqueid().intValue(), exp);
		}
		for (JobResultValueSummary jvs : jrvsList) {
			String key = jvs.getKey();
			JobResultResultSummary jrs = jrsMap.get(key);
			int expID = jvs.getExperimentID();
			int visitID = jvs.getVisitID();
			int resultGroupID = jvs.getGroupID();
			if (jrs == null) {
				String expName = expIDMap.get(jvs.getExperimentID()).getName();
				jrs = new JobResultResultSummary(jvs.getSubjectID(),
						String.valueOf(expID), expName);
				jrsMap.put(key, jrs);
			}
			VisitJobResultValues vjrv = jrs.getOrAdd(visitID, expID,
					resultGroupID);
			vjrv.add(jvs);
		}

		if (qsList != null) {
			Map<String, CBFQueryResultSummary> map = new HashMap<String, CBFQueryResultSummary>();
			for (CBFQueryResultSummary qs : qsList) {
				String key = null;
				if (qs.getArs() != null) {
					AssessmentResultSummary arso = qs.getArs();
					key = arso.getSubjectID() + ":" + arso.getExpID();
				} else if (qs.getPrs() != null) {
					ProvenanceResultSummary prs = qs.getPrs();
					key = prs.getSubjectID() + ":" + prs.getExpID();
				}
				map.put(key, qs);
			}
			for (JobResultResultSummary jrs : jrsMap.values()) {
				String key = jrs.getSubjectID() + ":" + jrs.getExpID();
				CBFQueryResultSummary qs = map.get(key);
				if (qs != null) {
					qs.setJrs(jrs);
				} else {
					qs = new CBFQueryResultSummary(jrs.getSubjectID(),
							jrs.getExpID());
					qs.setJrs(jrs);
					qsList.add(qs);
				}
			}
			return qsList;
		} else {
			List<CBFQueryResultSummary> newQsList = new ArrayList<CBFQueryResultSummary>(
					jrsMap.size());
			for (JobResultResultSummary jrs : jrsMap.values()) {
				CBFQueryResultSummary qs = new CBFQueryResultSummary(
						jrs.getSubjectID(), jrs.getExpID());
				qs.setJrs(jrs);
				newQsList.add(qs);
			}
			return newQsList;
		}
	}

	List<CBFQueryResultSummary> prepQsList4Provenance(
			List<JobProvenanceValueSummary> pvsList,
			List<CBFQueryResultSummary> qsList) throws Exception {
		Map<String, ProvenanceResultSummary> prsMap = new LinkedHashMap<String, ProvenanceResultSummary>();
		Map<Integer, Experiment> expIDMap = new HashMap<Integer, Experiment>();
		for (Experiment exp : expMap.values()) {
			expIDMap.put(exp.getUniqueid().intValue(), exp);
		}

		for (JobProvenanceValueSummary pvs : pvsList) {
			String key = pvs.getKey();
			ProvenanceResultSummary prs = prsMap.get(key);
			int expID = pvs.getExperimentID();
			int visitID = pvs.getVisitID();
			int jobUniqueID = pvs.getJobUniqueId();
			if (prs == null) {
				String expName = expIDMap.get(expID).getName();
				prs = new ProvenanceResultSummary(pvs.getSubjectID(),
						String.valueOf(expID), expName);
				prsMap.put(key, prs);
			}

			VisitProvResultValues vprv = prs.getOrAdd(visitID, expID,
					jobUniqueID);
			vprv.add(pvs);
		}

		if (qsList != null) {
			Map<String, CBFQueryResultSummary> map = new HashMap<String, CBFQueryResultSummary>();
			for (CBFQueryResultSummary qs : qsList) {
				AssessmentResultSummary arso = qs.getArs();
				String key = arso.getSubjectID() + ":" + arso.getExpID();
				map.put(key, qs);
			}
			for (ProvenanceResultSummary prs : prsMap.values()) {
				String key = prs.getSubjectID() + ":" + prs.getExpID();
				CBFQueryResultSummary qs = map.get(key);
				if (qs != null) {
					qs.setPrs(prs);
				} else {
					qs = new CBFQueryResultSummary(prs.getSubjectID(),
							prs.getExpID());
					qs.setPrs(prs);
					qsList.add(qs);
				}
			}
			return qsList;
		} else {
			List<CBFQueryResultSummary> newQsList = new ArrayList<CBFQueryResultSummary>(
					prsMap.size());
			for (ProvenanceResultSummary prs : prsMap.values()) {
				CBFQueryResultSummary qs = new CBFQueryResultSummary(
						prs.getSubjectID(), prs.getExpID());
				qs.setPrs(prs);
				newQsList.add(qs);
			}
			return newQsList;
		}

	}

	public static Map<String, Experiment> toMap(List<Experiment> experiments) {
		Map<String, Experiment> map = new HashMap<String, Experiment>();
		for (Iterator<Experiment> iter = experiments.iterator(); iter.hasNext();) {
			Experiment exp = iter.next();
			map.put(exp.getUniqueid().toString(), exp);
		}
		return map;
	}

	List<Integer> prepareExperimentIDs() {
		List<Integer> experimentIDs = new ArrayList<Integer>(10);
		for (Experiment exp : this.expMap.values()) {
			experimentIDs.add(exp.getUniqueid().intValue());
		}
		return experimentIDs;
	}

	class CAQueryWorker implements Callable<List<SubjectAsScoreValueSummary>> {
		private AssessmentQueryInfo aqi;
		private String dbID;
		private UserInfo ui;
		private List<SubjectAsScoreValueSummary> summaryList;
		private String primarySiteID;

		public CAQueryWorker(AssessmentQueryInfo aqi, String dbID, UserInfo ui,
				String primarySiteID) {
			this.aqi = aqi;
			this.dbID = dbID;
			this.ui = ui;
			this.primarySiteID = primarySiteID;
		}

		@Override
		public List<SubjectAsScoreValueSummary> call() throws Exception {
			try {
				IAssessmentService asService = ServiceFactory
						.getAssessmentService(dbID);
				Operator root = aqi.prepareOpTree();
				List<Integer> experimentIDs = prepareExperimentIDs();
				if (experimentIDs.isEmpty()) {
					// no matching experiment for this site don't bother
					// querying
					return new ArrayList<SubjectAsScoreValueSummary>(0);
				}
				summaryList = asService.queryForScores(ui, root,
						aqi.getAsiList(), experimentIDs,
						Constants.SEGMENT_SCOPE, primarySiteID);

			} catch (Exception x) {
				log.error("CAQueryWorker", x);
				summaryList = null;
				throw x;
			}
			return summaryList;
		}
	}

	class JobResultsQueryWorker implements
			Callable<List<JobResultValueSummary>> {
		private JobResultQueryInfo jrqi;
		private List<JobResultValueSummary> summaryList;
		private String dbID;
		private UserInfo ui;

		public JobResultsQueryWorker(JobResultQueryInfo jrqi, String dbID,
				UserInfo ui) {
			this.jrqi = jrqi;
			this.dbID = dbID;
			this.ui = ui;
		}

		@Override
		public List<JobResultValueSummary> call() throws Exception {
			List<Integer> experimentIDs = prepareExperimentIDs();
			if (experimentIDs.isEmpty()) {
				// no matching experiment for this site don't bother
				// querying
				return new ArrayList<JobResultValueSummary>(0);
			}
			Operator root = jrqi.prepareOpTree();
			IJobResultService jrs = ServiceFactory.getJobResultService(dbID);
			this.summaryList = jrs.queryForJobResults(ui, root,
					jrqi.getJrgiList(), experimentIDs);
			return summaryList;
		}
	}// ;

	class ProvenanceQueryWorker implements
			Callable<List<JobProvenanceValueSummary>> {
		private JobProvenanceQueryInfo pqi;
		private List<JobProvenanceValueSummary> summaryList;
		private String dbID;
		private UserInfo ui;

		public ProvenanceQueryWorker(JobProvenanceQueryInfo pqi, String dbID,
				UserInfo ui) {
			super();
			this.pqi = pqi;
			this.dbID = dbID;
			this.ui = ui;
		}

		@Override
		public List<JobProvenanceValueSummary> call() throws Exception {
			List<Integer> experimentIDs = prepareExperimentIDs();
			if (experimentIDs.isEmpty()) {
				// no matching experiment for this site don't bother
				// querying
				return new ArrayList<JobProvenanceValueSummary>(0);
			}
			Operator root = pqi.prepareOpTree();
			IJobProvenanceService jps = ServiceFactory
					.getJobProvenanceService(dbID);

			this.summaryList = jps.queryForJobProvenance(ui, root,
					experimentIDs);
			return summaryList;
		}
	}// ;
}
