package clinical.web.common.query;

import gnu.trove.TIntHashSet;
import gnu.trove.TIntIterator;
import gnu.trove.TIntObjectHashMap;
import gnu.trove.TIntObjectIterator;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import clinical.utils.GenUtils;
import clinical.web.common.query.SearchPredicate.DateRange;
import clinical.web.common.query.SearchPredicate.Range;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.vo.AssessmentQueryInfo;
import clinical.web.vo.AssessmentResultSummary;
import clinical.web.vo.CBFQueryResultSummary;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamInfo;
import clinical.web.vo.JobProvenanceQueryInfo;
import clinical.web.vo.JobProvenanceValueSummary;
import clinical.web.vo.JobResultQueryInfo;
import clinical.web.vo.ProvenanceResultSummary;
import clinical.web.vo.SynonymInfo;
import clinical.web.vo.ProvenanceResultSummary.VisitProvResultValues;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.vo.VisitSegAsResultValues;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class CBFSecondaryFilter {
	private AssessmentQueryInfo aqi;
	private JobProvenanceQueryInfo pqi;
	private JobResultQueryInfo jrqi;
	private Map<String, SynonymInfo> synonymMap = new HashMap<String, SynonymInfo>();
	private List<SearchPredicateWrapper> spwList = new ArrayList<SearchPredicateWrapper>(
			10);

	public static enum SearchPredicateGroup {
		ASSESSMENT, PROVENANCE, JOBRESULT
	}

	public static enum FilterOptions {
		REMOVE_NON_MATCHING_VISITS, LEAVE_NON_MATCHING_VISITS
	}

	public CBFSecondaryFilter(AssessmentQueryInfo aqi,
			JobProvenanceQueryInfo pqi, JobResultQueryInfo jrqi,
			List<SynonymInfo> siList) {
		this.aqi = aqi;
		this.pqi = pqi;
		this.jrqi = jrqi;
		for (SynonymInfo si : siList) {
			AssessmentSelectionInfo theASI = null;

			for (AssessmentSelectionInfo asi : aqi.getAsiList()) {
				if (asi.getName().equals(si.getAssessmentName())) {
					theASI = asi;
					break;
				}
			}
			if (theASI != null) {
				String key = prepSynonymInfoKey(theASI.getAssessmentID(),
						si.getScoreName());
				synonymMap.put(key, si);
			}
		}
	}

	private String prepSynonymInfoKey(BigDecimal asID, String scoreName) {
		StringBuilder sb = new StringBuilder();
		sb.append(scoreName).append(':').append(asID);
		return sb.toString();
	}

	public void buildFilter() {
		if (aqi != null) {
			for (QueryPartInfo qp : aqi.getQpiList()) {
				SearchPredicate sp = qp.toSearchPredicate();
				int partIdx = qp.getPartIdx();
				int logicalConnective = getSelectedConnector(qp.getConnector());
				SearchPredicateWrapper spw = new SearchPredicateWrapper(sp,
						partIdx, SearchPredicateGroup.ASSESSMENT,
						logicalConnective);
				spwList.add(spw);
			}
		}
		if (pqi != null) {
			for (JobProvQueryPartInfo qp : pqi.getQpiList()) {
				SearchPredicate sp = qp.toSearchPredicate();
				int logicalConnective = getSelectedConnector(qp.getConnector());
				SearchPredicateWrapper spw = new SearchPredicateWrapper(sp,
						qp.getPartIdx(), SearchPredicateGroup.PROVENANCE,
						logicalConnective);
				spwList.add(spw);
			}
		}
		if (jrqi != null) {
			for (JobResultQueryPartInfo qp : jrqi.getQpiList()) {
				SearchPredicate sp = qp.toSearchPredicate();
				int logicalConnective = getSelectedConnector(qp.getConnector());
				SearchPredicateWrapper spw = new SearchPredicateWrapper(sp,
						qp.getPartIdx(), SearchPredicateGroup.JOBRESULT,
						logicalConnective);
				spwList.add(spw);
			}
		}
		Collections.sort(spwList, new Comparator<SearchPredicateWrapper>() {
			@Override
			public int compare(SearchPredicateWrapper o1,
					SearchPredicateWrapper o2) {
				return o1.partIdx - o2.partIdx;
			}
		});
	}

	public boolean isSatisfied2(CBFQueryResultSummary qs) {
		boolean ok = false;
		if (spwList.isEmpty()) {
			return true;
		}
		List<VisitScope> visitScopes = buildVisitScopes(qs);
		List<VisitScope> matchedVsList = new ArrayList<VisitScope>(
				visitScopes.size());
		List<VisitScope> unmatchedVsList = new ArrayList<VisitScope>(
				visitScopes.size());

		for (VisitScope vs : visitScopes) {
			for (int i = 0; i < spwList.size(); i++) {
				SearchPredicateWrapper spw = spwList.get(i);
				boolean satisfied = false;
				satisfied = vs.isSatisfied(spw, qs);
				if (i == 0) {
					ok = satisfied;
				} else {
					int logicalConnective = spwList.get(i - 1).logicalConnective;
					if (logicalConnective == Operator.AND) {
						ok &= satisfied;
					} else {
						ok |= satisfied;
					}
				}
			}
			if (ok) {
				matchedVsList.add(vs);
			} else {
				unmatchedVsList.add(vs);
			}
		}
		if (matchedVsList.isEmpty()) {
			return false;
		}
		ok = !matchedVsList.isEmpty();

		// FIXME

		PredicateResult[] spResults = new PredicateResult[spwList.size()];
		for (int i = 0; i < spwList.size(); i++) {
			spResults[i] = new PredicateResult();
		}
		for (VisitScope vs : matchedVsList) {
			PredicateResult overallResult = new PredicateResult();
			for (int i = 0; i < spwList.size(); i++) {
				spResults[i].clear();
				SearchPredicateWrapper spw = spwList.get(i);
				vs.isSatisfied(spw, qs, spResults[i]);
				if (i == 0) {
					overallResult
							.initializeFromOtherVisitIdSet(spResults[i].visitIdSet);
					if (spw.group == SearchPredicateGroup.PROVENANCE) {
						overallResult
								.initializeFromOtherJobUniqueidSet(spResults[i].jobUniqueIdSet);
					}

				} else {
					int logicalConnective = spwList.get(i - 1).logicalConnective;
					if (logicalConnective == Operator.AND) {
						PredicateResult.intersect(spResults[i].visitIdSet,
								overallResult.visitIdSet);
						if (spw.group == SearchPredicateGroup.PROVENANCE) {
							if (spResults[i].isJobSetInitialized()) {
								PredicateResult.intersect(
										spResults[i].jobUniqueIdSet,
										overallResult.jobUniqueIdSet);
							} else {
								overallResult
										.initializeFromOtherJobUniqueidSet(spResults[i].jobUniqueIdSet);
							}
						}
					} else {
						PredicateResult.union(spResults[i].visitIdSet,
								overallResult.visitIdSet);
						if (spw.group == SearchPredicateGroup.PROVENANCE) {
							if (spResults[i].isJobSetInitialized()) {
								PredicateResult.union(
										spResults[i].jobUniqueIdSet,
										overallResult.jobUniqueIdSet);
							} else {
								overallResult
										.initializeFromOtherJobUniqueidSet(spResults[i].jobUniqueIdSet);
							}
						}
					}
				}
			}
			if (overallResult.visitIdSet.isEmpty()) {
				stripMatchingVisit(qs, overallResult, vs.visitID);
			} else {
				stripUnmatchedJobs(qs, overallResult, vs.visitID);
			}
		}
		// remove any visit not in matchedVsList
		removeUnmatchedVisits(qs, unmatchedVsList);
		return ok;
	}

	private void removeUnmatchedVisits(CBFQueryResultSummary qs,
			List<VisitScope> unmatchedVsList) {
		TIntHashSet visitIDSet = new TIntHashSet();
		for (VisitScope vs : unmatchedVsList) {
			visitIDSet.add(vs.visitID);
		}
		if (qs.getArs() != null) {

			List<String> keysToRemove = new ArrayList<String>(1);
			for (String key : qs.getArs().getVsarvMap().keySet()) {
				VisitSegAsResultValues vsarv = qs.getArs().getVsarvMap()
						.get(key);
				if (visitIDSet.contains(vsarv.getVisitID())) {
					keysToRemove.add(key);
				}
			}
			if (!keysToRemove.isEmpty()) {
				for (String key : keysToRemove) {
					qs.getArs().getVsarvMap().remove(key);
				}
			}
		}
		if (qs.getPrs() != null) {
			Map<String, VisitProvResultValues> vprvMap = qs.getPrs()
					.getVprvMap();
			List<String> keysToRemove = new ArrayList<String>(1);
			for (String key : vprvMap.keySet()) {
				VisitProvResultValues vprv = vprvMap.get(key);
				if (visitIDSet.contains(vprv.getVisitID())) {
					keysToRemove.add(key);
				}
			}
			if (!keysToRemove.isEmpty()) {
				for (String key : keysToRemove) {
					vprvMap.remove(key);
				}
			}
		}
	}

	private void stripUnmatchedJobs(CBFQueryResultSummary qs,
			PredicateResult overallResult, int visitID) {
		if (qs.getPrs() != null) {
			Map<String, VisitProvResultValues> vprvMap = qs.getPrs()
					.getVprvMap();
			List<String> keysToRemove = new ArrayList<String>(1);
			for (String key : vprvMap.keySet()) {
				VisitProvResultValues vprv = vprvMap.get(key);
				if (vprv.getVisitID() == visitID
						&& !overallResult.jobUniqueIdSet.contains(vprv
								.getJobUniqueID())) {
					keysToRemove.add(key);
				}
			}
			if (!keysToRemove.isEmpty()) {
				for (String key : keysToRemove) {
					vprvMap.remove(key);
				}
			}
		}

	}

	private void stripMatchingVisit(CBFQueryResultSummary qs,
			PredicateResult overallResult, int visitID) {
		if (qs.getArs() != null) {

			List<String> keysToRemove = new ArrayList<String>(1);
			for (String key : qs.getArs().getVsarvMap().keySet()) {
				VisitSegAsResultValues vsarv = qs.getArs().getVsarvMap()
						.get(key);
				if (vsarv.getVisitID() == visitID) {
					keysToRemove.add(key);
				}
			}
			if (!keysToRemove.isEmpty()) {
				for (String key : keysToRemove) {
					qs.getArs().getVsarvMap().remove(key);
				}
			}
		}
		if (qs.getPrs() != null) {
			Map<String, VisitProvResultValues> vprvMap = qs.getPrs()
					.getVprvMap();
			List<String> keysToRemove = new ArrayList<String>(1);
			for (String key : vprvMap.keySet()) {
				VisitProvResultValues vprv = vprvMap.get(key);
				if (vprv.getVisitID() == visitID
						&& !overallResult.jobUniqueIdSet.contains(vprv
								.getJobUniqueID())) {
					keysToRemove.add(key);
				}
			}
			if (!keysToRemove.isEmpty()) {
				for (String key : keysToRemove) {
					vprvMap.remove(key);
				}
			}
		}
	}

	public boolean isSatisfied(CBFQueryResultSummary qs) {
		boolean ok = false;
		if (spwList.isEmpty()) {
			return true;
		}

		PredicateResult[] spResults = new PredicateResult[spwList.size()];
		for (int i = 0; i < spwList.size(); i++) {
			spResults[i] = new PredicateResult();
			SearchPredicateWrapper spw = spwList.get(i);
			boolean satisfied = false;
			if (spw.group == SearchPredicateGroup.ASSESSMENT) {
				if (qs.getArs() == null) {
					return false;
				}
				satisfied = isSatisfied(qs.getArs(), spw, spResults[i]);
			} else if (spw.group == SearchPredicateGroup.PROVENANCE) {
				if (qs.getPrs() == null) {
					return false;
				}
				satisfied = isSatisfied(qs.getPrs(), spw);
			} else if (spw.group == SearchPredicateGroup.JOBRESULT) {
				// TODO
			}
			if (i == 0) {
				ok = satisfied;
			} else {
				int logicalConnective = spwList.get(i - 1).logicalConnective;
				if (logicalConnective == Operator.AND) {
					ok &= satisfied;
				} else {
					ok |= satisfied;
				}
			}
		}

		if (ok) {
			// TODO also jobUniqueIDs
			PredicateResult overallResult = new PredicateResult();
			for (int i = 0; i < spwList.size(); i++) {
				if (i == 0) {
					overallResult
							.initializeFromOtherVisitIdSet(spResults[i].visitIdSet);
				} else {
					int logicalConnective = spwList.get(i - 1).logicalConnective;
					if (logicalConnective == Operator.AND) {
						PredicateResult.intersect(spResults[i].visitIdSet,
								overallResult.visitIdSet);
					} else {
						PredicateResult.union(spResults[i].visitIdSet,
								overallResult.visitIdSet);
					}
				}
			}
			if (overallResult.visitIdSet.isEmpty()) {
				return false;
			} else {
				// remove all non matching visits from the CBFQueryResultSummary
				stripNonMatchingVisits(qs, overallResult);
			}

		}
		return ok;
	}

	private void stripNonMatchingVisits(CBFQueryResultSummary qs,
			PredicateResult overallResult) {
		if (qs.getArs() != null) {
			List<String> keysToRemove = new ArrayList<String>(qs.getArs()
					.getVsarvMap().size());
			for (String key : qs.getArs().getVsarvMap().keySet()) {
				VisitSegAsResultValues vsarv = qs.getArs().getVsarvMap()
						.get(key);
				if (!overallResult.visitIdSet.contains(vsarv.getVisitID())) {
					keysToRemove.add(key);
				}
			}
			if (!keysToRemove.isEmpty()) {
				for (String key : keysToRemove) {
					qs.getArs().getVsarvMap().remove(key);
				}
			}
		}
		if (qs.getPrs() != null) {
			Map<String, VisitProvResultValues> vprvMap = qs.getPrs()
					.getVprvMap();
			List<String> keysToRemove = new ArrayList<String>(vprvMap.size());
			for (String key : vprvMap.keySet()) {
				VisitProvResultValues vprv = vprvMap.get(key);
				if (!overallResult.visitIdSet.contains(vprv.getVisitID())) {
					keysToRemove.add(key);
				}
			}
			if (!keysToRemove.isEmpty()) {
				for (String key : keysToRemove) {
					vprvMap.remove(key);
				}
			}
		}
	}

	private List<VisitScope> buildVisitScopes(CBFQueryResultSummary qs) {
		TIntObjectHashMap map = new TIntObjectHashMap(17);
		if (qs.getArs() != null) {
			AssessmentResultSummary ars = qs.getArs();
			for (VisitSegAsResultValues vsarv : ars.getVsarvList()) {
				VisitScope vs = new VisitScope(vsarv);
				map.put(vsarv.getVisitID(), vs);
			}
		}
		if (qs.getPrs() != null) {
			ProvenanceResultSummary prs = qs.getPrs();
			List<VisitProvResultValues> vprvList = new ArrayList<VisitProvResultValues>(
					prs.getVprvMap().values());
			// TIntHashSet uniqVisitIDSet = new TIntHashSet();
			for (VisitProvResultValues vprv : vprvList) {
				VisitScope vs = (VisitScope) map.get(vprv.getVisitID());
				if (vs == null) {
					vs = new VisitScope(vprv.getVisitID());
					map.put(vprv.getVisitID(), vs);
				}
			}
			for (TIntObjectIterator it = map.iterator(); it.hasNext();) {
				it.advance();
				VisitScope vs = (VisitScope) it.value();
				vs.addVisitProvResultValues(vprvList);
			}
		}
		if (qs.getJrs() != null) {
			throw new RuntimeException("No derived data query support yet!");
		}

		List<VisitScope> vsList = new ArrayList<VisitScope>(map.size());

		for (TIntObjectIterator it = map.iterator(); it.hasNext();) {
			it.advance();
			VisitScope vs = (VisitScope) it.value();
			vsList.add(vs);
		}
		return vsList;
	}

	private class VisitScope {
		int visitID;
		VisitSegAsResultValues vsarv;
		List<VisitProvResultValues> vprvList = new ArrayList<VisitProvResultValues>(
				2);

		public VisitScope(int visitID) {
			this.visitID = visitID;
		}

		public VisitScope(VisitSegAsResultValues vsarv) {
			this.vsarv = vsarv;
			this.visitID = vsarv.getVisitID();
		}

		public void addVisitProvResultValues(
				List<VisitProvResultValues> fullVprvList) {
			for (VisitProvResultValues vprv : fullVprvList) {
				if (vprv.getVisitID() == visitID) {
					vprvList.add(vprv);
				}
			}
		}

		boolean isSatisfied(SearchPredicateWrapper spw, CBFQueryResultSummary qs) {
			boolean ok = false;
			if (vsarv != null && spw.group == SearchPredicateGroup.ASSESSMENT) {
				ok |= CBFSecondaryFilter.this.isSatisfied(qs.getArs(), spw,
						vsarv);
			}
			if (!vprvList.isEmpty()
					&& spw.group == SearchPredicateGroup.PROVENANCE) {
				for (VisitProvResultValues vprv : vprvList) {
					ok |= CBFSecondaryFilter.this.isSatisfied(qs.getPrs(), spw,
							vprv);
				}
			}
			// TODO derived data support

			return ok;
		}

		boolean isSatisfied(SearchPredicateWrapper spw,
				CBFQueryResultSummary qs, PredicateResult pr) {
			boolean ok = false;
			if (vsarv != null && spw.group == SearchPredicateGroup.ASSESSMENT) {
				boolean flag = CBFSecondaryFilter.this.isSatisfied(qs.getArs(),
						spw, vsarv);
				if (flag) {
					pr.addVisitId(vsarv.getVisitID());
				}
				ok |= flag;
			}
			if (!vprvList.isEmpty()
					&& spw.group == SearchPredicateGroup.PROVENANCE) {
				for (VisitProvResultValues vprv : vprvList) {
					boolean flag = CBFSecondaryFilter.this.isSatisfied(
							qs.getPrs(), spw, vprv);
					if (flag) {
						pr.addVisitId(vprv.getVisitID());
						pr.addJobUniqueId(vprv.getJobUniqueID());
					}
					ok |= flag;
				}
			}
			// TODO derived data support

			return ok;
		}
	}// ;

	private boolean isSatisfied(ProvenanceResultSummary prs,
			SearchPredicateWrapper spw, VisitProvResultValues vprv) {
		JobProvenanceParamInfo jppi = (JobProvenanceParamInfo) spw.sp
				.getAttribute();
		String jpName = jppi.getName();
		for (JobProvenanceValueSummary jpvs : vprv.getJpvsMap().values()) {
			if (jpvs.getName().equals(jpName)) {
				String strValue = jpvs.getValue();
				Object value = QueryUtils.convertToType(strValue,
						spw.sp.getType());
				return isPredicateSatisfied(spw.sp, value, null);
			}
		}
		return false;
	}

	private boolean isSatisfied(ProvenanceResultSummary prs,
			SearchPredicateWrapper spw) {
		JobProvenanceParamInfo jppi = (JobProvenanceParamInfo) spw.sp
				.getAttribute();
		String jpName = jppi.getName();
		// find a visit that satisfies the predicate
		boolean flag = false;
		for (VisitProvResultValues vprv : prs.getVprvMap().values()) {
			for (JobProvenanceValueSummary jpvs : vprv.getJpvsMap().values()) {
				if (jpvs.getName().equals(jpName)) {
					String strValue = jpvs.getValue();
					Object value = QueryUtils.convertToType(strValue,
							spw.sp.getType());
					flag |= isPredicateSatisfied(spw.sp, value, null);
					break;
				}
			}
		}
		return flag;
	}

	private static class PredicateResult {
		private TIntHashSet visitIdSet = new TIntHashSet(11);
		private TIntHashSet jobUniqueIdSet = new TIntHashSet(11);
		private boolean jobSetInitialized = false;

		void addVisitId(int visitId) {
			visitIdSet.add(visitId);
		}

		public void clear() {
			visitIdSet.clear();
			jobUniqueIdSet.clear();
			jobSetInitialized = false;
		}

		void initializeFromOtherJobUniqueidSet(TIntHashSet otherJUIDSet) {
			jobUniqueIdSet.clear();
			for (TIntIterator it = otherJUIDSet.iterator(); it.hasNext();) {
				jobUniqueIdSet.add(it.next());
			}
			jobSetInitialized = true;
		}

		void initializeFromOtherVisitIdSet(TIntHashSet otherVisitIdSet) {
			visitIdSet.clear();
			for (TIntIterator it = otherVisitIdSet.iterator(); it.hasNext();) {
				visitIdSet.add(it.next());
			}
		}

		void addJobUniqueId(int jobUniqueId) {
			jobUniqueIdSet.add(jobUniqueId);
		}

		static void union(TIntHashSet first, TIntHashSet second) {
			for (TIntIterator it = first.iterator(); it.hasNext();) {
				second.add(it.next());
			}
		}

		static void intersect(TIntHashSet first, TIntHashSet second) {
			for (TIntIterator it = first.iterator(); it.hasNext();) {
				int key = it.next();
				if (!second.contains(key)) {
					second.remove(key);
				}
			}
		}

		public boolean isJobSetInitialized() {
			return jobSetInitialized;
		}
	}// ;

	private boolean isSatisfied(AssessmentResultSummary ars,
			SearchPredicateWrapper spw, PredicateResult pr) {
		AsScoreInfo scoreInfo = (AsScoreInfo) spw.sp.getAttribute();
		String asIDStr = scoreInfo.getAssessmentID().toString();
		// find a visit that satisfies the predicate
		boolean flag = false;
		for (VisitSegAsResultValues vsarv : ars.getVsarvList()) {
			Collection<SubjectAsScoreValueSummary> values = vsarv.getSasvsMap()
					.values();
			for (SubjectAsScoreValueSummary sasvs : values) {
				if (sasvs.getScoreName().equals(scoreInfo.getName())) {
					if (sasvs.getAssessmentID().equals(asIDStr)) {
						Object value = sasvs.getValue();
						SynonymInfo si = hasSynonyms(scoreInfo.getName(),
								sasvs.getAssessmentID(), value);
						boolean ok = isPredicateSatisfied(spw.sp, value, si);
						if (ok) {
							pr.addVisitId(sasvs.getVisitID());
						}
						flag |= ok;
						break;
					}
				}
			}
		}
		return flag;
	}

	private SynonymInfo hasSynonyms(String scoreName, String asID, Object value) {
		String key = prepSynonymInfoKey(new BigDecimal(asID), scoreName);
		SynonymInfo si = synonymMap.get(key);
		if (si != null) {
			if (!si.matchesAny((value.toString()))) {
				return null;
			}
		}
		return si;
	}

	private boolean isSatisfied(AssessmentResultSummary ars,
			SearchPredicateWrapper spw, VisitSegAsResultValues vsarv) {
		AsScoreInfo scoreInfo = (AsScoreInfo) spw.sp.getAttribute();
		String asIDStr = scoreInfo.getAssessmentID().toString();
		Collection<SubjectAsScoreValueSummary> values = vsarv.getSasvsMap()
				.values();
		for (SubjectAsScoreValueSummary sasvs : values) {
			if (sasvs.getScoreName().equals(scoreInfo.getName())) {
				if (sasvs.getAssessmentID().equals(asIDStr)) {
					Object value = sasvs.getValue();
					// handle also synonymns
					SynonymInfo si = hasSynonyms(scoreInfo.getName(),
							sasvs.getAssessmentID(), value);
					return isPredicateSatisfied(spw.sp, value,si);
				}
			}
		}
		return false;
	}

	private boolean isPredicateSatisfied(SearchPredicate sp, Object value,
			SynonymInfo si) {
		Object predicateValue = sp.getValue();
		if (predicateValue instanceof SearchPredicate.Range) {
			SearchPredicate.Range range = (Range) predicateValue;
			Number nv = (Number) value;
			return (nv.doubleValue() >= range.getLowBound().doubleValue() && nv
					.doubleValue() <= range.getUppBound().doubleValue());
		} else if (predicateValue instanceof SearchPredicate.DateRange) {
			SearchPredicate.DateRange dateRange = (DateRange) predicateValue;
			Date dv = (Date) value;
			return (dv.compareTo(dateRange.getLowBound()) >= 0 && dv
					.compareTo(dateRange.getUppBound()) <= 0);
		} else {
			String strValue = predicateValue.toString();
			if (strValue.equals("*")) {
				return true;
			}
			// synonym expansion
			if (si == null) {
				return compare(sp, value, predicateValue);
			} else {
				for (String synonym : si.getSynonymArray()) {
					if (compare(sp, value, synonym)) {
						return true;
					}
				}
				return false;
			}
		}
	}

	private boolean compare(SearchPredicate sp, Object value,
			Object predicateValue) {
		if (value == null) {
			return false;
		}
		switch (sp.getType()) {
		case SearchPredicate.STRING:
			String s = (String) value;
			String pv = (String) predicateValue;
			switch (sp.getOperator()) {
			case SearchPredicate.EQUAL:
				return s.equals(pv);
			case SearchPredicate.STARTS_WITH:
				return s.startsWith(pv);
			case SearchPredicate.ENDS_WITH:
				return s.endsWith(pv);
			}
		case SearchPredicate.INTEGER:
		case SearchPredicate.FLOAT:
			Number n = toNumber(value);
			Number npv = (Number) predicateValue;
			switch (sp.getOperator()) {
			case SearchPredicate.EQUAL:
				if (value instanceof Integer) {
					return n.intValue() == npv.intValue();
				} else {
					return n.doubleValue() == npv.doubleValue();
				}
			case SearchPredicate.GREATER:
				return n.doubleValue() > npv.doubleValue();
			case SearchPredicate.LESS:
				return n.doubleValue() < npv.doubleValue();
			case SearchPredicate.GREATER_EQUAL:
				return n.doubleValue() >= npv.doubleValue();
			case SearchPredicate.LESS_EQUAL:
				return n.doubleValue() <= npv.doubleValue();
			case SearchPredicate.NOT_EQUAL:
				if (value instanceof Integer) {
					return n.intValue() != npv.intValue();
				} else {
					return n.doubleValue() != npv.doubleValue();
				}
			}
		case SearchPredicate.BOOLEAN:
			Boolean bv = (Boolean) value;
			Boolean bpv = (Boolean) predicateValue;
			return bv == bpv;
		case SearchPredicate.DATE:
			Date dv = (Date) value;
			Date dpv = (Date) predicateValue;
			if (sp.getOperator() == SearchPredicate.GREATER_EQUAL) {
				return dv.compareTo(dpv) >= 0;
			} else if (sp.getOperator() == SearchPredicate.LESS_EQUAL) {
				return dv.compareTo(dpv) <= 0;
			}
		}
		return false;
	}

	public static Number toNumber(Object value) {
		if (value instanceof Number) {
			return (Number) value;
		}
		return GenUtils.toNumber(value.toString());
	}

	public static int getSelectedConnector(String connector) {
		return (connector.equals("AND")) ? Operator.AND : Operator.OR;
	}

	public static class SearchPredicateWrapper {
		private SearchPredicate sp;
		private int partIdx;
		private SearchPredicateGroup group;
		private int logicalConnective;

		public SearchPredicateWrapper(SearchPredicate sp, int partIdx,
				SearchPredicateGroup group, int logicalConnective) {
			super();
			this.sp = sp;
			this.partIdx = partIdx;
			this.group = group;
			this.logicalConnective = logicalConnective;
		}
	}

}
