package clinical.web.common.query;

import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.web.ISQLDialect;
import clinical.web.vo.JobResultGroupInfo;
import clinical.web.vo.JobResultTypeInfo;
import clinical.web.vo.JobResultTypeInfo.ResultType;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class JobResultQueryBuilder extends AbstractQueryBuilder implements
		ISearchCriteriaVisitor {
	private Map<JobResultTypeInfo, JobResultGroupInfo> resultVarMap = new HashMap<JobResultTypeInfo, JobResultGroupInfo>(
			17);
	private List<Integer> experimentIDs;
	private StringBuilder query;
	private ISQLDialect sqlDialect;
	private static Log log = LogFactory.getLog(JobResultQueryBuilder.class);

	public JobResultQueryBuilder(ISQLDialect sqlDialect,
			List<JobResultGroupInfo> jrgiList, List<Integer> experimentIDs) {
		this.sqlDialect = sqlDialect;
		this.experimentIDs = experimentIDs;
		// FIXME assumption a job result variable is assumed unique and assigned
		// to a single job_result_group
		for (JobResultGroupInfo jrgi : jrgiList) {
			for (JobResultTypeInfo jrti : jrgi.getResultTypes()) {
				resultVarMap.put(jrti, jrgi);
			}
		}
	}

	public String getQuery() {
		return query.toString();
	}

	@Override
	public void visit(Operator op) throws Exception {
		StringBuilder inQuery = new StringBuilder(768);
		prepareInnerQuery(inQuery, op);
		this.query = new StringBuilder(1024);
		prepareOuterQuery(query, inQuery.toString(), op);
	}

	protected void prepareOuterQuery(StringBuilder outq, String innerQuery,
			Operator op) throws Exception {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			Object o = sp.getAttribute();
			if (!(o instanceof JobResultTypeInfo)) {
				throw new RuntimeException(
						"Attribute needs to be of type JobResultTypeInfo");
			}
			JobResultTypeInfo jrti = (JobResultTypeInfo) o;
			JobResultGroupInfo jrgi = resultVarMap.get(jrti);
			outq.append(getOuterStaticPart(jrgi.getId(), jrti.getName()));
			outq.append(innerQuery);
			outq.append(")");
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			prepareOuterQuery(outq, innerQuery, lhs);
			Operator rhs = bop.getRhs();
			if (rhs != null) {
				outq.append("\nUNION ALL\n");
				log.debug("rhs=" + rhs);
				prepareOuterQuery(outq, innerQuery, rhs);
			}
		}
	}

	protected String getOuterStaticPart(Integer jobResultGroupId,
			String jobResultVarName) {
		StringBuilder buf = new StringBuilder(256);
		buf.append("select b.subjectid, a.value, a.name,");
		buf.append("b.job_group_id, b.exp_id, b.visit_id, b.modtime, c.result_type ");
		buf.append("from nc_job_result a, nc_stored_job_result_group b,");
		buf.append("nc_job_result_type where ");
		buf.append("a.stored_job_result_group_id = b.uniqueid and ");
		buf.append("a.result_type_id = c.uniqueid and b.job_group_id =");
		buf.append(jobResultGroupId).append(" and  a.name=");
		buf.append(sqlDialect.prepareString(jobResultVarName));

		buf.append(" and b.subjectid in (");

		return buf.toString();
	}

	protected String getInnerStaticPart(Integer jobResultGroupId) {
		StringBuilder buf = new StringBuilder(256);
		buf.append("select b.subjectid from ");
		buf.append("nc_job_result a, nc_stored_job_result_group b ");
		buf.append("where a.stored_job_result_group_id = b.uniqueid and ");
		buf.append("b.job_group_id = ").append(jobResultGroupId);
		buf.append(' ');
		buf.append(prepareExperimentQueryPart());
		return buf.toString();
	}

	protected String prepareExperimentQueryPart() {
		if (experimentIDs == null || experimentIDs.isEmpty()) {
			return "";
		}
		StringBuilder buf = new StringBuilder(80);
		buf.append(" and ");
		if (experimentIDs.size() == 1) {
			buf.append("b.exp_id = ").append((Integer) experimentIDs.get(0));
			buf.append(' ');
		} else {
			buf.append("b.exp_id in (");
			for (Iterator<Integer> iter = experimentIDs.iterator(); iter
					.hasNext();) {
				Integer experimentID = iter.next();
				buf.append(experimentID);
				if (iter.hasNext())
					buf.append(',');
			}
			buf.append(") ");
		}
		return buf.toString();
	}

	void prepareInnerQuery(StringBuilder inq, Operator op) throws Exception {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			prepareQueryPart(sp, inq);
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			prepareInnerQuery(inq, lhs);
			int connective = bop.getLogicalOp();
			if (connective == Operator.AND) {
				inq.append("\nINTERSECT\n");
			} else if (connective == Operator.OR) {
				inq.append("\nUNION\n");
			}
			Operator rhs = bop.getRhs();
			prepareInnerQuery(inq, rhs);
		}
	}

	protected void prepareQueryPart(SearchPredicate sp, StringBuilder inq)
			throws Exception {
		Object o = sp.getAttribute();
		if (!(o instanceof JobResultTypeInfo)) {
			throw new RuntimeException(
					"Attribute needs to be of type JobResultTypeInfo");
		}
		JobResultTypeInfo jrti = (JobResultTypeInfo) o;
		JobResultGroupInfo jrgi = resultVarMap.get(jrti);

		inq.append(getInnerStaticPart(jrgi.getId()));
		inq.append(" and (a.name=");
		inq.append(sqlDialect.prepareString(jrti.getName())).append(" and ");
		if (sp.getValue() instanceof SearchPredicate.Range) {
			SearchPredicate.Range range = (SearchPredicate.Range) sp.getValue();
			inq.append("a.numeric_value between ").append(range.getLowBound());
			inq.append(" and ").append(range.getUppBound());
		} else {
			if (jrti.getType() == ResultType.FLOAT
					|| jrti.getType() == ResultType.INT) {
				inq.append(createPredicate("a.numeric_value", sp.getValue(),
						sp.getOperator(), sp.getType(), CastingOption.NONE));
			} else {
				inq.append(createPredicate("a.value", sp.getValue(),
						sp.getOperator(), sp.getType(), CastingOption.NONE));
			}
		}
		inq.append(')');
	}

}
