package clinical.web.common.query;

import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import clinical.web.Constants;
import clinical.web.DBUtils;
import clinical.web.ISQLDialect;
import clinical.web.ServiceFactory;
import clinical.web.common.AssessmentMapping;
import clinical.web.common.IDBPoolService;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: MultiSiteAssessmentQueryBuilder.java,v 1.17 2006/06/19 22:42:27
 *          bozyurt Exp $
 */
public class MultiSiteAssessmentQueryBuilder extends AssessmentQueryBuilder {
	protected Map<String, Map<Integer, AssessmentMapping>> theAsMapInfoMap;
	protected Map<String, Map<String, Map<Integer, AssessmentMapping>>> siteDBAsMap = new HashMap<String, Map<String, Map<Integer, AssessmentMapping>>>(
			7);
	protected String firstDBID;
	protected String dbID;
	protected Map<String, Map<String, Integer>> retrievedAsNameIDMap = new HashMap<String, Map<String, Integer>>(
			7);
	protected Map<String, Map<String, Integer>> dbID2ExperimentIDsMap = new HashMap<String, Map<String, Integer>>(
			3);
	protected Map<Integer, String> masterExpID2NameMap;

	public MultiSiteAssessmentQueryBuilder(ISQLDialect sqlDialect,
			List<AssessmentSelectionInfo> assessments,
			Map<Integer, List<AssessmentMapping>> siteAsMap, String firstDBID,
			String dbID) {
		this(sqlDialect, assessments, siteAsMap, firstDBID, dbID, null);
	}

	/**
	 * 
	 * @param sqlDialect
	 * @param assessments
	 * @param siteAsMap
	 * @param firstDBID
	 *            the reference database (i.e. the database to which the web
	 *            user is connected)
	 * @param dbID
	 *            the ID of the database for which the query will be built
	 */
	public MultiSiteAssessmentQueryBuilder(ISQLDialect sqlDialect,
			List<AssessmentSelectionInfo> assessments,
			Map<Integer, List<AssessmentMapping>> siteAsMap, String firstDBID,
			String dbID, Map<Integer, String> expID2NameMap) {
		super(sqlDialect, assessments);
		this.firstDBID = firstDBID;
		this.dbID = dbID;
		this.masterExpID2NameMap = expID2NameMap;

		// prepare appropriate experiment ids for the corresponding db the query
		// will be built
		this.experimentIDs = getMatchingExperimentIDList();

		if (siteAsMap == null) {
			return;
		}
		for (List<AssessmentMapping> amList : siteAsMap.values()) {

			for (Iterator<AssessmentMapping> it = amList.iterator(); it
					.hasNext();) {
				AssessmentMapping am = it.next();

				String fdbID = am.getFirstDB();
				// each site can have assessment mappings for more than one
				// other site
				Map<String, Map<Integer, AssessmentMapping>> otherSiteAsMaps = siteDBAsMap
						.get(fdbID);

				if (otherSiteAsMaps == null) {
					otherSiteAsMaps = new LinkedHashMap<String, Map<Integer, AssessmentMapping>>(
							7);
					siteDBAsMap.put(fdbID, otherSiteAsMaps);

				}
				if (theAsMapInfoMap == null && fdbID.equals(firstDBID)) {
					log.info("selected the assessment into Map for db "
							+ firstDBID);
					theAsMapInfoMap = otherSiteAsMaps;
				}

				Map<Integer, AssessmentMapping> asMapInfoMap = otherSiteAsMaps
						.get(am.getSecondDB());
				if (asMapInfoMap == null) {
					asMapInfoMap = new LinkedHashMap<Integer, AssessmentMapping>(
							7);
					otherSiteAsMaps.put(am.getSecondDB(), asMapInfoMap);
				}
				asMapInfoMap.put(new Integer(am.getAssessmentID()), am);
			}
		}
		if (log.isDebugEnabled()) {
			log.debug("theAsMapInfoMap=" + theAsMapInfoMap);
		}
	}

	/**
	 * 
	 * @param sp
	 * @param inq
	 *            string buffer for the the inner SQL query (contains the query
	 *            generated so far)
	 */
	protected void prepareQueryPart(SearchPredicate sp, StringBuffer inq)
			throws Exception {
		Object o = sp.getAttribute();
		if (!(o instanceof AsScoreInfo))
			throw new Exception("Attribute needs to be of type AsScoreInfo!");
		AsScoreInfo si = (AsScoreInfo) o;

		// do the necessary mapping for the other site's assessmentIDs

		inq.append(getInnerStaticPart(getMatchingScoreType(si),
				getMatchingScoreAsID(si)));

		inq.append(" AND ( a.SCORENAME=");
		inq.append(sqlDialect.prepareString(getMatchingScoreName(si))).append(
				" AND ");

		if (sp.getValue() instanceof SearchPredicate.Range) {
			SearchPredicate.Range range = (SearchPredicate.Range) sp.getValue();

			Object value = getMatchingSearchPredicateValue(range.getLowBound(),
					si);

			inq.append("a.DATAVALUE BETWEEN ").append(value);
			value = getMatchingSearchPredicateValue(range.getUppBound(), si);
			inq.append(" AND ").append(value);

		} else {
			inq.append(createPredicate("a.datavalue",
					getMatchingSearchPredicateValue(sp.getValue(), si), sp
							.getOperator(), sp.getType(), CastingOption.NONE));
		}
		inq.append(')');
	}

	protected void prepareOuterQuery(StringBuffer outq, String innerQuery,
			Operator op) throws Exception {
		log.info("in multisite  prepareOuterQuery");
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			Object o = sp.getAttribute();
			if (!(o instanceof AsScoreInfo)) {
				throw new Exception(
						"Attribute needs to be of type AsScoreInfo!");
			}

			AsScoreInfo si = (AsScoreInfo) o;
			outq.append(getOuterStaticPart(si.getType(),
					getMatchingScoreAsID(si), getMatchingScoreName(si)));

			outq.append(innerQuery);
			outq.append(")");
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			prepareOuterQuery(outq, innerQuery, lhs);
			Operator rhs = bop.getRhs();
			/** @todo check why rhs can be null sometimes */
			if (rhs != null) {
				outq.append("\nUNION ALL\n");
				log.debug("rhs=" + rhs);
				prepareOuterQuery(outq, innerQuery, rhs);
			}
		}
	}

	protected AssessmentMapping findAssessmentMapping(String secondDBID,
			int assessmentID) {
		if (theAsMapInfoMap == null)
			return null;
		Map<Integer, AssessmentMapping> otherAsMapInfoMap = theAsMapInfoMap
				.get(secondDBID);
		if (otherAsMapInfoMap == null)
			return null;
		AssessmentMapping am = otherAsMapInfoMap.get(new Integer(assessmentID));
		if (log.isDebugEnabled()) {
			if (am != null) {
				log.debug("secondDBID=" + secondDBID + " am=" + am.toString());
			} else {
				log.debug("secondDBID=" + secondDBID + " assessmentID="
						+ assessmentID);
			}
		}
		return am;
	}

	/**
	 * Does the (necessary) predicate value mapping between the original
	 * database and the database for which the clinical query is currently
	 * built.
	 * 
	 * @param sp
	 * @param si
	 * @return
	 */
	protected Object getMatchingSearchPredicateValue(Object value,
			AsScoreInfo si) {
		if (dbID.equals(firstDBID)) {
			// use the original score type
			return value;
		}
		AssessmentSelectionInfo asi = (AssessmentSelectionInfo) scoreMap
				.get(si);

		AssessmentMapping am = findAssessmentMapping(dbID, asi
				.getAssessmentID().intValue());

		if (am == null) {
			return value;
		}
		AssessmentMapping.ScoreMap sm = (AssessmentMapping.ScoreMap) am
				.getMappedScoreName(si.getName());
		if (sm.getTypeMap() == null
				|| sm.getTypeMap().getSecondDBValue(value) == null) {
			return value;
		}

		return sm.getTypeMap().getSecondDBValue(value);
	}

	protected List<Integer> getMatchingExperimentIDList() {
		if (masterExpID2NameMap == null) {
			return null;
		}

		if (dbID.equals(firstDBID)) {
			// use the original experimentList
			return new ArrayList<Integer>(masterExpID2NameMap.keySet());
		}
		if (this.dbID2ExperimentIDsMap.get(dbID) == null) {
			prepareDBID2ExperimentListMap(dbID);
		}
		Map<String, Integer> expName2IDMap = this.dbID2ExperimentIDsMap
				.get(dbID);

		List<Integer> expIDList = new ArrayList<Integer>(3);
		for (Object element : masterExpID2NameMap.values()) {
			String expName = (String) element;
			Integer matchingExpID = (Integer) expName2IDMap.get(expName);
			assert (matchingExpID != null);
			if (matchingExpID != null) {
				expIDList.add(matchingExpID);
			}
		}
		return expIDList;
	}

	protected String getMatchingScoreType(AsScoreInfo si) {
		if (dbID.equals(firstDBID)) {
			// use the original score type
			return si.getType();
		}
		AssessmentSelectionInfo asi = (AssessmentSelectionInfo) scoreMap
				.get(si);
		AssessmentMapping am = findAssessmentMapping(dbID, asi
				.getAssessmentID().intValue());

		if (am == null) {
			return si.getType();
		}
		AssessmentMapping.ScoreMap sm = (AssessmentMapping.ScoreMap) am
				.getMappedScoreName(si.getName());
		if (log.isDebugEnabled()) {
			log.debug("getMatchingScoreName: " + sm.getMappedName());
		}
		if (sm.getTypeMap() == null) {
			return si.getType();
		}
		return sm.getTypeMap().getSecondDbType();

	}

	protected String getMatchingScoreName(AsScoreInfo si) {
		if (dbID.equals(firstDBID)) {
			// use the original score name
			return si.getName();
		}
		AssessmentSelectionInfo asi = (AssessmentSelectionInfo) scoreMap
				.get(si);
		AssessmentMapping am = findAssessmentMapping(dbID, asi
				.getAssessmentID().intValue());
		if (am == null) {
			return si.getName();
		}
		AssessmentMapping.ScoreMap sm = (AssessmentMapping.ScoreMap) am
				.getMappedScoreName(si.getName());
		if (log.isDebugEnabled()) {
			log.debug("getMatchingScoreName: " + sm.getMappedName());
		}
		return sm.getMappedName();
	}

	protected int getMatchingScoreAsID(AsScoreInfo si) {

		AssessmentSelectionInfo asi = (AssessmentSelectionInfo) scoreMap
				.get(si);
		if (dbID.equals(firstDBID)) {
			// use the original assessment id
			return asi.getAssessmentID().intValue();
		}

		AssessmentMapping am = findAssessmentMapping(dbID, asi
				.getAssessmentID().intValue());
		if (am == null) {
			// there is no matching assessment mapping for this assessment
			// so assume the assessments have same name across databases and
			// retrieve
			// assessment id - name pairs from the corresponding database and
			// (if successful)
			// return the assessment id back
			if (retrievedAsNameIDMap.get(dbID) == null) {
				prepareAsNameIDMapForDB(dbID);
			}
			Map<?, ?> asNameIDMap = retrievedAsNameIDMap.get(dbID);
			if (asNameIDMap != null) {
				Integer asID = (Integer) asNameIDMap.get(asi.getName());
				if (asID != null) {
					return asID.intValue();
				}
			}
		}

		if (am == null) {
			// there is no matching assessment mapping for this assessment so
			// use the assessment id of the originating database
			// which would not return any results from non-originating databases
			return asi.getAssessmentID().intValue();
		}
		AssessmentMapping.ScoreMap sm = (AssessmentMapping.ScoreMap) am
				.getMappedScoreName(si.getName());
		return sm.getMappedAsID();
	}

	protected void prepareAsNameIDMapForDB(String aDbID) {
		IDBPoolService pool = ServiceFactory.getPoolService(aDbID);
		Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		try {
			con = pool.getConnection(Constants.ADMIN_USER);
			st = con.createStatement();
			rs = st
					.executeQuery("select assessmentid, name from nc_assessment");
			Map<String, Integer> asNameIDMap = new HashMap<String, Integer>(17);
			while (rs.next()) {
				asNameIDMap.put(rs.getString(2), new Integer(rs.getInt(1)));
			}

			this.retrievedAsNameIDMap.put(aDbID, asNameIDMap);
		} catch (Exception e) {
			log.error("prepareAsNameIDMapForDB", e);
		} finally {
			DBUtils.close(st, rs);
			if (con != null) {
				try {
					pool.releaseConnection(Constants.ADMIN_USER, con);
				} catch (Exception x) {
				}
			}
		}
	}

	protected void prepareDBID2ExperimentListMap(String aDbID) {
		IDBPoolService pool = ServiceFactory.getPoolService(aDbID);
		Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		try {
			con = pool.getConnection(Constants.ADMIN_USER);
			st = con.createStatement();
			rs = st.executeQuery("select uniqueid, name from nc_experiment");
			Map<String, Integer> expName2IDMap = new HashMap<String, Integer>(5);
			while (rs.next()) {
				expName2IDMap.put(rs.getString(2), new Integer(rs.getInt(1)));
			}
			rs.close();
			this.dbID2ExperimentIDsMap.put(aDbID, expName2IDMap);

		} catch (Exception e) {
			log.error("prepareDBID2ExperimentListMap", e);
		} finally {
			DBUtils.close(st, rs);
			if (con != null) {
				try {
					pool.releaseConnection(Constants.ADMIN_USER, con);
				} catch (Exception x) {
				}
			}
		}
	}

}
