package clinical.web.common.query;

import java.beans.BeanInfo;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.IOException;
import java.io.Reader;
import java.io.StreamTokenizer;
import java.io.StringReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Stack;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: TSQLParser.java 811 2013-05-21 22:46:43Z bozyurt $
 */
public class TSQLParser implements SQLTokens {
	protected Map<String, ColumnInfo> columnTable = new HashMap<String, ColumnInfo>();
	protected Map<String, VOInfo> voTable = new HashMap<String, VOInfo>();
	protected List<ISymbol> symbolList = new ArrayList<ISymbol>();
	protected boolean verbose = false;

	public TSQLParser() {
		super();
	}

	public static boolean isDeleteStatement(List<ISymbol> symbolList) {
		if (symbolList.isEmpty())
			return false;
		ISymbol symbol = symbolList.get(0);
		return (symbol.getName().equalsIgnoreCase("delete"));
	}

	public List<ISymbol> parse(String tsqlStr) throws TSQLParserException {
		TSQLTokenizer tokenizer = new TSQLTokenizer(new StringReader(tsqlStr));
		int tokCode = -1;
		boolean deleteStmt = false;
		try {
			while ((tokCode = tokenizer.nextToken()) != EOS) {
				if (verbose) {
					System.out.println(tokenizer.getTokenString() + " code:"
							+ tokCode);
				}
				symbolList.add(new SQLTokenInfo(tokenizer.getTokenString()));
				switch (tokCode) {
				case DELETE:
					deleteStmt = true;
					break;
				case SELECT:
					parseSelectList(tokenizer);
					break;
				case FROM:
					parseTableList(tokenizer);
					break;
				case WHERE:
					parseWhereCondition(tokenizer);
					break;
				case ORDER:
					parseOrderClause(tokenizer);
					break;
				default:
					throw new TSQLParserException("Unrecognized token: "
							+ tokenizer.getTokenString());
				}
			}
		} catch (Exception x) {
			handleException(x);
		}

		if (!deleteStmt) {
			expandWildcards();
			validateSelectList();
		}

		if (verbose) {
			for (Iterator<ISymbol> iter = symbolList.iterator(); iter.hasNext();) {
				ISymbol symbol = iter.next();
				System.out.print(symbol.toString() + " ");
			}
			System.out.println();
		}
		return symbolList;
	}

	protected void expandWildcards() {
		List<ISymbol> newSymbolList = new ArrayList<ISymbol>();
		for (Iterator<ISymbol> iter = symbolList.iterator(); iter.hasNext();) {
			ISymbol symbol = iter.next();
			if (symbol instanceof ColumnInfo) {
				ColumnInfo ci = (ColumnInfo) symbol;
				if (ci.name.endsWith("*")) {
					String[] properties = ci.voInfo.getProperties();
					for (int i = 0; i < properties.length; i++) {
						newSymbolList.add(new ColumnInfo(properties[i],
								ci.voInfo));
						if ((i + 1) < properties.length) {
							newSymbolList.add(new SQLTokenInfo(","));
						}
					}
				} else {
					newSymbolList.add(symbol);
				}
			} else {
				newSymbolList.add(symbol);
			}
		}
		this.symbolList = newSymbolList;
	}

	protected void validateSelectList() throws TSQLParserException {
		boolean inSelect = false;
		for (Iterator<ISymbol> iter = symbolList.iterator(); iter.hasNext();) {
			ISymbol symbol = iter.next();
			if (symbol.getName().equals("select")) {
				inSelect = true;
				continue;
			} else if (symbol.getName().equals("from")) {
				inSelect = false;
				break;
			}
			if (inSelect && symbol instanceof ColumnInfo) {
				ColumnInfo ci = (ColumnInfo) symbol;
				if (voTable.get(ci.getVoInfo().getName()) == null) {
					throw new TSQLParserException(
							"Select list of columns cannot contain columns from tables "
									+ "other than the ones in the <from> list of tables!: "
									+ ci.getVoInfo());
				}
			}
		}
	}

	protected void validateSelectListOld() throws TSQLParserException {
		boolean inSelect = false;
		VOInfo theVOI = null;
		for (Iterator<ISymbol> iter = symbolList.iterator(); iter.hasNext();) {
			ISymbol symbol = iter.next();
			if (symbol.getName().equals("select")) {
				inSelect = true;
				theVOI = null;
				continue;
			} else if (symbol.getName().equals("from")) {
				inSelect = false;
				continue;
			}
			if (inSelect && symbol instanceof ColumnInfo) {
				ColumnInfo ci = (ColumnInfo) symbol;
				if (theVOI == null) {
					theVOI = ci.getVoInfo();
				} else {
					if (theVOI != ci.getVoInfo()) {
						throw new TSQLParserException(
								"Only column(s) from one table is allowed in select list! : <"
										+ theVOI.getName()
										+ "> , second table (object) :"
										+ ci.getVoInfo());
					}
				}
			}
		}
	}

	public String getKey(VOInfo voInfo, String id) {
		if (voInfo.getName() != null) {
			return voInfo.getName() + "_" + id;
		} else {
			return voInfo.getAlias() + "_" + id;
		}
	}

	protected void parseSelectList(TSQLTokenizer tokenizer)
			throws TSQLParserException {
		int tokCode = -1;
		String id = null;
		try {
			while ((tokCode = tokenizer.nextToken()) != FROM) {
				if (tokCode == IDENTIFIER) {
					id = tokenizer.getTokenString();
					VOInfo voInfo = null;
					ISymbol symbol = voTable.get(id);
					if (symbol == null) {
						Class<?> clazz = getVOClass(id);
						if (clazz != null) {
							voInfo = new VOInfo(id,
									getFullyQualifedClassName(id));
						} else {
							// id is assumed to be an alias
							voInfo = new VOInfo(id);
						}
						voTable.put(id, voInfo);
					} else {
						voInfo = (VOInfo) symbol;
					}
					tokCode = tokenizer.nextToken();
					if (tokCode == PERIOD) {
						tokCode = tokenizer.nextToken();
						if (tokCode == IDENTIFIER) {
							id = tokenizer.getTokenString();
							String key = getKey(voInfo, id);
							ColumnInfo ci = columnTable.get(key);
							if (ci == null) {
								ci = new ColumnInfo(id, voInfo);
								columnTable.put(key, ci);
							}
							symbolList.add(ci);
						} else if (tokCode == STAR) {
							voInfo.setAllColumns(true);
							ColumnInfo ci = columnTable.get(voInfo.getAlias()
									+ "_*");
							assert (ci == null);
							ci = new ColumnInfo(voInfo.getAlias() + "_*",
									voInfo);
							columnTable.put(voInfo.getAlias() + "_*", ci);
							symbolList.add(ci);
						} else {
							throw new IOException(
									"Unexpected token for select list:"
											+ tokenizer.getTokenString());
						}
					} else {
						tokenizer.pushBack();
					}

				} else if (tokCode == COMMA || tokCode == LEFT_PAREN
						|| tokCode == RIGHT_PAREN || tokCode == MAX
						|| tokCode == MIN || tokCode == COUNT || tokCode == AVG
						|| tokCode == SUM || tokCode == DISTINCT
						|| tokCode == ALL) {
					symbolList
							.add(new SQLTokenInfo(tokenizer.getTokenString()));
				} else {
					throw new IOException("Unexpected token for select list:"
							+ tokenizer.getTokenString());
				}
			}
			// pushback FROM
			if (tokCode == FROM) {
				tokenizer.pushBack();
			}
		} catch (Exception x) {
			handleException(x);
		}
	}

	protected void parseTableList(TSQLTokenizer tokenizer)
			throws TSQLParserException {
		int tokCode = -1;
		String id = null;
		try {
			tokCode = tokenizer.nextToken();
			while (tokCode != EOS && tokCode != WHERE && tokCode != ORDER) {
				if (tokCode == IDENTIFIER) {
					id = tokenizer.getTokenString();
					VOInfo voInfo = null;
					String voName = id;
					if (isFullyQualifiedClass(id)) {
						voName = extractVOName(id);
					}
					ISymbol symbol = voTable.get(voName);
					if (symbol == null) {
						Class<?> clazz = getVOClass(id);
						if (clazz == null)
							throw new IOException("Not a valid value object:"
									+ id);
						voInfo = new VOInfo(voName,
								getFullyQualifedClassName(id));
						voTable.put(voName, voInfo);
					} else {
						voInfo = (VOInfo) symbol;
					}
					symbolList.add(voInfo);
					tokCode = tokenizer.nextToken();
					if (tokCode == AS) {
						tokCode = tokenizer.nextToken();
						if (tokCode != IDENTIFIER) {
							throw new IOException(
									"Unexpected token in from list:"
											+ tokenizer.getTokenString());
						}

						voInfo.setAlias(tokenizer.getTokenString());
					} else if (tokCode == COMMA) {
						tokenizer.pushBack();
					}
				}// IDENTIFIER
				else if (tokCode == COMMA) {
					symbolList
							.add(new SQLTokenInfo(tokenizer.getTokenString()));
				} else {
					throw new IOException("Unexpected token in from list:"
							+ tokenizer.getTokenString());
				}
				tokCode = tokenizer.nextToken();
			}
			if (tokCode == WHERE || tokCode == ORDER) {
				tokenizer.pushBack();
			}
			syncSymbolTables();
		} catch (Exception x) {
			handleException(x);
		}
	}

	protected void syncSymbolTables() {
		Map<String, VOInfo> map = new HashMap<String, VOInfo>(7);
		for (Iterator<VOInfo> iter = voTable.values().iterator(); iter
				.hasNext();) {
			VOInfo voInfo = iter.next();
			if (voInfo.isFullyInitialized()) {
				map.put(voInfo.getName(), voInfo);
				if (voInfo.getAlias() != null) {
					VOInfo ivi = voTable.get(voInfo.getAlias());
					if (ivi != null) {
						voInfo.setAllColumns(ivi.isAllColumns());

					}
					map.put(voInfo.getAlias(), voInfo);
				}
			}
		}
		this.voTable = map;
		for (Iterator<ColumnInfo> iter = columnTable.values().iterator(); iter
				.hasNext();) {
			ColumnInfo ci = iter.next();
			VOInfo ivi = ci.voInfo;
			if (ivi.name != null) {
				ci.voInfo = voTable.get(ivi.name);
			} else {
				ci.voInfo = voTable.get(ivi.getAlias());
			}
			assert (ci.voInfo != null);
		}
	}

	protected void parseWhereCondition(TSQLTokenizer tokenizer)
			throws TSQLParserException {
		int tokCode = -1;
		String id = null;
		boolean inSubSelect = false;
		try {
			tokCode = tokenizer.nextToken();
			while (tokCode != EOS && tokCode != ORDER && tokCode != GROUP) {
				if (inSubSelect) {
					symbolList
							.add(new SQLTokenInfo(tokenizer.getTokenString()));
				} else if (tokCode == IDENTIFIER) {
					id = tokenizer.getTokenString();
					VOInfo voInfo = voTable.get(id);
					if (voInfo == null) {
						if (id.equals("true") || id.equals("false")) {
							symbolList.add(new BoolTokenInfo(id));
							tokCode = tokenizer.nextToken();
							continue;
						} else if (id.equals("endsWith")) {
							// string function
							FunctionInfo fi = new FunctionInfo(id);
							tokCode = tokenizer.nextToken();
							if (tokCode == LEFT_PAREN) {
								parseFunctionParams(tokenizer, fi);
								symbolList.add(fi);
							}
							tokCode = tokenizer.nextToken();
							continue;
						} else {
							throw new TSQLParserException(
									"Unrecognized table or alias in where clause: "
											+ tokenizer.getTokenString());
						}
					}
					tokCode = tokenizer.nextToken();
					if (tokCode != PERIOD) {
						throw new TSQLParserException(
								"A period is expected after table or alias in where clause:"
										+ tokenizer.getTokenString());
					}
					tokCode = tokenizer.nextToken();
					if (tokCode == IDENTIFIER) {
						id = tokenizer.getTokenString();
						String key = getKey(voInfo, id);
						ColumnInfo ci = columnTable.get(key);
						if (ci == null) {
							if (!voInfo.isValidProperty(id)) {
								throw new IOException(
										"Not a valid property for VO '"
												+ voInfo.getName() + "' : "
												+ id);
							}
							ci = new ColumnInfo(id, voInfo);
							columnTable.put(key, ci);
						}
						symbolList.add(ci);
					} else {
						throw new IOException(
								"Unexpected token after '.' in where clause: "
										+ tokenizer.getTokenString());
					}

				} else {
					if (tokCode == STRING) {
						symbolList.add(new StringInfo(tokenizer
								.getTokenString()));
					} else {
						String tokStr = tokenizer.getTokenString();
						// assumption subselect is not coordinated and last in
						// the TSQL
						if (tokStr.equalsIgnoreCase("select")) {
							inSubSelect = true;
						}
						symbolList.add(new SQLTokenInfo(tokStr));
					}
				}
				tokCode = tokenizer.nextToken();
			}
			if (tokCode == ORDER || tokCode == GROUP) {
				tokenizer.pushBack();
			}
		} catch (Exception x) {
			handleException(x);
		}
	}

	protected void parseFunctionParams(TSQLTokenizer tokenizer, FunctionInfo fi)
			throws IOException, TSQLParserException {
		int tokCode = tokenizer.nextToken();
		while (tokCode != RIGHT_PAREN) {
			if (tokCode == IDENTIFIER) {
				String id = tokenizer.getTokenString();
				VOInfo voInfo = voTable.get(id);
				if (voInfo == null) {
					throw new TSQLParserException(
							"A column declaration or a string is expected as the argument for function "
									+ fi.getName());
				}
				tokCode = tokenizer.nextToken();
				if (tokCode != PERIOD) {
					throw new TSQLParserException(
							"A period is expected after table or alias in where clause:"
									+ tokenizer.getTokenString());
				}
				tokCode = tokenizer.nextToken();
				if (tokCode == IDENTIFIER) {
					id = tokenizer.getTokenString();
					String key = getKey(voInfo, id);
					ColumnInfo ci = columnTable.get(key);
					if (ci == null) {
						if (!voInfo.isValidProperty(id)) {
							throw new IOException(
									"Not a valid property for VO '"
											+ voInfo.getName() + "' : " + id);
						}
						ci = new ColumnInfo(id, voInfo);
						columnTable.put(key, ci);
					}
					fi.addParam(ci);
				} else {
					throw new TSQLParserException(
							"Unexpected token after '.' in function arguments: "
									+ tokenizer.getTokenString());
				}
			} else if (tokCode == STRING) {
				fi.addParam(new StringInfo(tokenizer.getTokenString()));
			} else if (tokCode != COMMA || tokCode == EOS) {
				throw new TSQLParserException(
						"Unexpected token in function arguments: "
								+ tokenizer.getTokenString());
			}
			tokCode = tokenizer.nextToken();

		}
	}

	protected void parseOrderClause(TSQLTokenizer tokenizer)
			throws TSQLParserException {
		int tokCode = -1;
		String id = null;
		try {
			tokCode = tokenizer.nextToken();
			if (tokCode != BY) {
				throw new TSQLParserException(
						"The keyword 'by' is expected after 'order'!");
			}
			symbolList.add(new SQLTokenInfo(tokenizer.getTokenString()));

			tokCode = tokenizer.nextToken();
			while (tokCode != EOS && tokCode != GROUP) {

				if (tokCode == IDENTIFIER) {
					id = tokenizer.getTokenString();
					VOInfo voInfo = voTable.get(id);
					if (voInfo == null) {
						throw new TSQLParserException(
								"Unrecognized table or alias in order clause: "
										+ tokenizer.getTokenString());
					}
					tokCode = tokenizer.nextToken();
					if (tokCode != PERIOD) {
						throw new TSQLParserException(
								"A period is expected after table or alias in order clause:"
										+ tokenizer.getTokenString());
					}
					tokCode = tokenizer.nextToken();
					if (tokCode == IDENTIFIER) {
						id = tokenizer.getTokenString();
						String key = getKey(voInfo, id);
						ColumnInfo ci = columnTable.get(key);
						if (ci != null) {
							symbolList.add(ci);
						} else {
							if (!voInfo.isValidProperty(id)) {
								throw new IOException(
										"Not a valid property for VO '"
												+ voInfo.getName() + "' : "
												+ id);
							}
							ci = new ColumnInfo(id, voInfo);
							columnTable.put(key, ci);
							symbolList.add(ci);
						}

					} else {
						throw new IOException(
								"Unexpected token after '.' in order clause: "
										+ tokenizer.getTokenString());
					}
				} else {
					symbolList
							.add(new SQLTokenInfo(tokenizer.getTokenString()));
				}
				tokCode = tokenizer.nextToken();
			}// while
			if (tokCode == GROUP) {
				tokenizer.pushBack();
			}
		} catch (Exception x) {
			handleException(x);
		}
	}

	protected void handleException(Exception x) throws TSQLParserException {
		if (x instanceof TSQLParserException) {
			throw (TSQLParserException) x;
		} else {
			throw new TSQLParserException(x);
		}
	}

	protected static String getFullyQualifedClassName(String voName) {
		if (isFullyQualifiedClass(voName)) {
			return voName;
		}
		StringBuilder buf = new StringBuilder();
		buf.append("clinical.server.vo.").append(voName);
		return buf.toString();
	}

	protected static boolean isFullyQualifiedClass(String className) {
		return className.indexOf('.') != -1;
	}

	protected static String extractVOName(String fqClassName) {
		int idx = fqClassName.lastIndexOf('.');
		return fqClassName.substring(idx + 1);
	}

	protected static Class<?> getVOClass(String voName) {
		String fqName = getFullyQualifedClassName(voName);
		Class<?> clazz = null;
		try {
			clazz = Class.forName(fqName);
			return clazz;
		} catch (ClassNotFoundException e) {
			return null;
		}
	}

	public static class FunctionInfo implements ISymbol {
		String name;
		List<ISymbol> params = new ArrayList<ISymbol>(2);

		public FunctionInfo(String name) {
			super();
			this.name = name;
		}

		public void addParam(ISymbol param) {
			params.add(param);
		}

		@Override
		public String getName() {
			// TODO Auto-generated method stub
			return name;
		}

		@Override
		public boolean isFullyInitialized() {
			return true;
		}

	}

	public static class ColumnInfo implements ISymbol {
		String name;
		VOInfo voInfo;

		public ColumnInfo(String name, VOInfo voInfo) {
			super();
			this.name = name;
			this.voInfo = voInfo;
		}

		public String getName() {
			return name;
		}

		public boolean isFullyInitialized() {
			return voInfo != null && voInfo.isFullyInitialized();
		}

		public String toString() {
			StringBuilder buf = new StringBuilder();
			buf.append("ColumnInfo[");
			buf.append(voInfo.getName()).append('.').append(name);
			buf.append(']');
			return buf.toString();
		}

		public VOInfo getVoInfo() {
			return voInfo;
		}
	}// ;

	public static class VOInfo implements ISymbol {
		String name;
		String fqName;
		String alias;
		Class<?> voClass;
		boolean allColumns = false;
		protected BeanInfo bi;
		protected Map<String, PropertyDescriptor> propMap = new LinkedHashMap<String, PropertyDescriptor>(
				19);

		public VOInfo(String alias) {
			this.alias = alias;
		}

		public VOInfo(String name, String fqName)
				throws ClassNotFoundException, IntrospectionException {
			this.name = name;
			this.fqName = fqName;
			voClass = Class.forName(fqName);
			bi = Introspector.getBeanInfo(voClass, java.lang.Object.class);
			PropertyDescriptor[] pds = bi.getPropertyDescriptors();
			for (int i = 0; i < pds.length; i++) {
				propMap.put(pds[i].getName().toLowerCase(), pds[i]);
			}
		}

		public String[] getProperties() {
			String[] propNames = new String[propMap.size()];
			int i = 0;
			for (Iterator<String> iter = propMap.keySet().iterator(); iter
					.hasNext();) {
				String propertyName = iter.next();
				propNames[i++] = propertyName;
			}
			return propNames;
		}

		public PropertyDescriptor getPD(String propertyName) {
			return propMap.get(propertyName);
		}

		public boolean isValidProperty(String propertyName) {
			return propMap.get(propertyName.toLowerCase()) != null;
		}

		public String getName() {
			return name;
		}

		public boolean isFullyInitialized() {
			return fqName != null && voClass != null;
		}

		public boolean isAllColumns() {
			return allColumns;
		}

		public void setAllColumns(boolean allColumns) {
			this.allColumns = allColumns;
		}

		public String getAlias() {
			return alias;
		}

		public void setAlias(String alias) {
			this.alias = alias;
		}

		public String toString() {
			StringBuilder buf = new StringBuilder();
			buf.append("VOInfo::[");
			buf.append(name);
			buf.append(']');
			return buf.toString();
		}

		public Class<?> getVoClass() {
			return voClass;
		}
	}// ;

	public static class StringInfo implements ISymbol {
		String name;

		public StringInfo(String name) {
			this.name = name;
		}

		public String getName() {
			return name;
		}

		public boolean isFullyInitialized() {
			return true;
		}

		public String toString() {
			StringBuilder buf = new StringBuilder();
			buf.append("StringInfo::[");
			buf.append(name);
			buf.append(']');
			return buf.toString();
		}
	}// ;

	public static class SQLTokenInfo implements ISymbol {
		String name;

		public SQLTokenInfo(String name) {
			this.name = name;
		}

		public String getName() {
			return name;
		}

		public boolean isFullyInitialized() {
			return true;
		}

		public String toString() {
			return name;
		}
	}// ;

	public static class BoolTokenInfo implements ISymbol {
		String name;

		public BoolTokenInfo(String name) {
			this.name = name;
		}

		public String getName() {
			return name;
		}

		public boolean isFullyInitialized() {
			return true;
		}

		public String toString() {
			return name;
		}
	}// ;

	public static class TSQLTokenizer implements SQLTokens {
		StreamTokenizer stok;
		Map<String, Integer> specialCharMap = new HashMap<String, Integer>();
		Map<String, Integer> keywordMap = new HashMap<String, Integer>();
		TokenInfo curTokenInfo;
		Stack<TokenInfo> tokenStack = new Stack<TokenInfo>();

		public TSQLTokenizer(Reader in) {
			stok = new StreamTokenizer(in);
			stok.resetSyntax();
			stok.eolIsSignificant(false);
			stok.wordChars('a', 'z');
			stok.wordChars('A', 'Z');
			stok.wordChars('_', '_');
			stok.wordChars('0', '9');

			stok.ordinaryChar('.');
			stok.ordinaryChar(',');
			stok.ordinaryChar(';');
			stok.ordinaryChar('*');
			stok.ordinaryChar('/');
			stok.ordinaryChar('+');
			stok.ordinaryChar('-');
			stok.ordinaryChar('(');
			stok.ordinaryChar(')');
			stok.ordinaryChar('%');
			stok.ordinaryChar('\'');

			String[] tokStrs = new String[] { ".", ",", ";", "*", "/", "+",
					"-", "(", ")", "%", ">", "<", "=", "<>", ">=", "<=", "'",
					"select", "from", "where", "and", "or", "like", "order",
					"by", "asc", "desc", "group", "having", "null", "is",
					"not", "in", "as", "distinct", "all", "max", "min", "avg",
					"count", "sum", "endsWith", "delete" };
			int[] tokCodes = new int[] { PERIOD, COMMA, SEMICOLON, STAR, SLASH,
					PLUS, MINUS, LEFT_PAREN, RIGHT_PAREN, PERCENT, GREATER,
					LESS, EQUAL, NOT_EQUAL, GE, LE, QUOTE, SELECT, FROM, WHERE,
					AND, OR, LIKE, ORDER, BY, ASC, DESC, GROUP, HAVING, NULL,
					IS, NOT, IN, AS, DISTINCT, ALL, MAX, MIN, AVG, COUNT, SUM,
					ENDSWITH, DELETE };
			for (int i = 0; i < 17; i++) {
				specialCharMap.put(tokStrs[i], new Integer(tokCodes[i]));
			}
			for (int i = 17; i < tokStrs.length; i++) {
				keywordMap.put(tokStrs[i], new Integer(tokCodes[i]));
			}
		}

		public void pushBack() {
			if (curTokenInfo != null) {
				tokenStack.push(curTokenInfo);
			}
		}

		public boolean isKeyword(String token) {
			return keywordMap.get(token.toLowerCase()) != null;
		}

		public Integer getSpecialChar(String token) {
			return specialCharMap.get(token);
		}

		public int nextToken() throws IOException {
			if (!tokenStack.isEmpty()) {
				this.curTokenInfo = tokenStack.pop();
				return curTokenInfo.tokCode;
			}

			int ttype = stok.nextToken();
			Integer sc = null;
			while (ttype == ' ' || ttype == '\t') {
				ttype = stok.nextToken();
			}

			if (ttype == StreamTokenizer.TT_WORD) {
				if (isAllDigits(stok.sval)) {
					String curTok = handleNumber(stok.sval);
					this.curTokenInfo = new TokenInfo(NUMBER, curTok);
					return NUMBER;
				} else if (isKeyword(stok.sval)) {
					Integer kw = keywordMap.get(stok.sval.toLowerCase());
					this.curTokenInfo = new TokenInfo(kw.intValue(), stok.sval);
					return kw.intValue();
				} else {
					this.curTokenInfo = new TokenInfo(IDENTIFIER, stok.sval);
					return IDENTIFIER;
				}
			} else if ((sc = getSpecialChar(String.valueOf((char) stok.ttype))) != null) {
				char prevCh = (char) stok.ttype;
				if (stok.ttype == '-' || stok.ttype == '+') {
					int curTType = stok.ttype;
					ttype = stok.nextToken();
					if (ttype == StreamTokenizer.TT_WORD) {
						if (isAllDigits(stok.sval)) {
							StringBuilder buf = new StringBuilder();
							buf.append((char) curTType).append(stok.sval);
							String curTok = handleNumber(buf.toString());
							this.curTokenInfo = new TokenInfo(NUMBER, curTok);
							return NUMBER;
						} else {
							stok.pushBack();
							this.curTokenInfo = new TokenInfo(sc.intValue(),
									String.valueOf((char) stok.ttype));
							return sc.intValue();
						}
					} else {
						stok.pushBack();
						this.curTokenInfo = new TokenInfo(sc.intValue(),
								String.valueOf((char) stok.ttype));
						return sc.intValue();
					}
				} else if (stok.ttype == '\'') {

					String curTok = handleString();
					this.curTokenInfo = new TokenInfo(STRING, curTok);
					return STRING;

				} else if ((prevCh == '<' || prevCh == '>')) {
					ttype = stok.nextToken();
					if (stok.ttype == '>' || stok.ttype == '=') {
						if (prevCh == '<' && stok.ttype == '<') {
							throw new IOException("Unrecognized token:<<");
						}
						char[] carr = { prevCh, (char) stok.ttype };
						String multiCharTok = new String(carr);
						sc = getSpecialChar(multiCharTok);
						this.curTokenInfo = new TokenInfo(sc.intValue(),
								multiCharTok);
						return sc.intValue();
					} else {
						stok.pushBack();
					}
				}
				this.curTokenInfo = new TokenInfo(sc.intValue(),
						String.valueOf((char) stok.ttype));

				return sc.intValue();

			} else if (stok.ttype == StreamTokenizer.TT_EOF) {
				return EOS;
			} else {
				throw new IOException("Unrecognized token:"
						+ String.valueOf((char) stok.ttype));
			}
		}

		protected String handleString() throws IOException {

			StringBuilder buf = new StringBuilder();
			int ttype = -1;
			while ((ttype = stok.nextToken()) != StreamTokenizer.TT_EOF) {
				if (ttype == '\'') {
					ttype = stok.nextToken();
					if (ttype == '\'') {
						buf.append('\'');
					} else {
						stok.pushBack();
						break;
					}
				} else {
					if (ttype == StreamTokenizer.TT_WORD) {
						buf.append(stok.sval);
					} else {
						buf.append((char) ttype);
					}
				}
			}
			return buf.toString();
		}

		protected String handleNumber(String numTok) throws IOException {
			StringBuilder buf = new StringBuilder();
			buf.append(numTok);
			int ttype = stok.nextToken();
			if (ttype == '.') {
				buf.append('.');
				ttype = stok.nextToken();
				if (ttype != StreamTokenizer.TT_WORD) {
					buf.append(ttype);
					throw new IOException("Not a valid number: "
							+ buf.toString());
				} else {
					buf.append(stok.sval);
					if (!isAllDigits(stok.sval)) {
						throw new IOException("Not a valid number: "
								+ buf.toString());
					}
				}
			} else {
				stok.pushBack();
			}
			return buf.toString();
		}

		public static boolean isAllDigits(String token) {
			boolean ok = true;
			char[] carr = token.toCharArray();
			for (int i = 0; i < carr.length; i++) {
				if (!Character.isDigit(carr[i]))
					return false;
			}
			return ok;
		}

		public String getTokenString() {
			return curTokenInfo.token;
		}
	}// ;

	public static class TokenInfo {
		int tokCode;
		String token;

		public TokenInfo(int code, String token) {
			tokCode = code;
			this.token = token;
		}

	}

	public boolean isVerbose() {
		return verbose;
	}

	public void setVerbose(boolean verbose) {
		this.verbose = verbose;
	}

	@SuppressWarnings("unused")
	public static void main(String[] args) throws Exception {

		TSQLParser parser = new TSQLParser();
		String s = "select r.* from Rawdata as r where r.subjectid in ('000894468383', '000862323007') order by r.componentid, r.segmentid asc";
		String s1 = "SELECT number, quantity  FROM Order  WHERE quantity > +5.6 ORDER BY number DESC";
		String s2 = "select a.scorename, a.textvalue from Assessmentdata as a where a.subjectid in ('000894468383', '000862323007') and a.isvalidated = 1 order by a.assessmentid, a.scorename asc";
		String s3 = "select a.* from Assessmentdata as a, Storedassessment as s "
				+ "where a.ncStoredassessmentUniqueid = s.uniqueid and s.isvalidated = 1 "
				+ "and s.subjectid in ('000894468383', '000862323007') "
				+ " and a.isvalidated = 1 order by a.assessmentid, a.scorename asc";

		String s4 = "select r.*, d.objectsize, d.objecttype from Rawdata as r, Dataobject as d where "
				+ " r.subjectid ='000602155404' and r.uniqueid = d.dataid and  d.objecttype = '3_LOCAL COL DICOM'"
				+ " order by r.ncExperimentUniqueid, r.subjectid, r.componentid, r.segmentid";

		String s5 = "select d.* from Deriveddata as d where endsWith(d.datauri,'.xml') and d.israw = false";

		String s6 = "select a.* from Jobs as a where a.jobstatus <> 'canceled'";
		parser.parse(s6);
	}

}
