package clinical.web.download;

import java.io.File;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

import org.apache.tools.ant.Project;
import org.apache.tools.ant.taskdefs.Tar;

import clinical.utils.Assertion;
import clinical.utils.BIRNURIException;
import clinical.utils.FileUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: Packager.java 366 2011-05-05 20:06:27Z bozyurt $
 */
public class Packager extends Tar implements Serializable {
	private static final long serialVersionUID = 1L;
	protected String cacheDir;
	protected List<PathWrapper> files;
	protected String bundleName;
	protected int action;
	protected boolean doCleanup = false;
	protected Map<String, List<PathWrapper>> fileSetMap = new HashMap<String, List<PathWrapper>>(
			7);
	protected boolean verbose = false;
	protected File tarFile;

	public Packager(String cacheDir, String bundleName, int action) {
		this(cacheDir, null, bundleName, action);
	}

	public Packager(String cacheDir, List<PathWrapper> files,
			String bundleName, int action) {
		super();
		this.cacheDir = cacheDir;
		this.files = files;
		this.bundleName = bundleName;
		this.action = action;
		setProject(new Project());
		setTaskName("tar");
		// setTaskType("tar");
		tarFile = null;
		if ((action & FileBundleConfig.GZIPPED) > 0) {
			tarFile = new File(cacheDir, bundleName + ".tar.gz");
			TarCompressionMethod tcm = new TarCompressionMethod();
			tcm.setValue("gzip");
			setCompression(tcm);
		} else {
			tarFile = new File(cacheDir, bundleName + ".tar");
		}
		setDestFile(tarFile);

		if (files != null) {
			setBasedir(new File(cacheDir));
			fileSetMap.put(cacheDir, files);
			StringBuffer buf = new StringBuffer(files.size() * 80);
			for (Iterator<PathWrapper> iter = files.iterator(); iter.hasNext();) {
				PathWrapper pw = iter.next();
				buf.append(pw.getRelPath());
				if (iter.hasNext()) {
					buf.append(',');
				}
			}
			setIncludes(buf.toString());
		}
	}

	public File getTarFile() {
		return tarFile;
	}
	public void includeFiles(List<PathWrapper> files) {
		Map<String, List<PathWrapper>> bySrcMap = new HashMap<String, List<PathWrapper>>(
				7);
		for (PathWrapper pw : files) {
			List<PathWrapper> files4Src = bySrcMap.get(pw.getSourceRootDir());
			if (files4Src == null) {
				files4Src = new LinkedList<PathWrapper>();
				bySrcMap.put(pw.getSourceRootDir(), files4Src);
			}
			files4Src.add(pw);
		}
		// for (Iterator iter = bySrcMap.entrySet().iterator(); iter.hasNext();)
		// {
		for (Map.Entry<String, List<PathWrapper>> entry : bySrcMap.entrySet()) {
			String rootDir = (String) entry.getKey();
			List<PathWrapper> files4Root = (List<PathWrapper>) entry.getValue();
			TarFileSet fs = createTarFileSet();
			if (rootDir != null) {
				fs.setDir(new File(rootDir));
			} else {
				fs.setDir(new File("/"));
			}
			StringBuffer buf = new StringBuffer(files4Root.size() * 80);
			int includedCount = 0;
			for (Iterator<PathWrapper> iterator = files4Root.iterator(); iterator
					.hasNext();) {
				PathWrapper pw = iterator.next();
				buf.append(pw.getRelPath());
				if (verbose) {
					System.out.println("including :" + pw.getRelPath());
				}
				++includedCount;
				if (iterator.hasNext()) {
					buf.append(',');
				}
			}
			System.out.println("including " + includedCount + " files from "
					+ rootDir);
			if (verbose) {
				System.out.println("rootDir:" + rootDir);
				System.out.println("includes: " + buf.toString());
			}

			fs.setIncludes(buf.toString());

			// used in cleanup
			List<PathWrapper> fileList = fileSetMap.get(rootDir);
			if (fileList == null) {
				fileSetMap.put(rootDir, files4Root);
			} else {
				fileList.addAll(files4Root);
			}
		}
	}

	public static boolean areAbsolutePaths(List<PathWrapper> files) {
		for(PathWrapper pw : files) {
			File f = new File(pw.getRelPath());
			if (f.isFile()) {
				return true;
			}
		}
		return false;
	}
	
	public static String findCommonPath(List<PathWrapper> files) {
		List<FileInfo> fileInfoList = new ArrayList<FileInfo>(files.size());
		try {
			for (PathWrapper pw : files) {
				FileInfo fi = new FileInfo(pw.getRelPath(), -1, -1, null, false);
				fileInfoList.add(fi);
			}
			return FileInfo.findCommonPath(fileInfoList);
		} catch (BIRNURIException be) {
			return null;
		}
	}

	public void includeFiles(String rootDir, List<PathWrapper> files) {
		TarFileSet fs = createTarFileSet();
		boolean noCache = false;
		String commonPath = null;
		if (rootDir != null) {
			// check if files are of absolute path type i.e. no intermediate
			// caching directory is used
			if (areAbsolutePaths(files)) {
				commonPath = findCommonPath(files);
				Assertion
						.assertNotNull(commonPath,
								"Internal error while packaging files! No common path!");
				fs.setDir(new File(commonPath));
				noCache = true;
				rootDir = commonPath;
			} else {
				fs.setDir(new File(rootDir));
			}
		} else {
			fs.setDir(new File("/"));
		}
		StringBuffer buf = new StringBuffer(files.size() * 80);
		int includedCount = 0;
		for (Iterator<PathWrapper> iter = files.iterator(); iter.hasNext();) {
			PathWrapper pw = iter.next();
			String relPath = pw.getRelPath();
			if (noCache) {
				if (commonPath != null && relPath.startsWith(commonPath)) {
					relPath = relPath.substring(commonPath.length());
				}
				// remove any leading / to comply with ant tar task
				relPath = relPath.replaceFirst("^\\/+", "");
			}
			buf.append(relPath);
			if (verbose) {
				System.out.println("including :" + relPath);
			}
			++includedCount;
			if (iter.hasNext()) {
				buf.append(',');
			}
		}
		System.out.println("including " + includedCount + " files from "
				+ rootDir);
		if (verbose) {
			System.out.println("rootDir:" + rootDir);
			System.out.println("includes: " + buf.toString());
		}
		fs.setIncludes(buf.toString());
		List<PathWrapper> fileList = fileSetMap.get(rootDir);
		if (fileList == null && !noCache) {
			fileSetMap.put(rootDir, files);
		} else {
			if (!noCache) {
				fileList.addAll(files);
			}
		}
	}

	protected void cleanupAfter() {
		for (Map.Entry<String, List<PathWrapper>> entry : fileSetMap.entrySet()) {
			String rootDir = entry.getKey();
			List<PathWrapper> files = entry.getValue();
			Map<String, String> uniqTopDirMap = new HashMap<String, String>(7);
			for (Iterator<PathWrapper> it = files.iterator(); it.hasNext();) {
				PathWrapper pw = it.next();
				String relPath = pw.getRelPath();
				File f = new File(rootDir, relPath);
				f.delete();
				String topDir = FileUtils.getTopDir(rootDir, f);
				if (topDir != null) {
					if (!uniqTopDirMap.containsKey(topDir)) {
						uniqTopDirMap.put(topDir, f.getAbsolutePath());
					}
				}
			}
			for (Iterator<String> iterator = uniqTopDirMap.values().iterator(); iterator
					.hasNext();) {
				String path = iterator.next();
				FileUtils.deleteSubdirs(rootDir, path);
			}
		}
	}

	public void pack() {
		super.setFollowSymlinks(true);
		super.execute();
		if (doCleanup) {
			cleanupAfter();
		}
	}

	public boolean isDoCleanup() {
		return doCleanup;
	}

	public void setDoCleanup(boolean doCleanup) {
		this.doCleanup = doCleanup;
	}
}
