package clinical.web.game;

import java.beans.BeanInfo;
import java.beans.IndexedPropertyDescriptor;
import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.io.File;
import java.io.FilenameFilter;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.URL;
import java.security.CodeSource;
import java.security.ProtectionDomain;
import java.text.NumberFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.action.ActionForm;
import org.apache.struts.util.LabelValueBean;

import clinical.server.vo.Dataclassification;
import clinical.server.vo.Storedassessment;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.ISubjectAssessmentManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.exception.SubjectAssessmentManagementException;
import clinical.web.exception.ValidationException;
import clinical.web.game.forms.DynamicDropDownSelector;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.AssessmentScoreValues;
import clinical.web.vo.ScoreValue;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentManagementHelper.java,v 1.32 2007/04/12 19:40:06
 *          bozyurt Exp $
 */
public class AssessmentManagementHelper {
	/** hash table which keys each form bean class with the assessment name */
	Map<String, FormBeanInfo> formBeanMap = new LinkedHashMap<String, FormBeanInfo>(
			17);
	/** hash table which keys each form bean class with the bean class */
	Map<Class<?>, FormBeanInfo> formBeanClassMap = new LinkedHashMap<Class<?>, FormBeanInfo>(
			17);

	private static Log log = LogFactory.getLog(AssessmentManagementHelper.class);
	protected static AssessmentManagementHelper instance = null;

	/** cache holding introspected beans keyed by the bean class */
	Map<Class<?>, Map<String, PropertyDescriptor>> propertyDescriptorMap = Collections
			.synchronizedMap(new LinkedHashMap<Class<?>, Map<String, PropertyDescriptor>>(
					17));
	protected static final SimpleDateFormat sdf = new SimpleDateFormat(
			"MM/dd/yyyy HH:mm");
	protected static final SimpleDateFormat df = new SimpleDateFormat(
			"MM/dd/yyyy");
	protected static final SimpleDateFormat tf = new SimpleDateFormat("HH:mm");
	protected static NumberFormat numberFormat = NumberFormat.getInstance();

	public final static int ADD_METHOD = 1;
	public final static int SIZE_METHOD = 2;
	public final static int REMOVE_LAST_METHOD = 3;

	protected AssessmentManagementHelper() {
	}

	public synchronized static AssessmentManagementHelper getInstance() {
		if (instance == null) {
			throw new RuntimeException(
					"AssessmentManagementHelper is not properly initialized!");
		}
		return instance;
	}

	/**
	 * 
	 * @param formBeanPackageName
	 * @return
	 */
	public synchronized static AssessmentManagementHelper getInstance(
			String formBeanPackageName) {
		if (instance == null) {
			instance = new AssessmentManagementHelper();
			instance.cacheAssessmentFormBeans(formBeanPackageName);
		}
		return instance;
	}

	/**
	 * caches Assessment Form beans for increased efficiency. The algorithm used
	 * is as follows;
	 * <ul>
	 * <li>From the package name, determine the full path of the class files
	 * for the form beans.</li>
	 * <li>For each form bean detected </li>
	 * <ul>
	 * <li>question the form bean class about its assessment data as stored in
	 * the database and form page variable mapping information</li>
	 * </ul>
	 * </ul>
	 * 
	 * @param formBeanPackageName
	 *            the Java package name for the assessment form beans for Struts
	 * 
	 * @see AssessmentManagementHelper#FormBeanInfo
	 */
	@SuppressWarnings("unchecked")
	protected void cacheAssessmentFormBeans(String formBeanPackageName) {

		ProtectionDomain pd = AssessmentManagementHelper.class
				.getProtectionDomain();
		CodeSource codeSource = pd.getCodeSource();
		URL codeLoc = codeSource.getLocation();
		String fileName = codeLoc.getFile();

		// infer the location of Assessment form beans from this path
		log.info("fileName=" + fileName);
		if (fileName.endsWith(".class")) {
			int idx = fileName.lastIndexOf("classes");
			if (idx != -1) {
				fileName = fileName.substring(0, idx + "classes".length() + 1);
			}
		}

		File packagePath = new File(fileName, formBeanPackageName.replace('.',
				File.separatorChar));
		log.info("package path=" + packagePath);

		File[] formBeanClassFiles = packagePath.listFiles(new FilenameFilter() {
			public boolean accept(File dir, String name) {
				if (!name.endsWith(".class")) {
					return false;
				}
				if (name.indexOf('$') != -1) {
					return false;
				}
				return true;
			}
		});
		if (formBeanClassFiles == null) {
			return;
		}

		// question each form bean class about its assessment data as stored in
		// the database
		// and form page variable mapping information
		for (int i = 0; i < formBeanClassFiles.length; i++) {
			log.info("formBeanClassFiles[" + i + "]=" + formBeanClassFiles[i]);

			String className = fromPath2FullyQualifiedClassName(
					formBeanClassFiles[i].getAbsolutePath(), fileName);
			log.info("className=" + className);
			if (className.endsWith("DynamicDropDownSelector")) {
				continue;
			}
			try {
				Class<?> c = Class.forName(className);

				Method m = c.getMethod("getAssessmentName", new Class[] {});
				String assessmentName = (String) m
						.invoke(null, new Object[] {});
				if (log.isDebugEnabled()) {
					log.debug("** got assessment name as " + assessmentName);

				}
				FormBeanInfo fbi = new FormBeanInfo(assessmentName, className);
				// populate PageVariableInfos
				m = c.getMethod("getVariableMap", new Class[] {});
				if (log.isDebugEnabled()) {
					log.debug("** got  getVariableMap method " + m);
				}
				Map<Object, PageVariableInfo> variableMap = (Map<Object, PageVariableInfo>) m
                  .invoke(null, new Object[]{});
				if (log.isDebugEnabled()) {
					log.debug("** got  getVariableMap.size() as  "
							+ variableMap.size());
				}
				for (Map.Entry<Object, PageVariableInfo> entry : variableMap
						.entrySet()) {
					Object varInfo = entry.getValue();
					PageVariableInfo pvi = populatePageVariableInfo(varInfo);
					fbi.addPageVariableInfo(pvi);
				}

				// populate PageQuestionInfos
				try {
					m = c.getMethod("getPageQuestionsMap", new Class[] {});
					Map<Integer, List<Object>> pageQuestionsMap = (Map<Integer, List<Object>>) m
							.invoke(null, new Object[] {});
					for (Map.Entry<Integer, List<Object>> entry : pageQuestionsMap.entrySet()) {
						List<Object> list = entry.getValue();
						for (Object qi : list) {
							PageQuestionInfo pqi = populatePageQuestionInfo(qi);
							fbi.addPageQuestionInfo(pqi);
						}
					}
				} catch (NoSuchMethodException nsme) {
					log.warn(nsme.getMessage());
				}

				// populate mandatory field metadata map
				try {
					m = c.getMethod("getMandatoryFieldMetaDataMap",
							new Class[] {});
					Map<String, Map<String, String>> mandatoryFieldMetaDataMap = (Map<String, Map<String, String>>) m
							.invoke(null, new Object[] {});
					for (Map.Entry<String, Map<String, String>> entry : mandatoryFieldMetaDataMap
							.entrySet()) {
						String fieldName = entry.getKey();
						Map<String, String> mdMap = entry.getValue();
						MandatoryFieldMetaData mfmd = new MandatoryFieldMetaData(
								fieldName, mdMap);
						fbi.addMandatoryFieldMetaData(mfmd);
					}

				} catch (NoSuchMethodException nsme) {
					log.warn(nsme.getMessage());
				}

				// log.info( fbi.toString() );
				formBeanMap.put(assessmentName, fbi);
				formBeanClassMap.put(c, fbi);
			} catch (Exception x) {
				log.error("", x);
				x.printStackTrace();
			}
		}
	}

	protected PageVariableInfo populatePageVariableInfo(Object varInfo)
			throws Exception {
		Class<?> clazz = varInfo.getClass();
		Method m = clazz.getMethod("getPageNumber", new Class[] {});
		Integer pageNum = (Integer) m.invoke(varInfo, new Object[] {});
		m = clazz.getMethod("getFormVarName", new Class[] {});
		String formVarName = (String) m.invoke(varInfo, new Object[] {});
		m = clazz.getMethod("getDbVarName", new Class[] {});
		String dbVarName = (String) m.invoke(varInfo, new Object[] {});
		PageVariableInfo pvi = new PageVariableInfo(pageNum.intValue(),
				formVarName, dbVarName);
		m = clazz.getMethod("getMetaDataMap", new Class[] {});
		@SuppressWarnings("unchecked")
		Map<String, String> metaDataMap = (Map<String, String>) m.invoke(
				varInfo, new Object[] {});
		if (metaDataMap != null) {
			for (Map.Entry<String, String> entry : metaDataMap.entrySet()) {
				pvi.addMetaData(entry.getKey(), entry.getValue());
			}
		}
		return pvi;
	}

	protected PageQuestionInfo populatePageQuestionInfo(Object questionInfo)
			throws Exception {
		Class<?> clazz = questionInfo.getClass();
		Method m = clazz.getMethod("getPageNumber", new Class[] {});
		Integer pageNum = (Integer) m.invoke(questionInfo, new Object[] {});
		m = clazz.getMethod("getQuestionNumber", new Class[] {});
		Integer questionNum = (Integer) m.invoke(questionInfo, new Object[] {});
		m = clazz.getMethod("getType", new Class[] {});
		String type = (String) m.invoke(questionInfo, new Object[] {});
		m = clazz.getMethod("getMinAnswer", new Class[] {});
		Integer minAnswer = (Integer) m.invoke(questionInfo, new Object[] {});
		m = clazz.getMethod("getMaxAnswer", new Class[] {});
		Integer maxAnswer = (Integer) m.invoke(questionInfo, new Object[] {});
		m = clazz.getMethod("getScoreNames", new Class[] {});
		@SuppressWarnings("unchecked")
		List<String> scoreNames = (List<String>) m.invoke(questionInfo,
				new Object[] {});

		PageQuestionInfo pqi = new PageQuestionInfo(pageNum.intValue(),
				questionNum.intValue(), type, minAnswer.intValue(), maxAnswer
						.intValue());

		boolean multipleAnswer = type.equals("multiple-answer");
		m = null;
		if (multipleAnswer) {
			m = clazz.getMethod("getIDForScoreName",
					new Class[] { String.class });
		}
		for (String scoreName : scoreNames) {
			pqi.addScoreName(scoreName);
			if (m != null) {
				Integer scoreID = (Integer) m.invoke(questionInfo,
						new Object[] { scoreName });
				pqi.addScoreNameIDAssoc(scoreName, scoreID);
			}

		}
		return pqi;
	}

	/**
	 * From a given absolute path extracts the fully qualified classname. A
	 * fully qualified classname includes the package path.
	 * 
	 * @param path
	 *            the absolute path to a Java class file
	 * @param pathRoot
	 *            the root (static) part of the <code>path</code>
	 * @return the fully qualified classname excluding <code>.class</code>
	 *         suffix
	 */
	public static String fromPath2FullyQualifiedClassName(String path,
			String pathRoot) {
		pathRoot = toPlatformIndependentPath(pathRoot);
		path = toPlatformIndependentPath(path);
		int idx = path.indexOf(pathRoot);
		if (idx != 0) {
			log.info("pathRoot=" + pathRoot + " - path=" + path);
			return null;
		}
		String part = path.substring(pathRoot.length());
		part = part.replace('/', '.');
		idx = part.lastIndexOf(".class");
		if (idx != -1) {
			part = part.substring(0, idx);
		}
		return part;
	}

	static String toPlatformIndependentPath(String aPath) {
		// String osName = System.getProperty("os.name").toLowerCase();
		aPath = aPath.replace(File.separatorChar, '/');

		if (aPath.startsWith("/") && aPath.indexOf(':') != -1) {
			aPath = aPath.substring(1);
		}
		return aPath;
	}

	/**
	 * Returns a list of assessment names for which there is form bean generated
	 * 
	 * @return a list of assessment names for which there is form bean generated
	 */
	public List<String> getAssessmentNames() {
		List<String> asNames = new LinkedList<String>();
		for (Iterator<String> iter = formBeanMap.keySet().iterator(); iter
				.hasNext();) {
			String asName = (String) iter.next();
			asNames.add(asName);
		}
		return asNames;
	}

	public String findFormBeanClassName(String assessmentName) {
		FormBeanInfo fbi = formBeanMap.get(assessmentName);
		if (fbi == null) {
			return null;
		}
		return fbi.getFormBeanClass();
	}

	/**
	 * given an assessment name gets the score name metadata as generated by
	 * CALM in the corresponding form bean for each score of the assessment.
	 * 
	 * @param assessmentName
	 *            String
	 * @return Map
	 */
	public Map<String, Map<String, String>> prepareDbVarMetaDataMap(
			String assessmentName) {
		FormBeanInfo fbi = formBeanMap.get(assessmentName);
		assert (fbi != null);
		Map<String, Map<String, String>> dbVarMetaDataMap = new HashMap<String, Map<String, String>>(
				23);
		for (Map<String, PageVariableInfo> pageVarsMap : fbi
				.getVarsPerPageMap().values()) {
			for (PageVariableInfo pvi : pageVarsMap.values()) {
				if (pvi.getMetaDataMap() != null) {
					dbVarMetaDataMap.put(pvi.getDbVarName(), pvi
							.getMetaDataMap());
				}
			}
		}
		return dbVarMetaDataMap;
	}

	/**
	 * Given a Java class, returns all its properties as
	 * <code>PropertyDescriptor</code> objects in a hash table keyed by the
	 * property name
	 * 
	 * @param clazz
	 *            aJava class
	 * @return a hash table of <code>PropertyDescriptor</code> objects keyed
	 *         by the property name
	 * @throws IntrospectionException
	 * 
	 * @see java.beans.PropertyDescriptor
	 */
	protected Map<String, PropertyDescriptor> getDescriptorsMap(Class<?> clazz)
			throws IntrospectionException {
		Map<String, PropertyDescriptor> descriptorsMap = propertyDescriptorMap
				.get(clazz);
		if (descriptorsMap == null) {
			descriptorsMap = new HashMap<String, PropertyDescriptor>(17);
			BeanInfo bi = Introspector.getBeanInfo(clazz);
			PropertyDescriptor[] descriptors = (PropertyDescriptor[]) bi
					.getPropertyDescriptors().clone();
			for (int i = 0; i < descriptors.length; i++) {
				descriptorsMap.put(descriptors[i].getName(), descriptors[i]);
			}
			propertyDescriptorMap.put(clazz, descriptorsMap);
		}
		return descriptorsMap;
	}

	protected static Class<?> getTypeClass(String type) {
		if (type.equalsIgnoreCase("integer")) {
			return int.class;
		}
		if (type.equalsIgnoreCase("float")) {
			return float.class;
		}
		if (type.equalsIgnoreCase("boolean")) {
			return boolean.class;
		}
		return String.class;
	}

	/**
	 * sets the value of a property in a form bean.
	 * 
	 * @param pageNumber
	 *            the page number of the assessment form
	 * @param dbVariableName
	 *            the name of the score as stored in the database
	 * @param object
	 *            instantiated Struts form bean object
	 * @param objectClazz
	 *            Struts form bean class
	 * @param value
	 *            the value to be set
	 * @param type
	 *            type of the property <code>integer</code>,<code>float</code>
	 *            or <code>boolean</code>
	 * @param idx
	 *            for indexed properties use -1 for regular setter
	 * 
	 * @throws IntrospectionException
	 * @throws java.lang.IllegalAccessException
	 * @throws InvocationTargetException
	 */
	public void setProperty(int pageNumber, String dbVariableName,
			Object object, Class<?> objectClazz, String value, String type,
			int idx) throws IntrospectionException, IllegalAccessException,
			InvocationTargetException, SecurityException, NoSuchMethodException {

		if (value == null) {
			return;
		}
		Class<?> clazz = objectClazz;
		if (objectClazz == null) {
			clazz = object.getClass();
		}

		Map<String, PropertyDescriptor> descriptorsMap = prepareAndGetDescriptorsMap(clazz);

		FormBeanInfo fbi = formBeanClassMap.get(clazz);

		Map<String, PageVariableInfo> pageVars = fbi.getVarsPerPageMap().get(
				new Integer(pageNumber));
		PageVariableInfo pvi = pageVars.get(dbVariableName);
		PropertyDescriptor pd = null;
		Object convertedValue = value;
		// type conversion
		if (value.length() > 0) {
			if (type.equalsIgnoreCase("integer")) {
				convertedValue = new Integer(value);
			} else if (type.equalsIgnoreCase("float")) {
				try {
					Number number = numberFormat.parse(value);
					convertedValue = new Float(number.floatValue());
				} catch (ParseException ex) {
					log.error("setProperty", ex);
					// convertedValue = new
					// Float(Constants.FLOAT_NOT_SET_VALUE);
					// the missing value is shown empty in the form now
					convertedValue = "";
				}
			} else if (type.equalsIgnoreCase("boolean")) {
				convertedValue = new Boolean(value);
			}
		}

		Method m = null;
		if (idx >= 0) {
			String methodName = GenUtils.createAccessorName(pvi
					.getFormVarName(), "set");
			Class<?> typeClazz = getTypeClass(type);
			try {
				m = objectClazz.getMethod(methodName, new Class[] { int.class,
						typeClazz });

				m.invoke(object, new Object[] { new Integer(idx),
						convertedValue });
			} catch (NoSuchMethodException nsme) {
				// try String class
				m = objectClazz.getMethod(methodName, new Class[] { int.class,
						String.class });
				m.invoke(object, new Object[] { new Integer(idx),
						convertedValue.toString() });
			}

		} else {
			pd = descriptorsMap.get(pvi.getFormVarName());
			if (pd != null) {
				m = pd.getWriteMethod();

				if (pd.getPropertyType() == boolean.class
						|| pd.getPropertyType() == Boolean.class) {

					// TODO: check if necessary for all String property form
					// beans

					if (convertedValue instanceof Integer) {
						// need to convert integer value to boolean
						if (((Integer) convertedValue).intValue() == 0) {
							convertedValue = new Boolean(false);
						} else {
							convertedValue = new Boolean(true);
						}
					}
				}
				try {
					if (m.getParameterTypes()[0] == String.class) {
						m.invoke(object, new Object[] { convertedValue
								.toString() });
					} else {
						// boolean values for checkboxes are exceptions, since
						// they have
						// only two state , checked and unchecked
						if (pd.getPropertyType() == boolean.class
								|| pd.getPropertyType() == Boolean.class) {
							if (convertedValue instanceof String) {
								// missing boolean value means false boolean
								// value
								m.invoke(object, new Object[] { new Boolean(
										false) });
							} else {
								m.invoke(object,
										new Object[] { convertedValue });
							}
						} else {
							m.invoke(object, new Object[] { convertedValue });
						}
					}
				} catch (IllegalArgumentException iae) {
					log.error("method=" + m.getName() + " convertedValue="
							+ convertedValue + " - "
							+ convertedValue.getClass());
					throw iae;
				}
			} else {
				log.info("*** pd was null pvi = " + pvi.toString());
			}
		}
	}

	private Map<String, PropertyDescriptor> prepareAndGetDescriptorsMap(
			Class<?> clazz) throws IntrospectionException {
		Map<String, PropertyDescriptor> descriptorsMap = propertyDescriptorMap
				.get(clazz);
		if (descriptorsMap == null) {
			descriptorsMap = new HashMap<String, PropertyDescriptor>(17);
			BeanInfo bi = Introspector.getBeanInfo(clazz);
			PropertyDescriptor[] descriptors = (PropertyDescriptor[]) bi
					.getPropertyDescriptors().clone();
			for (int i = 0; i < descriptors.length; i++) {
				descriptorsMap.put(descriptors[i].getName(), descriptors[i]);
			}
			propertyDescriptorMap.put(clazz, descriptorsMap);
		}
		return descriptorsMap;
	}

	protected PageQuestionInfo findPageQuestionInfo(FormBeanInfo fbi,
			int pageNumber, String dbVariableName) {
		Map<Integer, PageQuestionInfo> pageQuestions = fbi
				.getPageQuestions(pageNumber);
		if (pageQuestions == null) {
			return null;
		}
		for (PageQuestionInfo pqi : pageQuestions.values()) {
			for (String scoreName : pqi.getScoreNames()) {
				if (dbVariableName.equals(scoreName)) {
					return pqi;
				}
			}
		}
		return null;
	}

	public Map<String, PageQuestionInfo> prepareDbVar2PageQuestionInfoMap(
			int pageNumber, Object formBean, Class<?> fbClazz,
			List<String> dbVarNames) throws IntrospectionException {

		Map<String, PageQuestionInfo> dbVar2PqiMap = new LinkedHashMap<String, PageQuestionInfo>(
				11);
		Class<?> clazz = fbClazz;
		if (fbClazz == null) {
			clazz = formBean.getClass();
		}
		// prepares descriptors cache if necessary for this class
		getDescriptorsMap(clazz);

		FormBeanInfo fbi = formBeanClassMap.get(clazz);
		for (String dbVarName : dbVarNames) {
			PageQuestionInfo pqi = findPageQuestionInfo(fbi, pageNumber,
					dbVarName);
			if (pqi != null) {
				dbVar2PqiMap.put(dbVarName, pqi);
			}
		}
		return dbVar2PqiMap;
	}

	/**
	 * dynamically invokes the corresponding method of the specified type on the
	 * passed form bean. The available method types are
	 * <ul>
	 * <li>AssessmentManagementHelper.SIZE_METHOD
	 * <li>AssessmentManagementHelper.ADD_METHOD
	 * <li>AssessmentManagementHelper.REMOVE_LAST_METHOD
	 * </ul>
	 * 
	 * @param pageNumber
	 * @param dbVariableName
	 * @param methodType
	 * @param args
	 * @param object
	 * @param objectClazz
	 * @return
	 * @throws IntrospectionException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 * @throws SecurityException
	 * @throws NoSuchMethodException
	 */
	public Object runMethod(int pageNumber, String dbVariableName,
			int methodType, Object[] args, Object object, Class<?> objectClazz)
			throws IntrospectionException, IllegalAccessException,
			InvocationTargetException, SecurityException, NoSuchMethodException {
		Class<?> clazz = objectClazz;
		if (objectClazz == null) {
			clazz = object.getClass();
		}

		FormBeanInfo fbi = formBeanClassMap.get(clazz);

		Map<String, PageVariableInfo> pageVars = fbi.getPageVars(pageNumber);
		if (pageVars == null) {
			log.error("pageVars is null for page " + pageNumber);
		}
		PageVariableInfo pvi = pageVars.get(dbVariableName);
		if (pvi != null) {
			if (methodType == SIZE_METHOD) {
				String methodName = GenUtils.createAccessorName(pvi
						.getFormVarName(), "get")
						+ "Size";
				Method m = clazz.getMethod(methodName, new Class[0]);
				return m.invoke(object, new Object[0]);
			} else if (methodType == ADD_METHOD) {
				String methodName = GenUtils.createAccessorName(pvi
						.getFormVarName(), "add");

				Class<?>[] parameterTypes = new Class<?>[args.length];
				for (int i = 0; i < args.length; i++) {
					Class<?> c = args[i].getClass();
					if (c == Integer.class) {
						parameterTypes[i] = int.class;
					} else if (c == Float.class) {
						parameterTypes[i] = float.class;
					} else {
						parameterTypes[i] = c;
					}
				}

				Method m = clazz.getMethod(methodName, parameterTypes);
				return m.invoke(object, args);
			} else if (methodType == REMOVE_LAST_METHOD) {
				String methodName = GenUtils.createAccessorName(pvi
						.getFormVarName(), "removeLast");
				Method m = clazz.getMethod(methodName, new Class[0]);
				return m.invoke(object, args);
			} else {
				throw new UnsupportedOperationException(
						"Not a known method type:" + methodType);
			}
		} else {
			log.error("pvi is null for " + dbVariableName);
			return null;
		}
	}

	/**
	 * 
	 * Returns the value of a particular property from a form bean given
	 * <code>dbVariableName</code>. <br>
	 * A form variable is different than a database variable name (property). A
	 * <i>database variable</i> name is the score name for the particular
	 * assessment score. A form variable name is the name of the form bean
	 * property that corresponds to the score in the assessment. They may have
	 * different values.
	 * 
	 * 
	 * @param pageNumber
	 *            the page number of the assessment form
	 * @param dbVariableName
	 *            the name of the score as stored in the database
	 * @param object
	 *            instantiated Struts form bean object
	 * @param objectClazz
	 *            Struts form bean class
	 * @return the value of a particular property from a form bean
	 * @throws IntrospectionException
	 * @throws java.lang.IllegalAccessException
	 * @throws InvocationTargetException
	 */
	public Object getProperty(int pageNumber, String dbVariableName, int idx,
			Object object, Class<?> objectClazz) throws IntrospectionException,
			IllegalAccessException, InvocationTargetException,
			SecurityException, NoSuchMethodException {
		/** @todo refactor */
		Class<?> clazz = objectClazz;
		if (objectClazz == null) {
			clazz = object.getClass();
		}

		Map<String, PropertyDescriptor> descriptorsMap = getDescriptorsMap(clazz);

		FormBeanInfo fbi = formBeanClassMap.get(clazz);

		Map<String, PageVariableInfo> pageVars = fbi.getPageVars(pageNumber);
		if (pageVars == null) {
			log.error("pageVars is null for page " + pageNumber);
		}
		PageVariableInfo pvi = pageVars.get(dbVariableName);
		if (pvi != null) {
			PropertyDescriptor pd = descriptorsMap.get(pvi.getFormVarName());

			if (idx >= 0) {
				String methodName = GenUtils.createAccessorName(pd.getName(),
						"get");
				Method m = objectClazz.getMethod(methodName,
						new Class[] { int.class });
				return m.invoke(object, new Object[] { new Integer(idx) });
			} else {
				Method m = pd.getReadMethod();
				if (m == null) {
					log.error("no read method for form variable "
							+ pvi.getFormVarName());
				}
				return m.invoke(object, new Object[] {});
			}
		} else {
			log.error("pvi is null for " + dbVariableName);
			return null;
		}
	}

	/**
	 * Returns the attached metadata for a form variable for the given page of
	 * the assessment form and associated with the provided score name in the
	 * database.
	 * 
	 * @param pageNumber
	 * @param dbVariableName
	 * @param object
	 * @param objectClazz
	 * @return
	 * @throws IntrospectionException
	 */
	public Map<String, String> getPropertyMetaData(int pageNumber,
			String dbVariableName, Object object, Class<?> objectClazz)
			throws IntrospectionException {
		Class<?> clazz = objectClazz;
		if (objectClazz == null) {
			clazz = object.getClass();
		}

		FormBeanInfo fbi = formBeanClassMap.get(clazz);

		Map<String, PageVariableInfo> pageVars = fbi.getPageVars(pageNumber);
		if (pageVars == null) {
			log.error("pageVars is null for page " + pageNumber);
		}
		PageVariableInfo pvi = pageVars.get(dbVariableName);
		if (pvi != null) {
			return pvi.getMetaDataMap();
		} else {
			log.error("pvi is null for " + dbVariableName);
			return null;
		}
	}

	/**
	 * 
	 * @param fieldName
	 *            the mandatory field name
	 * @param objectClazz
	 *            the form bean class
	 * @return Map MandatoryFieldMetaData for the mandatory field (if any) as
	 *         name, value pairs or null
	 */
	public Map<String, String> getMandatoryFieldMetaData(String fieldName,
			Class<?> objectClazz) {
		FormBeanInfo fbi = formBeanClassMap.get(objectClazz);
		if (fbi == null) {
			log.warn("fbi was null!");
			return null;
		}
		MandatoryFieldMetaData mfmd = fbi.getMandatoryFieldMetaData(fieldName);
		return mfmd.getMetaDataMap();
	}

	/**
	 * Given a form bean as <code>object</code>, a database variable name and
	 * the the form page number on which to search for the corresponding form
	 * variable, find the type class of corresponding form variable by
	 * introspection.
	 * 
	 * @param dbVariableName
	 * @param object
	 * @param objectClazz
	 * @param pageNumber
	 * @return
	 * @throws IntrospectionException
	 * @throws IllegalAccessException
	 * @throws InvocationTargetException
	 */
	public Class<?> getPropertyClass(String dbVariableName, Object object,
			Class<?> objectClazz, Integer pageNumber)
			throws IntrospectionException, IllegalAccessException,
			InvocationTargetException {
		Class<?> clazz = objectClazz;
		if (objectClazz == null) {
			clazz = object.getClass();
		}

		Map<String, PropertyDescriptor> descriptorsMap = getDescriptorsMap(clazz);
		FormBeanInfo fbi = formBeanClassMap.get(clazz);

		Map<String, PageVariableInfo> pageVars = fbi.getPageVars(pageNumber);
		if (pageVars == null) {
			log.error("pageVars is null for page " + pageNumber);
		}
		PageVariableInfo pvi = pageVars.get(dbVariableName);
		PropertyDescriptor pd = descriptorsMap.get(pvi.getFormVarName());
		Class<?> type = pd.getPropertyType();
		if (type == null) {
			return ((IndexedPropertyDescriptor) pd).getIndexedPropertyType();
		}
		return type;
	}

	/**
	 * Returns the value of a particular property from a form bean.
	 * 
	 * @param propertyName
	 *            name of the Struts form bean property
	 * @param object
	 *            instantiated Struts form bean object
	 * @param objectClazz
	 *            Struts form bean class
	 * @return the value of a particular property from the form bean
	 * @throws IntrospectionException
	 * @throws java.lang.IllegalAccessException
	 * @throws InvocationTargetException
	 */
	public Object getFormProperty(String propertyName, Object object,
			Class<?> objectClazz) throws IntrospectionException,
			IllegalAccessException, InvocationTargetException {
		/** @todo refactor */
		Class<?> clazz = objectClazz;
		if (objectClazz == null) {
			clazz = object.getClass();
		}

		Map<String, PropertyDescriptor> descriptorsMap = getDescriptorsMap(clazz);

		PropertyDescriptor pd = descriptorsMap.get(propertyName);
		if (pd == null) {
			log.error("pd was null for property name " + propertyName);
		}
		Method m = pd.getReadMethod();
		return m.invoke(object, new Object[] {});
	}

	public boolean hasFormProperty(String propertyName, Object object,
			Class<?> objectClazz) throws IntrospectionException {
		Class<?> clazz = objectClazz;
		if (objectClazz == null) {
			clazz = object.getClass();
		}
		Map<String, PropertyDescriptor> descriptorsMap = getDescriptorsMap(clazz);
		PropertyDescriptor pd = descriptorsMap.get(propertyName);
		return (pd != null);
	}

	/**
	 * Sets the value of a property in a Struts form bean.
	 * 
	 * @param propertyName
	 *            name of the Struts form bean property
	 * @param object
	 *            instantiated Struts form bean object
	 * @param objectClazz
	 *            Struts form bean class
	 * @param value
	 * @throws IntrospectionException
	 * @throws java.lang.IllegalAccessException
	 * @throws InvocationTargetException
	 */
	public void setFormProperty(String propertyName, Object object,
			Class<?> objectClazz, Object value) throws IntrospectionException,
			IllegalAccessException, InvocationTargetException {
		Class<?> clazz = objectClazz;
		if (objectClazz == null) {
			clazz = object.getClass();
		}

		Map<String, PropertyDescriptor> descriptorsMap = getDescriptorsMap(clazz);
		PropertyDescriptor pd = descriptorsMap.get(propertyName);
		if (pd == null) {
			log.error("setFormProperty:" + propertyName + " " + object
					+ " value=" + value);
		}
		Method m = pd.getWriteMethod();
		m.invoke(object, new Object[] { value });
	}

	/**
	 * 
	 * @param pageNumber
	 * @param formBeanClazz
	 * @return
	 */
	public List<String> getDataVariableNames(int pageNumber,
			Class<?> formBeanClazz) {
		FormBeanInfo fbi = formBeanClassMap.get(formBeanClazz);
		if (fbi == null) {
			return null;
		}
		Map<String, PageVariableInfo> pageVars = fbi.getPageVars(pageNumber);
		if (pageVars == null) {
			return new ArrayList<String>(0);
		}
		List<String> dvNames = new ArrayList<String>(pageVars.size());
		for (PageVariableInfo pvi : pageVars.values()) {
			dvNames.add(pvi.getDbVarName());
		}
		return dvNames;
	}

	public Set<Integer> getPagesAffected(List<?> questionIDList,
			Class<?> formBeanClazz) {
		FormBeanInfo fbi = formBeanClassMap.get(formBeanClazz);
		if (fbi == null) {
			return null;
		}
		Set<Integer> pages = new TreeSet<Integer>();
		for (Object o : questionIDList) {
		   Integer qid = (Integer) o;
			PageQuestionInfo pqi = fbi.getPageQuestionInfo(qid);
			pages.add(new Integer(pqi.getPageNumber()));
		}
		return pages;
	}

	/**
	 * given a list of question IDs as <ode>Integer</code> objects returns the
	 * corresponding database variable names (score names).
	 * 
	 * @param pageNumber
	 * @param formBeanClazz
	 * @param questionIDList
	 * @return
	 */
	public List<String> getDataVariableNames(int pageNumber,
			Class<?> formBeanClazz, List<?> questionIDList) {
		FormBeanInfo fbi = formBeanClassMap.get(formBeanClazz);
		if (fbi == null) {
			return null;
		}
		Map<String, PageVariableInfo> pageVars = fbi.getPageVars(pageNumber);
		if (pageVars == null) {
			return new LinkedList<String>();
		}

		Map<String, String> questScoreMap = new HashMap<String, String>();
		for (Object o : questionIDList) {
		   Integer qid = (Integer) o;
			PageQuestionInfo pqi = fbi.getPageQuestionInfo(qid);
			for (String scoreName : pqi.getScoreNames()) {
				questScoreMap.put(scoreName, scoreName);
			}
		}
		List<String> dvNames = new ArrayList<String>();
		for (Iterator<PageVariableInfo> iter = pageVars.values().iterator(); iter
				.hasNext();) {
			PageVariableInfo pvi = iter.next();
			if (questScoreMap.get(pvi.getDbVarName()) != null) {
				dvNames.add(pvi.getDbVarName());
			}
		}
		return dvNames;
	}

	public List<Integer> getQuestionIDsForPage(int pageNumber,
			Class<?> formBeanClazz, List<Integer> questionIDList) {
		FormBeanInfo fbi = formBeanClassMap.get(formBeanClazz);
		if (fbi == null) {
			return null;
		}
		List<Integer> qidList = new ArrayList<Integer>(5);
		for (Integer qid : questionIDList) {
			PageQuestionInfo pqi = fbi.getPageQuestionInfo(qid);
			if (pqi.getPageNumber() == pageNumber) {
				qidList.add(qid);
			}
		}
		return qidList;
	}

	/**
	 * Given a Form Bean class returns the total number of pages in the online
	 * assessment if the form bean is not registered, returns -1
	 * 
	 * @param formBeanClazz
	 * @return the total number of pages in the online assessment , or -1 if the
	 *         form bean is not registered
	 */
	public int getTotalNumberOfPages(Class<?> formBeanClazz) {
		FormBeanInfo fbi = formBeanClassMap.get(formBeanClazz);
		if (fbi == null) {
			return -1;
		}
		return fbi.getVarsPerPageMapSize();
	}

	/**
	 * Given a Form Bean class returns true if the form has a cover page. Cover
	 * page is infered to exists if there is no variables (scores) for the first
	 * page.
	 * 
	 * @param formBeanClazz
	 * @return
	 */
	public boolean hasCoverPage(Class<?> formBeanClazz) {
		FormBeanInfo fbi = formBeanClassMap.get(formBeanClazz);
		if (fbi == null) {
			return false;
		}
		boolean hasPage1 = false;
		for (Integer pageNum : fbi.getVarsPerPageMap().keySet()) {
			if (pageNum.intValue() == 1) {
				hasPage1 = true;
				break;
			}
		}
		return (!hasPage1);
	}

	public Map<Integer, List<String>> getDataVarNamesPerPage(
			Class<?> formBeanClazz) {
		FormBeanInfo fbi = formBeanClassMap.get(formBeanClazz);
		if (fbi == null) {
			return null;
		}
		Map<Integer, List<String>> dbVarNamesPerPageMap = new LinkedHashMap<Integer, List<String>>(
				17);
		for (Integer pageNum : fbi.getVarsPerPageMap().keySet()) {
			Map<String, PageVariableInfo> pageVars = fbi.getPageVars(pageNum);
			List<String> dvNames = new ArrayList<String>(pageVars.size());
			for (PageVariableInfo pvi : pageVars.values()) {
				dvNames.add(pvi.getDbVarName());
			}
			dbVarNamesPerPageMap.put(pageNum, dvNames);
		}
		return dbVarNamesPerPageMap;
	}

	/**
	 * 
	 * @param pageNumber
	 * @param dbVarName
	 * @param formBeanClazz
	 * @return
	 */
	public String getFormBeanVariableName(int pageNumber, String dbVarName,
			Class<?> formBeanClazz) {
		FormBeanInfo fbi = formBeanClassMap.get(formBeanClazz);
		if (fbi == null) {
			return null;
		}
		Map<String, PageVariableInfo> pageVars = fbi.getPageVars(pageNumber);
		PageVariableInfo pvi = pageVars.get(dbVarName);
		return pvi.getFormVarName();
	}

	//
	// static methods
	//
	/**
	 * set the hidden form field value for the notes field of the score value in
	 * the db.
	 * 
	 * @param form
	 *            ActionForm
	 * @param amh
	 *            AssessmentManagementHelper
	 * @param dataClassifications
	 *            List
	 * @param formClazz
	 *            Class
	 * @param scoreValue
	 *            {@link clinical.web.vo.ScoreValue}
	 * @param pqi
	 *            {@link PageQuestionInfo}
	 * @param idx
	 *            int
	 * @param scoreID
	 *            int
	 * @throws Exception
	 */
	public static void setDataClassificationHiddenFormField(ActionForm form,
			AssessmentManagementHelper amh,
			List<Dataclassification> dataClassifications, Class<?> formClazz,
			ScoreValue scoreValue, PageQuestionInfo pqi, int idx, int scoreID)
			throws Exception {
		if (scoreValue.getClassification() != null) {
			String notesVarName = null;
			if (idx >= 0) {
				notesVarName = GenUtils.createHiddenNotesVariable(pqi
						.getQuestionNumber(), scoreID, idx);
			} else {
				notesVarName = GenUtils.createHiddenNotesVariable(pqi
						.getQuestionNumber(), idx);
			}

			String dataClassification = findDataClassification(scoreValue
					.getClassification().intValue(), dataClassifications);
			if (dataClassification != null) {

				log.info("found dataClassification=" + dataClassification
						+ " with id " + scoreValue.getClassification()
						+ " setting form variable " + notesVarName);
				amh.setFormProperty(notesVarName, form, formClazz,
						dataClassification);
			}
		}
	}

	public static String findDataClassification(int classification,
			List<Dataclassification> dataClassifications) {
		for (Dataclassification dc : dataClassifications) {
			if (dc.getUniqueid().intValue() == classification) {
				return dc.getName();
			}
		}
		return null;
	}

	public static class Params {
		String dbVarName;
		Object form;
		Class<?> formClazz;
		AssessmentManagementHelper amh;

		public Params(String dbVarName, Object form, Class<?> formClazz,
				AssessmentManagementHelper amh) {
			this.dbVarName = dbVarName;
			this.form = form;
			this.formClazz = formClazz;
			this.amh = amh;
		}
	}

	/**
	 * prepares and adds the score value of a score to the database from the
	 * corresponding form field. It checks if the user has provided a reason for
	 * not answering the question. If so a note is set for reason and the score
	 * value is ignored.
	 * 
	 * @param pqi
	 *            {@link PageQuestionInfo}
	 * @param asv
	 *            {@link clinical.web.vo.AssessmentScoreValues}
	 * @param dataClassifications
	 *            List
	 * @param pageNumber
	 *            int
	 * @param params
	 * @param reasonMap
	 * @throws Exception
	 */
	public static void addScoreValue(PageQuestionInfo pqi,
			AssessmentScoreValues asv,
			List<Dataclassification> dataClassifications, int pageNumber,
			Params params, Map<Integer, String> reasonMap) throws Exception {
		if (pqi != null && pqi.isMultiAnswer()) {
			Integer size = (Integer) params.amh.runMethod(pageNumber,
					params.dbVarName, AssessmentManagementHelper.SIZE_METHOD,
					null, params.form, params.formClazz);
			for (int i = 0; i < size.intValue(); i++) {
				Object value = params.amh.getProperty(pageNumber,
						params.dbVarName, i, params.form, params.formClazz);

				Integer scoreID = (Integer) pqi
						.getIDForScoreName(params.dbVarName);

				boolean hasNotes = AssessmentManagementHelper.hasNotes(pqi
						.getQuestionNumber(), i + 1, scoreID.intValue(),
						params.form, params.formClazz, params.amh);

				if (!hasNotes && AssessmentManagementHelper.isValueValid(value)) {
					ScoreValue sv = new ScoreValue(params.dbVarName, value
							.toString(), i + 1);
					asv.addScoreValue(sv);
				} else {

					handleNotes(params, pqi.getQuestionNumber(), i + 1, scoreID
							.intValue(), asv, dataClassifications, reasonMap);
					log.warn("No value for dbVarName=" + params.dbVarName);
				}
			}
		} else {
			Object value = params.amh.getProperty(pageNumber, params.dbVarName,
					-1, params.form, params.formClazz);
			boolean hasNotes = false;
			if (pqi != null) {
				hasNotes = AssessmentManagementHelper.hasNotes(pqi
						.getQuestionNumber(), -1, params.form,
						params.formClazz, params.amh);
			}

			if (!hasNotes && AssessmentManagementHelper.isValueValid(value)) {
				ScoreValue sv = new ScoreValue(params.dbVarName, value
						.toString(), 1);
				asv.addScoreValue(sv);
			} else {
				if (pqi != null) {
					handleNotes(params, pqi.getQuestionNumber(), -1, -1, asv,
							dataClassifications, reasonMap);
				}
				log.warn("No value for dbVarName=" + params.dbVarName);
			}
		}
	}

	protected static void handleNotes(Params params, int questionID, int idx,
			int scoreID, AssessmentScoreValues asv,
			List<Dataclassification> dataClassifications,
			Map<Integer, String> reasonMap) throws Exception {
		handleNotes(params, questionID, idx, scoreID, asv, dataClassifications,
				null, null, reasonMap);
	}

	/**
	 * 
	 * @param params
	 *            a {@link Params} object encapsulating common parameters passed
	 *            to many static methods
	 * @param questionID
	 *            the id of the question as given by CALM
	 * @param idx
	 *            if nonnegative, for multiple valued scores represents the
	 *            order index of the score value in the list of score values
	 * @param scoreID
	 *            if nonnegative indicates the id (key) of the score in a
	 *            multiscore question
	 * @param asv
	 *            {@link clinical.web.vo.AssessmentScoreValues}
	 * @param dataClassifications
	 * @param scoreValuesToBeAddedOrUpdated
	 * @param scoreValue
	 *            {@link clinical.web.vo.ScoreValue}
	 * @param reasonMap
	 * @throws Exception
	 */
	protected static void handleNotes(Params params, int questionID, int idx,
			int scoreID, AssessmentScoreValues asv,
			List<Dataclassification> dataClassifications,
			List<ScoreValue> scoreValuesToBeAddedOrUpdated, ScoreValue scoreValue,
			Map<Integer, String> reasonMap) throws Exception {
		String notesVarName = null;
		if (idx >= 0) {
			notesVarName = GenUtils.createHiddenNotesVariable(questionID,
					scoreID, idx);
		} else {
			notesVarName = GenUtils.createHiddenNotesVariable(questionID, idx);
		}

		String dataClassification = (String) params.amh.getFormProperty(
				notesVarName, params.form, params.formClazz);

		log.info("in handleNotes: notesVarName = " + notesVarName
				+ " dataClassification = " + dataClassification);

		if (dataClassification == null) {
			throw new ValidationException("no dataClassification for "
					+ notesVarName);
		}

		for (Dataclassification dc : dataClassifications) {
			if (dc.getName().equals(dataClassification)) {
				int scoreOrder = idx > 0 ? idx : 1;
				ScoreValue sv = null;
				if (scoreValue == null) {
					sv = new ScoreValue(params.dbVarName, null, scoreOrder);
					asv.addScoreValue(sv);
				} else {
					sv = scoreValue;
				}

				// unset and previous score value
				unsetScoreValue(sv);
				Integer cid = new Integer(dc.getUniqueid().intValue());
				sv.setClassification(cid);
				sv.setReason((String) reasonMap.get(cid));

				log.info("handleNotes set sv=" + sv.toString());
				// make sure the update knows about this also
				if (scoreValuesToBeAddedOrUpdated != null) {
					if (scoreValue == null) {
						scoreValuesToBeAddedOrUpdated.add(sv);
					}
				}
				break;
			}
		}
	}
	
	protected static void handleNotes4Updates(Params params, int questionID, int idx,
			int scoreID, AssessmentScoreValues asv,
			List<Dataclassification> dataClassifications,
			List<String> scoreValuesToBeAddedOrUpdated, ScoreValue scoreValue,
			Map<Integer, String> reasonMap) throws Exception {
		String notesVarName = null;
		if (idx >= 0) {
			notesVarName = GenUtils.createHiddenNotesVariable(questionID,
					scoreID, idx);
		} else {
			notesVarName = GenUtils.createHiddenNotesVariable(questionID, idx);
		}

		String dataClassification = (String) params.amh.getFormProperty(
				notesVarName, params.form, params.formClazz);

		log.info("in handleNotes: notesVarName = " + notesVarName
				+ " dataClassification = " + dataClassification);

		if (dataClassification == null) {
			throw new ValidationException("no dataClassification for "
					+ notesVarName);
		}

		for (Dataclassification dc : dataClassifications) {
			if (dc.getName().equals(dataClassification)) {
				int scoreOrder = idx > 0 ? idx : 1;
				ScoreValue sv = null;
				if (scoreValue == null) {
					sv = new ScoreValue(params.dbVarName, null, scoreOrder);
					asv.addScoreValue(sv);
				} else {
					sv = scoreValue;
				}

				// unset and previous score value
				unsetScoreValue(sv);
				Integer cid = new Integer(dc.getUniqueid().intValue());
				sv.setClassification(cid);
				sv.setReason((String) reasonMap.get(cid));

				log.info("handleNotes set sv=" + sv.toString());
				// make sure the update knows about this also
				if (scoreValuesToBeAddedOrUpdated != null) {
					if (scoreValue != null) {
						scoreValuesToBeAddedOrUpdated.add(params.dbVarName);
					} 
				}
				break;
			}
		}
	}

	public static void unsetScoreValue(ScoreValue sv) {
		if (sv.getValue() == null) {
			return;
		}
		sv.setValue(null);
	}

	/**
	 * Stores the mandatory field values gotten from the passed form bean to the
	 * passed {@link clinical.web.vo.AssessmentScoreValues} object.
	 * 
	 * @param form
	 * @param amh
	 * @param asv
	 *            {@link clinical.web.vo.AssessmentScoreValues}
	 * @param formClazz
	 * @throws ParseException
	 * @throws InvocationTargetException
	 * @throws IllegalAccessException
	 * @throws IntrospectionException
	 */
	public static void setMandatoryData(ActionForm form,
			AssessmentManagementHelper amh, AssessmentScoreValues asv,
			Class<?> formClazz) throws ParseException,
			InvocationTargetException, IllegalAccessException,
			IntrospectionException {
		String dateStr = (String) amh.getFormProperty(Constants.DATE_PROPERTY,
				form, formClazz);
		String timeStr = (String) amh.getFormProperty(Constants.TIME_PROPERTY,
				form, formClazz);
		String informantID = (String) amh.getFormProperty(
				Constants.INFORMANTID_PROPERTY, form, formClazz);
		String informantRelation = (String) amh.getFormProperty(
				Constants.INFORMANTRELATION_PROPERTY, form, formClazz);
		boolean hasClinicalRater = amh.hasFormProperty(
				Constants.CLINICALRATER_PROPERTY, form, formClazz);

		asv.setTimeStamp(convertToDate(dateStr, timeStr));
		asv.setInformantID(informantID);
		asv.setInformantRelation(informantRelation);

		if (hasClinicalRater) {
			log.info("*** hasClinicalRater");
			DynamicDropDownSelector dds = (DynamicDropDownSelector) amh
					.getFormProperty(Constants.CLINICALRATER_PROPERTY, form,
							formClazz);

			String clinicalRater = dds.getPossibleSelection(dds
					.getSelectedItemIdx());
			asv.setClinicalRater(clinicalRater);
			log.info("***** SET CLINICALRATER=" + clinicalRater);
		}
	}

	/**
	 * 
	 * @param dbID
	 * @param form
	 * @param userInfo
	 * @param amh
	 * @param isam
	 *            {@link clinical.web.ISubjectAssessmentManagement}
	 * @param subjectID
	 * @param experimentID
	 * @param visitID
	 * @param segmentID
	 * @param theAsi
	 *            {@link clinical.web.vo.AssessmentInfo}
	 * @param asv
	 *            {@link clinical.web.vo.AssessmentScoreValues}
	 * @throws Exception
	 */
	public static void setMandatoryFormInfo(String dbID, ActionForm form,
			UserInfo userInfo, AssessmentManagementHelper amh,
			ISubjectAssessmentManagement isam, String subjectID,
			int experimentID, int visitID, int segmentID,
			AssessmentInfo theAsi, AssessmentScoreValues asv) throws Exception {
		Date tsmp = asv.getTimeStamp();
		amh.setFormProperty(Constants.TIME_PROPERTY, form, form.getClass(), tf
				.format(tsmp));
		amh.setFormProperty(Constants.DATE_PROPERTY, form, form.getClass(), df
				.format(tsmp));
		Class<? extends ActionForm> formClazz = form.getClass();

		String informantID = (String) amh.getFormProperty(
				Constants.INFORMANTID_PROPERTY, form, formClazz);
		String informantRelation = (String) amh.getFormProperty(
				Constants.INFORMANTRELATION_PROPERTY, form, formClazz);

		if ((informantID == null || informantID.length() == 0)
				|| (informantRelation == null || informantRelation.length() == 0)) {
			// set informant info
			List<Storedassessment> saList = isam.getStoredAssessmentRecs(dbID,
					userInfo, subjectID, experimentID, visitID, segmentID,
					theAsi.getAssessmentID());
			Storedassessment sa = saList.get(0);
			amh.setFormProperty(Constants.INFORMANTID_PROPERTY, form,
					formClazz, sa.getInformantid());
			amh.setFormProperty(Constants.INFORMANTRELATION_PROPERTY, form,
					formClazz, sa.getInformantrelation());
		}

		boolean hasClinicalRater = amh.hasFormProperty(
				Constants.CLINICALRATER_PROPERTY, form, formClazz);
		if (hasClinicalRater) {
			// set the selected clinical rater value in the online assessment
			// from the db
			String storedClinicalRater = asv.getClinicalRater();
			if (storedClinicalRater != null
					&& storedClinicalRater.trim().length() > 0) {
				DynamicDropDownSelector dds = (DynamicDropDownSelector) amh
						.getFormProperty(Constants.CLINICALRATER_PROPERTY,
								form, formClazz);
				int idx = 0;
				for (LabelValueBean lvb : dds.getPossibleSelections()) {
					String cr = lvb.getValue();
					if (storedClinicalRater.equals(cr)) {
						dds.setSelectedItemIdx(idx);
						break;
					}
					idx++;
				}
			}
		}
	}

	public static Date convertToDate(String date, String time)
			throws ParseException {
		String dateString = date + " " + time;
		return sdf.parse(dateString);
	}

	/**
	 * checks if the passed object is a valid value. If the passed object is not
	 * an empty string or null, then it will be a valid value.
	 * 
	 * @param value
	 * @return
	 */
	public static boolean isValueValid(Object value) {
		if (value == null) {
			return false;
		}
		if (value.toString().trim().length() == 0) {
			return false;
		}
		return true;
	}

	public static boolean hasNotes(int questionID, int idx, Object form,
			Class<?> formClazz, AssessmentManagementHelper amh)
			throws Exception {
		return hasNotes(questionID, idx, -1, form, formClazz, amh);
	}

	public static boolean hasNotes(int questionID, List<Integer> questionIDList)
			throws Exception {
		for (Integer qid : questionIDList) {
			if (qid.intValue() == questionID) {
				return true;
			}
		}
		return false;
	}

	/**
	 * 
	 * @param pageNumber
	 *            the page number for the online form
	 * @param form
	 *            the Struts form bean
	 * @param amh
	 *            an instance of <code>AssessmentManagementHelper</code>
	 *            object
	 * @param dbVarNames
	 *            the list score names as represented in the database
	 * @param dataClassification
	 * @throws Exception
	 */
	public static void setNoteForQuestions(int pageNumber, Object form,
			AssessmentManagementHelper amh, List<String> dbVarNames,
			String dataClassification) throws Exception {
		Class<?> formClazz = form.getClass();
		Map<String, PageQuestionInfo> dbVar2PqiMap = amh
				.prepareDbVar2PageQuestionInfoMap(pageNumber, form, formClazz,
						dbVarNames);

		for (String dbVarName : dbVarNames) {
			PageQuestionInfo pqi = dbVar2PqiMap.get(dbVarName);
			if (pqi == null) {
				log.error("pqi = null dbVarName=" + dbVarName);
			}
			if (pqi != null && pqi.isMultiAnswer()) {
				Integer size = (Integer) amh.runMethod(pageNumber, dbVarName,
						AssessmentManagementHelper.SIZE_METHOD, null, form,
						formClazz);
				for (int i = 0; i < size.intValue(); i++) {
					Integer scoreID = (Integer) pqi
							.getIDForScoreName(dbVarName);
					setNote(pqi.getQuestionNumber(), i + 1, scoreID.intValue(),
							form, formClazz, amh, dataClassification);
				}
			} else {
				// single answer
				setNote(pqi.getQuestionNumber(), -1, -1, form, formClazz, amh,
						dataClassification);
			}
		}
	}

	/**
	 * Sets the value of the corresponding hidden form field for the particular
	 * score form field indicating a reason note for missing value.
	 * 
	 * @param questionID
	 *            the id of the question as given by CALM
	 * @param idx
	 *            if nonnegative, for multiple valued scores represents the
	 *            order index of the score value in the list of score values
	 * @param scoreID
	 *            if nonnegative indicates the id (key) of the score in a
	 *            multiscore question
	 * @param form
	 * @param formClazz
	 * @param amh
	 * @param dataClassification
	 * @throws Exception
	 */
	public static void setNote(int questionID, int idx, int scoreID,
			Object form, Class<?> formClazz, AssessmentManagementHelper amh,
			String dataClassification) throws Exception {
		String notesVarName = null;

		if (scoreID >= 0) {
			notesVarName = GenUtils.createHiddenNotesVariable(questionID,
					scoreID, idx);
		} else {
			notesVarName = GenUtils.createHiddenNotesVariable(questionID, idx);
		}
		amh.setFormProperty(notesVarName, form, formClazz, dataClassification);
	}

	public static boolean hasNotes(int questionID, int idx, int scoreID,
			Object form, Class<?> formClazz, AssessmentManagementHelper amh)
			throws Exception {
		String notesVarName = null;

		if (scoreID >= 0) {
			notesVarName = GenUtils.createHiddenNotesVariable(questionID,
					scoreID, idx);
		} else {
			notesVarName = GenUtils.createHiddenNotesVariable(questionID, idx);
		}

		String dataClassification = (String) amh.getFormProperty(notesVarName,
				form, formClazz);
		log.info("in hasNotes: notesVArName = " + notesVarName
				+ " dataClassification = " + dataClassification);

		if (dataClassification == null
				|| dataClassification.trim().length() == 0) {
			return false;
		}
		return true;
	}

	/**
	 * 
	 * @param ui
	 * @param assessmentID
	 * @param subjectID
	 * @param experimentID
	 * @param dbID
	 * @return
	 * @throws Exception
	 */
	public static AssessmentInfo findTheAssesment(UserInfo ui,
			int assessmentID, String subjectID, int experimentID, String dbID)
			throws Exception {
		ISubjectAssessmentManagement isam = ServiceFactory
				.getSubjectAssessmentManagement(dbID);

		List<AssessmentInfo> asInfos = isam.getAssessmentsForSubject(ui,
				subjectID, experimentID);
		AssessmentInfo theAsi = null;
		for (AssessmentInfo asi : asInfos) {
			if (asi.getAssessmentID() == assessmentID
					&& !asi.getScores().isEmpty()) {
				theAsi = asi;
				break;
			}
		} // for
		return theAsi;
	}

	public void prepareClinicalRaterProperty(ActionForm form,
			UserInfo userInfo, Class<?> formClazz,
			ISubjectAssessmentManagement isam) throws IntrospectionException,
			IllegalAccessException, InvocationTargetException,
			SubjectAssessmentManagementException {
		Object crValue = getFormProperty(Constants.CLINICALRATER_PROPERTY,
				form, form.getClass());
		if (crValue == null) {
			// prepare the dynamic dropdown
			Map<String, String> mdMap = getMandatoryFieldMetaData(
					Constants.CLINICALRATER_PROPERTY, form.getClass());
			if (mdMap != null && mdMap.get(Constants.QUERY_PROPERTY) != null) {
				String sqlQuery = (String) mdMap.get(Constants.QUERY_PROPERTY);
				List<String> values = isam.getDynamicQueryResults(userInfo,
						sqlQuery);
				DynamicDropDownSelector dddSelector = new DynamicDropDownSelector(
						values);
				setFormProperty(Constants.CLINICALRATER_PROPERTY, form,
						formClazz, dddSelector);
			}
		}
	}

	public Object prepareDefaultValue(ActionForm form, Class<?> formClazz,
			Integer pageNumber, String scoreName) throws Exception {
		Class<?> typeClass = getPropertyClass(scoreName, form, formClazz,
				pageNumber);
		Object defaultValue = null;
		if (typeClass == int.class) {
			defaultValue = new Integer(-10000);
		} else if (typeClass == float.class) {
			defaultValue = new Float(-10000);
		} else {
			defaultValue = "";
		}
		return defaultValue;
	}

	public static void main(String[] args) {
		AssessmentManagementHelper.getInstance("clinical.web.game.forms");
	}
}
