package clinical.web.scheduler;

import java.io.File;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: JobRunner.java 786 2013-03-20 02:21:25Z bozyurt $
 */
public class JobRunner implements Runnable {
	protected JobInfo jobInfo;
	private final JobScheduler scheduler;
	protected int stageId = -1;
	protected int numStages;
	protected boolean resumed = false;
	private long maxWaitingSecs = 3600l; // default (one hour before shelving)
	private Log log = LogFactory.getLog(JobRunner.class);

	public JobRunner(JobScheduler scheduler, JobInfo jobInfo) {
		this.scheduler = scheduler;
		this.jobInfo = jobInfo;
		numStages = jobInfo.getJob().getNumberOfStages();
		stageId = 1;
	}

	public synchronized void resume() {
		resumed = true;
	}

	public void run() {
		IJob job = jobInfo.getJob();
		if (jobInfo.getStatus() != null
				&& jobInfo.getStatus().equals(JobInfo.WAITING)) {
			// handle recovered job
			try {
				while (!resumed) {
					synchronized (this) {
						try {
							this.wait();
						} catch (InterruptedException e) {
						}
					}

					log.info("JobRunner (recovery): resumed:" + resumed);
				}
				jobInfo.setStatus(JobInfo.RUNNING);
				job.setStatus(JobInfo.RUNNING);

				scheduler.updateJobDB(jobInfo, job.getID());

				if (!jobInfo.getJob().isCanceled()) {
					if (job.getNumberOfStages() != 2) {
						throw new Exception(
								"Cannot revive a job with more than one human intervention!");
					}
					job.execute(2);
					File f = new File(scheduler.getOutputRootDir(), job.getID()
							+ ".csv");

					job.saveResults(f);
					jobInfo.setStatus(JobInfo.FINISHED);
					job.setStatus(JobInfo.FINISHED);
				} else {
					jobInfo.setStatus(JobInfo.CANCELED);
					job.setStatus(JobInfo.CANCELED);
				}

			} catch (JobException e) {
				log.error("", e);
				jobInfo.setStatus(JobInfo.FINISHED_WITH_ERR);
				job.setStatus(JobInfo.FINISHED_WITH_ERR);
				String msg = getErrorMessage(e);
				jobInfo.setErrorMsg(msg);
			} catch (Throwable t) {
				log.error("", t);
				t.printStackTrace();
				jobInfo.setStatus(JobInfo.FINISHED_WITH_ERR);
				job.setStatus(JobInfo.FINISHED_WITH_ERR);
				String msg = getErrorMessage(t);
				jobInfo.setErrorMsg(msg);
			} finally {
				synchronized (scheduler) {
					scheduler.notify();
				}
			}
			return;
		} // recovery case

		try {
			jobInfo.setStatus(JobInfo.RUNNING);
			job.setStatus(JobInfo.RUNNING);

			scheduler.updateJobDB(jobInfo, job.getID());
			boolean canceled = false;

			for (int i = 1; i <= numStages; i++) {
				job.execute(i);
				if (jobInfo.getJob().isCanceled()) {
					canceled = true;
					break;
				}
				if (job.getNumberOfStages() == 1) {
					// optimization short cut
					jobInfo.setStatus(JobInfo.FINISHED);
					job.setStatus(JobInfo.FINISHED);
					scheduler.updateJobDB(jobInfo, job.getID());
					break;
				}

				if (i < numStages) {

					resumed = false;
					jobInfo.setStatus(JobInfo.WAITING);
					job.setStatus(JobInfo.WAITING);
					scheduler.updateJobDB(jobInfo, job.getID());
					long maxWait = getMaxWaitingSecs() * 1000;
					long waitStart = System.currentTimeMillis();

					while (!resumed) {
						while (!resumed) {
							synchronized (this) {
								try {
									this.wait(10000l);
								} catch (InterruptedException e) {
								}
							}
							long diff = System.currentTimeMillis() - waitStart;
							if (jobInfo.getJob().isCanceled()) {
								break;
							}
							if (diff >= maxWait) {
								// time to shelf the job an return
								log.info("selving the job " + job.getID());
								jobInfo.setStatus(JobInfo.SHELVED);
								job.setStatus(JobInfo.SHELVED);
								scheduler.updateJobDB(jobInfo, job.getID());
								return;
							}

						}

						log.info("JobRunner: resumed:" + resumed);
						if (jobInfo.getJob().isCanceled()) {
							canceled = true;
							break;
						}
					}
				}
				if (!canceled) {
					// job is running again for the next stage
					jobInfo.setStatus(JobInfo.RUNNING);
					job.setStatus(JobInfo.RUNNING);
					scheduler.updateJobDB(jobInfo, job.getID());
				} else {
					log.info("JobRunner: canceling job:" + job.getID());
					break;
				}
			}
			// job has finished

			if (!canceled) {
				File f = new File(scheduler.getOutputRootDir(), job.getID()
						+ ".csv");

				job.saveResults(f);
				jobInfo.setStatus(JobInfo.FINISHED);
				job.setStatus(JobInfo.FINISHED);
			} else {
				jobInfo.setStatus(JobInfo.CANCELED);
				job.setStatus(JobInfo.CANCELED);
			}

		} catch (JobException e) {
			log.error("", e);
			jobInfo.setStatus(JobInfo.FINISHED_WITH_ERR);
			job.setStatus(JobInfo.FINISHED_WITH_ERR);
			String msg = getErrorMessage(e);
			jobInfo.setErrorMsg(msg);
		} catch (Throwable t) {
			log.error("", t);
			t.printStackTrace();
			jobInfo.setStatus(JobInfo.FINISHED_WITH_ERR);
			job.setStatus(JobInfo.FINISHED_WITH_ERR);
			String msg = getErrorMessage(t);
			jobInfo.setErrorMsg(msg);
		} finally {
			synchronized (scheduler) {
				scheduler.notify();
			}
		}
	}

	protected String getErrorMessage(Throwable t) {
        return t.getMessage() != null ? t.getMessage()
                : "Unspecified internal error";
	}

	public synchronized void setMaxWaitingSecs(long maxWaitingSecs) {
		this.maxWaitingSecs = maxWaitingSecs;
	}

	public synchronized long getMaxWaitingSecs() {
		return maxWaitingSecs;
	}

}
