package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import clinical.cache.CacheUtils;
import clinical.server.dao.AnalysisDAO;
import clinical.server.vo.Analysis;
import clinical.server.vo.Analysisresult;
import clinical.server.vo.Experiment;
import clinical.server.vo.Extendedtuple;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Site;
import clinical.server.vo.Tuplecolumns;
import clinical.server.vo.Tupledata;
import clinical.server.vo.Tuplevarchar;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.GenUtils;
import clinical.web.DAOFactory;
import clinical.web.IAnalysisResultService;
import clinical.web.IRemoteDBServices;
import clinical.web.ISQLDialect;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.AnalysisDDQueryBuilder;
import clinical.web.common.query.Operator;
import clinical.web.common.query.TSQLParserException;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.query.AnalysisDDQueryBuilder.MeasurementGroupColInfo;
import clinical.web.common.vo.ARValueKey;
import clinical.web.common.vo.AnalysisExTupleInfo;
import clinical.web.common.vo.AnalysisResultSummary;
import clinical.web.common.vo.VisitSegAnalysisResultValues;
import clinical.web.exception.BaseException;
import clinical.web.vo.VisitDateQuery;

/**
 * @author I. Burak Ozyurt
 * @version $Id: AnalysisResultServiceImpl.java,v 1.1.2.2 2008/06/05 23:17:08
 *          bozyurt Exp $
 */
public class AnalysisResultServiceImpl extends AbstractServiceImpl implements
      IAnalysisResultService {
   protected ISQLDialect sqlDialect;

   public AnalysisResultServiceImpl(String dbID) throws BaseException {
      super(dbID);
      sqlDialect = ServiceFactory.getSQLDialect(theDBID);
   }

   public boolean hasAnalysisMetaData() throws Exception {
      IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
      String siteID = ServiceFactory.getSecurityService().findSiteIDByDbID(
            this.theDBID);
      List<Extendedtuple> etList = rds.getExtendedTuples(siteID);

      return !etList.isEmpty();
   }

   public List<Tuplecolumns> getTupleColumnsForET(UserInfo ui,
         List<String> exTupleNames) throws Exception {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         return getTupleColumnsForET(con, exTupleNames);
      } finally {
         releaseConnection(con, ui);
      }
   }

   public List<Tuplecolumns> getTupleColumnsForET(Connection con,
         List<String> exTupleNames) throws Exception {
      if (exTupleNames == null || exTupleNames.isEmpty())
         return new ArrayList<Tuplecolumns>(0);

      TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
      StringBuilder qBuf = new StringBuilder(128);
      qBuf.append("select a.* from Tuplecolumns as a,");
      qBuf.append("Extendedtuple as b where ");
      qBuf.append(" a.extendedtupleid = b.uniqueid and b.name ");
      if (exTupleNames.size() == 1) {
         qBuf.append(" = '").append(exTupleNames.get(0)).append("'");
      } else {
         qBuf.append(" in (");
         for (Iterator<String> it = exTupleNames.iterator(); it.hasNext();) {
            String name = it.next();
            qBuf.append("'").append(name).append("'");
            if (it.hasNext())
               qBuf.append(',');
         }
         qBuf.append(')');
      }
      qBuf.append(" order by a.extendedtupleid, a.columnname");

      List<?> results = tsp.executeQuery(con, qBuf.toString());
      List<Tuplecolumns> tcList = new ArrayList<Tuplecolumns>(results.size());
      for (Iterator<?> it = results.iterator(); it.hasNext();) {
         tcList.add((Tuplecolumns) it.next());
      }
      return tcList;
   }

   public List<Tuplevarchar> getMeasurementGroupNamesForET(UserInfo ui,
         String mgName, List<Integer> extTupleIDList) throws Exception {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         return getMeasurementGroupNamesForET(con, mgName, extTupleIDList);
      } finally {
         releaseConnection(con, ui);
      }
   }

   public List<Tuplevarchar> getMeasurementGroupNamesForET(Connection con,
         String mgName, List<Integer> extTupleIDList) throws Exception {
      if (extTupleIDList == null || extTupleIDList.isEmpty())
         return new ArrayList<Tuplevarchar>(0);
      TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
      StringBuilder qBuf = new StringBuilder(128);
      qBuf.append("select distinct a.textvalue,a.columnname,");
      qBuf.append("a.extendedtupleid from Tuplevarchar as a where ");
      qBuf.append("a.columnname = '").append(mgName).append("' and ");
      if (extTupleIDList.size() == 1) {
         qBuf.append("a.extendedtupleid =").append(extTupleIDList.get(0));
      } else {
         qBuf.append("a.extendedtupleid in (");
         for (Iterator<Integer> it = extTupleIDList.iterator(); it.hasNext();) {
            Integer etID = it.next();
            qBuf.append(etID);
            if (it.hasNext())
               qBuf.append(',');
         }
         qBuf.append(")");
      }
      qBuf.append(" order by a.extendedtupleid, a.textvalue");

      List<?> results = tsp.executeQuery(con, qBuf.toString());
      List<Tuplevarchar> tvList = new ArrayList<Tuplevarchar>(results.size());
      for (Iterator<?> it = results.iterator(); it.hasNext();) {
         Tuplevarchar tv = (Tuplevarchar) it.next();
         tvList.add(tv);
      }
      return tvList;
   }

   public List<Analysis> getAnalyses(Connection con) throws Exception {
      AnalysisDAO dao = DAOFactory.createAnalysisDAO(this.theDBID);
      return dao.find(con, new Analysis());
   }

   public Map<String, AnalysisExTupleInfo> getExTuplesPerAnalysisAvail(
         UserInfo ui) throws Exception {
      Connection con = null;
      try {
         con = pool.getConnection(ui.getName());
         return getExTuplesPerAnalysisAvail(con);
      } finally {
         releaseConnection(con, ui);
      }
   }

   protected Map<String, AnalysisExTupleInfo> getExTuplesPerAnalysisAvailOld(
         Connection con) throws Exception, TSQLParserException {
      Map<String, AnalysisExTupleInfo> map = new HashMap<String, AnalysisExTupleInfo>(
            7);
      List<Analysis> analysisList = getAnalyses(con);
      Map<BigDecimal, Analysis> analysisMap = new HashMap<BigDecimal, Analysis>(
            7);
      for (Analysis an : analysisList) {
         analysisMap.put(an.getUniqueid(), an);
      }
      TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
      StringBuilder qBuf = new StringBuilder(128);
      qBuf
            .append("select distinct a.analysisid, e.uniqueid, e.name from Analysisresult as a, ");
      qBuf.append("Extendedtuple as e, Storedtuple as s where ");
      qBuf.append("s.basetupleid = a.uniqueid and ");
      qBuf.append("s.extendedtupleid = e.uniqueid");

      List<?> results = tsp.executeQuery(con, qBuf.toString());
      for (Iterator<?> it = results.iterator(); it.hasNext();) {
         Object[] row = (Object[]) it.next();
         Analysisresult ar = (Analysisresult) row[0];
         Extendedtuple et = (Extendedtuple) row[1];
         Analysis an = analysisMap.get(ar.getAnalysisid());
         if (an != null) {
            AnalysisExTupleInfo aeti = map.get(an.getName());
            if (aeti == null) {
               aeti = new AnalysisExTupleInfo(an);
               map.put(an.getName(), aeti);
            }
            aeti.addExTuple(et);
         }
      }

      return map;
   }

   /**
    * ASSUMPTION: the first word of analysis name and the corresponding extended
    * tuples' first word of the name matches.
    *
    * @param con
    * @return
    * @throws Exception
    * @throws TSQLParserException
    */
   protected Map<String, AnalysisExTupleInfo> getExTuplesPerAnalysisAvail(
         Connection con) throws Exception, TSQLParserException {
      Map<String, AnalysisExTupleInfo> map = new HashMap<String, AnalysisExTupleInfo>(
            7);
      List<Analysis> analysisList = getAnalyses(con);
      Map<String, Analysis> analysisMap = new HashMap<String, Analysis>(7);
      for (Analysis an : analysisList) {
         String[] toks = an.getName().split("\\s+");
         analysisMap.put(toks[0], an);
      }
      TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
      StringBuilder qBuf = new StringBuilder(128);
      qBuf.append("select e.uniqueid, e.name from Extendedtuple as e");

      List<?> results = tsp.executeQuery(con, qBuf.toString());
      for (Iterator<?> it = results.iterator(); it.hasNext();) {
         Extendedtuple et = (Extendedtuple) it.next();
         String[] toks = et.getName().split("\\s+");

         Analysis an = analysisMap.get(toks[0]);
         if (an != null) {
            AnalysisExTupleInfo aeti = map.get(an.getName());
            if (aeti == null) {
               aeti = new AnalysisExTupleInfo(an);
               map.put(an.getName(), aeti);
            }
            aeti.addExTuple(et);
         }
      }
      return map;
   }

   public List<AnalysisResultSummary> queryForAnalysisResults(UserInfo ui,
         Operator root, List<Integer> experimentIDs, String queryScope,
         String primarySiteID) throws Exception {
      Connection con = null;
      Statement st = null;
      try {
         con = pool.getConnection(ui.getName());

         ISecurityService secService = ServiceFactory.getSecurityService();
         IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();

         String dbID = CacheUtils.getDBID4SiteID(primarySiteID);

         IDBCache dbCache = ServiceFactory.getDBCache(dbID);
         List<Site> sites = dbCache.getSites(ui, false);
         AnalysisDDQueryBuilder builder = new AnalysisDDQueryBuilder(
               ServiceFactory.getSQLDialect(this.theDBID), experimentIDs);

         builder.visit(root);
         String query = builder.getQuery();
         // log the query
         log.info(GenUtils.applyWordWrapping(query));

         st = con.createStatement();
         ResultSet rs = st.executeQuery(query);
         List<MatchedSubjectRec> msrList = new ArrayList<MatchedSubjectRec>();
         Set<VisitDateQuery> vdqList = new HashSet<VisitDateQuery>();
         Map<Integer, MatchedSubjectRec> msrMap = new HashMap<Integer, MatchedSubjectRec>();
         while (rs.next()) {
            MatchedSubjectRec msr = new MatchedSubjectRec();
            msr.subjectID = rs.getString(1);
            msr.storedTupleID = rs.getInt(2);
            msr.extTupleID = rs.getInt(3);
            msr.expID = rs.getInt(4);
            msr.visitID = rs.getInt(5);
            msr.segmentID = rs.getInt(6);
            msr.analysisID = rs.getInt(7);
            msr.analysisName = rs.getString(8);
            msrList.add(msr);
            msrMap.put(msr.storedTupleID, msr);
            vdqList.add(new VisitDateQuery(msr.subjectID, msr.visitID,
                  msr.expID));
         }
         Set<String> uniqColNames = new HashSet<String>();
         List<MeasurementGroupColInfo> mgciList = builder.getMgciList();
         for (MeasurementGroupColInfo mgci : mgciList) {
            if (!uniqColNames.contains(mgci.getMeasGroupFieldName())) {
               uniqColNames.add(mgci.getMeasGroupFieldName());
            }
            if (!uniqColNames.contains(mgci.getMeasName())) {
               uniqColNames.add(mgci.getMeasName());
            }
         }

         TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
         Map<String, AnalysisResultSummary> asrMap = new HashMap<String, AnalysisResultSummary>();
         String siteID = secService.findSiteIDByDbID(theDBID);
         if (siteID == null) {
            siteID = CacheUtils.getDBID2SiteIDMap().get(theDBID);
         }
         System.out.println("queryForAnalysisResults:: siteID:" + siteID);

         if (siteID != null && !msrList.isEmpty()) {

            StringBuilder qBuf = new StringBuilder(1024);
            qBuf.append("select d.columnname, d.columntype,");
            qBuf.append("d.extendedtupleid, d.storedtupleid, d.textvalue ");
            qBuf.append("from Tupledata as d ");
            qBuf.append("where d.storedtupleid in (");
            for (Iterator<MatchedSubjectRec> it = msrList.iterator(); it
                  .hasNext();) {
               MatchedSubjectRec msr = it.next();
               qBuf.append(msr.storedTupleID);
               if (it.hasNext())
                  qBuf.append(',');
            }
            qBuf.append(") and d.columnname in (");
            for (Iterator<String> it = uniqColNames.iterator(); it.hasNext();) {
               String colName = it.next();
               qBuf.append('\'').append(colName).append('\'');
               if (it.hasNext())
                  qBuf.append(',');
            }

            qBuf.append(") order by d.extendedtupleid");

            Map<String, AnalysisExTupleInfo> anETIMap = getExTuplesPerAnalysisAvail(con);
            Map<Integer, AnalysisExTupleInfo> aetiMap = new HashMap<Integer, AnalysisExTupleInfo>(
                  7);
            for (AnalysisExTupleInfo aeti : anETIMap.values()) {
               aetiMap.put(aeti.getAnalysis().getUniqueid().intValue(), aeti);
            }
            anETIMap = null;
            List<?> results = tsp.executeQuery(con, qBuf.toString());
            Map<Integer, String> storedTuple2AnatomicalEntityMap = new HashMap<Integer, String>(
                  31);
            for (Iterator<?> it = results.iterator(); it.hasNext();) {
               Tupledata td = (Tupledata) it.next();
               // TODO more general representation??
               if (td.getColumnname().equals("anatomicalEntity")) {
                  storedTuple2AnatomicalEntityMap.put(td.getStoredtupleid()
                        .intValue(), td.getTextvalue());
               }
            }

            Map<String, Experiment> expMap = rds.getAllExperiments(siteID);

            for (Iterator<?> it = results.iterator(); it.hasNext();) {
               Tupledata td = (Tupledata) it.next();
               if (td.getColumnname().equals("anatomicalEntity")) {
                  continue;
               }
               MatchedSubjectRec msr = msrMap.get(td.getStoredtupleid()
                     .intValue());
               Assertion.assertNotNull(msr);
               AnalysisExTupleInfo aeti = aetiMap.get(msr.analysisID);
               Assertion.assertNotNull(aeti);

               String key = AnalysisResultSummary.getKey(msr.subjectID,
                     msr.expID, siteID);
               AnalysisResultSummary asr = asrMap.get(key);
               if (asr == null) {
                  // FIXME timestamp
                  Experiment exp = expMap.get(String.valueOf(msr.expID));
                  String expName = exp.getName();
                  String timeStamp = null;
                  asr = new AnalysisResultSummary(msr.subjectID, String
                        .valueOf(msr.expID), expName, siteID, timeStamp);
                  asrMap.put(key, asr);
               }
               VisitSegAnalysisResultValues vsarv = asr
                     .getOrPutVisitSegAnalysisResultValues(msr.visitID,
                           msr.segmentID, msr.expID, siteID);
               String mgName = storedTuple2AnatomicalEntityMap.get(td
                     .getStoredtupleid().intValue());
               Assertion.assertNotNull(mgName);
               ARValueKey avKey = new ARValueKey(aeti, mgName, td
                     .getColumnname(), msr.extTupleID);
               vsarv.addValue(avKey, td.getTextvalue());
            }
         }
         List<AnalysisResultSummary> arsList = new ArrayList<AnalysisResultSummary>(
               asrMap.values());

         Collections.sort(arsList, new Comparator<AnalysisResultSummary>() {
            public int compare(AnalysisResultSummary o1,
                  AnalysisResultSummary o2) {
               int rc = o1.getSubjectID().compareTo(o2.getSubjectID());
               if (rc == 0) {
                  return o1.getExpID().compareTo(o2.getExpID());
               }
               return rc;
            }
         });

         if (AnalysisResultSummary.needsFiltering(root)) {
            for (Iterator<AnalysisResultSummary> it = arsList.iterator(); it
                  .hasNext();) {
               AnalysisResultSummary ars = it.next();
               boolean ok2Drop = ars.filter(root, queryScope);
               if (ok2Drop)
                  it.remove();
            }
         }

         if (arsList.isEmpty())
            return arsList;

         Map<String, Date> visitDateMap = rds.getVisitDates(siteID,
               new ArrayList<VisitDateQuery>(vdqList), true);

         // attach subject site info and visit dates
         Map<String, SubjectInfo> subjectInfoMap = getSubjectInfos(con, msrMap,
               sites, primarySiteID);
         for (AnalysisResultSummary ars : arsList) {
            SubjectInfo si = subjectInfoMap.get(ars.getSubjectID());
            ars.setSiteID(si.siteID);
            ars.setSiteName(si.siteName);
            ars.setRemote(si.remote);
            for (VisitSegAnalysisResultValues vsarv : ars.getVsarvList()) {
               String key = ars.getSubjectID() + "_" + ars.getExpID() + "_"
                     + vsarv.getVisitID();
               Date visitDate = visitDateMap.get(key);
               if (visitDate != null) {
                  vsarv.setTimeStamp(DateTimeUtils.formatDate(visitDate));
               }
            }
         }

         return arsList;
      } finally {
         releaseConnection(con, ui);
      }
   }

   protected Map<String, SubjectInfo> getSubjectInfos(Connection con,
         Map<Integer, MatchedSubjectRec> msrMap, List<Site> sites,
         String primarySiteID) throws Exception {
      Set<String> uniqSubjectIDSet = new HashSet<String>();
      for (MatchedSubjectRec msr : msrMap.values()) {
         uniqSubjectIDSet.add(msr.subjectID);
      }
      Map<BigDecimal, Site> siteMap = new HashMap<BigDecimal, Site>(17);

      Site primarySite = null;
      for (Site site : sites) {
         siteMap.put(site.getUniqueid(), site);
         if (site.getSiteid().equals(primarySiteID))
            primarySite = site;
      }
      Assertion.assertNotNull(primarySite);
      siteMap.put(null, primarySite);

      Map<String, SubjectInfo> subjectMap = new HashMap<String, SubjectInfo>();
      TSQLProcessor tsp = new TSQLProcessor(this.sqlDialect);
      StringBuilder sb = new StringBuilder(512);
      sb.append("select s.subjectid, s.siteid, s.isremote ");
      sb.append("from Humansubject as s where s.subjectid in (");
      for (Iterator<String> it = uniqSubjectIDSet.iterator(); it.hasNext();) {
         String subjectID = it.next();
         sb.append("'").append(subjectID).append("'");
         if (it.hasNext())
            sb.append(',');
      }
      sb.append(')');
      List<?> results = tsp.executeQuery(con, sb.toString());
      for (Iterator<?> it = results.iterator(); it.hasNext();) {
         Humansubject hs = (Humansubject) it.next();
         Site site = siteMap.get(hs.getSiteid());
         Assertion.assertNotNull(site);
         boolean isRemote = hs.getIsremote() == null ? false : hs.getIsremote();
         SubjectInfo si = new SubjectInfo(hs.getSubjectid(), site.getSiteid(),
               site.getSitename(), isRemote);
         subjectMap.put(si.subjectID, si);
      }
      return subjectMap;
   }

   public static class SubjectInfo {
      String subjectID;
      String siteID;
      String siteName;
      boolean remote;

      public SubjectInfo(String subjectID, String siteID, String siteName,
            boolean remote) {
         super();
         this.subjectID = subjectID;
         this.siteID = siteID;
         this.siteName = siteName;
         this.remote = remote;
      }
   }

   public static class MatchedSubjectRec {
      String subjectID;
      int storedTupleID;
      int extTupleID;
      int expID;
      int visitID;
      int segmentID;
      int analysisID;
      String analysisName;

   }// ;
}
