package clinical.web.services;

import gnu.trove.TIntObjectHashMap;

import java.math.BigDecimal;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.ExperimentDAO;
import clinical.server.dao.ScorevalidatorDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Experiment;
import clinical.server.vo.Scorevalidator;
import clinical.server.vo.Storedassessment;
import clinical.utils.Assertion;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IAssessmentService;
import clinical.web.ISQLDialect;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.BinaryOperator;
import clinical.web.common.query.Operator;
import clinical.web.common.query.QueryUtils;
import clinical.web.common.query.SearchPredicate;
import clinical.web.common.query.UnaryOperator;
import clinical.web.common.query.SearchPredicate.DateRange;
import clinical.web.common.security.DBConfig;
import clinical.web.common.vo.AsScoreInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.download.SubjectImageDataBundle;
import clinical.web.exception.BaseException;
import clinical.web.helpers.SubCorticalVarInfo;
import clinical.web.vo.QuerySummary;
import clinical.web.vo.RangeInfo;
import clinical.web.vo.SegmentScoreValuesSummary;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.vo.SynonymInfo;
import clinical.web.vo.VisitScoreValuesSummary;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentServiceHelper.java,v 1.1 2007/04/12 20:16:16 bozyurt
 *          Exp $
 */
public class AssessmentServiceHelper {
	private static Log log = LogFactory.getLog(AssessmentServiceHelper.class);

	protected AssessmentServiceHelper() {
	}

	public static void prepareSubjectAsScoreSummaryRec(
			List<SubjectAsScoreValueSummary> summaryList, ResultSet rs,
			Map<String, SubjectAsScoreValueSummary> summaryMap)
			throws SQLException {
		String subjectID = rs.getString(1);
		Object value = rs.getObject(2);
		int scoreOrder = rs.getInt(3);
		String scoreName = rs.getString(4);
		String asID = rs.getObject(5).toString();
		int visitID = rs.getInt(6);
		int segmentID = rs.getInt(7);
		int expID = rs.getInt(8);
		Timestamp timeStamp = rs.getTimestamp(9);
		String scoreType = rs.getString(10);

		String timeStampStr = "";
		if (timeStamp != null) {
			timeStampStr = QueryUtils.formatDate(new java.sql.Date(timeStamp
					.getTime()));
		}
		String key = prepareSubjectAsScoreValueSummaryKey(subjectID, visitID,
				segmentID, expID, scoreName);
		SubjectAsScoreValueSummary ssv = summaryMap.get(key);
		if (ssv == null) {
			ssv = new SubjectAsScoreValueSummary();
			summaryMap.put(key, ssv);

			ssv.setSubjectID(subjectID);
			ssv.setScoreName(scoreName);
			ssv.setAssessmentID(asID);
			ssv.setVisitID(visitID);
			ssv.setSegmentID(segmentID);
			ssv.setExperimentID(expID);
			ssv.setTimeStamp(timeStampStr);
			ssv.setScoreType(scoreType);
			summaryList.add(ssv);
		}

		if (scoreOrder == 1) {
			ssv.setValue(value);
			ssv.addValue(0, value);
		} else {
			ssv.addValue(scoreOrder - 1, value);
			ssv.setMultiValued(true);
		}
	}

	public static String prepareSubjectAsScoreValueSummaryKey(String subjectID,
			int visitID, int segmentID, int expID, String scoreName) {
		StringBuffer buf = new StringBuffer();
		buf.append(subjectID).append('_').append(visitID).append('_');
		buf.append(segmentID).append('_').append(expID);
		buf.append('_').append(scoreName);
		return buf.toString();
	}

	public static boolean isFiltered(SegmentScoreValuesSummary ssvs,
			Operator op, Map<String, SynonymInfo> synMap) {
		ResultWrapper result = new ResultWrapper();
		checkCondition(ssvs, op, result, synMap);
		// filter out anything that does not match
		return !result.result;
	}

	public static boolean isFiltered(VisitScoreValuesSummary vsvs, Operator op) {
		ResultWrapper result = new ResultWrapper();
		checkCondition(vsvs, op, result);
		// filter out anything that does not match
		return !result.result;
	}

	static class ResultWrapper {
		boolean result = true;
	}

	public static void checkCondition(VisitScoreValuesSummary vsvs,
			Operator op, ResultWrapper result) {
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			AsScoreInfo si = (AsScoreInfo) sp.getAttribute();
			String asID = si.getAssessmentID().toString();
			SubjectAsScoreValueSummary ssv = vsvs.get(asID, si.getName());

			if (ssv != null) {
				String value2Check = (String) ssv.getValue();
				if (sp.getValue() instanceof SearchPredicate.Range) {
					SearchPredicate.Range range = (SearchPredicate.Range) sp
							.getValue();
					boolean satisfied = QueryUtils.condSatisfied(range,
							value2Check);
					result.result = satisfied;
				} else if (sp.getValue() instanceof SearchPredicate.DateRange) {
					SearchPredicate.DateRange range = (DateRange) sp.getValue();
					boolean satisfied = QueryUtils.condSatisfied(range,
							value2Check);
					result.result = satisfied;
				} else {
					boolean satisfied = QueryUtils.condSatisfied(sp.getValue(),
							sp.getOperator(), sp.getType(), value2Check);
					result.result = satisfied;
				}
			} else {
				// missing SubjectAsScoreValueSummary does not satisfy condition
				// by default
				result.result = false;
			}
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			int connective = bop.getLogicalOp();
			checkCondition(vsvs, lhs, result);
			boolean satisfied = result.result;
			if (connective == Operator.AND && !satisfied) {
				return;
			}
			Operator rhs = bop.getRhs();
			checkCondition(vsvs, rhs, result);
			if (connective == Operator.AND) {
				result.result = satisfied && result.result;
			} else {
				result.result = satisfied || result.result;
			}
		}
	}

	public static SynonymInfo hasSynonyms(String scoreName, String asID,
			Object value, Map<String, SynonymInfo> synonymMap) {
		String key = prepSynonymInfoKey(new BigDecimal(asID), scoreName);
		SynonymInfo si = synonymMap.get(key);
		if (si != null) {
			if (!si.matchesAny((value.toString()))) {
				return null;
			}
		}
		return si;
	}

	public static void checkCondition(SegmentScoreValuesSummary ssvs,
			Operator op, ResultWrapper result, Map<String, SynonymInfo> synMap) {
		/** TODO handle multi-valued scores */
		/* pass all records for a segment for condition checking */
		if (op instanceof UnaryOperator) {
			UnaryOperator uop = (UnaryOperator) op;
			SearchPredicate sp = uop.getPredicate();
			AsScoreInfo si = (AsScoreInfo) sp.getAttribute();
			String asID = si.getAssessmentID().toString();
			SubjectAsScoreValueSummary ssv = ssvs.get(asID, si.getName());

			if (ssv != null) {
				String value2Check = (String) ssv.getValue();
				if (sp.getValue() instanceof SearchPredicate.Range) {
					SearchPredicate.Range range = (SearchPredicate.Range) sp
							.getValue();
					boolean satisfied = QueryUtils.condSatisfied(range,
							value2Check);
					result.result = satisfied;
				} else if (sp.getValue() instanceof SearchPredicate.DateRange) {
					SearchPredicate.DateRange range = (DateRange) sp.getValue();
					boolean satisfied = QueryUtils.condSatisfied(range,
							value2Check);
					result.result = satisfied;
				} else {
					// synonym expansion
					SynonymInfo synonymInfo = hasSynonyms(si.getName(), asID,
							sp.getValue(), synMap);
					if (synonymInfo == null) {
						boolean satisfied = QueryUtils.condSatisfied(
								sp.getValue(), sp.getOperator(), sp.getType(),
								value2Check);
						result.result = satisfied;
					} else {
						boolean satisfied = false;
						for (String synonym : synonymInfo.getSynonymArray()) {
							satisfied |= QueryUtils
									.condSatisfied(synonym, sp.getOperator(),
											sp.getType(), value2Check);
						}
						result.result = satisfied;
					}
				}
			} else {
				// missing SubjectAsScoreValueSummary does not satisfy condition
				// by default
				result.result = false;
			}
		} else if (op instanceof BinaryOperator) {
			BinaryOperator bop = (BinaryOperator) op;
			Operator lhs = bop.getLhs();
			int connective = bop.getLogicalOp();
			checkCondition(ssvs, lhs, result, synMap);
			boolean satisfied = result.result;
			if (connective == Operator.AND && !satisfied) {
				return;
			}
			Operator rhs = bop.getRhs();
			checkCondition(ssvs, rhs, result, synMap);
			if (connective == Operator.AND) {
				result.result = satisfied && result.result;
			} else {
				result.result = satisfied || result.result;
			}
		}
	}

	public static String prepSynonymInfoKey(BigDecimal asID, String scoreName) {
		StringBuilder sb = new StringBuilder();
		sb.append(scoreName).append(':').append(asID);
		return sb.toString();
	}

	public static List<SubjectAsScoreValueSummary> filterSearchResults(
			List<SubjectAsScoreValueSummary> summaryList, Operator op,
			Map<String, SynonymInfo> synMap) {
		List<SubjectAsScoreValueSummary> filteredSummaryList = new LinkedList<SubjectAsScoreValueSummary>();

		List<SegmentScoreValuesSummary> ssvsList = prepareSegmentScoreValuesSummaryRecs(summaryList);
		for (Iterator<SegmentScoreValuesSummary> iter = ssvsList.iterator(); iter
				.hasNext();) {
			SegmentScoreValuesSummary ssvs = iter.next();
			if (!AssessmentServiceHelper.isFiltered(ssvs, op, synMap)) {
				filteredSummaryList.addAll(ssvs
						.getSubjectAsScoreValueSummaryList());
			}
		}
		return filteredSummaryList;
	}

	public static List<SubjectAsScoreValueSummary> filterSearchResultsGroupedByVisit(
			List<SubjectAsScoreValueSummary> summaryList, Operator op) {
		List<SubjectAsScoreValueSummary> filteredSummaryList = new LinkedList<SubjectAsScoreValueSummary>();

		List<VisitScoreValuesSummary> vsvsList = prepareVisitScoreValuesSummaryRecs(summaryList);
		for (Iterator<VisitScoreValuesSummary> iter = vsvsList.iterator(); iter
				.hasNext();) {
			VisitScoreValuesSummary vsvs = iter.next();
			if (!AssessmentServiceHelper.isFiltered(vsvs, op)) {
				filteredSummaryList.addAll(vsvs
						.getSubjectAsScoreValueSummaryList());
			}
		}
		return filteredSummaryList;
	}

	public static List<VisitScoreValuesSummary> prepareVisitScoreValuesSummaryRecs(
			List<SubjectAsScoreValueSummary> summaryList) {
		Map<String, VisitScoreValuesSummary> vsvsMap = new LinkedHashMap<String, VisitScoreValuesSummary>();
		for (Iterator<SubjectAsScoreValueSummary> iter = summaryList.iterator(); iter
				.hasNext();) {
			SubjectAsScoreValueSummary ssv = iter.next();
			String key = VisitScoreValuesSummary.prepareKey(ssv.getSubjectID(),
					ssv.getVisitID(), ssv.getExperimentID(), ssv.getSiteID());
			VisitScoreValuesSummary vsvs = vsvsMap.get(key);
			if (vsvs == null) {
				vsvs = new VisitScoreValuesSummary(ssv.getSubjectID(),
						ssv.getVisitID(), ssv.getExperimentID(),
						ssv.getSiteID());
				vsvsMap.put(key, vsvs);
			}
			vsvs.addScoreValueSummary(ssv);
		}
		return new ArrayList<VisitScoreValuesSummary>(vsvsMap.values());

	}

	public static List<SegmentScoreValuesSummary> prepareSegmentScoreValuesSummaryRecs(
			List<SubjectAsScoreValueSummary> summaryList) {
		Map<String, SegmentScoreValuesSummary> ssvsMap = new LinkedHashMap<String, SegmentScoreValuesSummary>();
		for (Iterator<SubjectAsScoreValueSummary> iter = summaryList.iterator(); iter
				.hasNext();) {
			SubjectAsScoreValueSummary ssv = iter.next();
			String key = SegmentScoreValuesSummary.prepareKey(
					ssv.getSubjectID(), ssv.getVisitID(), ssv.getSegmentID(),
					ssv.getExperimentID(), ssv.getSiteID());
			SegmentScoreValuesSummary ssvs = ssvsMap.get(key);
			if (ssvs == null) {
				ssvs = new SegmentScoreValuesSummary(ssv.getSubjectID(),
						ssv.getVisitID(), ssv.getSegmentID(),
						ssv.getExperimentID(), ssv.getSiteID());
				ssvsMap.put(key, ssvs);
			}
			ssvs.addScoreValueSummary(ssv);
		}
		return new ArrayList<SegmentScoreValuesSummary>(ssvsMap.values());
	}

	public static Map<Integer, String> prepareExpID2NameMap(String dbID,
			String username, List<Integer> experimentIDs) throws Exception {
		java.sql.Connection con = null;
		Statement st = null;
		ResultSet rs = null;
		IDBPoolService aPool = null;
		if (experimentIDs == null) {
			return null;
		}
		Map<Integer, String> map = new LinkedHashMap<Integer, String>(7);
		try {
			aPool = ServiceFactory.getPoolService(dbID);
			con = aPool.getConnection(username);

			ExperimentDAO dao = DAOFactory.createExperimentDAO(dbID);
			List<Experiment> exps = dao.find(con, new Experiment());
			Map<Integer, String> expIDMap = new HashMap<Integer, String>(7);
			for (Experiment exp : exps) {
				expIDMap.put(new Integer(exp.getUniqueid().toString()),
						exp.getName());
			}
			for (Integer expID : experimentIDs) {
				String name = expIDMap.get(expID);
				assert (name != null);
				map.put(expID, name);
			}
		} finally {
			DBUtils.close(st, rs);
			if (con != null) {
				aPool.releaseConnection(username, con);
			}
		}

		return map;
	}

	public static Map<String, Integer> getValidatorIDs(String dbID,
			java.sql.Connection con, BigDecimal rangeValidatorTableID)
			throws Exception {
		ScorevalidatorDAO dao = DAOFactory.createScorevalidatorDAO(dbID);
		Scorevalidator criteria = new Scorevalidator();
		criteria.setValidatortable(rangeValidatorTableID);
		List<?> validators = dao.find(con, criteria);
		log.info("getting validator ids for rangevalidatortableid  "
				+ rangeValidatorTableID);
		Map<String, Integer> rangeValidatorMap = new HashMap<String, Integer>();
		for (Iterator<?> iter = validators.iterator(); iter.hasNext();) {
			Scorevalidator sv = (Scorevalidator) iter.next();
			rangeValidatorMap.put(
					createScoreKey(sv.getScorename(), sv.getAssessmentid()
							.intValue()), new Integer(sv.getValidatorid()
							.intValue()));
		}

		return rangeValidatorMap;
	}

	public static String createScoreKey(String scoreName, int assessmentID) {
		StringBuffer sb = new StringBuffer();
		sb.append(scoreName).append("_").append(assessmentID);
		return sb.toString();
	}

	public static RangeInfo findRangeForSubcorticalVariable(
			java.sql.Connection con, SubCorticalVarInfo scvi) throws Exception {
		StringBuffer sb = new StringBuffer(128);

		String regionName = scvi.getBrainRegionName();
		if (regionName.startsWith("Left ")) {
			regionName = regionName.substring(5);
		} else if (regionName.startsWith("Right ")) {
			regionName = regionName.substring(6);
		}

		sb.append("select min(measurement), max(measurement) "
				+ "from nc_brainsegmentdata where brainregionname = '");
		sb.append(regionName).append("' and laterality='");
		sb.append(scvi.getLaterality()).append("'");
		ResultSet rs = null;
		Statement st = null;
		try {
			st = con.createStatement();
			log.info("subcortical var query=" + sb.toString());
			rs = st.executeQuery(sb.toString());
			if (rs.next()) {
				if (rs.getObject(1) == null || rs.getObject(2) == null) {
					return null;
				}

				String lowBound = rs.getBigDecimal(1).toString();
				String uppBound = rs.getBigDecimal(2).toString();
				// log.info("lowBound=" + lowBound + ", uppBound=" + uppBound);
				RangeInfo ri = new RangeInfo(lowBound, uppBound, true, true,
						scvi);
				return ri;
			}
		} finally {
			DBUtils.close(st, rs);
		}
		return null;
	}

	/**
	 * FIXME use TSQLProcessor
	 * 
	 * @param con
	 * @param saList
	 * @param sqlDialect
	 * @return
	 * @throws SQLException
	 */
	public static List<SubjectAsScoreValueSummary> getFullAssessments(
			Connection con, List<Storedassessment> saList,
			ISQLDialect sqlDialect) throws SQLException {
		if (saList.isEmpty()) {
			return new ArrayList<SubjectAsScoreValueSummary>(0);
		}
		String selectSect = "select scoreorder, assessmentid, scorename, scoretype, textvalue,"
				+ "nc_storedassessment_uniqueid, subjectid from nc_assessmentdata where ";
		StringBuilder queryBuf = new StringBuilder(800);
		queryBuf.append(selectSect);
		queryBuf.append(sqlDialect.prepareBooleanPredicate(" isvalidated = ",
				true));
		queryBuf.append(" and nc_storedassessment_uniqueid in (");
		for (Iterator<Storedassessment> iter = saList.iterator(); iter
				.hasNext();) {
			Storedassessment sa = iter.next();
			queryBuf.append(sa.getUniqueid());
			if (iter.hasNext()) {
				queryBuf.append(',');
			}
		}
		queryBuf.append(") order by subjectid, assessmentid");
		TIntObjectHashMap saIDMap = new TIntObjectHashMap();
		for (Storedassessment sa : saList) {
			saIDMap.put(sa.getUniqueid().intValue(), sa);
		}

		Statement st = null;
		ResultSet rs = null;
		try {
			st = con.createStatement();
			System.out.println("** getFullAssessments: \n"
					+ queryBuf.toString());
			rs = st.executeQuery(queryBuf.toString());
			List<SubjectAsScoreValueSummary> summaryList = new LinkedList<SubjectAsScoreValueSummary>();
			// TODO multiple values
			while (rs.next()) {
				SubjectAsScoreValueSummary svs = new SubjectAsScoreValueSummary();
				svs.setAssessmentID(rs.getString(2));
				svs.setScoreName(rs.getString(3));
				svs.setScoreType(rs.getString(4));
				svs.setValue(rs.getString(5));
				int saID = rs.getInt(6);
				Storedassessment sa = (Storedassessment) saIDMap.get(saID);
				svs.setExperimentID(sa.getNcExperimentUniqueid().intValue());
				svs.setVisitID(sa.getComponentid().intValue());
				svs.setSegmentID(sa.getSegmentid().intValue());
				svs.setSubjectID(rs.getString(7));

				summaryList.add(svs);
			}
			return summaryList;
		} finally {
			DBUtils.close(st, rs);
		}

	}

	public static Map<String, SiteAsiInfo> findSitesWithData(UserInfo ui,
			List<QuerySummary> qsList, List<AssessmentSelectionInfo> asiList)
			throws Exception {
		Map<String, SiteAsiInfo> siteMap = new HashMap<String, SiteAsiInfo>(17);

		ISecurityService secService = ServiceFactory.getSecurityService();
		Set<String> asNameSet = new HashSet<String>(17);
		for (Iterator<AssessmentSelectionInfo> iter = asiList.iterator(); iter
				.hasNext();) {
			AssessmentSelectionInfo asi = iter.next();
			asNameSet.add(asi.getName());
		}

		for (QuerySummary qs : qsList) {
			String siteID = qs.getSiteID();
			SiteAsiInfo sai = siteMap.get(siteID);
			if (sai == null) {
				DBConfig dbConfig = secService.findBySiteID(siteID);
				sai = new SiteAsiInfo(siteID, dbConfig.getId());
				sai.sqlDialect = ServiceFactory.getSQLDialect(dbConfig.getId());
				siteMap.put(qs.getSiteID(), sai);
			}
		}
		addAssessments2SAI(ui, siteMap, asNameSet);

		return siteMap;
	}

	/*
	 * public static Map<String, SiteAsiInfo> findSitesWithData(UserInfo ui,
	 * List<SubjectAsScoreValueSummary> summaryList,
	 * List<AssessmentSelectionInfo> asiList) throws Exception { Map<String,
	 * SiteAsiInfo> siteMap = new HashMap<String, SiteAsiInfo>(17);
	 * 
	 * ISecurityService secService = ServiceFactory.getSecurityService();
	 * Set<String> asNameSet = new HashSet<String>(17); for
	 * (Iterator<AssessmentSelectionInfo> iter = asiList.iterator(); iter
	 * .hasNext();) { AssessmentSelectionInfo asi = iter.next();
	 * asNameSet.add(asi.getName()); }
	 * 
	 * for (SubjectAsScoreValueSummary sasvs : summaryList) { String siteID =
	 * sasvs.getSiteID(); SiteAsiInfo sai = siteMap.get(siteID); if (sai ==
	 * null) { DBConfig dbConfig = secService.findBySiteID(siteID); sai = new
	 * SiteAsiInfo(siteID, dbConfig.getId()); sai.sqlDialect =
	 * ServiceFactory.getSQLDialect(dbConfig.getId());
	 * siteMap.put(sasvs.getSiteID(), sai); } } addAssessments2SAI(ui, siteMap,
	 * asNameSet);
	 * 
	 * return siteMap; }
	 */

	public static Map<String, SiteAsiInfo> findSites4AssessmentDownload(
			UserInfo ui, List<SubjectImageDataBundle> sidbList,
			List<AssessmentSelectionInfo> asiList) throws Exception {
		Map<String, SiteAsiInfo> siteMap = new HashMap<String, SiteAsiInfo>(17);
		ISecurityService secService = ServiceFactory.getSecurityService();
		Set<String> asNameSet = new HashSet<String>(17);
		for (AssessmentSelectionInfo asi : asiList) {
			asNameSet.add(asi.getName());
		}

		for (SubjectImageDataBundle sidb : sidbList) {
			String siteID = sidb.getSiteID();
			SiteAsiInfo sai = siteMap.get(siteID);
			if (sai == null) {
				String dbID = secService.findDBForSiteID(siteID);
				Assertion.assertNotNull(dbID);
				sai = new SiteAsiInfo(siteID, dbID);
				sai.sqlDialect = ServiceFactory.getSQLDialect(dbID);
				siteMap.put(siteID, sai);
			}
		}

		addAssessments2SAI(ui, siteMap, asNameSet);

		return siteMap;
	}

	protected static void addAssessments2SAI(UserInfo ui,
			Map<String, SiteAsiInfo> siteMap, Set<String> asNameSet)
			throws BaseException, Exception {
		for (Iterator<SiteAsiInfo> iter = siteMap.values().iterator(); iter
				.hasNext();) {
			SiteAsiInfo sai = iter.next();
			UserInfo adminUI = new UserInfo(Constants.ADMIN_USER,
					ui.getRoles(), ui.getAvailableTables());
			IAssessmentService ias = ServiceFactory
					.getAssessmentService(sai.dbID);
			List<Assessment> asList = ias.getAllAssessments(adminUI);
			for (Assessment as : asList) {
				if (asNameSet.contains(as.getName())) {
					AssessmentSelectionInfo asi = new AssessmentSelectionInfo(
							as.getName(), as.getAssessmentid());
					sai.addASI(asi);
				}
			}
		}
	}

	public static class SiteAsiInfo {
		String siteID;
		String dbID;
		ISQLDialect sqlDialect;
		protected List<AssessmentSelectionInfo> asiList = new ArrayList<AssessmentSelectionInfo>(
				10);

		public SiteAsiInfo(String siteid, String dbID) {
			super();
			siteID = siteid;
			this.dbID = dbID;
		}

		public void addASI(AssessmentSelectionInfo asi) {
			asiList.add(asi);
		}

		public List<AssessmentSelectionInfo> getAsiList() {
			return asiList;
		}

		public String getDbID() {
			return dbID;
		}

		public String getSiteID() {
			return siteID;
		}

		public ISQLDialect getSqlDialect() {
			return sqlDialect;
		}

	}// ;

	public static Map<Integer, List<String>> prepareExpSubjectIDListMapFromSASVS(
			List<SubjectAsScoreValueSummary> summaryList) {
		Map<Integer, List<String>> expSubjectIDListMap = new HashMap<Integer, List<String>>(
				7);

		for (SubjectAsScoreValueSummary sasvs : summaryList) {
			Integer expID = new Integer(sasvs.getExperimentID());
			List<String> subjectIDList = expSubjectIDListMap.get(expID);
			if (subjectIDList == null) {
				subjectIDList = new ArrayList<String>();
				expSubjectIDListMap.put(expID, subjectIDList);
			}
			subjectIDList.add(sasvs.getSubjectID());
		}
		return expSubjectIDListMap;
	}

	public static Map<Integer, List<String>> prepareExpSubjectIDListMapFromQuerySummaries(
			List<QuerySummary> qsList) {
		Map<Integer, List<String>> expSubjectIDListMap = new HashMap<Integer, List<String>>(
				7);

		for (QuerySummary qs : qsList) {
			Integer expID = new Integer(qs.getExpID());
			List<String> subjectIDList = expSubjectIDListMap.get(expID);
			if (subjectIDList == null) {
				subjectIDList = new ArrayList<String>();
				expSubjectIDListMap.put(expID, subjectIDList);
			}
			subjectIDList.add(qs.getSubjectID());
		}
		return expSubjectIDListMap;
	}

	public static Map<Integer, List<String>> prepareExpSubjectIDListMapFromSIDB(
			List<SubjectImageDataBundle> sidbList) {
		Map<Integer, List<String>> expSubjectIDListMap = new HashMap<Integer, List<String>>(
				7);

		for (SubjectImageDataBundle sidb : sidbList) {
			for (Iterator<String> it = sidb.getExpIds().iterator(); it
					.hasNext();) {
				String expIDStr = it.next();
				Integer expID = new Integer(expIDStr);
				List<String> subjectIDList = expSubjectIDListMap.get(expID);
				if (subjectIDList == null) {
					subjectIDList = new ArrayList<String>();
					expSubjectIDListMap.put(expID, subjectIDList);
				}
				subjectIDList.add(sidb.getSubjectID());
			}
		}
		return expSubjectIDListMap;
	}

	public static List<Storedassessment> getStoredAsRecords(Connection con,
			Map<Integer, List<String>> expSubjectIDListMap,
			List<AssessmentSelectionInfo> asiList, ISQLDialect sqlDialect)
			throws SQLException {
		String selectSect = "select uniqueid, assessmentid, componentid, nc_experiment_uniqueid,"
				+ "segmentid, subjectid from nc_storedassessment where ";

		StringBuffer queryBuf = new StringBuffer(150);
		queryBuf.append(selectSect);
		queryBuf.append(sqlDialect.prepareBooleanPredicate(" isvalidated = ",
				true));

		StringBuffer asIDInBuffer = new StringBuffer();
		asIDInBuffer.append(" assessmentid in (");
		for (Iterator<AssessmentSelectionInfo> it = asiList.iterator(); it
				.hasNext();) {
			AssessmentSelectionInfo asi = it.next();
			asIDInBuffer.append(asi.getAssessmentID());
			if (it.hasNext()) {
				asIDInBuffer.append(',');
			}
		}
		asIDInBuffer.append(") ");

		List<Storedassessment> masterSAList = new ArrayList<Storedassessment>();
		for (Iterator<Map.Entry<Integer, List<String>>> iter = expSubjectIDListMap
				.entrySet().iterator(); iter.hasNext();) {
			StringBuffer buf = new StringBuffer(200);
			buf.append(queryBuf);
			Map.Entry<Integer, List<String>> entry = iter.next();
			Integer expID = entry.getKey();
			List<String> subjectIDList = entry.getValue();
			buf.append(" and nc_experiment_uniqueid = ").append(expID);
			buf.append(" and subjectid in ");
			buf.append(createInList(subjectIDList, sqlDialect));
			buf.append(" and ").append(asIDInBuffer.toString());
			List<Storedassessment> saList = retrieve(buf.toString(), con);
			masterSAList.addAll(saList);
		}

		return masterSAList;
	}

	protected static List<Storedassessment> retrieve(String storedAsQuery,
			Connection con) throws SQLException {
		Statement st = null;
		ResultSet rs = null;
		try {
			st = con.createStatement();
			System.out.println("retrieve Query: " + storedAsQuery);
			rs = st.executeQuery(storedAsQuery);
			List<Storedassessment> saList = new LinkedList<Storedassessment>();
			while (rs.next()) {
				Storedassessment sa = new Storedassessment();
				// assessmentid, componentid, nc_experiment_uniqueid," +
				// "segmentid, subjectid
				sa.setUniqueid(rs.getBigDecimal(1));
				sa.setAssessmentid(rs.getBigDecimal(2));
				sa.setComponentid(rs.getBigDecimal(3));
				sa.setNcExperimentUniqueid(rs.getBigDecimal(4));
				sa.setSegmentid(rs.getBigDecimal(5));
				sa.setSubjectid(rs.getString(6));
				saList.add(sa);
			}
			return saList;
		} finally {
			DBUtils.close(st, rs);
		}
	}

	public static String createInList(List<String> strList,
			ISQLDialect sqlDialect) {
		StringBuffer buf = new StringBuffer();
		buf.append(" (");
		for (Iterator<String> iter = strList.iterator(); iter.hasNext();) {
			String str = iter.next();
			buf.append(sqlDialect.prepareString(str));
			if (iter.hasNext()) {
				buf.append(',');
			}
		}
		buf.append(") ");
		return buf.toString();
	}
}
