package clinical.web.services;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Assessment;
import clinical.server.vo.Experiment;
import clinical.server.vo.Storedassessment;
import clinical.server.vo.Subjexperiment;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.ISubjectVisitManagement;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.vo.AssessmentSelectionInfo;
import clinical.web.exception.BaseException;
import clinical.web.exception.SubjectVisitManagementException;
import clinical.web.vo.SubjectAsScoreValueSummary;
import clinical.web.workflow.common.WFGenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class BatchAssessmentServiceImpl extends AbstractServiceImpl implements
		IBatchAssessmentService {
	protected IDBCache dbCache;

	public BatchAssessmentServiceImpl(String dbID) throws BaseException {
		super(dbID);
		log = LogFactory.getLog(BatchAssessmentServiceImpl.class);
		dbCache = ServiceFactory.getDBCache(dbID);
	}

	public Map<Integer, List<String>> prepExpSubjectIDListMap(UserInfo ui,
			List<String> expNames) throws Exception {

		List<Experiment> experiments = dbCache.getExperiments(ui, false);
		List<Experiment> selExpList = new ArrayList<Experiment>(expNames.size());
		Set<String> expNameSet = new HashSet<String>(expNames);
		for (Experiment exp : experiments) {
			if (expNameSet.contains(exp.getName())) {
				selExpList.add(exp);
			}
		}

		return prepExpSubjectIDListMapFromExpList(ui, selExpList);
	}

	public Map<Integer, List<String>> prepExpSubjectIDListMapFromExpIDs(
			UserInfo ui, List<Integer> expIds) throws Exception {

		List<Experiment> experiments = dbCache.getExperiments(ui, false);
		List<Experiment> selExpList = new ArrayList<Experiment>(expIds.size());
		Set<Integer> expIDSet = new HashSet<Integer>(expIds);
		for (Experiment exp : experiments) {
			if (expIDSet.contains(exp.getUniqueid().intValue())) {
				selExpList.add(exp);
			}
		}

		return prepExpSubjectIDListMapFromExpList(ui, selExpList);
	}

	private Map<Integer, List<String>> prepExpSubjectIDListMapFromExpList(
			UserInfo ui, List<Experiment> selExpList) throws BaseException,
			SubjectVisitManagementException {
		Map<Integer, List<String>> expSubjectIDListMap = new HashMap<Integer, List<String>>(
				7);
		ISubjectVisitManagement isvm = ServiceFactory
				.getSubjectVisitManagement(theDBID);
		for (Experiment selExp : selExpList) {
			List<Subjexperiment> seList = isvm.getSubjectsForExperiment(ui,
					selExp.getUniqueid().intValue());
			List<String> subjectIDList = new ArrayList<String>(seList.size());
			for (Subjexperiment se : seList) {
				subjectIDList.add(se.getSubjectid());
			}
			expSubjectIDListMap.put(selExp.getUniqueid().intValue(),
					subjectIDList);
		}
		return expSubjectIDListMap;
	}

	public List<SubjectAsScoreValueSummary> getAllAssessmentValues(UserInfo ui,
			Map<Integer, List<String>> expSubjectIDListMap) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			List<Assessment> asList = dbCache.getAssessments(ui, false);
			List<AssessmentSelectionInfo> asiList = new ArrayList<AssessmentSelectionInfo>(
					asList.size());
			for (Assessment as : asList) {
				// assessmentID is same as uniqueid
				AssessmentSelectionInfo asi = new AssessmentSelectionInfo(
						as.getName(), as.getAssessmentid());
				asiList.add(asi);
			}
			List<Storedassessment> saList = AssessmentServiceHelper
					.getStoredAsRecords(con, expSubjectIDListMap, asiList,
							this.sqlDialect);

			List<SubjectAsScoreValueSummary> fullList = AssessmentServiceHelper
					.getFullAssessments(con, saList, this.sqlDialect);
			return fullList;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void prepareCSVFile(UserInfo ui, File file,
			List<SubjectAsScoreValueSummary> sasvsList) throws Exception {
		BufferedWriter out = null;
		CSVTable table = prepCSVTable(ui, sasvsList);

		try {
			out = new BufferedWriter(new FileWriter(file));
			out.write(table.header.toCSV());
			out.newLine();
			for (String rowKey : table.rowKeys) {
				CSVTableRow row = table.rowMap.get(rowKey);
				out.write(row.toCSV(table.allScoreLabels));
				out.newLine();
			}
		} finally {
			FileUtils.close(out);
		}

	}

	public CSVTable prepCSVTable(UserInfo ui,
			List<SubjectAsScoreValueSummary> sasvsList) throws Exception {
		List<Experiment> experiments = dbCache.getExperiments(ui, false);
		Map<Integer, Experiment> expMap = new HashMap<Integer, Experiment>();
		for (Experiment exp : experiments) {
			expMap.put(exp.getUniqueid().intValue(), exp);
		}

		List<Assessment> asList = dbCache.getAssessments(ui, false);
		Map<String, Assessment> asMap = new HashMap<String, Assessment>();
		for (Assessment as : asList) {
			asMap.put(String.valueOf(as.getUniqueid()), as);
		}

		Map<String, AsScores> asScoreMap = new HashMap<String, AsScores>();

		Map<String, CSVTableRow> rowMap = new LinkedHashMap<String, CSVTableRow>();

		for (SubjectAsScoreValueSummary sasvs : sasvsList) {
			StringBuilder sb = new StringBuilder();
			sb.append(sasvs.getSubjectID()).append(':');
			Experiment exp = expMap.get(sasvs.getExperimentID());
			sb.append(exp.getName()).append(':');
			sb.append(sasvs.getVisitID());
			String key = sb.toString();
			CSVTableRow row = rowMap.get(key);
			if (row == null) {
				row = new CSVTableRow(sasvs.getSubjectID(), exp.getName(),
						String.valueOf(sasvs.getVisitID()));
				rowMap.put(key, row);
			}
			AsScores asc = asScoreMap.get(sasvs.getAssessmentID());
			if (asc == null) {
				Assessment a = asMap.get(sasvs.getAssessmentID());
				asc = new AsScores(a.getName());
				asScoreMap.put(sasvs.getAssessmentID(), asc);
			}
			asc.addScore(sasvs.getScoreName());
			String scoreLabel = prepScoreLabel(asc.asName, sasvs.getScoreName());
			row.addScore(scoreLabel, sasvs.getValue() != null ? sasvs
					.getValue().toString() : null);
		}

		List<String> allScoreLabels = new ArrayList<String>();
		List<AsScores> ascList = new ArrayList<AsScores>(asScoreMap.values());
		Collections.sort(ascList, new Comparator<AsScores>() {
			@Override
			public int compare(AsScores o1, AsScores o2) {
				if (o1.asName.equals("Subject Information")) {
					return -1;
				} else if (o2.asName.equals("Subject Information")) {
					return 1;
				}
				return o1.asName.compareTo(o2.asName);
			}
		});
		for (AsScores asc : ascList) {
			allScoreLabels.addAll(asc.getOrderedScoreLabels());
		}

		List<String> rowKeys = new ArrayList<String>(rowMap.keySet());
		Collections.sort(rowKeys, new Comparator<String>() {
			@Override
			public int compare(String o1, String o2) {
				String[] toks1 = o1.split(":");
				String[] toks2 = o2.split(":");
				int rc = toks1[0].compareTo(toks2[0]);
				if (rc == 0) {
					rc = toks1[2].compareTo(toks2[2]);
				}
				if (rc == 0) {
					rc = toks1[1].compareTo(toks2[1]);
				}

				return rc;
			}
		});

		CSVTableHeader header = new CSVTableHeader();
		for (String scoreLabel : allScoreLabels) {
			header.addScoreHeader(scoreLabel);
		}
		
		CSVTable table = new CSVTable(header, rowMap, rowKeys, allScoreLabels);
		return table;
	}

	public static String prepScoreLabel(String asName, String scoreName) {
		StringBuilder sb = new StringBuilder(50);
		sb.append(asName.replaceAll("\\s+", "_"));
		sb.append("__");
		sb.append(scoreName.replaceAll("\\s+", "_"));

		return sb.toString();
	}

	public static class AsScores {
		String asName;
		List<String> scores;
		Set<String> scoreSet = new HashSet<String>();

		public AsScores(String asName) {
			this.asName = asName;
		}

		public void addScore(String scoreName) {
			scoreSet.add(scoreName);
		}

		public void orderScores() {
			scores = new ArrayList<String>(scoreSet);
			Collections.sort(scores);
		}

		public List<String> getOrderedScores() {
			return scores;
		}

		public List<String> getOrderedScoreLabels() {
			List<String> scoreLabels = new ArrayList<String>(scoreSet.size());
			orderScores();
			for (String scoreName : scores) {
				String scoreLabel = prepScoreLabel(asName, scoreName);
				scoreLabels.add(scoreLabel);
			}
			return scoreLabels;
		}
	}

	public static class CSVTable {
		CSVTableHeader header;
		Map<String, CSVTableRow> rowMap;
		List<String> rowKeys;
		List<String> allScoreLabels;

		public CSVTable(CSVTableHeader header, Map<String, CSVTableRow> rowMap,
				List<String> rowKeys, List<String> allScoreLabels) {
			super();
			this.header = header;
			this.rowMap = rowMap;
			this.rowKeys = rowKeys;
			this.allScoreLabels = allScoreLabels;
		}

		public CSVTableHeader getHeader() {
			return header;
		}

		public Map<String, CSVTableRow> getRowMap() {
			return rowMap;
		}

		public List<String> getRowKeys() {
			return rowKeys;
		}

		public List<String> getAllScoreLabels() {
			return allScoreLabels;
		}
	}// ;

	public static class CSVTableRow {
		String subjectID;
		String expName;
		String visitID;
		Map<String, String> svMap = new HashMap<String, String>();

		public CSVTableRow(String subjectID, String expName, String visitID) {
			super();
			this.subjectID = subjectID;
			this.expName = expName;
			this.visitID = visitID;
		}

		public void addScore(String asScoreName, String value) {
			svMap.put(asScoreName, value);
		}
		
		public String getValue(String asScoreName) {
			return svMap.get(asScoreName);
		}

		public String toCSV(List<String> asScoreNameList) {
			StringBuilder sb = new StringBuilder(4096);
			sb.append(subjectID).append(',').append(expName).append(',');
			sb.append(visitID);
			for (String asScoreName : asScoreNameList) {
				sb.append(',');
				String value = svMap.get(asScoreName);
				if (value != null) {
					// no comma is allowed in a CSV column
					value = WFGenUtils.replaceCommas(value, ";");
					sb.append(value);
				}
			}
			return sb.toString();
		}

		public String getSubjectID() {
			return subjectID;
		}

		public String getExpName() {
			return expName;
		}

		public String getVisitID() {
			return visitID;
		}

	}// ;

	public static class CSVTableHeader {
		List<String> scoreHeaders = new ArrayList<String>();

		public CSVTableHeader() {
		}

		public void addScoreHeader(String scoreHeader) {
			scoreHeaders.add(scoreHeader);
		}
		
		public String toCSV() {
			StringBuilder sb = new StringBuilder(4096);
			sb.append("SubjectID").append(',').append("Project Name")
					.append(',');
			sb.append("Visit ID");
			for (String scoreHeader : scoreHeaders) {
				sb.append(',').append(scoreHeader);
			}
			return sb.toString();
		}

		public List<String> getScoreHeaders() {
			return scoreHeaders;
		}
		
		public static String getScoreName(String scoreHeader) {
			int idx = scoreHeader.indexOf("__");
			Assertion.assertTrue(idx != -1);
			return scoreHeader.substring(idx + 2);
		}
		
		public static String getAssessmentName(String scoreHeader) {
			int idx = scoreHeader.indexOf("__");
			Assertion.assertTrue(idx != -1);
			return scoreHeader.substring(0, idx);
		}
	}
}
