package clinical.web.services;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.LogFactory;

import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.services.BatchAssessmentServiceImpl.CSVTable;
import clinical.web.services.BatchAssessmentServiceImpl.CSVTableRow;
import clinical.web.vo.GroupAnalysisCBFJobInfo;
import clinical.web.workflow.common.WFGenUtils;

/**
 * Implementation of the common group analysis database services
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class CBFGroupAnalysisServiceImpl extends AbstractServiceImpl implements
		ICBFGroupAnalysisService {
	protected IDBCache dbCache;

	public CBFGroupAnalysisServiceImpl(String dbID) throws BaseException {
		super(dbID);
		log = LogFactory.getLog(CBFGroupAnalysisServiceImpl.class);
		dbCache = ServiceFactory.getDBCache(dbID);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see clinical.web.services.ICBFGroupAnalysisService#
	 * prepareBaselineAugmentedCSVFile(clinical.web.common.UserInfo,
	 * java.io.File, java.util.List)
	 */
	@Override
	public File prepareBaselineAugmentedCSVFile(UserInfo ui, File inputRootDir,
			List<GroupAnalysisCBFJobInfo> gajiList) throws Exception {
		Assertion.assertTrue(inputRootDir.isDirectory());
		List<BaselineGARec> recList = new ArrayList<BaselineGARec>();
		for (GroupAnalysisCBFJobInfo gaji : gajiList) {
			String subjectID = gaji.getSubjectId();
			String site = null;
			String expName = gaji.getJvi().getExpName();
			int idx = expName.lastIndexOf('_');
			if (idx != -1) {
				site = expName.substring(idx + 1);
				if (!CBFWFManagementService.isAKnownSite(site)) {
					site = null;
				}
			}
			List<File> allFiles = FileUtils.getAllFilesUnderDir(gaji.getJvi()
					.getDataURI());
			String gmCBFValue = null;
			for (File f : allFiles) {
				String name = f.getName();
				if (name.equals("gm.txt")) {
					String text = null;
					try {
						text = FileUtils.loadTextFile(f.getAbsolutePath());
						if (text != null) {
							gmCBFValue = text.trim();
						}
					} catch (IOException iox) {
						// no op
					}
					break;
				}
			}

			BaselineGARec rec = new BaselineGARec(subjectID, expName,
					gaji.getVisitDate(), gaji.getJvi().getVisitID(), gaji
							.getJvi().getJobID(), site, gmCBFValue);
			recList.add(rec);

		}

		ICBFStandardSpaceGroupAnalysisService ssgaService = ServiceFactory
				.getCBFStandardSpaceGroupAnalysisService(theDBID);

		CSVTable asDataTable = ssgaService.loadAssessmentData(ui, gajiList);
		List<CSVColumnMapping> headerMappings = prepHeaderMappings(asDataTable
				.getHeader().getScoreHeaders());

		Map<String, CSVTableRow> asRowMap = new HashMap<String, CSVTableRow>();

		for (CSVTableRow tr : asDataTable.getRowMap().values()) {
			// subjectID ':' expName ':' visitID
			String key = prepKey(tr);
			asRowMap.put(key, tr);
		}

		List<String> augHeadRow = new ArrayList<String>(headerMappings.size());
		for (CSVColumnMapping mapping : headerMappings) {
			augHeadRow.add(mapping.destKey);
		}

		List<List<String>> augRows = new ArrayList<List<String>>(recList.size());
		augRows.add(augHeadRow);

		for (BaselineGARec rec : recList) {
			String key = prepKey(rec);
			CSVTableRow asRow = asRowMap.get(key);
			List<String> augRow = new ArrayList<String>(headerMappings.size());
			augRows.add(augRow);
			for (CSVColumnMapping mapping : headerMappings) {
				if (mapping.srcType == CSVColumnMapping.OTHER) {
					if (mapping.srcKey.equals("SubjectID")) {
						augRow.add(asString(rec.subjectID));
					} else if (mapping.srcKey.equals("JobID")) {
						augRow.add(asString(rec.jobID));
					} else if (mapping.srcKey.equals("Project")) {
						augRow.add(rec.projectName);
					} else if (mapping.srcKey.equals("VisitID")) {
						augRow.add(String.valueOf(rec.visitID));
					} else if (mapping.srcKey.equals("Visit Date")) {
						augRow.add(rec.visitDate);
					} else if (mapping.srcKey.equals("GM CBF")) {
						augRow.add(rec.gmCBF != null ? rec.gmCBF : ".");
					} else if (mapping.destKey.equals("Systolic_BP")) {
						String value = asRow.getValue(mapping.srcKey);
						augRow.add( parseBloodPressure(value, true));
					} else if (mapping.destKey.equals("Diastolic_BP")) {
						String value = asRow.getValue(mapping.srcKey);
						augRow.add( parseBloodPressure(value, false));
					}
				} else {
					// CSVColumnMapping.ASSESSMENT
					if (asRow != null) {
						String value = asRow.getValue(mapping.srcKey);
						value = WFGenUtils.replaceCommas(value, ";");
						augRow.add(value != null ? value : ".");
					} else {
						// missing value
						augRow.add(".");
					}
				}
			}
		}
		File outCSVFile = new File(inputRootDir, "baseline_group_analysis.csv");
		BufferedWriter out = null;
		try {
			out = new BufferedWriter(new FileWriter(outCSVFile));
			for (List<String> augRow : augRows) {
				String csvRow = GenUtils.join(augRow, ",");
				out.write(csvRow);
				out.newLine();
			}
		} finally {
			FileUtils.close(out);
		}
		return outCSVFile;
	}

	public static String parseBloodPressure(String bpValue, boolean systolic) {
		if (bpValue == null || bpValue.length() == 0
				|| bpValue.equalsIgnoreCase("dk")) {
			return ".";
		}
		String[] toks = bpValue.split("\\/");
		if (toks.length != 2) {
			return ".";
		}
        return systolic ? toks[0] : toks[1];
	}

	static String asString(String value) {
		StringBuilder sb = new StringBuilder(value.length() + 2);
		sb.append('"').append(value).append('"');
		return sb.toString();
	}

	protected static String prepKey(CSVTableRow tr) {
		StringBuilder sb = new StringBuilder();
		sb.append(tr.getSubjectID()).append(':').append(tr.getExpName())
				.append(':').append(tr.getVisitID());
		return sb.toString();
	}

	protected static String prepKey(BaselineGARec rec) {
		StringBuilder sb = new StringBuilder();
		sb.append(rec.subjectID).append(':').append(rec.projectName)
				.append(':').append(rec.visitID);
		return sb.toString();
	}

	protected List<CSVColumnMapping> prepHeaderMappings(List<String> asHeaderRow) {
		List<CSVColumnMapping> mappings = new ArrayList<CSVColumnMapping>();
		mappings.add(new CSVColumnMapping("SubjectID", "SubjectID", 0, 0,
				CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("JobID", "JobID", 1, 1,
				CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("Project", "Project", 2, 2,
				CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("VisitID", "VisitID", 3, 3,
				CSVColumnMapping.OTHER));
		mappings.add(new CSVColumnMapping("Visit Date", "Visit Date", 4, 4,
				CSVColumnMapping.OTHER));
		int startIdx = -1;
		int i = 0;
		for (String scoreHeader : asHeaderRow) {
			if (scoreHeader.startsWith("Subject_Information__")) {
				startIdx = i;
				break;
			}
			i++;
		}

		// sDEMOG_BP (Demographics)
		int destOffset = 5;
		if (startIdx != -1) {
			int j = 0;
			for (i = startIdx; i < asHeaderRow.size(); i++) {
				String scoreHeader = asHeaderRow.get(i);
				if (!scoreHeader.startsWith("Subject_Information__")) {
					destOffset += j;
					break;
				}
				mappings.add(new CSVColumnMapping(scoreHeader, scoreHeader, i,
						destOffset + j, CSVColumnMapping.ASSESSMENT));
				j++;
			}
		}
		// check for sDEMOG_BP, if found create two additional variables
		if (startIdx != -1) {
			for (i = startIdx; i < asHeaderRow.size(); i++) {
				String scoreHeader = asHeaderRow.get(i);
				if (scoreHeader.indexOf("sDEMOG_BP") != -1) {
					mappings.add(new CSVColumnMapping(scoreHeader,
							"Systolic_BP", i, destOffset,
							CSVColumnMapping.OTHER));
					destOffset++;
					mappings.add(new CSVColumnMapping(scoreHeader,
							"Diastolic_BP", i, destOffset,
							CSVColumnMapping.OTHER));
					destOffset++;
					break;
				}
			}
		}

		mappings.add(new CSVColumnMapping("GM CBF", "GM CBF", -1, destOffset,
				CSVColumnMapping.OTHER));
		destOffset++;
		// now the rest of the assessments
		int j = 0;
		for (i = 3; i < asHeaderRow.size(); i++) {
			String scoreHeader = asHeaderRow.get(i);
			if (!scoreHeader.startsWith("Subject_Information__")) {
				mappings.add(new CSVColumnMapping(scoreHeader, scoreHeader, i,
						destOffset + j, CSVColumnMapping.ASSESSMENT));
				j++;
			}
		}

		return mappings;
	}

	public static class BaselineGARec {
		private String subjectID;
		private String projectName;
		private String visitDate;
		private int visitID;
		private String jobID;
		@SuppressWarnings("unused")
		private String site;
		private String gmCBF;

		public BaselineGARec(String subjectID, String projectName,
				String visitDate, int visitID, String jobID, String site,
				String gmCBF) {
			super();
			this.subjectID = subjectID;
			this.projectName = projectName;
			this.visitDate = visitDate;
			this.visitID = visitID;
			this.jobID = jobID;
			this.site = site;
			this.gmCBF = gmCBF;
		}
		// TODO
	}// ;

	public static class CSVColumnMapping {
		String srcKey;
		String destKey;
		/** column index in the source */
		int srcIdx;
		int destIdx;
		int srcType;
		public static final int OTHER = 1;
		public static final int ASSESSMENT = 2;

		public CSVColumnMapping(String srcKey, String destKey, int srcIdx,
				int destIdx, int srcType) {
			super();
			this.srcKey = srcKey;
			this.destKey = destKey;
			this.srcIdx = srcIdx;
			this.destIdx = destIdx;
			this.srcType = srcType;
		}
	}// ;

}
