package clinical.web.services;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.jdom.Element;

import clinical.utils.FileUtils;
import clinical.utils.GenUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */

public class CBFROILookupService {
	private String roiLabelSetXmlFile;
	public  final static String FREESURFER = "freesurfer";
	private Map<String, RoiLabelSet> roisLabelSetMap = new HashMap<String, RoiLabelSet>();
	private static CBFROILookupService instance = null;

	/**
	 * intentionally not thread safe. Intended to be called once at the server
	 * startup
	 * 
	 * @param roiLabelSetXmlFile
	 * @return
	 */
	public static CBFROILookupService getInstance(String roiLabelSetXmlFile) throws Exception {
		if (instance == null) {
			instance = new CBFROILookupService(roiLabelSetXmlFile);
		}
		return instance;
	}

	/**
	 * intentionally not thread safe. Just returns the already (at startup)
	 * created instance
	 * 
	 * @return
	 */
	public static CBFROILookupService getInstance() {
		if (instance == null) {
			throw new RuntimeException(
					"CBFROILookupService is not initialized properly!");
		}
		return instance;
	}

	private CBFROILookupService(String roiLabelSetXmlFile) throws Exception {
		this.roiLabelSetXmlFile = roiLabelSetXmlFile;
		loadAndPopulate();
	}
	
	public String getLabel(String lutTypeName, Integer idx) {
		RoiLabelSet roiLabelSet = this.roisLabelSetMap.get(lutTypeName);
		if (roiLabelSet == null) {
			return null;
		}
		return roiLabelSet.getLabel(idx);
	}
	
	public Integer getROINumber(String lutTypeName, String label) {
		RoiLabelSet roiLabelSet = this.roisLabelSetMap.get(lutTypeName);
		if (roiLabelSet == null) {
			return null;
		}
		return roiLabelSet.getIndex(label);
	}

	private void loadAndPopulate() throws Exception {
		Element rootEl = FileUtils.loadXML(roiLabelSetXmlFile);
		List<?> rsEls = rootEl.getChildren("roi-label-set");
		for (Object o : rsEls) {
			Element rsEl = (Element) o;
			String name = rsEl.getAttributeValue("name");
			String lutFilePath = rsEl.getTextTrim();
			
			IROILUTParserStrategy strategy = LUTParserStrategyFactory.newStrategy(name);
			RoiLabelSet roiLabelSet = new RoiLabelSet(name, strategy);
			
			roiLabelSet.populate(lutFilePath);
			roisLabelSetMap.put(name, roiLabelSet);
		}
	}
	
	
	public static class LUTParserStrategyFactory {
		private LUTParserStrategyFactory() {
		}
		
		public static IROILUTParserStrategy newStrategy(String lutTypeName) {
			if (lutTypeName != null && lutTypeName.equals("freesurfer")) {
				return new FreesurferLUTParserStrategy();
			} else {
				throw new RuntimeException("Unsupported ROI LUT type!");
			}
		}
	}

	public static class FreesurferLUTParserStrategy implements
			IROILUTParserStrategy {

		@Override
		public Map<Integer, String> loadTable(File lutFile) throws Exception {
			Map<Integer, String> idx2LabelMap = new HashMap<Integer, String>();
			BufferedReader in = null;
			try {
				in = new BufferedReader( new FileReader(lutFile));
				String line = null;
				while((line = in.readLine()) != null) {
					line = line.trim();
					if (line.length() == 0 || line.startsWith("#")) {
						continue;
					}
					String[] toks = line.split("\\s+");
					if (toks.length == 6) {
						int idx = GenUtils.toInt(toks[0], -1);
						if (idx >= 0) {
							idx2LabelMap.put(new Integer(idx), toks[1]);
						}
					}
				}
			} finally {
				FileUtils.close(in);
			}
			return idx2LabelMap;
		}

	}

	public static class RoiLabelSet {
		private String name;
		IROILUTParserStrategy strategy;
		private Map<String, Integer> label2IdxMap;
		private Map<Integer, String> idx2LabelMap;

		public RoiLabelSet(String name, IROILUTParserStrategy strategy) {
			this.name = name;
			this.strategy = strategy;
		}

		public void populate(String lutFile) throws Exception {
			this.idx2LabelMap = strategy.loadTable(new File(lutFile));
			this.label2IdxMap = new HashMap<String, Integer>();
			for (Map.Entry<Integer, String> entry : this.idx2LabelMap
					.entrySet()) {
				label2IdxMap.put(entry.getValue(), entry.getKey());
			}
		}

		public String getLabel(Integer idx) {
			return idx2LabelMap.get(idx);
		}

		public Integer getIndex(String label) {
			return label2IdxMap.get(label);
		}

		public String getName() {
			return name;
		}
	}
}
