package clinical.web.services;

import java.io.File;
import java.util.Date;
import java.util.Iterator;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.vo.Jobs;
import clinical.utils.FileUtils;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.IAppConfigService;
import clinical.web.IJobManagementService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.download.DownloadCachePolicy;
import clinical.web.exception.BaseException;
import clinical.web.exception.JobManServiceException;
import clinical.web.scheduler.JobInfo;

/**
 * @author I. Burak Ozyurt
 * @version $Id: DownloadCacheCleanupService.java,v 1.5 2008/01/12 01:00:33
 *          bozyurt Exp $
 */
public class DownloadCacheCleanupService implements Runnable {
	protected String cacheRoot;
	protected long cacheCheckInterval;
	protected String dbID;
	protected IAppConfigService configService;
	IJobManagementService jms;
	protected UserInfo ui;
	protected DownloadCachePolicy policy;
	private static Log log = LogFactory.getLog(DownloadCacheCleanupService.class);

	public DownloadCacheCleanupService(String cacheRoot,
			IAppConfigService configService, String dbID) throws BaseException {
		this.cacheRoot = cacheRoot;
		this.configService = configService;
		
		// default 5 minutes
		cacheCheckInterval = GenUtils.toLong(configService
				.getParamValue("download.cache.cache_check_interval"), 300); 
		cacheCheckInterval *= 1000; // to millisecs
		this.jms = ServiceFactory.getJobManagementService(dbID);
		this.ui = new UserInfo(Constants.ADMIN_USER, null, null);
		this.policy = DownloadCachePolicy.getInstance(cacheRoot);
	}

	public void run() {
		while (true) {
			try {
				if (log.isDebugEnabled()) {
					log.debug("doing routine download cache sweeping...");					
				}
				doCleanup();
			} catch (Throwable t) {
				log.error("", t);
			}
			synchronized (this) {
				try {
					wait(cacheCheckInterval);
				} catch (InterruptedException iex) {
				}
			}
		}
	}

	protected void doCleanup() throws JobManServiceException {
		List<Jobs> jobs = jms.getAllJobs(ui);
		if (jobs.isEmpty())
			return;
		for (Iterator<Jobs> it = jobs.iterator(); it.hasNext();) {
			Jobs job = (Jobs) it.next();
			if (job.getLastdownloadtime() == null) {
				File dir = new File(cacheRoot, job.getJobid());
				if (!dir.exists()) {
					it.remove();
				} else {
					job.setLastdownloadtime(new Date(dir.lastModified()));
				}
			}
		}

		long now = System.currentTimeMillis();
		boolean isLow = policy.isBelowLowMark(new File(cacheRoot)
				.getUsableSpace());
		for (Iterator<Jobs> it = jobs.iterator(); it.hasNext();) {
			Jobs job = (Jobs) it.next();
			String status = job.getJobstatus();
			if (status.endsWith("_expired")) {
				continue;
			}
			long age = now - job.getLastdownloadtime().getTime();
			int downloadCount = job.getDownloadcount() != null ? job.getDownloadcount().intValue() : 0; 
			boolean ok = canbeEvicted(status, age, isLow, downloadCount);
			if ( ok) {
				if (!JobInfo.FINISHED_WITH_ERR.equals(status)) {
					jms.expireJob(ui, job.getJobid());
				}
				File dir = new File(cacheRoot, job.getJobid());
				log.info("deleting " + dir);
				FileUtils.deleteRecursively(dir);
				FileUtils.deleteRecursively(dir);
			}
			/*
			if (age > maxAge) {
				jms.expireJob(ui, job.getJobid());
				File dir = new File(cacheRoot, job.getJobid());
				log.info("deleting " + dir);
				FileUtils.deleteRecursively(dir);
			}
			*/
		}
	}

	protected boolean canbeEvicted(String status, long age, boolean isLow,
			int downloadCount) {
		if (JobInfo.FINISHED_WITH_ERR.equals(status)) {
			return true;
		}
		if (isLow) {
			if (JobInfo.FINISHED.equals(status)) {
				if (downloadCount == 0)
					return (age > policy.getFinishedLowAge() * 3600 * 1000);
				else
					return (age > policy.getDownloadedLowAge() * 3600 * 1000);
			}
		} else {
			if (JobInfo.FINISHED.equals(status)) {
				if (downloadCount == 0)
					return (age > policy.getFinishedNormalAge() * 3600 * 1000);
				else
					return (age > policy.getDownloadedNormalAge() * 3600 * 1000);
			}
		}

		return false;
	}

}
