package clinical.web.services;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Callable;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONException;

import clinical.cache.DBChangeEvent;
import clinical.cache.DBChangeNotifyListener;
import clinical.utils.ThreadPoolMan;
import clinical.web.ServiceFactory;
import clinical.web.common.ISecurityService;
import clinical.web.exception.BaseException;

/**
 *
 * @author I. Burak Ozyurt
 * @version $Id: FederatedDBCacheServiceImpl.java,v 1.1.2.1 2008/08/14 01:08:26
 *          bozyurt Exp $
 */
public class FederatedDBCacheServiceImpl implements DBChangeNotifyListener {
	private ThreadPoolMan threadPoolMan;
	private ISecurityService ss;
	private Log log = LogFactory.getLog(FederatedDBCacheServiceImpl.class);

	public FederatedDBCacheServiceImpl() throws BaseException {
		threadPoolMan = ThreadPoolMan.getInstance();
		ss = ServiceFactory.getSecurityService();
	}

	public List<String> getTargetURLs() {
		List<String> targetURLs = new ArrayList<String>(10);
		String[] dbIDs = ss.getAllDBIDs();
		Map<String, String> siteURLMap = ss.getSiteURLs();
		for (int i = 0; i < dbIDs.length; i++) {
			if (!dbIDs[i].equals(ss.getDefaultDBID())) {
				String siteURL = siteURLMap.get(dbIDs[i]);
				if (siteURL != null) {
					StringBuilder sb = new StringBuilder(siteURL.length() + 60);
					sb.append(siteURL);
					if ( !siteURL.endsWith("/")) sb.append("/");
					sb.append("cacheevent.do?action=changed");
					targetURLs.add(sb.toString());
				}
			}
		}
		return targetURLs;
	}

	@Override
	public void dbChanged(DBChangeEvent event) {
		List<String> targetURLs = getTargetURLs();
		if (targetURLs.isEmpty())
			return;
		try {
			String dbChangeJSON = event.getDbChange().toJSON().toString();
			Notifier notifier = new Notifier(targetURLs, dbChangeJSON);
			threadPoolMan.getExecutorService().submit(notifier);
		} catch (JSONException e) {
			log.error("dbChanged", e);
		}
	}

	public static class Notifier implements Callable<Integer> {
		List<String> targetURLs;
		String dbChangeJSON;

		public Notifier(List<String> targetURLs, String dbChangeJSON) {
			this.targetURLs = targetURLs;
			this.dbChangeJSON = dbChangeJSON;
		}

		@Override
		public Integer call() throws Exception {
			int count = 0;
			for (String targetURL : targetURLs) {
				try {
					notifySite(targetURL, dbChangeJSON);
					count++;
				} catch (Exception x) {
					// ignore
				}
			}
			return new Integer(targetURLs.size() - count);
		}

		protected void notifySite(String targetURL, String dbChangeJSON)
				throws Exception {
			PostMethod post = new PostMethod(targetURL);

			try {
				post.addParameter("dbChangeJSON", dbChangeJSON);
				HttpClient client = new HttpClient();
				client.getHttpConnectionManager().getParams()
						.setConnectionTimeout(5000);

				int status = client.executeMethod(post);
				if (status == HttpStatus.SC_OK) {
					System.out.println("Succesfull notification of site: "
							+ targetURL);
				} else {
					throw new Exception("problem in notification of site: "
							+ targetURL + "\nHTTP Status: " + status);
				}
			} finally {
				post.releaseConnection();
			}
		}

	}

}
