package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import clinical.server.dao.DataarchiveDAO;
import clinical.server.vo.Assessment;
import clinical.server.vo.Assessmentdata;
import clinical.server.vo.Dataarchive;
import clinical.server.vo.Dataobject;
import clinical.server.vo.Dataobjecttype;
import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Rawdata;
import clinical.utils.Assertion;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.IImageDataService;
import clinical.web.IRemoteDBServices;
import clinical.web.ISQLDialect;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.BaseException;
import clinical.web.vo.AssessmentInfo;
import clinical.web.vo.RawDataObject;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: ImageDataServices.java,v 1.1.2.1 2007/06/28 01:49:01 bozyurt
 *          Exp $
 */
public class ImageDataServices implements IImageDataService {
	private IDBPoolService pool = null;
	private String theDBID;
	private ISQLDialect sqlDialect;
	private Log log = LogFactory.getLog(ImageDataServices.class);

	public ImageDataServices(String dbID) throws BaseException {
		pool = ServiceFactory.getPoolService(dbID);
		theDBID = dbID;
		sqlDialect = ServiceFactory.getSQLDialect(theDBID);
	}

	public List<RawDataObject> getRawData(String siteID, UserInfo userInfo,
			int[] expIDs, List<String> subjectIDList, String[] objectTypes,
			boolean verbose) throws Exception {
		ISecurityService ss = ServiceFactory.getSecurityService();
		String dbID = null;
		if (siteID == null) {
			// default site
			dbID = ss.getDefaultDBID();
			siteID = ss.findSiteIDByDbID(dbID);
			Assertion.assertNotNull(siteID);
		} else {
			dbID = ss.findDBForSiteID(siteID);
		}
		Assertion.assertNotNull(dbID);
		IDBPoolService pool = ServiceFactory.getPoolService(dbID);
		IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();

		List<String> dotIDList = new ArrayList<String>(objectTypes.length);
		Map<BigDecimal, Dataobjecttype> dotIDMap = new HashMap<BigDecimal, Dataobjecttype>(
				7);

		Connection con = null;
		try {
			con = pool.getConnection(Constants.ADMIN_USER);
			Map<String, Dataobjecttype> dotMap = rds
					.getAllDataObjectTypes(siteID);
			for (int i = 0; i < objectTypes.length; i++) {
				Dataobjecttype dot = dotMap.get(objectTypes[i]);
				if (dot != null) {
					dotIDList.add(dot.getUniqueid().toString());
					dotIDMap.put(dot.getUniqueid(), dot);
				}
			}
			String[] dotIDArr = new String[dotIDList.size()];
			dotIDArr = dotIDList.toArray(dotIDArr);

			ISQLDialect dialect = ServiceFactory.getSQLDialect(dbID);
			TSQLProcessor tp = new TSQLProcessor(dialect);
			StringBuffer qBuf = prepQuery(expIDs, subjectIDList, objectTypes,
					dotIDArr);

			List<?> results = tp.executeQuery(con, qBuf.toString());
			List<RawDataObject> rdoList = prepResults(dotIDMap, results,
					verbose);
			results = null;
			return rdoList;

		} finally {
			if (con != null)
				pool.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public Map<String, Map<String, Expsegment>> getSegments(String siteID,
			UserInfo userInfo, List<String> expNames, List<String> subjectIDList)
			throws Exception {
		ISecurityService ss = ServiceFactory.getSecurityService();
		String dbID = theDBID;
		if (siteID != null) {
			dbID = ss.findDBForSiteID(siteID);
			Assertion.assertNotNull(dbID);
		}
		IDBPoolService pool = ServiceFactory.getPoolService(dbID);
		Map<String, Map<String, Expsegment>> segmentMap = new HashMap<String, Map<String, Expsegment>>(
				7);
		Connection con = null;
		try {
			con = pool.getConnection(Constants.ADMIN_USER);
			StringBuilder sb = new StringBuilder(400);
			sb
					.append("select s.*, e.name from Expsegment as s, Experiment as e where ");
			sb.append("s.ncExperimentUniqueid = e.uniqueid and e.name in (");
			sb.append(GenUtils.join(expNames, ",", "'"));
			sb.append(") and s.subjectid in (");
			sb.append(GenUtils.join(subjectIDList, ",", "'"));
			sb.append(")");

			System.out.println("query: " + sb.toString());
			ISQLDialect dialect = ServiceFactory.getSQLDialect(dbID);
			TSQLProcessor tp = new TSQLProcessor(dialect);
			List<?> results = tp.executeQuery(con, sb.toString());
			for (Object element : results) {
				Object[] row = (Object[]) element;
				Expsegment es = (Expsegment) row[0];
				Experiment exp = (Experiment) row[1];
				Map<String, Expsegment> segList = segmentMap.get(exp.getName());
				if (segList == null) {
					segList = new HashMap<String, Expsegment>(29);
					segmentMap.put(exp.getName(), segList);
				}
				segList.put(prepKey(es), es);
			}

			return segmentMap;

		} finally {
			if (con != null)
				pool.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	protected String prepKey(Expsegment es) {
		StringBuilder sb = new StringBuilder(100);
		sb.append(es.getSubjectid()).append('_');
		sb.append(es.getComponentid()).append('_');
		sb.append(es.getSegmentid());
		return sb.toString();
	}

	public List<RawDataObject> getRawData(UserInfo userInfo, int[] expIDs,
			List<String> subjectIDList, String[] objectTypes, boolean verbose)
			throws Exception {

		IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
		List<String> dotIDList = new ArrayList<String>(objectTypes.length);
		Map<BigDecimal, Dataobjecttype> dotIDMap = new HashMap<BigDecimal, Dataobjecttype>(
				7);
		for (int i = 0; i < objectTypes.length; i++) {
			Dataobjecttype dot = rds.findObjectType(objectTypes[i], false);
			if (dot != null) {
				dotIDList.add(dot.getUniqueid().toString());
				dotIDMap.put(dot.getUniqueid(), dot);
			}
		}

		String[] dotIDArr = new String[dotIDList.size()];
		dotIDArr = dotIDList.toArray(dotIDArr);

		TSQLProcessor tp = new TSQLProcessor(sqlDialect);
		StringBuffer qBuf = prepQuery(expIDs, subjectIDList, objectTypes,
				dotIDArr);
		Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			List<?> results = tp.executeQuery(con, qBuf.toString());
			List<RawDataObject> rdoList = prepResults(dotIDMap, results,
					verbose);
			results = null;

			return rdoList;
		} finally {
			pool.releaseConnection(userInfo.getName(), con);
		}
	}

	private List<RawDataObject> prepResults(
			Map<BigDecimal, Dataobjecttype> dotIDMap, List<?> results,
			boolean verbose) {
		List<RawDataObject> rdoList = new ArrayList<RawDataObject>(results
				.size());
		for (Object element : results) {
			Object[] row = (Object[]) element;
			Rawdata rd = (Rawdata) row[0];
			Dataobject dataObject = (Dataobject) row[1];
			Dataobjecttype dot = dotIDMap.get(dataObject.getObjecttypeid());
			if (verbose) {
				System.out.println("dataObject=" + dataObject.toString());
				System.out.println("rd=" + rd + " dot=" + dot);
			}
			if (dot != null) {
				Assertion.assertNotNull(dot);
				RawDataObject rdo = new RawDataObject(rd, dot.getObjecttype(),
						dataObject.getObjectsize().longValue());

				rdoList.add(rdo);
			}
		}
		return rdoList;
	}

	protected StringBuffer prepQuery(int[] expIDs, List<String> subjectIDList,
			String[] objectTypes, String[] dotIDArr) {
		StringBuffer qBuf = new StringBuffer();
		qBuf
				.append("select r.*, d.objectsize, d.objecttypeid from Rawdata as r, Dataobject as d where ");
		if (expIDs != null) {
			if (expIDs.length == 1) {
				qBuf.append("r.ncExperimentUniqueid = ").append(expIDs[0])
						.append(" and ");
			} else {
				qBuf.append("r.ncExperimentUniqueid in (").append(
						GenUtils.join(expIDs, ",")).append(") and ");
			}
		}
		qBuf.append("r.subjectid in (").append(
				GenUtils.join(subjectIDList, ",", "'")).append(")");
		qBuf.append(" and r.uniqueid = d.dataid ");
		if (objectTypes != null && objectTypes.length > 0
				&& dotIDArr.length > 0) {
			qBuf.append(" and d.objecttypeid in (").append(
					GenUtils.join(dotIDArr, ",", null)).append(")");
		}
		qBuf
				.append(" order by r.ncExperimentUniqueid, r.subjectid, r.componentid, r.segmentid");
		return qBuf;
	}

	public List<?> getDerivedData(UserInfo userInfo, int[] expIDs,
			List<String> subjectIDList) throws Exception {
		TSQLProcessor tp = new TSQLProcessor(sqlDialect);
		StringBuffer qBuf = new StringBuffer();
		qBuf.append("select d.* from Deriveddata as d where ");
		if (expIDs != null) {
			if (expIDs.length == 1) {
				qBuf.append("d.ncExperimentUniqueid = ").append(expIDs[0])
						.append(" and ");
			} else {
				qBuf.append("d.ncExperimentUniqueid in (").append(
						GenUtils.join(expIDs, ",")).append(") and ");
			}
		}
		qBuf.append("d.subjectid in (").append(
				GenUtils.join(subjectIDList, ",", "'")).append(")");
		qBuf.append(" order by r.ncExperimentUniqueid, r.subjectid");
		Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			log.info(qBuf.toString());
			List<?> derivedDataList = tp.executeQuery(con, qBuf.toString());
			return derivedDataList;
		} finally {
			pool.releaseConnection(userInfo.getName(), con);
		}
	}

	public List<AssessmentInfo> getAssessmentInfos(String siteID, int[] expIDs,
			List<String> subjectIDList) throws Exception {
		IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();
		List<Assessment> assessments = rds.getAllAssessments(siteID);
		Map<BigDecimal, Assessment> asIDMap = new HashMap<BigDecimal, Assessment>();
		for (Assessment as : assessments) {
			asIDMap.put(as.getAssessmentid(), as);
		}
		assessments = null;
		ISecurityService ss = ServiceFactory.getSecurityService();
		String dbID = ss.findDBForSiteID(siteID);
		Assertion.assertNotNull(dbID);
		IDBPoolService dbPool = ServiceFactory.getPoolService(dbID);
		TSQLProcessor tp = new TSQLProcessor(ServiceFactory.getSQLDialect(dbID));
		StringBuffer qBuf = prepQuery(expIDs, subjectIDList);
		Connection con = null;
		try {
			con = dbPool.getConnection(Constants.ADMIN_USER);
			List<?> adList = tp.executeQuery(con, qBuf.toString());
			List<AssessmentInfo> asiList = prepareAsInfoList(asIDMap, adList);
			return asiList;
		} finally {
			if (con != null)
				dbPool.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public List<AssessmentInfo> getAssessmentInfos(UserInfo userInfo,
			int[] expIDs, List<String> subjectIDList) throws Exception {
		IDBCache dbCache = ServiceFactory.getDBCache(theDBID);
		List<Assessment> assessments = dbCache.getAssessments(userInfo, false);
		Map<BigDecimal, Assessment> asIDMap = new HashMap<BigDecimal, Assessment>();
		for (Assessment as : assessments) {
			asIDMap.put(as.getAssessmentid(), as);
		}
		assessments = null;

		TSQLProcessor tp = new TSQLProcessor(sqlDialect);
		StringBuffer qBuf = prepQuery(expIDs, subjectIDList);
		Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			log.info("getAssessmentInfos: " + qBuf.toString());
			List<?> adList = tp.executeQuery(con, qBuf.toString());
			List<AssessmentInfo> asiList = prepareAsInfoList(asIDMap, adList);
			return asiList;
		} finally {
			pool.releaseConnection(userInfo.getName(), con);
		}
	}

	protected List<AssessmentInfo> prepareAsInfoList(
			Map<BigDecimal, Assessment> asIDMap, List<?> adList) {
		List<AssessmentInfo> asiList = new ArrayList<AssessmentInfo>(adList
				.size());
		for (Iterator<?> iter = adList.iterator(); iter.hasNext();) {
			Assessmentdata ad = (Assessmentdata) iter.next();
			Assessment as = asIDMap.get(ad.getAssessmentid());
			AssessmentInfo asi = new AssessmentInfo(ad.getAssessmentid()
					.intValue(), as.getName(), null, -1, -1);
			asiList.add(asi);
		}
		if (asiList.size() > 1) {
			Collections.sort(asiList, new Comparator<AssessmentInfo>() {
				public int compare(AssessmentInfo o1, AssessmentInfo o2) {
					return o1.getName().compareToIgnoreCase(o2.getName());
				}

			});
		}
		return asiList;
	}

	private StringBuffer prepQuery(int[] expIDs, List<String> subjectIDList) {
		StringBuffer qBuf = new StringBuffer();
		qBuf
				.append("select distinct a.assessmentid, a.subjectid from Assessmentdata as a, "
						+ "Storedassessment as s where ");
		if (expIDs != null) {
			if (expIDs.length == 1) {
				qBuf.append("s.ncExperimentUniqueid = ").append(expIDs[0])
						.append(" and ");
			} else {
				qBuf.append("s.ncExperimentUniqueid in (").append(
						GenUtils.join(expIDs, ",")).append(") and ");
			}
		}
		if (subjectIDList.size() == 1) {
			qBuf.append("s.subjectid = '").append(subjectIDList.get(0)).append(
					"'");
		} else {
			qBuf.append("s.subjectid in (").append(
					GenUtils.join(subjectIDList, ",", "'")).append(")");
		}
		qBuf.append(" and a.ncStoredassessmentUniqueid = s.uniqueid");
		qBuf.append(" and a.isvalidated = true order by a.subjectid");
		return qBuf;
	}

	public Map<String, List<Dataarchive>> getArchivedDataByProject(
			UserInfo userInfo) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(userInfo.getName());
			DataarchiveDAO dao = DAOFactory.createDataarchiveDAO(theDBID);

			List<?> list = dao.find(con, new Dataarchive());
			if (list == null) {
				return new HashMap<String, List<Dataarchive>>(0);
			}
			Map<String, List<Dataarchive>> byProjectMap = new LinkedHashMap<String, List<Dataarchive>>(
					7);
			for (Iterator<?> it = list.iterator(); it.hasNext();) {
				Dataarchive da = (Dataarchive) it.next();
				List<Dataarchive> childList = byProjectMap.get(da
						.getArchivename());
				if (childList == null) {
					childList = new ArrayList<Dataarchive>(10);
					byProjectMap.put(da.getArchivename(), childList);
				}
				childList.add(da);
			}

			for (List<Dataarchive> childList : byProjectMap.values()) {
				Collections.sort(childList, new Comparator<Dataarchive>() {
					public int compare(Dataarchive d1, Dataarchive d2) {
						return d1.getArchivename().compareTo(
								d2.getArchivename());
						// return d1.getDatauri().compareTo(d2.getDatauri());
					}
				});
			}
			return byProjectMap;
		} finally {
			pool.releaseConnection(userInfo.getName(), con);
		}
	}
}
