package clinical.web.services;

import java.math.BigDecimal;
import java.sql.Connection;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import clinical.server.dao.AnalysisDAO;
import clinical.server.dao.AnalysiscomponentDAO;
import clinical.server.dao.AnalysisflowDAO;
import clinical.server.dao.ArgumenttypeDAO;
import clinical.server.dao.DeriveddataDAO;
import clinical.server.dao.ExecutedtransformDAO;
import clinical.server.dao.ExecutedtransformflowDAO;
import clinical.server.dao.ExternaldataDAO;
import clinical.server.dao.HumansubjectDAO;
import clinical.server.dao.MachineDAO;
import clinical.server.dao.TransformationDAO;
import clinical.server.dao.TransformationargumentDAO;
import clinical.server.dao.TransformeddataDAO;
import clinical.server.dao.TransforminputdataDAO;
import clinical.server.vo.Analysis;
import clinical.server.vo.Analysiscomponent;
import clinical.server.vo.Analysisflow;
import clinical.server.vo.Animalspecies;
import clinical.server.vo.Argumenttype;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Deriveddata;
import clinical.server.vo.Executedtransform;
import clinical.server.vo.Executedtransformflow;
import clinical.server.vo.Experiment;
import clinical.server.vo.Expsegment;
import clinical.server.vo.Externaldata;
import clinical.server.vo.Humansubject;
import clinical.server.vo.Machine;
import clinical.server.vo.Site;
import clinical.server.vo.Transformation;
import clinical.server.vo.Transformationargument;
import clinical.server.vo.Transformeddata;
import clinical.server.vo.Transforminputdata;
import clinical.utils.Assertion;
import clinical.utils.BIRNURIException;
import clinical.utils.BIRNURIUtils;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IDerivedImageDataService;
import clinical.web.IProvenanceService;
import clinical.web.IRemoteDBServices;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBCache;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.exception.BaseException;
import clinical.web.services.AnalysisInstanceDataResult.ExTransformData;
import clinical.web.vo.AnalysisCompInfo;
import clinical.web.vo.AnalysisInfo;
import clinical.web.vo.ArgumentTypeInfo;
import clinical.web.vo.DerivedDataInfo;
import clinical.web.vo.ExecutedTransformInfo;
import clinical.web.vo.ExternalDataInfo;
import clinical.web.vo.MachineInfo;
import clinical.web.vo.TransformArgInfo;
import clinical.web.vo.TransformationInfo;
import clinical.xml.XMLUtils;
import clinical.xml.xcede2.AnalysisT;
import clinical.xml.xcede2.LevelDataRefsT;
import clinical.xml.xcede2.ProcessStepT;
import clinical.xml.xcede2.ProvenanceT;

/**
 * @author I. Burak Ozyurt
 * @version $Id: ProvenanceServiceImpl.java,v 1.1 2008/02/02 01:50:57 bozyurt
 *          Exp $
 */
public class ProvenanceServiceImpl extends AbstractServiceImpl implements
		IProvenanceService {
	protected IDBCache dbCache;

	public ProvenanceServiceImpl(String dbID) throws BaseException {
		super(dbID);
		dbCache = ServiceFactory.getDBCache(dbID);
	}

	public void deleteProvenanceForAnalysisInstance(Connection con,
			UserInfo ui, AnalysisT at) throws Exception {
		IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();

		String expName = at.getProjectID();
		String subjectID = at.getSubjectID();
		String segmentName = at.getEpisodeID();

		String[] toks = at.getVisitID().split("__");
		String v = toks[toks.length - 1];
		int visitID = GenUtils.toInt(v, -1);
		Assertion.assertTrue(visitID != -1);
		SecurityService ss = (SecurityService) ServiceFactory
				.getSecurityService();
		String primarySiteID = ss.getSiteID(theDBID);

		// TODO assumption getting originating siteID from the subjectID
		String originatingSiteID = subjectID.substring(0, 4);

		Expsegment segment = rds.getSegmentInfo(ui, originatingSiteID, expName,
				subjectID, visitID, segmentName);
		Assertion.assertNotNull(segment);
		List<Experiment> expList = dbCache.getExperiments(ui, false);
		Experiment theExp = null;
		for (Experiment exp : expList) {
			if (exp.getName().equals(expName)) {
				theExp = exp;
				break;
			}
		}
		Assertion.assertNotNull(theExp);

		deleteMatchingExecutedTransforms(con, ui, subjectID, theExp
				.getUniqueid().intValue(), visitID, segment.getSegmentid()
				.intValue());

		if (!primarySiteID.equals(originatingSiteID)) {
			Integer segmentID = null;
			if (segment.getSegmentid() != null) {
				segmentID = segment.getSegmentid().intValue();
			}
			deleteExternalDataRec(con, theExp.getName(), subjectID, visitID,
					segmentID);
		}

	}

	public void deleteProvenanceForAnalysisInstance(UserInfo ui, AnalysisT at)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			deleteProvenanceForAnalysisInstance(con, ui, at);

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "deleteProvenanceForAnalysisInstance",
					x, true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	public List<Executedtransform> addProvenanceForAnalysisInstance(
			Connection con, UserInfo ui, AnalysisT at,
			String analysisShortName, String snapshotID) throws Exception {
		IRemoteDBServices rds = ServiceFactory.getRemoteDBServices();

		String expName = at.getProjectID();
		String subjectID = at.getSubjectID();
		String segmentName = at.getEpisodeID();
		String visitIDLabel = at.getVisitID();
		String[] toks = visitIDLabel.split("__");
		String v = toks[toks.length - 1];
		int visitID = GenUtils.toInt(v, -1);
		Assertion.assertTrue(visitID != -1);
		SecurityService ss = (SecurityService) ServiceFactory
				.getSecurityService();
		String primarySiteID = ss.getSiteID(theDBID);

		// TODO assumption getting originating siteID from the subjectID
		String originatingSiteID = subjectID.substring(0, 4);

		Expsegment segment = rds.getSegmentInfo(ui, originatingSiteID, expName,
				subjectID, visitID, segmentName);
		Assertion.assertNotNull(segment);
		List<Experiment> expList = dbCache.getExperiments(ui, false);
		Experiment theExp = null;
		for (Experiment exp : expList) {
			if (exp.getName().equals(expName)) {
				theExp = exp;
				break;
			}
		}
		Assertion.assertNotNull(theExp);
		String analysisTypeName = at.getType();
		// TODO hard-coded analysis version
		String version = "0001_0001";
		AnalysisInfo ai = new AnalysisInfo(analysisTypeName, version);
		ProvenanceT pt = at.getProvenance().get(0);

		List<ExecutedTransformInfo> etiList = new ArrayList<ExecutedTransformInfo>(
				pt.getProcessStep().size());

		List<ExecutedTransformInfo> flowList = new ArrayList<ExecutedTransformInfo>(
				pt.getProcessStep().size());

		int acID = 1;
		for (ProcessStepT pst : pt.getProcessStep()) {
			ExecutedTransformInfo eti = prepETI(pst, ai);
			AnalysisCompInfo aci = createACI(acID, 1, ai, eti.getTransform(),
					null);
			ai.addComponent(aci);
			eti.setComponent(aci);
			eti.setNodeLevel(aci.getNodeLevel());
			etiList.add(eti);
			if (acID == 1) {

				addDerivedDataInfos(eti, at, subjectID, theExp, visitID,
						segment.getSegmentid().intValue(), analysisShortName,
						snapshotID);
			}
			flowList.add(eti);
			acID++;
		}

		List<Executedtransform> etList = addExecutedTransforms(con, ui,
				etiList, flowList);

		if (!primarySiteID.equals(originatingSiteID)) {
			// TODO assumption the first step contains input/output URI
			// information
			ExecutedTransformInfo eti = etiList.get(0);
			addExternalData(con, ui, originatingSiteID, eti, theExp);
		}
		return etList;
	}

	public List<Executedtransform> addProvenanceForAnalysisInstance(
			UserInfo ui, AnalysisT at, String analysisShortName,
			String snapshotID) throws Exception {

		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			List<Executedtransform> etList = addProvenanceForAnalysisInstance(
					con, ui, at, analysisShortName, snapshotID);
			con.commit();
			return etList;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addProvenanceForAnalysisInstance", x,
					true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected void addExternalData(Connection con, UserInfo ui, String siteID,
			ExecutedTransformInfo eti, Experiment exp) throws Exception {
		List<Site> sites = dbCache.getSites(ui, false);
		Map<String, Site> siteMap = new HashMap<String, Site>(17);
		for (Site site : sites) {
			siteMap.put(site.getSiteid(), site);
		}
		Map<String, String> subjectID2SiteIDMap = new HashMap<String, String>(7);
		for (DerivedDataInfo ddi : eti.getInputDDIList()) {
			ExternalDataInfo edi = new ExternalDataInfo();
			edi.setDataURI(ddi.getDataURI());
			edi.setExperimentID(ddi.getExperimentID());
			edi.setExpName(exp.getName());
			edi.setSubjectID(ddi.getSubjectID());
			edi.setVisitID(ddi.getVisitID());
			edi.setSegmentID(ddi.getSegmentID());
			edi.setSiteIDStr(siteID);
			edi.setRaw(ddi.isRaw());
			subjectID2SiteIDMap.put(ddi.getSubjectID(), siteID);

			addExternalDataRec(con, ui, edi, siteMap);
		}

		for (DerivedDataInfo ddi : eti.getOutputDDIList()) {
			ExternalDataInfo edi = new ExternalDataInfo();
			edi.setDataURI(ddi.getDataURI());
			edi.setExperimentID(ddi.getExperimentID());
			edi.setExpName(exp.getName());
			edi.setSubjectID(ddi.getSubjectID());
			edi.setVisitID(ddi.getVisitID());
			edi.setSegmentID(ddi.getSegmentID());
			edi.setSiteIDStr(siteID);
			edi.setRaw(ddi.isRaw());
			subjectID2SiteIDMap.put(ddi.getSubjectID(), siteID);

			addExternalDataRec(con, ui, edi, siteMap);
		}
		// add remote human subject records if not present

		HumansubjectDAO hsDAO = DAOFactory.createHumansubjectDAO(theDBID);
		Humansubject cr = new Humansubject();
		cr.setIsremote(true);
		for (Map.Entry<String, String> entry : subjectID2SiteIDMap.entrySet()) {
			String subjectID = entry.getKey();
			String siteIDStr = entry.getValue();
			Site site = siteMap.get(siteIDStr);
			cr.setSubjectid(subjectID);
			List<Humansubject> hsList = hsDAO.find(con, cr);
			if (hsList.isEmpty()) {
				addRemoteHumanSubject(con, ui, site, subjectID);
			}
		}
	}

	protected ExecutedTransformInfo prepETI(ProcessStepT pst, AnalysisInfo ai) {
		MachineInfo mi = new MachineInfo();
		mi.setArchitecture(pst.getArchitecture());
		mi.setAddress(pst.getHostName());
		mi.setOpSystem(pst.getPlatform().getValue());
		if (pst.getPlatform().getVersion() != null) {
			mi.setOsVersion(pst.getPlatform().getVersion());
		} else {
			mi.setOsVersion("UNKNOWN");
		}
		ExecutedTransformInfo eti = new ExecutedTransformInfo();

		Assertion.assertNotNull(pst.getTimeStamp());
		Date ts = XMLUtils.toDate(pst.getTimeStamp());
		eti.setTimestamp(ts);

		String packageName = "NO-PACKAGE";
		String packageVersion = "NO-PACKAGE-VERSION";
		if (pst.getPackage() != null) {
			packageName = pst.getPackage().getValue();
			if (pst.getPackage().getVersion() != null)
				packageVersion = pst.getPackage().getVersion();
		}
		String program = pst.getProgram().getValue();
		String programVersion = pst.getProgram().getVersion();
		if (programVersion == null) {
			programVersion = pst.getCvs();
		}
		if (programVersion == null) {
			programVersion = "NO-PROGRAM-VERSION";
		}
		TransformationInfo ti = new TransformationInfo(packageName,
				packageVersion, program, programVersion);

		eti.setTransform(ti);
		eti.setMachine(mi);
		eti.setAnalysis(ai);
		if (pst.getProgramArguments() != null) {
			eti.setArgUsed(pst.getProgramArguments().getValue());
		}
		return eti;
	}

	protected AnalysisCompInfo createACI(int id, int nodeLevel,
			AnalysisInfo ai, TransformationInfo ti, AnalysisCompInfo parent) {
		AnalysisCompInfo aci = new AnalysisCompInfo(id);
		aci.setTransform(ti);
		aci.setAnalysis(ai);
		aci.setNodeLevel(nodeLevel);
		aci.setParent(parent);
		return aci;
	}

	protected void addDerivedDataInfos(ExecutedTransformInfo eti, AnalysisT at,
			String subjectID, Experiment exp, int visitID, int segmentID,
			String analysisShortName, String snapshotID) throws BIRNURIException {
		Assertion.assertNotNull(at.getInput());
		String workURI = null;
		for (LevelDataRefsT ldr : at.getInput()) {
			Assertion.assertTrue(subjectID.equals(ldr.getSubjectID()));
			String inDDIURI = prepDataURI(exp, ldr);
			if (workURI == null)
				workURI = inDDIURI;

			DerivedDataInfo inDDI = new DerivedDataInfo();
			inDDI.setDataURI(inDDIURI);
			inDDI.setExperimentID(exp.getUniqueid().intValue());
			inDDI.setSubjectID(ldr.getSubjectID());
			inDDI.setVisitID(visitID);
			inDDI.setSegmentID(segmentID);
			// TODO assumption input is raw data
			inDDI.setRaw(true);

			eti.addInputDerivedDataInfo(inDDI);
		}
		if (at.getOutput() != null) {
			for (LevelDataRefsT ldr : at.getOutput()) {
				StringBuffer buf = new StringBuffer(256);
				buf.append(workURI).append("/Analysis/");
				// TODO which siteID?
				String siteID = subjectID.substring(0, 4);
				buf.append(analysisShortName).append("__");
				buf.append(siteID).append("__").append(snapshotID);
				buf.append('/').append(ldr.getAnalysisURI());
				String outDDIURI = buf.toString().trim();
				outDDIURI = outDDIURI.replaceAll("/+", "/");
				DerivedDataInfo outDDI = new DerivedDataInfo();
				outDDI.setDataURI(outDDIURI);
				outDDI.setExperimentID(exp.getUniqueid().intValue());
				outDDI.setSubjectID(subjectID);
				outDDI.setVisitID(visitID);
				outDDI.setSegmentID(segmentID);
				outDDI.setRaw(false);

				eti.addOutputDerivedDataInfo(outDDI);
			}
		}
	}

	protected String prepDataURI(Experiment exp, LevelDataRefsT ldr) throws BIRNURIException {
		StringBuffer buf = new StringBuffer(256);
		String baseURI = exp.getBaseuri();
		if (Constants.USE_BIRN_URIS) {
			baseURI = BIRNURIUtils.getPath(baseURI);
		}
		
		buf.append(baseURI).append('/');
		buf.append(ldr.getSubjectID()).append('/');
		String siteID = ldr.getSubjectID().substring(0, 4);
		String visitIDStr = ldr.getVisitID();
		String[] toks = visitIDStr.split("__");
		// TODO which siteID?
		buf.append(toks[0]).append("__").append(siteID);
		buf.append("__").append(toks[1]).append('/');
		buf.append(ldr.getStudyID()).append('/');
		buf.append(ldr.getEpisodeID());
		String inDDIURI = buf.toString();
		inDDIURI = inDDIURI.replaceAll("/+", "/");
		return inDDIURI;
	}

	public List<Executedtransform> addExecutedTransforms(Connection con,
			UserInfo ui, List<ExecutedTransformInfo> etiList,
			List<ExecutedTransformInfo> flowList) throws Exception {
		AnalysisInfo ai = findAnalysisInfo(etiList);
		Analysis analysis = findAnalyis(con, ai);
		Assertion.assertNotNull(analysis);
		Set<TransformationInfo> tiSet = new HashSet<TransformationInfo>(11);
		Set<MachineInfo> miSet = new HashSet<MachineInfo>(7);
		for (ExecutedTransformInfo eti : etiList) {
			tiSet.add(eti.getTransform());
			miSet.add(eti.getMachine());
		}
		Map<String, Transformation> trMap = prepTransformationMap(con, tiSet);
		Map<String, Machine> machineMap = prepMachineMap(con, miSet);

		List<ExecutedTransformInfo> forrestRootList = new ArrayList<ExecutedTransformInfo>(
				1);

		List<ExecutedTransformInfo> remList = new ArrayList<ExecutedTransformInfo>(
				10);
		for (ExecutedTransformInfo eti : etiList) {
			if (eti.getParent() == null) {
				forrestRootList.add(eti);
			} else {
				remList.add(eti);
			}
		}
		List<Executedtransform> etList = new ArrayList<Executedtransform>(
				etiList.size());
		Map<Executedtransform, ExecutedTransformInfo> etiMap = new HashMap<Executedtransform, ExecutedTransformInfo>();
		Map<ExecutedTransformInfo, Executedtransform> parentMap = new HashMap<ExecutedTransformInfo, Executedtransform>(
				7);
		for (ExecutedTransformInfo eti : forrestRootList) {
			List<List<ExecutedTransformInfo>> levelList = arrangeBreadthFirst(
					eti, remList);
			for (List<ExecutedTransformInfo> levelNodes : levelList) {
				for (ExecutedTransformInfo levelNode : levelNodes) {
					Executedtransform parent = parentMap.get(levelNode
							.getParent());
					String key = eti.getTransform().getTransform() + "_"
							+ eti.getTransform().getTransformVersion();
					Transformation tr = trMap.get(key);

					key = prepMachineKey(eti.getMachine());
					Machine m = machineMap.get(key);
					if (m == null) {
						m = addMachineRec(con, ui, eti.getMachine());
						machineMap.put(key, m);
					}

					Assertion.assertNotNull(m);
					Executedtransform et = addExecutedTransform(con, ui,
							levelNode, analysis, tr, m, parent);
					parentMap.put(levelNode, et);
					etList.add(et);
					etiMap.put(et, levelNode);
				}
			}
		}

		// input and output files for each transformation
		List<Integer> etIDList = new ArrayList<Integer>(etList.size());
		for (Executedtransform et : etList) {
			etIDList.add(et.getUniqueid().intValue());
		}
		Map<BigDecimal, Map<Transforminputdata, Deriveddata>> existingTIDDMap = getInputDerivedData(
				con, ui, etIDList);

		Map<BigDecimal, Map<Transformeddata, Deriveddata>> existingTODDMap = getOutputDerivedData(
				con, ui, etIDList);

		for (Executedtransform et : etList) {
			ExecutedTransformInfo eti = etiMap.get(et);
			Map<Transforminputdata, Deriveddata> tiddMap = existingTIDDMap
					.get(et.getUniqueid());
			Map<Transformeddata, Deriveddata> toddMap = existingTODDMap.get(et
					.getUniqueid());
			List<Deriveddata> inDDList = new ArrayList<Deriveddata>(eti
					.getInputDDIList().size());
			List<Deriveddata> outDDList = new ArrayList<Deriveddata>(eti
					.getOutputDDIList().size());
			if (tiddMap != null) {
				for (Deriveddata dd : tiddMap.values()) {
					inDDList.add(dd);
				}
			}
			if (toddMap != null) {
				for (Deriveddata dd : toddMap.values()) {
					outDDList.add(dd);
				}
			}
			addInputAndOutputData4ExecutedTransform(con, ui, eti, et, inDDList,
					outDDList);
		}

		if (flowList != null) {
			Executedtransform theStartET = etList.get(0);
			Executedtransform priorET = null;
			for (ExecutedTransformInfo eti : flowList) {
				Executedtransform et = parentMap.get(eti);
				Assertion.assertNotNull(et);
				addExecutedTransformFlow(con, et, priorET, theStartET
						.getUniqueid().intValue());
				priorET = et;
			}
		}

		return etList;
	}

	public List<Executedtransform> addExecutedTransforms(UserInfo ui,
			List<ExecutedTransformInfo> etiList,
			List<ExecutedTransformInfo> flowList) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			List<Executedtransform> etList = addExecutedTransforms(con, ui,
					etiList, flowList);
			con.commit();
			return etList;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addExecutedTransforms", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public Map<BigDecimal, Map<Transformeddata, Deriveddata>> getOutputDerivedData(
			UserInfo ui, List<Integer> executedTransformIDList)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			Map<BigDecimal, Map<Transformeddata, Deriveddata>> map = getOutputDerivedData(
					con, ui, executedTransformIDList);

			con.commit();
			return map;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "getOutputDerivedData", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected Map<BigDecimal, Map<Transformeddata, Deriveddata>> getOutputDerivedData(
			Connection con, UserInfo ui, List<Integer> executedTransformIDList)
			throws Exception {
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		String inPart = DBUtils.toInList(executedTransformIDList);
		StringBuffer qb = new StringBuffer(256);
		qb.append("select t.* , d.* from Transformeddata as t , ");
		qb.append("Deriveddata as d where ");
		qb.append(" t.ncResearchdataUniqueid = d.uniqueid and ");
		qb.append("t.ncExecutedtransformUniqueid in ");
		qb.append(inPart);

		List<?> results = tsp.executeQuery(con, qb.toString());
		Map<BigDecimal, Map<Transformeddata, Deriveddata>> map = new HashMap<BigDecimal, Map<Transformeddata, Deriveddata>>(
				17);

		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Object[] row = (Object[]) iter.next();
			Transformeddata td = (Transformeddata) row[0];
			Deriveddata dd = (Deriveddata) row[1];

			Map<Transformeddata, Deriveddata> tdMap = map.get(td
					.getNcExecutedtransformUniqueid());
			if (tdMap == null) {
				tdMap = new HashMap<Transformeddata, Deriveddata>(17);
				map.put(td.getNcExecutedtransformUniqueid(), tdMap);
			}
			tdMap.put(td, dd);
		}
		return map;
	}

	public Map<BigDecimal, Map<Transforminputdata, Deriveddata>> getInputDerivedData(
			UserInfo ui, List<Integer> executedTransformIDList)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Map<BigDecimal, Map<Transforminputdata, Deriveddata>> map = getInputDerivedData(
					con, ui, executedTransformIDList);
			con.commit();

			return map;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "getInputDerivedData", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected Map<BigDecimal, Map<Transforminputdata, Deriveddata>> getInputDerivedData(
			Connection con, UserInfo ui, List<Integer> executedTransformIDList)
			throws Exception {
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		String inPart = DBUtils.toInList(executedTransformIDList);
		StringBuffer qb = new StringBuffer(256);
		qb.append("select t.*, d.* from Transforminputdata as t,");
		qb.append(" Deriveddata as d where ");
		qb.append("t.ncResearchdataUniqueid = d.uniqueid and ");
		qb.append("t.ncExecutedtransformUniqueid in ");
		qb.append(inPart);

		List<?> results = tsp.executeQuery(con, qb.toString());
		Map<BigDecimal, Map<Transforminputdata, Deriveddata>> map = new HashMap<BigDecimal, Map<Transforminputdata, Deriveddata>>(
				17);

		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Object[] row = (Object[]) iter.next();
			Transforminputdata td = (Transforminputdata) row[0];
			Deriveddata dd = (Deriveddata) row[1];

			Map<Transforminputdata, Deriveddata> tdMap = map.get(td
					.getNcExecutedtransformUniqueid());
			if (tdMap == null) {
				tdMap = new HashMap<Transforminputdata, Deriveddata>(17);
				map.put(td.getNcExecutedtransformUniqueid(), tdMap);
			}
			tdMap.put(td, dd);
		}
		return map;
	}

	public List<Transformation> addTransformations(UserInfo ui,
			List<TransformationInfo> tiList) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			List<Transformation> tList = new ArrayList<Transformation>(tiList
					.size());
			for (TransformationInfo ti : tiList) {
				Transformation t = addTransformation(con, ui, ti);
				tList.add(t);
			}
			con.commit();
			return tList;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addTransformations", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public AnalysisInstanceDataResult getMatchingExecutedTransforms(
			UserInfo ui, String subjectID, Integer expID, Integer visitID,
			Integer segmentID) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			Map<Executedtransform, Map<Transformeddata, Deriveddata>> outMap = getExecutedTransformsAndOutputDerivedData(
					con, ui, subjectID, expID, visitID, segmentID);
			Map<Executedtransform, Map<Transforminputdata, Deriveddata>> inMap = getExecutedTransformsAndInputDerivedData(
					con, ui, subjectID, expID, visitID, segmentID);

			AnalysisInstanceDataResult aids = new AnalysisInstanceDataResult(
					subjectID, expID, visitID, segmentID);

			aids.addExecutedTransformsAndInputDerivedData(inMap);
			aids.addExecutedTransformsAndOutputDerivedData(outMap);
			con.commit();
			return aids;

		} catch (Exception x) {
			handleErrorAndRollBack(con, "getMatchingExecutedTransforms", x,
					true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void deleteMatchingExecutedTransforms(Connection con, UserInfo ui,
			String subjectID, Integer expID, Integer visitID, Integer segmentID)
			throws Exception {
		Map<Executedtransform, Map<Transformeddata, Deriveddata>> outMap = getExecutedTransformsAndOutputDerivedData(
				con, ui, subjectID, expID, visitID, segmentID);
		Map<Executedtransform, Map<Transforminputdata, Deriveddata>> inMap = getExecutedTransformsAndInputDerivedData(
				con, ui, subjectID, expID, visitID, segmentID);

		AnalysisInstanceDataResult aids = new AnalysisInstanceDataResult(
				subjectID, expID, visitID, segmentID);

		aids.addExecutedTransformsAndInputDerivedData(inMap);
		aids.addExecutedTransformsAndOutputDerivedData(outMap);

		TransformeddataDAO tdDAO = DAOFactory.createTransformeddataDAO(theDBID);
		TransforminputdataDAO tidDAO = DAOFactory
				.createTransforminputData(theDBID);
		DeriveddataDAO ddDAO = DAOFactory.createDeriveddataDAO(theDBID);
		ExecutedtransformDAO etDAO = DAOFactory
				.createExecutedtransformDAO(theDBID);
		ExecutedtransformflowDAO etfDAO = DAOFactory
				.createExecutedtransformflowDAO(theDBID);

		Transforminputdata tid = new Transforminputdata();
		Transformeddata td = new Transformeddata();
		for (ExTransformData etd : aids.getEtdList()) {
			if (!etd.getInDataMap().isEmpty()) {
				tid.setNcExecutedtransformUniqueid(etd.getEt().getUniqueid());
				tidDAO.delete(con, tid);
				for (Deriveddata dd : etd.getInDataMap().values()) {
					Deriveddata cr = new Deriveddata();
					cr.setUniqueid(dd.getUniqueid());
					ddDAO.delete(con, cr);
				}
			}
			if (!etd.getOutDataMap().isEmpty()) {
				td.setNcExecutedtransformUniqueid(etd.getEt().getUniqueid());
				tdDAO.delete(con, td);
				for (Deriveddata dd : etd.getOutDataMap().values()) {
					Deriveddata cr = new Deriveddata();
					cr.setUniqueid(dd.getUniqueid());
					ddDAO.delete(con, cr);
				}
			}

			List<Executedtransformflow> etfList = getExecutedTransformFlow(con,
					etd);
			Executedtransformflow etfCR = new Executedtransformflow();
			Executedtransform etCR = new Executedtransform();
			for (Executedtransformflow etf : etfList) {
				etfCR.setExecutedtransformationid(etf
						.getExecutedtransformationid());
				etfDAO.delete(con, etfCR);
				etCR.setUniqueid(etf.getExecutedtransformationid());
				etDAO.delete(con, etCR);
			}
		}
	}

	public void deleteMatchingExecutedTransforms(UserInfo ui, String subjectID,
			Integer expID, Integer visitID, Integer segmentID) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);
			deleteMatchingExecutedTransforms(con, ui, subjectID, expID,
					visitID, segmentID);

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "deleteMatchingExecutedTransforms", x,
					true);
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected List<Executedtransformflow> getExecutedTransformFlow(
			Connection con, ExTransformData firstETD) throws Exception {
		// TODO assumption: the first process in a pipeline has attached input
		// and output which is used indirectly to find an executed
		// transformation
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		StringBuffer qb = new StringBuffer(256);
		qb.append("select f.* from Executedtransformflow as f,");
		qb.append("Analysisflow as a where ");
		qb.append("f.componentid = a.componentid and ");
		qb.append("f.analysisid = a.analysisid and ");
		qb.append("a.analysisid = ").append(firstETD.getEt().getAnalysisid());
		qb.append(" and f.startexecutedtransformid = ").append(
				firstETD.getEt().getUniqueid());
		List<?> results = tsp.executeQuery(con, qb.toString());
		List<Executedtransformflow> etfList = new ArrayList<Executedtransformflow>(
				results.size());

		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Executedtransformflow et = (Executedtransformflow) iter.next();
			etfList.add(et);
		}
		return etfList;
	}

	protected List<Executedtransformflow> getExecutedTransformFlow2(
			Connection con, ExTransformData firstETD) throws Exception {
		// TODO assumption: the first process in a pipeline has attached input
		// and output
		// which is used indirectly to find an executed transformation

		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		StringBuffer qb = new StringBuffer(256);
		qb.append("select f.* from Executedtransformflow as f,");
		qb.append("Analysisflow as a where ");
		qb.append("f.componentid = a.componentid and ");
		qb.append("f.analysisid = a.analysisid and ");
		qb.append("a.analysisid = ").append(firstETD.getEt().getAnalysisid());
		// TODO needs analysisResult ID to differentiate between different
		// instances of the executed analysis
		List<?> results = tsp.executeQuery(con, qb.toString());
		Map<BigDecimal, Executedtransformflow> etMap = new HashMap<BigDecimal, Executedtransformflow>(
				17);
		BigDecimal etID = firstETD.getEt().getUniqueid();
		Map<BigDecimal, Executedtransformflow> priorMap = new HashMap<BigDecimal, Executedtransformflow>();
		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Executedtransformflow et = (Executedtransformflow) iter.next();
			if (et.getExecutedtransformationid().equals(etID)) {
				etMap.put(et.getExecutedtransformationid(), et);
			}
			if (et.getPriorexecutedtransformid() != null) {
				priorMap.put(et.getPriorexecutedtransformid(), et);
			}
		}
		List<Executedtransformflow> etfList = new ArrayList<Executedtransformflow>(
				10);
		for (Executedtransformflow parentEtf : etMap.values()) {
			etfList.add(parentEtf);
			Executedtransformflow p = parentEtf;
			while (true) {
				Executedtransformflow next = priorMap.get(p
						.getExecutedtransformationid());
				if (next == null) {
					break;
				}
				etfList.add(next);
				p = next;
			}
		}

		return etfList;
	}

	protected Map<Executedtransform, Map<Transformeddata, Deriveddata>> getExecutedTransformsAndOutputDerivedData(
			Connection con, UserInfo ui, String subjectID, Integer expID,
			Integer visitID, Integer segmentID) throws Exception {
		Assertion.assertNotNull(subjectID);
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		StringBuffer qb = new StringBuffer(256);
		qb.append("select e.*, t.*, d.* from Executedtransform as e,");
		qb.append("Transformeddata as t, Deriveddata as d where ");
		qb.append("t.ncResearchdataUniqueid = d.uniqueid and ");
		qb.append("t.ncExecutedtransformUniqueid = e.uniqueid and ");
		qb.append("d.isbad = false and ");
		qb.append("d.subjectid = '").append(subjectID).append("'");
		if (expID != null) {
			qb.append(" and d.ncExperimentUniqueid = ").append(expID);
		}
		if (visitID != null) {
			qb.append(" and d.componentid = ").append(visitID);
		}
		if (segmentID != null) {
			qb.append(" and d.segmentid = ").append(segmentID);
		}

		List<?> results = tsp.executeQuery(con, qb.toString());
		Map<Executedtransform, Map<Transformeddata, Deriveddata>> map = new HashMap<Executedtransform, Map<Transformeddata, Deriveddata>>(
				17);
		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Object[] row = (Object[]) iter.next();

			Executedtransform et = (Executedtransform) row[0];
			Transformeddata td = (Transformeddata) row[1];
			Deriveddata dd = (Deriveddata) row[2];
			Map<Transformeddata, Deriveddata> tdMap = map.get(et);
			if (tdMap == null) {
				tdMap = new HashMap<Transformeddata, Deriveddata>(17);
				map.put(et, tdMap);
			}
			tdMap.put(td, dd);
		}
		return map;
	}

	protected Map<Executedtransform, Map<Transforminputdata, Deriveddata>> getExecutedTransformsAndInputDerivedData(
			Connection con, UserInfo ui, String subjectID, Integer expID,
			Integer visitID, Integer segmentID) throws Exception {
		Assertion.assertNotNull(subjectID);
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		StringBuffer qb = new StringBuffer(256);
		qb.append("select e.*, t.*, d.* from Executedtransform as e,");
		qb.append("Transforminputdata as t, Deriveddata as d where ");
		qb.append("t.ncResearchdataUniqueid = d.uniqueid and ");
		qb.append("t.ncExecutedtransformUniqueid = e.uniqueid and ");
		qb.append("d.isbad = false and ");
		qb.append("d.subjectid = '").append(subjectID).append("'");
		if (expID != null) {
			qb.append(" and d.ncExperimentUniqueid = ").append(expID);
		}
		if (visitID != null) {
			qb.append(" and d.componentid = ").append(visitID);
		}
		if (segmentID != null) {
			qb.append(" and d.segmentid = ").append(segmentID);
		}

		List<?> results = tsp.executeQuery(con, qb.toString());
		Map<Executedtransform, Map<Transforminputdata, Deriveddata>> map = new HashMap<Executedtransform, Map<Transforminputdata, Deriveddata>>(
				17);
		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Object[] row = (Object[]) iter.next();

			Executedtransform et = (Executedtransform) row[0];
			Transforminputdata td = (Transforminputdata) row[1];
			Deriveddata dd = (Deriveddata) row[2];
			Map<Transforminputdata, Deriveddata> tdMap = map.get(et);
			if (tdMap == null) {
				tdMap = new HashMap<Transforminputdata, Deriveddata>(17);
				map.put(et, tdMap);
			}
			tdMap.put(td, dd);
		}
		return map;
	}

	protected List<List<ExecutedTransformInfo>> arrangeBreadthFirst(
			ExecutedTransformInfo root, List<ExecutedTransformInfo> etiList) {
		List<List<ExecutedTransformInfo>> levelList = new ArrayList<List<ExecutedTransformInfo>>(
				10);
		Set<ExecutedTransformInfo> etiSet = new HashSet<ExecutedTransformInfo>();
		List<ExecutedTransformInfo> levelNodes = new ArrayList<ExecutedTransformInfo>(
				1);
		levelNodes.add(root);
		levelList.add(levelNodes);
		for (ExecutedTransformInfo eti : etiList) {
			etiSet.add(eti);
		}
		List<ExecutedTransformInfo> parentNodes = levelNodes;
		while (!etiSet.isEmpty()) {
			levelNodes = new ArrayList<ExecutedTransformInfo>(2);
			for (ExecutedTransformInfo parent : parentNodes) {
				List<ExecutedTransformInfo> children = getImmediateChildren(
						parent, etiSet);
				levelNodes.addAll(children);
			}
			if (levelNodes.isEmpty())
				break;
			levelList.add(levelNodes);
			parentNodes = levelNodes;
		}

		return levelList;
	}

	protected List<ExecutedTransformInfo> getImmediateChildren(
			ExecutedTransformInfo root, Set<ExecutedTransformInfo> etiSet) {
		List<ExecutedTransformInfo> children = new ArrayList<ExecutedTransformInfo>(
				2);
		for (Iterator<ExecutedTransformInfo> it = etiSet.iterator(); it
				.hasNext();) {
			ExecutedTransformInfo eti = it.next();
			if (eti.getParent() == root) {
				children.add(eti);
				it.remove();
			}
		}
		return children;
	}

	protected Executedtransform addExecutedTransform(Connection con,
			UserInfo ui, ExecutedTransformInfo eti, Analysis analysis,
			Transformation tr, Machine machine, Executedtransform parent)
			throws Exception {

		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		ExecutedtransformDAO dao = DAOFactory
				.createExecutedtransformDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);
		Executedtransform et = new Executedtransform();
		et.setAnalysisid(analysis.getUniqueid());
		et.setAnalysiscomponentid(new BigDecimal(eti.getComponent().getId()));
		et.setLogicaltransformid(tr.getUniqueid());
		et.setMachineid(machine.getUniqueid());
		et.setNodelevel(new BigDecimal(eti.getNodeLevel()));
		if (parent != null) {
			et.setParentexecutedtransformationid(parent.getUniqueid());
		}
		et.setTimeStamp(new java.sql.Timestamp(eti.getTimestamp().getTime()));
		et.setArgumentsused(eti.getArgUsed());

		et.setNcDatabaseuserUniqueid(dbUser.getUniqueid());

		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.EXECUTED_TRANSFORM, "uniqueid");
		et.setUniqueid(uniqueid);
		et.setOwner(dbUser.getUniqueid());
		et.setModuser(dbUser.getUniqueid());
		et.setModtime(new Date());
		et.setTableid(getTableID(ui, Constants.EXECUTED_TRANSFORM));

		dao.insert(con, et);
		return et;
	}

	public List<Analysiscomponent> findComponents(UserInfo ui, AnalysisInfo ai)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			AnalysisDAO anDAO = DAOFactory.createAnalysisDAO(theDBID);
			Analysis acr = new Analysis();
			acr.setName(ai.getName());
			acr.setVersion(ai.getVersion());

			List<Analysis> anList = anDAO.find(con, acr);
			if (anList.isEmpty()) {
				return new ArrayList<Analysiscomponent>(0);
			}
			Assertion.assertTrue(anList.size() == 1);
			Analysis analysis = anList.get(0);
			AnalysiscomponentDAO dao = DAOFactory
					.createAnalysiscomponentDAO(theDBID);
			Analysiscomponent cr = new Analysiscomponent();
			cr.setAnalysisid(analysis.getUniqueid());
			List<Analysiscomponent> acList = dao.find(con, cr);

			return acList;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "findComponents", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected AnalysisInfo findAnalysisInfo(List<ExecutedTransformInfo> etiList)
			throws Exception {
		AnalysisInfo ai = null;
		for (ExecutedTransformInfo eti : etiList) {
			if (ai == null) {
				ai = eti.getAnalysis();
			} else if (ai != eti.getAnalysis()) {
				throw new Exception(
						"All executed transformations need to belong to the same analysis!");
			}
		}
		return ai;
	}

	protected Analysis findAnalyis(Connection con, AnalysisInfo ai)
			throws Exception {
		AnalysisDAO dao = DAOFactory.createAnalysisDAO(theDBID);
		Analysis cr = new Analysis();
		cr.setName(ai.getName());
		if (ai.getName() != null) {
			cr.setVersion(ai.getVersion());
		}
		List<Analysis> list = dao.find(con, cr);
		if (list.isEmpty())
			return null;
		return list.get(0);
	}

	protected Map<String, Transformation> prepTransformationMap(Connection con,
			Set<TransformationInfo> tiSet) throws Exception {
		Set<String> tiKeySet = new HashSet<String>(17);
		for (TransformationInfo ti : tiSet) {
			String key = ti.getTransform() + "_" + ti.getTransformVersion();
			tiKeySet.add(key);
		}

		TransformationDAO dao = DAOFactory.createTransformationDAO(theDBID);
		List<Transformation> trList = dao.find(con, new Transformation());
		Map<String, Transformation> trMap = new HashMap<String, Transformation>(
				11);
		for (Transformation tr : trList) {
			String key = tr.getTransform() + "_" + tr.getTransformversion();
			if (tiKeySet.contains(key)) {
				trMap.put(key, tr);
			}
		}
		return trMap;
	}

	protected Map<String, Machine> prepMachineMap(Connection con,
			Set<MachineInfo> miSet) throws Exception {
		Set<String> miKeySet = new HashSet<String>(11);
		for (MachineInfo mi : miSet) {
			String key = prepMachineKey(mi);
			miKeySet.add(key);
		}
		MachineDAO dao = DAOFactory.createMachineDAO(theDBID);
		List<Machine> mList = dao.find(con, new Machine());
		Map<String, Machine> machineMap = new HashMap<String, Machine>(11);
		for (Machine m : mList) {
			String key = prepMachineKey(m);
			if (miKeySet.contains(key)) {
				machineMap.put(key, m);
			}
		}
		return machineMap;
	}

	protected String prepMachineKey(MachineInfo mi) {
		StringBuffer buf = new StringBuffer();
		buf.append(mi.getAddress()).append('_');
		buf.append(mi.getOpSystem()).append('_');
		buf.append(mi.getOsVersion()).append('_');
		buf.append(mi.getArchitecture());
		String key = buf.toString();
		return key;
	}

	protected String prepMachineKey(Machine m) {
		StringBuffer buf = new StringBuffer();
		buf.append(m.getAddress()).append('_');
		buf.append(m.getOs()).append('_');
		buf.append(m.getOsversion()).append('_');
		buf.append(m.getArchitecture());
		String key = buf.toString();
		return key;
	}

	protected void addInputAndOutputData4ExecutedTransform(Connection con,
			UserInfo ui, ExecutedTransformInfo eti, Executedtransform et,
			List<Deriveddata> existingInputDDList,
			List<Deriveddata> existingOutputDDList) throws Exception {
		IDerivedImageDataService dds = ServiceFactory
				.getDerivedImageDataService(theDBID);

		Map<String, Deriveddata> eiDDMap = new HashMap<String, Deriveddata>();
		for (Deriveddata dd : existingInputDDList) {
			eiDDMap.put(dd.getDatauri(), dd);
		}
		List<DerivedDataInfo> inDDIList = new ArrayList<DerivedDataInfo>(eti
				.getInputDDIList().size());
		for (DerivedDataInfo ddi : eti.getInputDDIList()) {
			if (!eiDDMap.containsKey(ddi.getDataURI())) {
				inDDIList.add(ddi);
			}
		}
		if (!inDDIList.isEmpty()) {
			List<Deriveddata> ddList = dds.addDerivedData(con, ui, inDDIList);
			for (Deriveddata dd : ddList) {
				addTransformInputData(con, ui, dd, et);
			}
		}
		eiDDMap = null;
		// for output
		Map<String, Deriveddata> eoDDMap = new HashMap<String, Deriveddata>();
		for (Deriveddata dd : existingOutputDDList) {
			eoDDMap.put(dd.getDatauri(), dd);
		}
		List<DerivedDataInfo> outDDIList = new ArrayList<DerivedDataInfo>(eti
				.getOutputDDIList().size());
		for (DerivedDataInfo ddi : eti.getOutputDDIList()) {
			if (!eoDDMap.containsKey(ddi.getDataURI())) {
				outDDIList.add(ddi);
			}
		}
		if (!outDDIList.isEmpty()) {
			List<Deriveddata> ddList = dds.addDerivedData(con, ui, outDDIList);
			for (Deriveddata dd : ddList) {
				addTransformedData(con, ui, dd, et);
			}
		}
	}

	protected Transformeddata addTransformedData(Connection con, UserInfo ui,
			Deriveddata dd, Executedtransform et) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		TransformeddataDAO dao = DAOFactory.createTransformeddataDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);

		Transformeddata td = new Transformeddata();
		td.setNcExecutedtransformUniqueid(et.getUniqueid());
		td.setNcResearchdataUniqueid(dd.getUniqueid());

		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.TRANSFORMED_DATA, "uniqueid");
		td.setUniqueid(uniqueid);
		td.setOwner(dbUser.getUniqueid());
		td.setModuser(dbUser.getUniqueid());
		td.setModtime(new Date());
		td.setTableid(getTableID(ui, Constants.TRANSFORMED_DATA));

		dao.insert(con, td);
		return td;
	}

	protected Transforminputdata addTransformInputData(Connection con,
			UserInfo ui, Deriveddata dd, Executedtransform et) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		TransforminputdataDAO dao = DAOFactory
				.createTransforminputData(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);

		Transforminputdata td = new Transforminputdata();
		td.setNcExecutedtransformUniqueid(et.getUniqueid());
		td.setNcResearchdataUniqueid(dd.getUniqueid());

		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.TRANSFORMED_DATA, "uniqueid");
		td.setUniqueid(uniqueid);
		td.setOwner(dbUser.getUniqueid());
		td.setModuser(dbUser.getUniqueid());
		td.setModtime(new Date());
		td.setTableid(getTableID(ui, Constants.TRANSFORMED_DATA));

		dao.insert(con, td);
		return td;
	}

	protected Transformation addTransformation(Connection con, UserInfo ui,
			TransformationInfo ti) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		TransformationDAO dao = DAOFactory.createTransformationDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);

		Transformation t = new Transformation();
		t.setTransform(ti.getTransform());
		t.setTransformversion(ti.getTransformVersion());
		t.setPackage_(ti.getPackage());
		t.setPackageversion(ti.getPackageVersion());
		t.setDescription(ti.getDescription());
		t.setOntologysource(Constants.DEFAULT_ONT_SRC);
		t.setConceptid(Constants.DEFAULT_ONT_CONCEPT);
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.TRANSFORMATION, "uniqueid");
		t.setUniqueid(uniqueid);
		t.setOwner(dbUser.getUniqueid());
		t.setModuser(dbUser.getUniqueid());
		t.setModtime(new Date());
		t.setTableid(getTableID(ui, Constants.TRANSFORMATION));

		dao.insert(con, t);
		return t;
	}

	protected Transformationargument addTransformationArgument(Connection con,
			UserInfo ui, TransformArgInfo argInfo, Transformation tr,
			Map<String, Argumenttype> argTypeMap) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);

		TransformationargumentDAO dao = DAOFactory
				.createTransformationargumentDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);

		Transformationargument ta = new Transformationargument();
		ta.setArgname(argInfo.getArgName());
		ta.setIsinputdata(new Boolean(argInfo.isInputData()));
		ta.setIsoutputdata(new Boolean(argInfo.isInputData()));
		ta.setArgumentdatatype(argInfo.getArgDataType());
		ta.setTransformid(tr.getUniqueid());
		Argumenttype argType = argTypeMap.get(argInfo.getArgType()
				.getArgumentType());
		if (argType == null) {
			// add new argument type record
			argType = addArgumentType(con, ui, argInfo.getArgType());
		}

		ta.setArgumenttype(argType.getArgumenttype());

		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.TRANSFORMATION_ARG, "uniqueid");
		ta.setUniqueid(uniqueid);
		ta.setTableid(getTableID(ui, Constants.TRANSFORMATION_ARG));
		ta.setOwner(dbUser.getUniqueid());
		ta.setModuser(dbUser.getUniqueid());
		ta.setModtime(new Date());

		dao.insert(con, ta);
		return ta;
	}

	protected Argumenttype addArgumentType(Connection con, UserInfo ui,
			ArgumentTypeInfo ati) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		ArgumenttypeDAO dao = DAOFactory.createArgumenttypeDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);
		Argumenttype at = new Argumenttype();
		at.setArgumenttype(ati.getArgumentType());
		at.setDescription(ati.getDescription());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.ARGUMENT_TYPE, "uniqueid");
		at.setUniqueid(uniqueid);
		at.setTableid(getTableID(ui, Constants.ARGUMENT_TYPE));
		at.setOwner(dbUser.getUniqueid());
		at.setModuser(dbUser.getUniqueid());
		at.setModtime(new Date());

		dao.insert(con, at);
		return at;
	}

	protected Analysiscomponent addAnalysisComponent(Connection con,
			UserInfo ui, AnalysisCompInfo aci, Analysis analysis,
			Transformation tr) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		AnalysiscomponentDAO dao = DAOFactory
				.createAnalysiscomponentDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);

		Analysiscomponent ac = new Analysiscomponent();
		ac.setAnalysisid(analysis.getUniqueid());
		ac.setTransformationid(tr.getUniqueid());
		ac.setComponentid(new BigDecimal(aci.getId()));
		ac.setNodelevel(new BigDecimal(aci.getNodeLevel()));
		if (aci.getParent() != null) {
			ac.setParentcomponentid(new BigDecimal(aci.getParent().getId()));
		}

		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.ANALYSIS_COMPONENT, "uniqueid");
		ac.setUniqueid(uniqueid);
		ac.setTableid(getTableID(ui, Constants.ANALYSIS_COMPONENT));
		ac.setOwner(dbUser.getUniqueid());
		ac.setModuser(dbUser.getUniqueid());
		ac.setModtime(new Date());

		dao.insert(con, ac);
		return ac;
	}

	public Analysis addAnalysisRec(UserInfo ui, AnalysisInfo ai)
			throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			return addAnalysisRec(con, ui, ai);
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addAnalysisRec", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	protected Analysis addAnalysisRec(Connection con, UserInfo ui,
			AnalysisInfo ai) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		AnalysisDAO dao = DAOFactory.createAnalysisDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);

		Analysis an = new Analysis();
		an.setName(ai.getName());
		an.setVersion(ai.getVersion());
		if (ai.getDescription() != null) {
			an.setDescription(ai.getDescription());
		} else {
			an.setDescription(ai.getName());
		}
		an.setOwner(dbUser.getUniqueid());
		an.setModuser(dbUser.getUniqueid());
		an.setModtime(new Date());
		an.setOntologysource(Constants.DEFAULT_ONT_SRC);
		an.setConceptid(Constants.DEFAULT_ONT_CONCEPT);
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.ANALYSIS, "uniqueid");
		an.setUniqueid(uniqueid);
		an.setTableid(getTableID(ui, Constants.ANALYSIS));

		dao.insert(con, an);
		return an;
	}

	public Analysis addAnalysisRecWithComponents(UserInfo ui, AnalysisInfo ai,
			List<AnalysisCompInfo> flowList) throws Exception {
		Connection con = null;
		try {
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			AnalysisDAO dao = DAOFactory.createAnalysisDAO(theDBID);
			List<Analysis> anList = dao.find(con, new Analysis());
			for (Analysis an : anList) {
				if (an.getName().equals(ai.getName())
						&& an.getVersion().equals(ai.getVersion())) {
                     log.info("Analysis already exists: " + an);
                     con.commit();
                     return an;
				}
			}

			Analysis analysis = addAnalysisRec(con, ui, ai);

			List<Transformation> trList = getTransformations(con, ui);
			Map<TransformationInfo, Transformation> ti2trMap = new HashMap<TransformationInfo, Transformation>(
					17);
			Map<String, Transformation> trMap = new HashMap<String, Transformation>(
					17);
			for (Transformation tr : trList) {
				String key = tr.getTransform() + "_" + tr.getTransformversion();
				trMap.put(key, tr);
			}
			List<AnalysisCompInfo> forrestRootList = new ArrayList<AnalysisCompInfo>(
					1);

			List<AnalysisCompInfo> remList = new ArrayList<AnalysisCompInfo>(10);
			for (AnalysisCompInfo aci : ai.getComponentList()) {
				if (aci.getParent() == null) {
					forrestRootList.add(aci);
				} else {
					remList.add(aci);
				}
				String key = aci.getTransform().getTransform() + "_"
						+ aci.getTransform().getTransformVersion();
				Transformation tr = trMap.get(key);
				Assertion.assertNotNull(tr);
				ti2trMap.put(aci.getTransform(), tr);

			}
			Map<AnalysisCompInfo, Analysiscomponent> parentMap = new HashMap<AnalysisCompInfo, Analysiscomponent>(
					7);
			for (AnalysisCompInfo aci : forrestRootList) {

				List<List<AnalysisCompInfo>> levelList = arrangeBreadthFirst(
						aci, remList);
				for (List<AnalysisCompInfo> levelNodes : levelList) {
					Collections.sort(levelNodes,
							new Comparator<AnalysisCompInfo>() {
								public int compare(AnalysisCompInfo o1,
										AnalysisCompInfo o2) {
									return o1.getId() - o2.getId();
								}
							});

					for (AnalysisCompInfo levelNode : levelNodes) {
						// Analysiscomponent parent = parentMap.get(levelNode
						// .getParent());
						Transformation tr = ti2trMap.get(levelNode
								.getTransform());
						Analysiscomponent ac = addAnalysisComponent(con, ui,
								levelNode, analysis, tr);
						parentMap.put(levelNode, ac);
					}
				}
			}

			// add analysis flow sequence
			if (flowList != null) {
				Analysiscomponent priorAC = null;
				for (AnalysisCompInfo aci : flowList) {
					Analysiscomponent ac = parentMap.get(aci);
					Assertion.assertNotNull(ac);
					addAnalysisflowRec(con, ac, priorAC);
					priorAC = ac;
				}
			}

			con.commit();
			return null;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "addAnalysisRec", x, true);
			return null;
		} finally {
			releaseConnection(con, ui);
		}
	}

	public void deleteAnalysis(UserInfo ui, AnalysisInfo ai) throws Exception {
		Connection con = null;
		try {
			System.out.println("In deleteAnalysis");
			con = pool.getConnection(ui.getName());
			con.setAutoCommit(false);

			AnalysisDAO anDAO = DAOFactory.createAnalysisDAO(theDBID);
			AnalysiscomponentDAO acDAO = DAOFactory
					.createAnalysiscomponentDAO(theDBID);
			AnalysisflowDAO afDAO = DAOFactory.createAnalysisflowDAO(theDBID);

			Analysis cr = new Analysis();
			cr.setName(ai.getName());
			cr.setVersion(ai.getVersion());
			List<Analysis> anList = anDAO.find(con, cr);
			if (anList.isEmpty()) {
				System.out.println("No analysis matching criteria has been found! " + cr);
				return;
			}
			for (Analysis an : anList) {
				Analysisflow afCR = new Analysisflow();
				afCR.setAnalysisid(an.getUniqueid());
				afDAO.delete(con, afCR);
				Analysiscomponent acCR = new Analysiscomponent();
				acCR.setAnalysisid(an.getUniqueid());
				acDAO.delete(con, acCR);

				cr = new Analysis();
				cr.setUniqueid(an.getUniqueid());
				anDAO.delete(con, cr);
				System.out.println("deleted " + an);
			}
			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "deleteAnalysis", x, true);
		} finally {
			releaseConnection(con, ui);
		}

	}

	protected List<Transformation> getTransformations(Connection con,
			UserInfo ui) throws Exception {
		TransformationDAO dao = DAOFactory.createTransformationDAO(theDBID);
		List<Transformation> list = dao.find(con, new Transformation());
		return list;
	}

	protected List<List<AnalysisCompInfo>> arrangeBreadthFirst(
			AnalysisCompInfo root, List<AnalysisCompInfo> aciList) {
		List<List<AnalysisCompInfo>> levelList = new ArrayList<List<AnalysisCompInfo>>(
				10);
		Set<AnalysisCompInfo> aciSet = new HashSet<AnalysisCompInfo>();
		List<AnalysisCompInfo> levelNodes = new ArrayList<AnalysisCompInfo>(1);
		levelNodes.add(root);
		levelList.add(levelNodes);
		for (AnalysisCompInfo aci : aciList) {
			aciSet.add(aci);
		}

		List<AnalysisCompInfo> parentNodes = levelNodes;
		while (!aciSet.isEmpty()) {
			levelNodes = new ArrayList<AnalysisCompInfo>(2);
			for (AnalysisCompInfo parent : parentNodes) {
				List<AnalysisCompInfo> children = getImmediateChildren(parent,
						aciSet);
				levelNodes.addAll(children);
			}
			if (levelNodes.isEmpty()) {
				break;
			}
			levelList.add(levelNodes);
			parentNodes = levelNodes;
		}

		return levelList;
	}

	protected List<AnalysisCompInfo> getImmediateChildren(
			AnalysisCompInfo root, Set<AnalysisCompInfo> aciSet) {
		List<AnalysisCompInfo> children = new ArrayList<AnalysisCompInfo>(2);
		for (Iterator<AnalysisCompInfo> it = aciSet.iterator(); it.hasNext();) {
			AnalysisCompInfo eti = it.next();
			if (eti.getParent() == root) {
				children.add(eti);
				it.remove();
			}
		}
		return children;
	}

	protected Machine addMachineRec(Connection con, UserInfo ui, MachineInfo mi)
			throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		MachineDAO dao = DAOFactory.createMachineDAO(theDBID);

		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);

		Machine m = new Machine();
		m.setOs(mi.getOpSystem());
		m.setArchitecture(mi.getArchitecture());
		m.setOsversion(mi.getOsVersion());
		m.setAddress(mi.getAddress());
		m.setDescription(mi.getDescription());

		m.setOwner(dbUser.getUniqueid());
		m.setModuser(dbUser.getUniqueid());
		m.setModtime(new Date());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con, Constants.MACHINE,
				"uniqueid");
		m.setUniqueid(uniqueid);
		m.setTableid(getTableID(ui, Constants.MACHINE));

		dao.insert(con, m);
		return m;
	}

	protected void removeRemoteHumanSubject(Connection con, String subjectID)
			throws Exception {
		HumansubjectDAO hsDAO = DAOFactory.createHumansubjectDAO(theDBID);
		Humansubject hs = new Humansubject();
		hs.setSubjectid(subjectID);
		hs.setIsremote(true);
		hsDAO.delete(con, hs);
	}

	protected Humansubject addRemoteHumanSubject(Connection con, UserInfo ui,
			Site site, String subjectID) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		HumansubjectDAO hsDAO = DAOFactory.createHumansubjectDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);
		Humansubject hs = new Humansubject();

		List<Animalspecies> asList = dbCache.getAnimalSpecies(ui, false);
		Animalspecies theAS = null;
		for (Animalspecies as : asList) {
			if (as.getName().equals("researchSubject")) {
				theAS = as;
				break;
			}
		}
		Assertion.assertNotNull(theAS);
		hs.setSubjectid(subjectID);
		hs.setSiteid(site.getUniqueid());
		hs.setIsremote(true);
		hs.setExtensionname("humanSubject");
		hs.setNcAnimalspeciesUniqueid(theAS.getUniqueid());

		hs.setOwner(dbUser.getUniqueid());
		hs.setModuser(dbUser.getUniqueid());
		hs.setModtime(new Date());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.HUMANSUBJECT_DB_TABLE, "uniqueid");
		hs.setUniqueid(uniqueid);
		hs.setTableid(getTableID(ui, Constants.HUMANSUBJECT_DB_TABLE));

		hsDAO.insert(con, hs);
		return hs;
	}

	protected Externaldata addExternalDataRec(Connection con, UserInfo ui,
			ExternalDataInfo edi, Map<String, Site> siteMap) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(theDBID);
		ExternaldataDAO dao = DAOFactory.createExternaldataDAO(theDBID);
		Databaseuser dbUser = getDatabaseUser(ui, Constants.USERCLASS_ADMIN);
		Assertion.assertNotNull(dbUser);
		Externaldata ed = new Externaldata();
		Site site = siteMap.get(edi.getSiteIDStr());
		Assertion.assertNotNull(site);

		ed.setDatauri(edi.getDataURI());
		ed.setSubjectid(edi.getSubjectID());

		if (edi.getVisitID() != null)
			ed.setComponentid(new BigDecimal(edi.getVisitID()));
		if (edi.getSegmentID() != null)
			ed.setSegmentid(new BigDecimal(edi.getSegmentID()));
		ed.setExperimentid(new BigDecimal(edi.getExperimentID()));
		ed.setExperimentname(edi.getExpName());
		ed.setIsraw(edi.isRaw());
		ed.setSiteid(site.getUniqueid());
		ed.setExtensionname("nc_externaldata");

		ed.setOntologysource(Constants.DEFAULT_ONT_SRC);
		ed.setConceptid(Constants.DEFAULT_ONT_CONCEPT);

		ed.setOwner(dbUser.getUniqueid());
		ed.setModuser(dbUser.getUniqueid());
		ed.setModtime(new Date());
		BigDecimal uniqueid = sequenceHelper.getNextUID(con,
				Constants.EXTERNAL_DATA, "uniqueid");
		ed.setUniqueid(uniqueid);
		ed.setTableid(getTableID(ui, Constants.EXTERNAL_DATA));

		dao.insert(con, ed);
		return ed;
	}

	protected void deleteExternalDataRec(Connection con, String expName,
			String subjectID, Integer visitID, Integer segmentID)
			throws Exception {
		ExternaldataDAO dao = DAOFactory.createExternaldataDAO(theDBID);

		Externaldata cr = new Externaldata();
		cr.setExperimentname(expName);
		cr.setSubjectid(subjectID);
		cr.setComponentid(new BigDecimal(visitID));
		if (segmentID != null) {
			cr.setSegmentid(new BigDecimal(segmentID));
		}
		dao.delete(con, cr);
	}

	protected Analysisflow addAnalysisflowRec(Connection con,
			Analysiscomponent ac, Analysiscomponent priorAC) throws Exception {
		Assertion.assertTrue(priorAC == null
				|| ac.getAnalysisid().equals(priorAC.getAnalysisid()));

		AnalysisflowDAO dao = DAOFactory.createAnalysisflowDAO(theDBID);
		Analysisflow af = new Analysisflow();
		af.setAnalysisid(ac.getAnalysisid());
		af.setComponentid(ac.getComponentid());
		if (priorAC != null) {
			af.setPriorcomponentid(priorAC.getComponentid());
		}
		dao.insert(con, af);
		return af;
	}

	protected Executedtransformflow addExecutedTransformFlow(Connection con,
			Executedtransform et, Executedtransform priorET, Integer startETID)
			throws Exception {

		Assertion.assertTrue(priorET == null
				|| et.getAnalysisid().equals(priorET.getAnalysisid()));
		ExecutedtransformflowDAO dao = DAOFactory
				.createExecutedtransformflowDAO(theDBID);
		Executedtransformflow etf = new Executedtransformflow();
		etf.setAnalysisid(et.getAnalysisid());
		etf.setComponentid(et.getAnalysiscomponentid());
		etf.setExecutedtransformationid(et.getUniqueid());
		if (priorET != null) {
			etf.setPriorexecutedtransformid(priorET.getUniqueid());
		}
		etf.setStartexecutedtransformid(new BigDecimal(startETID));
		dao.insert(con, etf);
		return etf;
	}
}
