package clinical.web.services;

import java.sql.Connection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;

import clinical.server.dao.HumansubjectDAO;
import clinical.server.vo.Humansubject;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IDBPoolService;

/**
 * Generates and maintains BIRN ids. A BIRN ID is a 4 digit zero-padded
 * institute ID followed by 8 digit zero padded random number. The random number
 * generated is not immune to time based attacks for guessing next random number
 * to be generated.
 * 
 * @author I. Burak Ozyurt
 * @version $Id: RandomBIRNIDGenerator.java 366 2011-05-05 20:06:27Z bozyurt $
 * 
 * @see SecureBIRNIDGenerator
 */
public class RandomBIRNIDGenerator implements IBIRNIDGenerationService {
	private Random rng;
	private IDBPoolService dbPoolService;
	private String dbID;
	private Map<String, String> birnIDsMap = Collections
			.synchronizedMap(new LinkedHashMap<String, String>());

	private static IBIRNIDGenerationService instance = null;
	public final static int NUMBER_LEN = 8;
	
	public static synchronized IBIRNIDGenerationService getInstance(String dbID)  {
		if (instance == null) {
			instance = new RandomBIRNIDGenerator(dbID);
		}
		return instance;
	}
	public static synchronized IBIRNIDGenerationService getInstance() {
		if (instance == null) {
			throw new RuntimeException(
					"BIRNIDGenerator not initialized properly!");
		}
		return instance;
	}
	
	private RandomBIRNIDGenerator(String dbID) {
		dbPoolService = ServiceFactory.getPoolService(dbID);
		this.dbID = dbID;
		prepare();
	}
	
	private void prepare() {
		rng = new Random();
	}

	@Override
	public String createBIRNID(String prefix) throws Exception {
		return createBIRNID(prefix, NUMBER_LEN);
	}

	public String createBIRNID(String prefix, int numDigits) throws Exception {
		String birnID = prefix + createRandomID(numDigits);
		loadBIRNIDs();
		while (birnIDsMap.get(birnID) != null) {
			birnID = prefix + createRandomID(numDigits);
		}
		
		birnIDsMap.put(birnID, birnID);
		return birnID;
	}
	
	private void loadBIRNIDs() throws Exception {
		Connection con = null;
		try {
			con = dbPoolService.getConnection(Constants.ADMIN_USER);
			HumansubjectDAO dao = DAOFactory.createHumansubjectDAO(this.dbID);
			List<Humansubject> hsList = dao.find(con, new Humansubject());
			for (Humansubject hs : hsList) {
				if (birnIDsMap.get(hs.getSubjectid()) == null) {
					birnIDsMap.put(hs.getSubjectid(), hs.getSubjectid());
				}
			}
		} finally {
			if (con != null)
				dbPoolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
	
	private String createRandomID(int length) {
		long maxNumber = 1;
		int i = length;
		while (i-- > 0) {
			maxNumber *= 10;
			if (maxNumber > Integer.MAX_VALUE) {
				break;
			}
		}

		StringBuilder buf = new StringBuilder(length);
		int number = rng.nextInt((int) maxNumber);
		buf.append(formatNumber(length, number));
		return buf.toString();
	}
	
	private String formatNumber(int noDigits, int number) {
		String s = Integer.toString(number);
		if (noDigits > s.length()) {
			// zero padding
			StringBuffer buf = new StringBuffer(noDigits);
			int diff = noDigits - s.length();
			for (int i = 0; i < diff; ++i)
				buf.append('0');
			buf.append(s);
			return buf.toString();
		} else
			return s;
	}

}
