package clinical.web.services;

import java.math.BigDecimal;
import java.security.NoSuchAlgorithmException;
import java.sql.Connection;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.globus.myproxy.MyProxy;
import org.ietf.jgss.GSSCredential;

import clinical.cache.DBChangeNotifySupport;
import clinical.server.dao.ConfDatabaseDAO;
import clinical.server.dao.ConfDbuserDAO;
import clinical.server.dao.ConfPrivilegeDAO;
import clinical.server.dao.ConfRoleDAO;
import clinical.server.dao.ConfRolePrivsDAO;
import clinical.server.dao.ConfWebuserDAO;
import clinical.server.dao.ConfWebuserPrivDAO;
import clinical.server.dao.ConfWebuserProjPrivDAO;
import clinical.server.dao.DatabaseuserDAO;
import clinical.server.dao.SiteDAO;
import clinical.server.dao.UserclassDAO;
import clinical.server.vo.ConfDatabase;
import clinical.server.vo.ConfDbuser;
import clinical.server.vo.ConfPrivilege;
import clinical.server.vo.ConfRole;
import clinical.server.vo.ConfRolePrivs;
import clinical.server.vo.ConfWebuser;
import clinical.server.vo.ConfWebuserPriv;
import clinical.server.vo.ConfWebuserProjPriv;
import clinical.server.vo.Databaseuser;
import clinical.server.vo.Experiment;
import clinical.server.vo.Site;
import clinical.server.vo.Userclass;
import clinical.utils.Assertion;
import clinical.utils.DBTableInfo;
import clinical.utils.GenUtils;
import clinical.web.Constants;
import clinical.web.DAOFactory;
import clinical.web.DBUtils;
import clinical.web.IAppConfigService;
import clinical.web.IDBUserManService;
import clinical.web.ISQLDialect;
import clinical.web.ISequenceHelper;
import clinical.web.MinimalServiceFactory;
import clinical.web.ServiceFactory;
import clinical.web.common.IAuthenticationService;
import clinical.web.common.IAuthorizationService;
import clinical.web.common.IDBCache;
import clinical.web.common.IDBPoolService;
import clinical.web.common.ISecurityAdminService;
import clinical.web.common.ISecurityService;
import clinical.web.common.UserInfo;
import clinical.web.common.query.TSQLProcessor;
import clinical.web.common.security.DBConfig;
import clinical.web.common.security.Privilege;
import clinical.web.common.security.ProjectPrivilege;
import clinical.web.common.security.User;
import clinical.web.common.security.User.PrivilegeExperiments;
import clinical.web.common.security.User.ProjectPrivilegeStatus;
import clinical.web.exception.AuthenticationException;
import clinical.web.exception.BaseException;
import clinical.web.exception.DBPoolServiceException;
import clinical.web.exception.UnknownUserException;
import clinical.web.helpers.security.AuthorizationHelper;
import clinical.web.helpers.security.UserPrivilegeSet;

/**
 * @author I. Burak Ozyurt
 * @version $Id: SecurityService.java 782 2013-03-13 02:02:28Z bozyurt $
 */
public class SecurityService implements IAuthenticationService,
		IAuthorizationService, ISecurityService, ISecurityAdminService {
	public static final String USEGSI = "usegsi";
	private Map<String, Privilege> privilegesMap = new HashMap<String, Privilege>(
			23);
	private Map<String, DBConfig> dbConfigMap = Collections
			.synchronizedMap(new HashMap<String, DBConfig>(7));
	private Map<String, Map<String, String>> masterTableMap = new HashMap<String, Map<String, String>>();
	private DBConfig currentDBConfig;
	private final static String PUBLIC_USER = "public";
	private final static String DEFAULT_USER = "admin";
	private final static String GUEST_USER = "guest";

	private String dbID;
	private IDBPoolService poolService;
	private String dbType;
	private Log log = LogFactory.getLog(SecurityService.class);
	private static SecurityService instance = null;

	private static final boolean USE_TABLE_MD_CACHE = false;

	private SecurityService(IDBPoolService poolService, String dbID,
			String dbType) throws BaseException {
		this.poolService = poolService;
		this.dbID = dbID;
		this.dbType = dbType;

	}

	public static synchronized SecurityService getInstance(
			IDBPoolService poolService, String dbID, String dbType)
			throws BaseException {
		if (instance == null) {
			instance = new SecurityService(poolService, dbID, dbType);
		}
		return instance;
	}

	/**
	 * 
	 * @return the singleton SecurityService
	 * @throws BaseException
	 *             if not created before
	 */
	public static synchronized SecurityService getInstance()
			throws BaseException {
		if (instance == null)
			throw new BaseException("Service is not initialized!");
		return instance;
	}

	public synchronized void startup() throws BaseException {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);

			retrieveUserDBAdminData(con);
		} catch (Exception e) {
			if (e instanceof BaseException) {
				throw (BaseException) e;
			}
			throw new BaseException(e);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public UserInfo authenticate(String user, String pwd, String dbID)
			throws AuthenticationException {
		System.out.println("authenticating user " + user + " for dbID:" + dbID);
		DBConfig dbConfig = dbConfigMap.get(dbID);
		if (dbConfig == null) {
			throw new AuthenticationException("Unknown dbID:" + dbID);
		}
		System.out.println(">>" + dbConfig.toString());
		User u = dbConfig.getUser(user);
		if (u == null) {
			throw new UnknownUserException("User '" + user
					+ "' is not recognized!");
		}
		try {
			String digest = GenUtils.getMD5Digest(pwd);
			if (!u.getPwd().equals(digest)) {
				throw new AuthenticationException("Incorrect Password!");
			}

		} catch (NoSuchAlgorithmException e1) {
			e1.printStackTrace();
			throw new AuthenticationException("Incorrect Password!");
		}

		try {
			AuthorizationHelper helper = new AuthorizationHelper(dbID);
			UserPrivilegeSet userPrivilegeSet = helper.prepareUserPrivileges(
					user, Constants.ADMIN_USER);
			u.setUserPrivSet(userPrivilegeSet);
		} catch (Exception x) {
			x.printStackTrace();
			throw new AuthenticationException("Authentication Error!");
		}

		GSSCredential credential = null;
		if (u.isUseGSI()) {
			System.out.println("using Myproxy authentication for web user:"
					+ u.getName());
			try {
				IAppConfigService configService = ServiceFactory
						.getAppConfigService();
				// pass user provided password instead of MD5 digest
				User clearUser = new User(user, pwd, null, u.isUseGSI());
				credential = getCredential(configService, clearUser);
			} catch (Exception e) {
				e.printStackTrace();
				throw new AuthenticationException(
						"Problem in getting user certificate!");
			}
			if (credential == null) {
				throw new AuthenticationException(
						"Cannot get certificate for user:" + u.getName());
			}
		}

		// find all available tables for this user and cache them
		// for the DAO adapters so that they can fetch from the correct view or
		// table
		// and return appropriate messages for the hidden columns
		if (USE_TABLE_MD_CACHE) {
			prepareAvailableTables(u, dbID);
		}
		if (u.isUseGSI()) {
			return new UserInfo(user, null, u.getAvailableTables(), credential);
		} else {
			return new UserInfo(user, null, u.getAvailableTables());
		}
	}

	protected GSSCredential getCredential(IAppConfigService configService,
			User u) throws Exception {
		String myproxyHost = configService
				.getParamValue("globus.myproxy.server.primary");
		int port = GenUtils.toInt(configService
				.getParamValue("globus.myproxy.server.port"), 7512);
		MyProxy proxy = new MyProxy(myproxyHost, port);
		GSSCredential credential = proxy.get(u.getName(), u.getPwd(), 0);
		return credential;
	}

	// FIXME how to handle GSI for anonymous users
	public UserInfo authenticateAnonymous(String email, String dbID)
			throws AuthenticationException {
		DBConfig dbConfig = dbConfigMap.get(dbID);
		if (dbConfig == null) {
			throw new AuthenticationException("Unknown dbID:" + dbID);
		}
		// check for email format (very primitive)
		if (email.indexOf("@") == -1) {
			throw new AuthenticationException("Not a valid email address");
		}

		User u = dbConfig.getUser(PUBLIC_USER);
		if (u == null)
			throw new UnknownUserException("errors.notrecognized", PUBLIC_USER);

		// find all available tables for this user and cache them
		// for the DAO adapters so that they can fetch from the correct view or
		// table and return appropriate messages for the hidden columns
		if (USE_TABLE_MD_CACHE) {
			prepareAvailableTables(u, dbID);
		}

		return new UserInfo(PUBLIC_USER, email, null, u.getAvailableTables());
	}

	public UserInfo getDefaultUser(String dbID, UserInfo ui)
			throws AuthenticationException {
		// a simple security check with the assumption that if the passed ui is
		// a valid user info possible it is safe the return the default user
		// information for database connection
		DBConfig dbConfig = dbConfigMap.get(dbID);
		// log.info("dbID=" + dbID + " dbConfig=" + dbConfig);
		User u = dbConfig.getUser(DEFAULT_USER);

		if (USE_TABLE_MD_CACHE) {
			if (u.getAvailableTables() == null
					|| u.getAvailableTables().isEmpty()) {
				// occurs if the default user from a secondary database is
				// requested
				prepareAvailableTables(u, dbID);
			}
		}

		return new UserInfo(DEFAULT_USER, null, u.getAvailableTables(), ui
				.getCredential());
	}

	public synchronized boolean isAuthorized(UserInfo userInfo, String dbId,
			PrivilegeLabel privName, Integer experimentID) {
		if (privName == null) {
			return false;
		}
		User u = null;
		DBConfig dbConf = this.dbConfigMap.get(dbId);
		if (dbConf != null) {
			u = dbConf.getUser(userInfo.getName());
		}
		if (u == null)
			return false;
		boolean hasPriv = u.hasPrivilege(privName.toString(), experimentID);

		if (log.isDebugEnabled()) {
			log.debug("privName:" + privName + " User u:" + u.getName()
					+ " dbId:" + dbId + " hasPrivilege:" + hasPriv);
		}
		return hasPriv;
	}

	/**
	 * Assumption: only one database per site dbID is in the form
	 * 
	 * 
	 * @param siteID
	 * @return
	 */
	public DBConfig findBySiteID(String siteID) {
		for (Iterator<DBConfig> iter = dbConfigMap.values().iterator(); iter
				.hasNext();) {
			DBConfig dbConfig = iter.next();
			if (dbConfig.getSiteID().equals(siteID)) {
				return dbConfig;
			}
		}
		return null;
	}

	public synchronized String findSiteIDByDbID(String dbID) {
		for (DBConfig dbConfig : dbConfigMap.values()) {
			if (dbConfig.getId().equals(dbID)) {
				return dbConfig.getSiteID();
			}
		}
		return null;
	}

	public synchronized Map<String, String> getSiteURLs() {
		Map<String, String> map = new HashMap<String, String>(17);
		for (DBConfig dbConfig : dbConfigMap.values()) {
			map.put(dbConfig.getId(), dbConfig.getSiteURL());
		}
		return map;
	}

	public String[] getAllDBIDs() {
		String[] arr = new String[dbConfigMap.size()];
		int i = 0;
		for (Iterator<String> iter = dbConfigMap.keySet().iterator(); iter
				.hasNext();) {
			String dbID = iter.next();
			arr[i++] = dbID;
		}
		return arr;
	}

	public Map<String, User> getAllNamedUsers(String dbID) {
		return dbConfigMap.get(dbID).getDBUserMap();
	}

	public Map<String, User> getAllUsers(String dbID) {
		return dbConfigMap.get(dbID).getUserMap();
	}

	public String getDBType(String dbID) {
		DBConfig dbConfig = dbConfigMap.get(dbID);
		if (dbConfig == null) {
			System.out.println("dbConfig is null");
			if (this.dbID == dbID) {
				return this.dbType;
			} else {
				return null;
			}
		}
		return dbConfig.getDbType();
	}

	public String getDefaultDBID() {
		return currentDBConfig.getId();
	}

	public User getRestrictedUser(String dbID) {
		DBConfig dbConfig = dbConfigMap.get(dbID);
		// log.info("dbID=" + dbID + " dbConfig=" + dbConfig);

		User guestUser = dbConfig.getUser(GUEST_USER);
		if (guestUser == null) {
			guestUser = new User(GUEST_USER, "", null);
			guestUser.setDbUser(getDefaultDBUser(dbConfig));
			dbConfig.addUser(guestUser);

			if (USE_TABLE_MD_CACHE) {
				if (guestUser.getAvailableTables() == null
						|| guestUser.getAvailableTables().isEmpty()) {
					// occurs if the default user from a secondary database is
					// requested
					prepareAvailableTables(guestUser, dbID);
				}
			}
		}
		return guestUser;
	}

	public Map<String, DBConfig> getDBConfigMap() {
		return dbConfigMap;
	}

	public List<Privilege> getPrivileges() {
		List<Privilege> privList = new ArrayList<Privilege>(privilegesMap
				.values());
		Collections.sort(privList, new Comparator<Privilege>() {
			public int compare(Privilege p1, Privilege p2) {
				return p1.getName().compareTo(p2.getName());
			}
		});
		return privList;
	}

	public synchronized void updateUserPoolForDatabase(DBConfig dbConfig)
			throws SecurityException {
		throw new UnsupportedOperationException();
	}

	public DBConfig getCurrentDBConfig() {
		return currentDBConfig;
	}

	private User getDefaultDBUser(DBConfig dbConfig) {
		// just return the first dbUser encountered as the default user
		for (User dbUser : dbConfig.getDBUserMap().values()) {
			return dbUser;
		}
		return null;
	}

	/**
	 * Gets the master table/view list from the database meta data and caches.
	 * 
	 * @param dbID
	 *            the ID of the database to get the table/view list
	 * @throws java.lang.Exception
	 */
	public void prepareTableCache(String dbID) throws Exception {
		if (!USE_TABLE_MD_CACHE) {
			return;
		}

		@SuppressWarnings("unused")
		final UserInfo ui = new UserInfo("admin", null, null);

		DBConfig adbc = dbConfigMap.get(dbID);

		// get the master table/view list and cache
		Map<String, String> dbMasterTableMap = new HashMap<String, String>(67);

		Map<String, DBTableInfo> tableInfoMap = DBUtils.prepareTableInfoCache(
				ui, dbID, false, adbc.getDbType());

		for (DBTableInfo dbti : tableInfoMap.values()) {
			log.info("prepareTableCache: Adding " + dbti.getTableName());
			dbMasterTableMap.put(dbti.getTableName().toUpperCase(), dbti
					.getTableName());
		}

		masterTableMap.put(dbID, dbMasterTableMap);
	}

	protected void prepareAvailableTables(User user, String dbID) {
		Connection con = null;
		IDBPoolService pool = null;
		if (!user.getAvailableTables().isEmpty())
			return;
		try {
			pool = MinimalServiceFactory.getPoolService(dbID);
			con = pool.getConnection(user.getName());
			Map<String, String> dbMasterTableMap = masterTableMap.get(dbID);
			for (String tableName : dbMasterTableMap.keySet()) {
				if (canAccessTable(con, tableName)) {
					user.addAvailableTable(tableName);
				} else {
					log.error(user.getName() + " cannot access table: "
							+ tableName);
				}
			}
		} catch (Exception x) {
			x.printStackTrace();
			log.error(x, x);
		} finally {
			try {
				log.info("****** Releasing connection " + user.getName()
						+ " con=" + con + " dbID=" + dbID);
				pool.releaseConnection(user.getName(), con);
			} catch (Exception ex) {
				log.error(ex, ex);
			}
		}
	}

	/**
	 * Checks if the given table name can be accessed by using the given
	 * connection.
	 * 
	 * @param con
	 *            database connection
	 * @param tableName
	 *            database table name
	 * @return true if the given table name can be accessed by using the given
	 *         connection.
	 */
	public static boolean canAccessTable(Connection con, String tableName) {
		Statement st = null;
		try {
			st = con.createStatement();
			ResultSet rs = st.executeQuery("select 1 from " + tableName);
			rs.close();
			return true;
		} catch (SQLException x) {
			return false;
		} finally {
			DBUtils.close(st);
		}
	}

	public List<Site> getAllSites() throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			SiteDAO dao = DAOFactory.createSiteDAO(dbID);
			List<Site> sites = dao.find(con, new Site());
			return sites;
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
			return null;
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public String getEmail(String userName) {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			ConfWebuserDAO dao = DAOFactory.createConfWebuserDAO(dbID);
			ConfWebuser cr = new ConfWebuser();
			cr.setName(userName);
			List<ConfWebuser> list = dao.find(con, cr);
			if (!list.isEmpty()) {
				return list.get(0).getEmail();
			}
			return null;
		} catch (Exception x) {
			log.error("getEmail:" + x);
			return null;
		} finally {
			try {
				poolService.releaseConnection(Constants.ADMIN_USER, con);
			} catch (DBPoolServiceException e) {
				e.printStackTrace();
			}
		}
	}

	public List<ConfWebuser> findByEmail(String email) {
		if (email == null || email.isEmpty()) {
			return null;
		}
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			ConfWebuserDAO dao = DAOFactory.createConfWebuserDAO(dbID);
			ConfWebuser cr = new ConfWebuser();
			cr.setEmail(email);
			List<ConfWebuser> list = dao.find(con, cr);
			if (!list.isEmpty()) {
				return list;
			}
			return null;
		} catch (Exception x) {
			log.error("findByEmail:" + x);
			return null;
		} finally {
			try {
				poolService.releaseConnection(Constants.ADMIN_USER, con);
			} catch (DBPoolServiceException e) {
				e.printStackTrace();
			}
		}
	}

	public Site getSite(String siteUniqueId) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			return getSite(con, siteUniqueId);
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
			return null;
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public Site getSite(Connection con, String siteUniqueId) throws Exception {
		SiteDAO dao = DAOFactory.createSiteDAO(dbID);
		Site cr = new Site();
		cr.setUniqueid(new BigDecimal(siteUniqueId));
		List<Site> sites = dao.find(con, cr);
		if (sites.isEmpty()) {
			return null;
		}
		return sites.get(0);
	}

	public Site getSiteByID(Connection con, String siteID) throws Exception {
		SiteDAO dao = DAOFactory.createSiteDAO(dbID);
		Site cr = new Site();
		cr.setSiteid(siteID);
		List<Site> sites = dao.find(con, cr);
		if (sites.isEmpty()) {
			return null;
		}
		return sites.get(0);
	}

	public synchronized String getSiteID(String dbID) {
		DBConfig dbc = this.dbConfigMap.get(dbID);
		if (dbc == null)
			return null;
		return dbc.getSiteID();
	}

	public synchronized String findDBForSiteID(String siteID) {
		for (DBConfig dbConf : this.dbConfigMap.values()) {
			if (dbConf.getSiteID().equals(siteID)) {
				return dbConf.getId();
			}
		}
		return null;
	}

	public void removePrivileges(DBConfig dbConf, String webUserName,
			List<String> privNames) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);

			removePrivileges(con, dbConf, webUserName, privNames);
			con.commit();

			for (User webUser : dbConf.getWebUsers()) {
				if (webUser.getName().equals(webUserName)) {
					for (String privName : privNames) {
						webUser.removePrivilege(privName);
					}
				}
			}
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void addPrivileges(DBConfig dbConf, String webUserName,
			List<String> privNames) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);

			Set<String> privSet = addPrivileges(con, dbConf, webUserName,
					privNames);
			con.commit();
			if (privSet != null) {
				for (User webUser : dbConf.getWebUsers()) {
					if (webUser.getName().equals(webUserName)) {
						for (String privName : privNames) {
							if (privSet.contains(privName))
								webUser.addPrivilege(privilegesMap
										.get(privName));
						}
					}
				}
			}
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
	
	public void saveProjectPrivileges(DBConfig dbConf, User webUser)
			throws Exception {
		saveProjectPrivileges(dbConf, webUser, false);
	}

	public void saveProjectPrivileges(DBConfig dbConf, User webUser, boolean forceRefresh)
			throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			ConfWebuserDAO dao = DAOFactory.createConfWebuserDAO(dbID);
			ConfWebuser cr = new ConfWebuser();
			cr.setName(webUser.getName());
			List<ConfWebuser> wuList = dao.find(con, cr);
			if (!wuList.isEmpty() && wuList.size() == 1) {
				ConfWebuser wu = wuList.get(0);
				updateUserProjPrivileges(con, wu, webUser, dbConf);
			}
			con.commit();
			if (forceRefresh) {
				try {
					AuthorizationHelper helper = new AuthorizationHelper(dbID);
					UserPrivilegeSet userPrivilegeSet = helper.prepareUserPrivileges(
							webUser.getName(), Constants.ADMIN_USER);
					webUser.setUserPrivSet(userPrivilegeSet);
				} catch (Exception x) {
					x.printStackTrace();
				}
				// refreshWebUserPrivileges(dbConf);
			}
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public synchronized void refreshWebUserPrivileges(DBConfig dbConf)
			throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			// get project specific privileges list
			List<ConfPrivilege> ppList = getProjectSpecPrivilegeList(con);
			Set<String> ppSet = new HashSet<String>(7);
			for (ConfPrivilege pp : ppList) {
				ppSet.add(pp.getName());
			}
			ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(dbID);
			List<ConfWebuser> wuList = wuDAO.find(con, new ConfWebuser());
			IDBCache dbCache = ServiceFactory.getDBCache(dbID);
			List<Experiment> experiments = dbCache.getExperiments(new UserInfo(
					Constants.ADMIN_USER, null, null), true);

			// and prepare project specific data matrix
			for (ConfWebuser wu : wuList) {
				User webUser = dbConf.findUser(wu.getName());
				prepProjectPrivData(con, wu, experiments, ppList, webUser);
			}
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void addWebUser(DBConfig dbConf, User webUser) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			webUser.setFullPrivilegesMap(privilegesMap);
			ConfWebuser wu = addUpdateWebUser(con, webUser, dbConf);
			// get project specific privileges list
			List<ConfPrivilege> ppList = getProjectSpecPrivilegeList(con);
			Set<String> ppSet = new HashSet<String>(7);
			for (ConfPrivilege pp : ppList) {
				ppSet.add(pp.getName());
			}

			IDBCache dbCache = ServiceFactory.getDBCache(dbID);
			List<Experiment> experiments = dbCache.getExperiments(new UserInfo(
					Constants.ADMIN_USER, null, null), false);

			// and prepare project specific data matrix
			prepProjectPrivData(con, wu, experiments, ppList, webUser);

			con.commit();
			dbConf.addUser(webUser);
			IDBPoolService aPoolService = DBPoolService.getPoolService(dbConf
					.getId());
			Assertion.assertNotNull(aPoolService);
			aPoolService.addUser(webUser);

			// drop any database user cache
			// notify both local and remote systems
			DBChangeNotifySupport.getInstance().fireDBChangeNotifyEvent(
					getSiteID(this.dbID), Constants.DATABASEUSER_DB_TABLE);

		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void removeWebUser(DBConfig dbConf, String webUserName)
			throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			removeWebUser(con, dbConf, webUserName);
			con.commit();
			User webUser = dbConf.removeUser(webUserName);
			IDBPoolService aPoolService = DBPoolService.getPoolService(dbConf
					.getId());
			Assertion.assertNotNull(aPoolService);
			aPoolService.removeUser(webUser);

		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void editWebUser(DBConfig dbConf, User webUser) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			webUser.setFullPrivilegesMap(privilegesMap);
			addUpdateWebUser(con, webUser, dbConf);
			con.commit();
			// sync current state in memory
			// FIXME race condition?
			User user = dbConf.findUser(webUser.getName());
			user.setDbUser(webUser.getDbUser());
			user.setPwd(webUser.getPwd());
			user.setUseGSI(webUser.isUseGSI());
			user.setEmail(webUser.getEmail());
			user.setInvited(webUser.isInvited());

		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}
	
	public ConfWebuser getWebUserById(BigDecimal webUserId) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			ConfWebuser wuCR = new ConfWebuser();
			wuCR.setUniqueId(webUserId);
			ConfWebuserDAO dao = DAOFactory.createConfWebuserDAO(dbID);
			List<ConfWebuser> list = dao.find(con, wuCR);
			if (list.isEmpty()) {
				return null;
			}
			Assertion.assertTrue(list.size() == 1);
			return list.get(0);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void editDBUser(DBConfig dbConf, User dbUser) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			dbUser.setFullPrivilegesMap(privilegesMap);
			addUpdateDBUser(con, dbUser, dbConf);
			con.commit();
			User user = dbConf.findDBUser(dbUser.getName());
			user.setPwd(dbUser.getPwd());

		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void addDbUser(DBConfig dbConf, User dbUser) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);

			ConfDbuser du = addUpdateDBUser(con, dbUser, dbConf);
			// default password needs to changed
			AuthorizationHelper helper = new AuthorizationHelper(dbConf.getId());
			UserPrivilegeSet userPrivSet = helper.prepareUserPrivileges(
					Constants.ADMIN_USER, Constants.ADMIN_USER);
			User webUser = new User(Constants.ADMIN_USER, Constants.ADMIN_USER,
					userPrivSet);
			webUser.setDbUser(dbUser);

			for (String privName : privilegesMap.keySet()) {
				if (!privName.equalsIgnoreCase("admin")) {
					webUser.addPrivilege(privilegesMap.get(privName));
				}
			}
			addWebUser(con, webUser, dbConf, du);
			dbConf.addDbUser(dbUser);
			dbConf.addUser(webUser);
			DBPoolService.addDatabase(dbConf);

			con.commit();
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void removeDbUser(DBConfig dbConf, String dbUserName)
			throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);

			removeDBUser(con, dbConf, dbUserName);

			con.commit();
			dbConf.removeDBUser(dbUserName);

		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public DBConfig addUpdateDatabase(String dbId, String dbURL, String dbType,
			String siteUniqueId, String siteURL) throws Exception {
		Connection con = null;
		try {
			Site site = getSite(siteUniqueId);
			Assertion.assertNotNull(site);
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);

			DBConfig dbConf = new DBConfig(dbId, site.getSitename(), site
					.getSiteid());
			dbConf.setDbType(dbType);
			dbConf.setDbURL(dbURL);
			dbConf.setSiteURL(siteURL);

			addUpdateDatabase(con, dbConf);
			con.commit();
			if (!dbConfigMap.containsKey(dbId)) {
				this.dbConfigMap.put(dbId, dbConf);
				return dbConf;
			} else {
				DBConfig origDBConf = dbConfigMap.get(dbId);
				origDBConf.setDbType(dbType);
				origDBConf.setDbURL(dbURL);
				origDBConf.setSiteURL(siteURL);
				DBPoolService.removeDatabase(origDBConf.getId());
				DBPoolService.addDatabase(origDBConf);
				return null;
			}
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
			return null;
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	public void removeDatabase(DBConfig dbConfig) throws Exception {
		Connection con = null;
		try {
			con = poolService.getConnection(Constants.ADMIN_USER);
			con.setAutoCommit(false);
			removeDatabase(con, dbConfig);
			con.commit();
			this.dbConfigMap.remove(dbConfig.getId());
		} catch (Exception x) {
			handleErrorAndRollBack(con, "", x);
		} finally {
			poolService.releaseConnection(Constants.ADMIN_USER, con);
		}
	}

	protected synchronized void removePrivileges(Connection con,
			DBConfig dbConf, String webUserName, List<String> privNames)
			throws Exception {

		ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(dbID);
		ConfWebuserPrivDAO wupDAO = DAOFactory.createConfWebuserPrivDAO(dbID);

		ConfDatabase cd = getDatabaseRec(con, dbConf);
		if (cd == null) {
			return;
		}

		ConfWebuser wuCR = new ConfWebuser();
		wuCR.setDbUniqueId(cd.getUniqueId());
		wuCR.setName(webUserName);
		List<ConfWebuser> wuList = wuDAO.find(con, wuCR);
		if (wuList.isEmpty()) {
			// no web user matching (should not happen, indicates a bug or
			// tampering with db outside the web app)
			System.err.println("Found no ConfWebuser matching " + wuCR);
			return;
		}
		Assertion.assertTrue(wuList.size() == 1);
		ConfWebuserPriv cr = new ConfWebuserPriv();
		cr.setWebuserId(wuList.get(0).getUniqueId());
		for (String privName : privNames) {
			cr.setPrivName(privName);
			wupDAO.delete(con, cr);
		}
	}

	protected synchronized void removeWebUser(Connection con, DBConfig dbConf,
			String webUserName) throws Exception {
		if (webUserName == null || dbConf == null) {
			throw new IllegalArgumentException();
		}
		ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(dbID);
		ConfWebuserPrivDAO wupDAO = DAOFactory.createConfWebuserPrivDAO(dbID);

		ConfDatabase cd = getDatabaseRec(con, dbConf);
		if (cd == null) {
			return;
		}
		ConfWebuser wuCR = new ConfWebuser();
		wuCR.setDbUniqueId(cd.getUniqueId());
		wuCR.setName(webUserName);
		List<ConfWebuser> wuList = wuDAO.find(con, wuCR);
		Assertion.assertTrue(wuList.size() == 1);
		ConfWebuser wu = wuList.get(0);
		ConfWebuserPriv cr = new ConfWebuserPriv();
		cr.setWebuserId(wu.getUniqueId());
		wupDAO.delete(con, cr);

		wuCR = new ConfWebuser();
		wuCR.setUniqueId(wu.getUniqueId());
		wuDAO.delete(con, wuCR);

		// also remove from nc_databaseuser table
		Databaseuser duCR = new Databaseuser();
		duCR.setName(webUserName.toUpperCase());
		DatabaseuserDAO duDAO = DAOFactory.createDatabaseuserDAO(dbID);
		duDAO.delete(con, duCR);
	}

	protected synchronized void removeDBUser(Connection con, DBConfig dbConf,
			String dbUserName) throws Exception {
		if (dbUserName == null || dbConf == null) {
			throw new IllegalArgumentException();
		}
		ConfDbuserDAO duDAO = DAOFactory.createConfDBuserDAO(dbID);
		ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(dbID);
		ConfWebuserPrivDAO wupDAO = DAOFactory.createConfWebuserPrivDAO(dbID);

		ConfDatabase cd = getDatabaseRec(con, dbConf);
		if (cd == null) {
			return;
		}
		ConfDbuser duCR = new ConfDbuser();
		duCR.setName(dbUserName);
		duCR.setDbUniqueId(cd.getUniqueId());
		List<ConfDbuser> duList = duDAO.find(con, duCR);
		if (duList.isEmpty()) {
			return;
		}
		Assertion.assertTrue(duList.size() == 1);
		ConfDbuser du = duList.get(0);

		ConfWebuser wuCR = new ConfWebuser();
		wuCR.setDbUniqueId(cd.getUniqueId());
		wuCR.setDbuserId(du.getUniqueId());
		List<ConfWebuser> wuList = wuDAO.find(con, wuCR);
		for (ConfWebuser wu : wuList) {
			ConfWebuserPriv cr = new ConfWebuserPriv();
			cr.setWebuserId(wu.getUniqueId());
			wupDAO.delete(con, cr);
			wuCR = new ConfWebuser();
			wuCR.setUniqueId(wu.getUniqueId());
			wuDAO.delete(con, wuCR);
		}
		duCR = new ConfDbuser();
		duCR.setUniqueId(du.getUniqueId());
		duDAO.delete(con, duCR);
	}

	protected synchronized Set<String> addPrivileges(Connection con,
			DBConfig dbConf, String webUserName, List<String> privNames)
			throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfPrivilegeDAO privDAO = DAOFactory.createConfPrivilegeDAO(dbID);
		ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(dbID);
		ConfWebuserPrivDAO wupDAO = DAOFactory.createConfWebuserPrivDAO(dbID);

		ConfDatabase cd = getDatabaseRec(con, dbConf);
		if (cd == null) {
			return null;
		}

		ConfPrivilege cpCR = new ConfPrivilege();
		cpCR.setProjectSpecific(Boolean.FALSE);
		List<ConfPrivilege> npPrivList = privDAO.find(con, cpCR);
		Set<String> npPrivSet = new HashSet<String>(7);
		for (ConfPrivilege npp : npPrivList) {
			npPrivSet.add(npp.getName());
		}
		npPrivList = null;

		ConfWebuser wuCR = new ConfWebuser();
		wuCR.setDbUniqueId(cd.getUniqueId());
		wuCR.setName(webUserName);
		List<ConfWebuser> wuList = wuDAO.find(con, wuCR);
		if (wuList.isEmpty()) {
			// no web user matching (should not happen, indicates a bug or
			// tampering with db outside the web app)
			System.err.println("Found no ConfWebuser matching " + wuCR);
			return null;
		}
		Assertion.assertTrue(wuList.size() == 1);
		Databaseuser d = getDatabaseUser(con);
		ConfWebuserPriv wup = new ConfWebuserPriv();
		for (String privName : privNames) {
			if (!npPrivSet.contains(privName)) {
				continue;
			}
			BigDecimal uniqueId = sequenceHelper.getNextUID(con,
					"nc_conf_webuser_priv", "uniqueId");
			wup.setPrivName(privName);
			wup.setWebuserId(wuList.get(0).getUniqueId());
			wup.setUniqueId(uniqueId);
			wup.setOwner(d.getUniqueid());
			wup.setModUser(d.getUniqueid());
			wup.setModTime(new Date());
			wupDAO.insert(con, wup);
		}

		return npPrivSet;
	}

	protected synchronized void addUpdateDatabase(Connection con,
			DBConfig dbConf) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfDatabaseDAO cdDAO = DAOFactory.createConfDatabaseDAO(dbID);
		Site site = getSiteByID(con, dbConf.getSiteID());

		StringBuilder sb = new StringBuilder(200);
		sb.append(dbConf.getDbURL());
		if (dbConf.getSiteURL() != null && dbConf.getSiteURL().length() > 0) {
			sb.append(';');
			sb.append(dbConf.getSiteURL());
		}
		ConfDatabase cd = getDatabaseRec(con, dbConf.getId());
		if (cd == null) {
			cd = new ConfDatabase();
			cd.setDbId(dbConf.getId());
			cd.setDbType(dbConf.getDbType());
			cd.setDbUrl(sb.toString());
			cd.setIsDefault(dbConf.isDefaultDB());
			cd.setForceSchemaOwnerCheck(new Boolean(false));
			cd.setSiteUniqueId(site.getUniqueid());
			BigDecimal uniqueId = sequenceHelper.getNextUID(con,
					"nc_conf_database", "uniqueId");

			cd.setUniqueId(uniqueId);
			Databaseuser d = getDatabaseUser(con);
			cd.setModUser(d.getUniqueid());
			cd.setOwner(d.getUniqueid());
			cd.setModTime(new Date());
			cdDAO.insert(con, cd);
		} else {
			ConfDatabase bean = new ConfDatabase();
			bean.setDbId(dbConf.getId());
			bean.setDbType(dbConf.getDbType());
			bean.setDbUrl(sb.toString());
			bean.setIsDefault(dbConf.isDefaultDB());
			ConfDatabase cdCR = new ConfDatabase();
			cdCR.setUniqueId(cd.getUniqueId());
			cdDAO.update(con, bean, cdCR);
		}
	}

	protected synchronized void removeDatabase(Connection con, DBConfig dbConf)
			throws Exception {
		ConfDatabaseDAO cdDAO = DAOFactory.createConfDatabaseDAO(dbID);
		ConfDbuserDAO duDAO = DAOFactory.createConfDBuserDAO(dbID);
		ConfDatabase cd = getDatabaseRec(con, dbConf.getId());
		if (cd == null) {
			// no db conf to delete
			return;
		}
		ConfDbuser cr = new ConfDbuser();
		cr.setDbUniqueId(cd.getUniqueId());
		List<ConfDbuser> duList = duDAO.find(con, cr);
		for (ConfDbuser du : duList) {
			removeDBUser(con, dbConf, du.getName());
		}
		ConfDatabase cdCR = new ConfDatabase();
		cdCR.setUniqueId(cd.getUniqueId());
		cdDAO.delete(con, cdCR);
	}

	protected synchronized ConfDbuser addUpdateDBUser(Connection con,
			User dbUser, DBConfig dbConf) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfDbuserDAO duDAO = DAOFactory.createConfDBuserDAO(dbID);

		ConfDatabase cd = getDatabaseRec(con, dbConf);

		ConfDbuser cr = new ConfDbuser();
		cr.setName(dbUser.getName());
		cr.setDbUniqueId(cd.getUniqueId());
		List<ConfDbuser> duList = duDAO.find(con, cr);
		if (duList.isEmpty()) {
			ConfDbuser du = new ConfDbuser();
			du.setName(dbUser.getName());
			du.setPwd(dbUser.getPwd());
			du.setDbUniqueId(cd.getUniqueId());
			Databaseuser d = getDatabaseUser(con);
			du.setModUser(d.getUniqueid());
			du.setOwner(d.getUniqueid());
			du.setModTime(new Date());
			BigDecimal uniqueId = sequenceHelper.getNextUID(con,
					"nc_conf_dbuser", "uniqueId");
			du.setUniqueId(uniqueId);
			log.info("inserting " + du);
			duDAO.insert(con, du);
			return du;
		} else {
			ConfDbuser du = duList.get(0);
			ConfDbuser bean = new ConfDbuser();
			bean.setPwd(dbUser.getPwd());
			ConfDbuser duCR = new ConfDbuser();
			duCR.setUniqueId(du.getUniqueId());
			duDAO.update(con, bean, duCR);
			return du;
		}
	}

	protected synchronized void addWebUser(Connection con, User webUser,
			DBConfig dbConf, ConfDbuser du) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(dbID);

		ConfDatabase cd = getDatabaseRec(con, dbConf);
		List<Privilege> privList = webUser.listPrivileges();
		// add new web user
		ConfWebuser wu = new ConfWebuser();
		wu.setName(webUser.getName());
		String digest = GenUtils.getMD5Digest(webUser.getPwd());
		wu.setPwd(digest);
		wu.setDbuserId(du.getUniqueId());
		wu.setDbUniqueId(cd.getUniqueId());
		Databaseuser dbUser = getDatabaseUser(con);
		wu.setOwner(dbUser.getUniqueid());
		wu.setModUser(dbUser.getUniqueid());
		wu.setModTime(new Date());
		if (webUser.isUseGSI()) {
			wu.setOptionFlags(USEGSI);
		}
		wu.setInvited(webUser.isInvited() ? Boolean.TRUE : Boolean.FALSE);
		// set sequence
		BigDecimal dbUniqueId = sequenceHelper.getNextUID(con,
				"nc_conf_webuser", "uniqueId");
		wu.setUniqueId(dbUniqueId);

		wuDAO.insert(con, wu);
		dbConf.addUser(webUser);
		// user privileges
		if (privList != null && !privList.isEmpty()) {
			updateUserPrivileges(con, wu, privList, dbConf);
		}
	}

	protected synchronized void addKeyer(Connection con, User webUser,
			DBConfig dbConf) throws Exception {
		Userclass ucCrit = new Userclass();
		ucCrit.setUserclass(Constants.USERCLASS_DATA_PROCESSING);
		UserclassDAO ucDAO = DAOFactory.createUserclassDAO(dbID);
		List<Userclass> ucList = ucDAO.find(con, ucCrit);
		Assertion.assertTrue(ucList.size() == 1);
		Userclass uc = ucList.get(0);

		Databaseuser duCrit = new Databaseuser();
		duCrit.setIsgroup(new Boolean(false));
		duCrit.setUserclass(uc.getUniqueid());
		duCrit.setName(webUser.getName().toUpperCase());

		DatabaseuserDAO duDAO = DAOFactory.createDatabaseuserDAO(dbID);
		List<Databaseuser> duList = duDAO.find(con, duCrit);
		if (duList.isEmpty()) {
			UserInfo ui = new UserInfo("admin", null, null);
			IDBUserManService dbums = ServiceFactory
					.getDBUserManService(this.dbID);
			User adminUser = dbConf.findUser("admin");
			dbums
					.insertDatabaseUser(con, this.dbID, webUser, uc, ui,
							adminUser);
		}
	}

	protected synchronized ConfWebuser addUpdateWebUser(Connection con,
			User webUser, DBConfig dbConf) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(dbID);

		ConfDatabase cd = getDatabaseRec(con, dbConf);
		List<Privilege> privList = webUser.listPrivileges();

		ConfDbuserDAO duDAO = DAOFactory.createConfDBuserDAO(dbID);
		ConfDbuser duCR = new ConfDbuser();
		duCR.setDbUniqueId(cd.getUniqueId());
		List<ConfDbuser> duList = duDAO.find(con, duCR);
		Map<String, ConfDbuser> duMap = new HashMap<String, ConfDbuser>(7);
		for (ConfDbuser du : duList) {
			duMap.put(du.getName(), du);
		}
		duList = null;

		ConfWebuser criteria = new ConfWebuser();
		criteria.setName(webUser.getName());
		ConfDbuser du = duMap.get(webUser.getDbUser().getName());

		criteria.setDbUniqueId(du.getDbUniqueId());
		List<ConfWebuser> wuList = wuDAO.find(con, criteria);
		if (wuList.isEmpty()) {
			// add new web user
			ConfWebuser wu = new ConfWebuser();
			wu.setName(webUser.getName());
			String digest = GenUtils.getMD5Digest(webUser.getPwd());
			wu.setPwd(digest);
			// store digest instead of the clear text pwd
			webUser.setPwd(digest);

			wu.setEmail(webUser.getEmail());
			wu.setDbuserId(du.getUniqueId());
			wu.setDbUniqueId(cd.getUniqueId());
			Databaseuser dbUser = getDatabaseUser(con);
			wu.setOwner(dbUser.getUniqueid());
			wu.setModUser(dbUser.getUniqueid());
			wu.setModTime(new Date());
			if (webUser.isUseGSI()) {
				wu.setOptionFlags(USEGSI);
			}
			wu.setInvited(webUser.isInvited() ? Boolean.TRUE : Boolean.FALSE);

			// set sequence
			BigDecimal dbUniqueId = sequenceHelper.getNextUID(con,
					"nc_conf_webuser", "uniqueId");
			wu.setUniqueId(dbUniqueId);

			wuDAO.insert(con, wu);
			// add a keyer record for the web user also
			addKeyer(con, webUser, dbConf);
			dbConf.addUser(webUser);
			// user privileges
			if (privList != null && !privList.isEmpty()) {
				updateUserPrivileges(con, wu, privList, dbConf);
			}
			return wu;
		} else {
			ConfWebuser theWU = null;
			for (ConfWebuser wu : wuList) {
				if (wu.getDbUniqueId().equals(cd.getUniqueId())) {
					theWU = wu;
					break;
				}
			}
			// only password, useGSI, email and invited flags can be changed
			ConfWebuser update = new ConfWebuser();
			if (GenUtils.isNotEmpty(webUser.getPwd())) {
				String digest = GenUtils.getMD5Digest(webUser.getPwd());
				update.setPwd(digest);

				// store digest instead of the clear text pwd
				webUser.setPwd(digest);
			}
			if (webUser.isUseGSI()) {
				update.setOptionFlags(USEGSI);
			} else {
				update.setOptionFlags("");
			}
			update.setInvited(webUser.isInvited() ? Boolean.TRUE
					: Boolean.FALSE);
			if (GenUtils.isNotEmpty(webUser.getEmail())) {
				update.setEmail(webUser.getEmail());
			}

			ConfWebuser wuCR = new ConfWebuser();
			wuCR.setUniqueId(theWU.getUniqueId());
			wuDAO.update(con, update, wuCR);

			if (privList != null && !privList.isEmpty()) {
				updateUserPrivileges(con, theWU, privList, dbConf);
			}
			return theWU;
		}

	}

	protected void updateUserPrivileges(Connection con, ConfWebuser wu,
			List<Privilege> privList, DBConfig dbConf) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfWebuserPrivDAO wupDAO = DAOFactory.createConfWebuserPrivDAO(dbID);
		ConfWebuserPriv cr = new ConfWebuserPriv();
		cr.setWebuserId(wu.getUniqueId());
		List<ConfWebuserPriv> wupList = wupDAO.find(con, cr);
		if (!wupList.isEmpty()) {
			wupDAO.delete(con, cr);
		}
		for (Privilege priv : privList) {
			ConfWebuserPriv wup = new ConfWebuserPriv();
			wup.setPrivName(priv.getName());
			wup.setWebuserId(wu.getUniqueId());
			BigDecimal uniqueId = sequenceHelper.getNextUID(con,
					"nc_conf_webuser_priv", "uniqueId");
			wup.setUniqueId(uniqueId);
			Databaseuser dbUser = getDatabaseUser(con);
			wup.setModUser(dbUser.getUniqueid());
			wup.setOwner(dbUser.getUniqueid());
			wup.setModTime(new Date());
			wupDAO.insert(con, wup);
		}
	}

	protected synchronized ConfRole addRole(Connection con, String role,
			String descr) throws Exception {
		ConfRole criterion = new ConfRole();
		criterion.setRole(role);

		ConfRoleDAO dao = DAOFactory.createConfRoleDAO(dbID);
		List<ConfRole> list = dao.find(con, criterion);
		if (!list.isEmpty()) {
			throw new Exception("Role already exists in the database:" + role);
		}

		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);

		ConfRole cr = new ConfRole();
		cr.setRole(role);
		BigDecimal uniqueId = sequenceHelper.getNextUID(con, "nc_conf_role",
				"uniqueId");
		cr.setRoleId(uniqueId);
		cr.setDescr(descr);
		Databaseuser dbUser = getDatabaseUser(con);
		cr.setModUser(dbUser.getUniqueid());
		cr.setOwner(dbUser.getUniqueid());
		cr.setModTime(new Date());
		dao.insert(con, cr);
		return cr;
	}

	protected synchronized void addPrivileges2Role(Connection con,
			ConfRole role, List<ProjectPrivilege> ppList,
			List<Privilege> privList) throws Exception {
		ConfRolePrivsDAO dao = DAOFactory.createConfRolePrivsDAO(dbID);
		Date modTime = new Date();
		Databaseuser dbUser = getDatabaseUser(con);
		for (ProjectPrivilege pp : ppList) {
			ConfRolePrivs crp = new ConfRolePrivs();
			crp.setExpId(GenUtils.toBigDecimal(pp.getExpID()));
			crp.setPrivName(pp.getPrivilege().getName());
			crp.setRoleId(role.getRoleId());
			crp.setOwner(dbUser.getUniqueid());
			crp.setModUser(dbUser.getUniqueid());
			crp.setModTime(modTime);
			dao.insert(con, crp);
		}
		for (Privilege p : privList) {
			ConfRolePrivs crp = new ConfRolePrivs();
			crp.setExpId(GenUtils.toBigDecimal(-1));
			crp.setPrivName(p.getName());
			crp.setRoleId(role.getRoleId());
			crp.setOwner(dbUser.getUniqueid());
			crp.setModUser(dbUser.getUniqueid());
			crp.setModTime(modTime);
			dao.insert(con, crp);
		}
	}

	protected void updateUserProjPrivileges(Connection con, ConfWebuser wu,
			User webUser, DBConfig dbConf) throws Exception {
		ISequenceHelper sequenceHelper = MinimalServiceFactory
				.getSequenceHelper(dbID);
		ConfWebuserPrivDAO wupDAO = DAOFactory.createConfWebuserPrivDAO(dbID);
		ConfWebuserProjPrivDAO wuppDAO = DAOFactory
				.createConfWebuserProjPrivDAO(dbID);

		List<ConfPrivilege> privList = getProjectSpecPrivilegeList(con);
		Set<String> privSet = new HashSet<String>(7);
		for (ConfPrivilege p : privList) {
			privSet.add(p.getName());
		}
		privList = null;

		ConfWebuserPriv cr = new ConfWebuserPriv();
		cr.setWebuserId(wu.getUniqueId());

		List<ConfWebuserPriv> wupList = wupDAO.find(con, cr);
		if (!wupList.isEmpty()) {
			ConfWebuserProjPriv wuppCR = new ConfWebuserProjPriv();
			wuppCR.setWebuserId(wu.getUniqueId());
			wuppDAO.delete(con, wuppCR);
			for (String privName : privSet) {
				cr.setPrivName(privName);
				wupDAO.delete(con, cr);
			}
		}

		Databaseuser dbUser = getDatabaseUser(con);
		for (PrivilegeExperiments pe : webUser.getPrivExperiments()) {
			if (!pe.hasGranted()) {
				continue;
			}
			ConfWebuserPriv wup = new ConfWebuserPriv();
			wup.setPrivName(pe.getPrivilege());
			wup.setWebuserId(wu.getUniqueId());
			BigDecimal uniqueId = sequenceHelper.getNextUID(con,
					"nc_conf_webuser_priv", "uniqueId");
			wup.setUniqueId(uniqueId);

			wup.setModUser(dbUser.getUniqueid());
			wup.setOwner(dbUser.getUniqueid());
			wup.setModTime(new Date());
			wupDAO.insert(con, wup);
			Date modTime = new Date();
			for (ProjectPrivilegeStatus pps : pe.getPpStatusList()) {
				if (pps.isGranted()) {
					ConfWebuserProjPriv wupp = new ConfWebuserProjPriv();
					int expID = pps.getExpId();
					wupp.setExpId(GenUtils.toBigDecimal(expID));
					wupp.setWebuserId(wu.getUniqueId());
					wupp.setPrivName(pe.getPrivilege());
					wupp.setOwner(dbUser.getUniqueid());
					wupp.setModUser(dbUser.getUniqueid());
					wupp.setModTime(modTime);
					wuppDAO.insert(con, wupp);
				}
			}
		}
	}

	private List<ConfPrivilege> getProjectSpecPrivilegeList(Connection con)
			throws Exception {
		ConfPrivilegeDAO privDAO = DAOFactory.createConfPrivilegeDAO(dbID);
		ConfPrivilege cp = new ConfPrivilege();
		cp.setProjectSpecific(Boolean.TRUE);
		List<ConfPrivilege> privList = privDAO.find(con, cp);
		return privList;
	}

	protected ConfDatabase getDatabaseRec(Connection con, DBConfig dbConf)
			throws Exception {
		ConfDatabaseDAO cdDAO = DAOFactory.createConfDatabaseDAO(dbID);
		ConfDatabase cr = new ConfDatabase();
		cr.setDbId(dbConf.getId());
		List<ConfDatabase> list = cdDAO.find(con, cr);
		if (list.isEmpty() || list.size() != 1) {
			throw new Exception(
					"Exactly one database record is required for dbID:"
							+ dbConf.getId());
		}
		return list.get(0);
	}

	protected ConfDatabase getDatabaseRec(Connection con, String databaseId)
			throws Exception {
		ConfDatabaseDAO cdDAO = DAOFactory.createConfDatabaseDAO(dbID);
		ConfDatabase cr = new ConfDatabase();
		cr.setDbId(databaseId);
		List<ConfDatabase> list = cdDAO.find(con, cr);
		if (list.isEmpty())
			return null;
		return list.get(0);
	}

	protected Databaseuser getDatabaseUser(Connection con) throws Exception {
		ISQLDialect sqlDialect = MinimalServiceFactory.getSQLDialect(dbID);
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		List<?> results = tsp
				.executeQuery(con,
						"select u.* from Databaseuser as u where u.name ='ADMIN' and u.isgroup = false");
		Assertion.assertFalse(results.isEmpty());
		Databaseuser dbUser = (Databaseuser) results.get(0);
		return dbUser;
	}

	protected synchronized void retrieveUserDBAdminData(Connection con)
			throws Exception {
		ISQLDialect sqlDialect = MinimalServiceFactory.getSQLDialect(dbID);
		TSQLProcessor tsp = new TSQLProcessor(sqlDialect);
		ConfDatabaseDAO cdDAO = DAOFactory.createConfDatabaseDAO(dbID);
		ConfPrivilegeDAO privDAO = DAOFactory.createConfPrivilegeDAO(dbID);
		ConfDbuserDAO dbUserDAO = DAOFactory.createConfDBuserDAO(dbID);

		List<ConfPrivilege> privList = privDAO.find(con, new ConfPrivilege());
		for (ConfPrivilege priv : privList) {
			Privilege p = new Privilege(priv.getName(), priv
					.getProjectSpecific());
			p.setDescription(priv.getDescription());
			privilegesMap.put(p.getName(), p);
		}

		List<ConfDatabase> cdList = cdDAO.find(con, new ConfDatabase());
		Map<BigDecimal, ConfDatabase> cdMap = new HashMap<BigDecimal, ConfDatabase>(
				7);
		this.currentDBConfig = null;
		boolean first = true;
		for (ConfDatabase cd : cdList) {
			Site site = getSite(cd.getSiteUniqueId().toString());
			Assertion.assertNotNull(site);
			DBConfig dbConfig = new DBConfig(cd.getDbId(), site.getSitename(),
					site.getSiteid());

			String[] toks = cd.getDbUrl().split(";");
			String dbURL = null;
			String siteURL = null;
			Assertion.assertTrue(toks.length >= 1);
			dbURL = toks[0];
			if (toks.length == 2) {
				siteURL = toks[1];
			}
			dbConfig.setDbType(cd.getDbType());
			dbConfig.setDbURL(dbURL);
			dbConfig.setSiteURL(siteURL);
			dbConfig.setDefaultDB(cd.getIsDefault());
			dbConfigMap.put(dbConfig.getId(), dbConfig);
			cdMap.put(cd.getUniqueId(), cd);
			if (first) {
				this.currentDBConfig = dbConfig;
			}
			if (dbConfig.isDefaultDB()) {
				this.currentDBConfig = dbConfig;
			}
			first = false;
		}

		List<ConfDbuser> dbUserList = dbUserDAO.find(con, new ConfDbuser());
		Map<BigDecimal, User> dbUserMap = new HashMap<BigDecimal, User>();
		for (ConfDbuser cdu : dbUserList) {
			User dbUser = new User(cdu.getName(), cdu.getPwd(), null);
			ConfDatabase cd = cdMap.get(cdu.getDbUniqueId());
			DBConfig dbConfig = dbConfigMap.get(cd.getDbId());
			dbConfig.addDbUser(dbUser);
			dbUserMap.put(cdu.getUniqueId(), dbUser);
		}

		// get project specific privileges list
		List<ConfPrivilege> ppList = getProjectSpecPrivilegeList(con);
		Set<String> ppSet = new HashSet<String>(7);
		for (ConfPrivilege pp : ppList) {
			ppSet.add(pp.getName());
		}

		IDBCache dbCache = ServiceFactory.getDBCache(dbID);

		List<Experiment> experiments = dbCache.getExperiments(new UserInfo(
				Constants.ADMIN_USER, null, null), false);

		StringBuffer qBuf = new StringBuffer(128);
		qBuf
				.append("select w.*, p.* from ConfWebuser as w, ConfWebuserPriv as p where ");
		qBuf.append("w.uniqueId = p.webuserId order by w.name");
		List<?> results = tsp.executeQuery(con, qBuf.toString());
		Map<BigDecimal, User> webUserMap = new HashMap<BigDecimal, User>(7);
		AuthorizationHelper helper = new AuthorizationHelper(dbID);

		for (Iterator<?> iter = results.iterator(); iter.hasNext();) {
			Object[] row = (Object[]) iter.next();
			ConfWebuser wu = (ConfWebuser) row[0];
			ConfWebuserPriv wup = (ConfWebuserPriv) row[1];

			User webUser = webUserMap.get(wu.getUniqueId());
			// GenUtils.dumpList(new ArrayList( dbConfigMap.values()), log);
			if (webUser == null) {
				boolean useGSI = useGSI(wu.getOptionFlags());
				System.out.println(wu.getName() + " useGSI=" + useGSI);
				UserPrivilegeSet userPrivSet = helper.prepareUserPrivileges(wu
						.getName(), Constants.ADMIN_USER);
				webUser = new User(wu.getName(), wu.getPwd(), userPrivSet,
						useGSI);
				webUser.setInvited(wu.getInvited().booleanValue());
				webUser.setEmail(wu.getEmail());

				webUserMap.put(wu.getUniqueId(), webUser);
				ConfDatabase cd = cdMap.get(wu.getDbUniqueId());
				// System.out.println("cd:" + cd);

				DBConfig dbConfig = dbConfigMap.get(cd.getDbId());
				Assertion.assertNotNull(dbConfig);
				dbConfig.addUser(webUser);
				// add corresponding DBUser to webUser
				User dbUser = dbUserMap.get(wu.getDbuserId());
				Assertion.assertNotNull(dbUser);
				webUser.setDbUser(dbUser);

				webUser.setFullPrivilegesMap(privilegesMap);
				if (cd.getDbId().equals(this.dbID)) {
					this.poolService.addUser(webUser);
				}

				// also prepare project specific privileges
				prepProjectPrivData(con, wu, experiments, ppList, webUser);
			}
			Privilege priv = privilegesMap.get(wup.getPrivName());
			assert (priv != null);
			// only non-project specific privileges
			if (!ppSet.contains(wup.getPrivName())) {
				webUser.addPrivilege(priv);
			}
		}

		// get users without any privileges also
		ConfWebuserDAO wuDAO = DAOFactory.createConfWebuserDAO(dbID);
		List<ConfWebuser> wuList = wuDAO.find(con, new ConfWebuser());
		for (ConfWebuser wu : wuList) {
			if (webUserMap.get(wu.getUniqueId()) == null) {
				boolean useGSI = useGSI(wu.getOptionFlags());
				System.out.println("useGSI=" + useGSI);
				UserPrivilegeSet userPrivSet = helper.prepareUserPrivileges(wu
						.getName(), Constants.ADMIN_USER);
				User webUser = new User(wu.getName(), wu.getPwd(), userPrivSet,
						useGSI);
				webUser.setInvited(wu.getInvited().booleanValue());
				webUser.setEmail(wu.getEmail());

				ConfDatabase cd = cdMap.get(wu.getDbUniqueId());
				DBConfig dbConfig = dbConfigMap.get(cd.getDbId());
				Assertion.assertNotNull(dbConfig);
				dbConfig.addUser(webUser);

				User dbUser = dbUserMap.get(wu.getDbuserId());
				Assertion.assertNotNull(dbUser);
				webUser.setDbUser(dbUser);

				// also prepare project specific privileges
				prepProjectPrivData(con, wu, experiments, ppList, webUser);

				webUser.setFullPrivilegesMap(privilegesMap);
				if (cd.getDbId().equals(this.dbID)) {
					this.poolService.addUser(webUser);
				}
			}
		}

		// make sure all connection pools are initialized
		for (DBConfig dbConfig : dbConfigMap.values()) {
			DBPoolService.addDatabase(dbConfig);
		}
	}

	protected synchronized void prepProjectPrivData(Connection con,
			ConfWebuser wu, List<Experiment> experiments,
			List<ConfPrivilege> ppList, User webUser) throws Exception {

		ConfWebuserProjPrivDAO wuppDAO = DAOFactory
				.createConfWebuserProjPrivDAO(dbID);
		ConfWebuserProjPriv cr = new ConfWebuserProjPriv();
		cr.setWebuserId(wu.getUniqueId());
		List<ConfWebuserProjPriv> wuppList = wuppDAO.find(con, cr);
		Set<String> wuppSet = new HashSet<String>(17);
		for (ConfWebuserProjPriv wupp : wuppList) {
			StringBuilder sb = new StringBuilder();
			sb.append(wupp.getPrivName()).append('.');
			sb.append(wupp.getExpId());
			wuppSet.add(sb.toString());
		}

		List<PrivilegeExperiments> peList = new ArrayList<PrivilegeExperiments>(
				ppList.size());
		for (ConfPrivilege priv : ppList) {
			List<ProjectPrivilegeStatus> ppStatusList = new ArrayList<ProjectPrivilegeStatus>(
					experiments.size());
			for (Experiment exp : experiments) {
				StringBuilder sb = new StringBuilder();
				sb.append(priv.getName()).append('.');
				sb.append(exp.getUniqueid());
				String key = sb.toString();
				boolean granted = wuppSet.contains(key);
				ProjectPrivilegeStatus pps = new ProjectPrivilegeStatus(exp
						.getUniqueid().intValue(), granted, exp.getName());
				ppStatusList.add(pps);
			}
			PrivilegeExperiments pe = new PrivilegeExperiments(priv.getName(),
					ppStatusList);
			peList.add(pe);
		}
		webUser.setPrivExperiments(peList);
	}

	public static boolean useGSI(String optionFlags) {
		if (optionFlags == null || optionFlags.trim().isEmpty())
			return false;
		optionFlags = optionFlags.trim();
		String[] toks = optionFlags.split("::");
		for (String tok : toks) {
			if (tok.equalsIgnoreCase(USEGSI)) {
				return true;
			}
		}
		return false;
	}

	protected void handleErrorAndRollBack(Connection con, String msg,
			Exception x) throws BaseException {
		if (con != null) {
			try {
				con.rollback();
			} catch (SQLException se) {
				log.error("", se);
				throw new BaseException(se);
			}
		}
		log.error(msg, x);
		if (x instanceof BaseException)
			throw (BaseException) x;
		else
			throw new BaseException(x);
	}

}
