package clinical.web.vo;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;

import clinical.utils.DateTimeUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AssessmentResultSummary.java,v 1.1.2.1 2008/07/24 01:08:35
 *          bozyurt Exp $
 */
public class AssessmentResultSummary implements Serializable {
	private static final long serialVersionUID = -5870226531466636501L;
	private String subjectID;
	private String expID;
	private String expName = "unknown";
	private String siteID;
	private String timeStamp;
	private Map<String, VisitSegAsResultValues> vsarvMap = new LinkedHashMap<String, VisitSegAsResultValues>(
			3);
	private Set<String> uniqSiteIDs = new TreeSet<String>();
	private Map<String, String> paramMap = new HashMap<String, String>(3);

	public AssessmentResultSummary(String subjectID, String expID,
			String expName, String siteID, String timeStamp) {
		super();
		this.subjectID = subjectID;
		this.expID = expID;
		this.expName = expName;
		this.siteID = siteID;
		this.timeStamp = timeStamp;
	}

	public String getKey() {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(":").append(expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}

	public static String getKey(String subjectID, int expID, String siteID) {
		StringBuilder sb = new StringBuilder();
		sb.append(subjectID).append(":").append(expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}
	
	public Set<Integer> getUniqueVisitIdSet() {
		Set<Integer> visitIdSet = new HashSet<Integer>(7);
		for(VisitSegAsResultValues vsarv : vsarvMap.values()) {
			visitIdSet.add( vsarv.getVisitID() );
		}
		return visitIdSet;
	}
	
	public boolean hasSameTimeStamp(String timestamp) {
		for(VisitSegAsResultValues vsarv : vsarvMap.values()) {
			if (vsarv.getTimeStamp().equals(timestamp)) {
				return true;
			}
		}
		return false;
	}

	public VisitSegAsResultValues add(int visitID, int segmentID,
			int experimentID, String siteID) {
		if (!this.expID.equals(String.valueOf(experimentID))) {
			throw new RuntimeException("Invalid expID:" + experimentID
					+ " Expected:" + this.expID);
		}
		String key = getKey(visitID, segmentID, experimentID, siteID);
		VisitSegAsResultValues vsarv = new VisitSegAsResultValues(visitID,
				segmentID, experimentID, siteID);
		vsarvMap.put(key, vsarv);
		if (siteID != null)
			uniqSiteIDs.add(siteID);
		return vsarv;
	}

	public VisitSegAsResultValues getOrAdd(int visitID, int segmentID,
			int experimentID, String siteID) {
		String key = getKey(visitID, segmentID, experimentID, siteID);
		VisitSegAsResultValues vsarv = vsarvMap.get(key);
		if (vsarv == null) {
			vsarv = new VisitSegAsResultValues(visitID, segmentID,
					experimentID, siteID);
			vsarvMap.put(key, vsarv);
			if (siteID != null)
				uniqSiteIDs.add(siteID);

		}
		return vsarv;
	}

	public VisitSegAsResultValues get(int visitID, int segmentID, int expID,
			String siteID, String asID, String scoreName) {
		String key = getKey(visitID, segmentID, expID, siteID);
		return vsarvMap.get(key);
	}
	
	

	public Set<ScoreMetaData> getScoreMetadata() {
		Set<ScoreMetaData> uniqSet = new HashSet<ScoreMetaData>(17);
		for (VisitSegAsResultValues vsarv : vsarvMap.values()) {
			uniqSet.addAll(vsarv.getScoreMetadata());
		}
		return uniqSet;
	}

	/**
	 * if there are matching score values for more than one visit, then use the
	 * newest visit.
	 * 
	 * @param varList
	 * @return keyed by scoreName ':' asName ':' visitID
	 */
	public Map<String, String> getMatchingValues(List<ScoreMetaData> varList) {
		Map<String, String> map = new HashMap<String, String>(17);
		VisitSegAsResultValues newest = null;
		Date newestDate = null;
		for (VisitSegAsResultValues vsarv : vsarvMap.values()) {
			if (newest == null) {
				newest = vsarv;
				newestDate = DateTimeUtils.toDate(vsarv.getTimeStamp());
			} else {
				if (newestDate
						.before(DateTimeUtils.toDate(vsarv.getTimeStamp()))) {
					newest = vsarv;
					newestDate = DateTimeUtils.toDate(vsarv.getTimeStamp());
				}
			}
		}

		Map<String, SubjectAsScoreValueSummary> sasvsMap = newest
				.getScore2SavsMap();
		for (ScoreMetaData smd : varList) {
			String key = smd.getScoreName() + ":" + smd.getAsName();
			SubjectAsScoreValueSummary sasvs = sasvsMap.get(key);
			if (sasvs != null && sasvs.getValue() != null) {
				String aKey = smd.getScoreName() + ":" + smd.getAsName() + ":"
						+ sasvs.getVisitID();
				map.put(aKey, sasvs.getValue().toString());
			}
		}
		return map;
	}

	public static String getKey(int visitID, int segmentID, int expID,
			String siteID) {
		StringBuilder sb = new StringBuilder();
		sb.append(visitID).append(':').append(segmentID).append(':')
				.append(expID);
		sb.append(':').append(siteID);
		return sb.toString();
	}

	public Map<String, String> getParamMap() {
		if (paramMap.isEmpty()) {
			paramMap.put("sid", getSubjectID());
			paramMap.put("siteid", getSiteID());
		}
		return paramMap;
	}

	public String getAllSiteIDs() {
		StringBuilder buf = new StringBuilder();
		for (Iterator<String> it = uniqSiteIDs.iterator(); it.hasNext();) {
			String sid = it.next();
			buf.append(sid);
			if (it.hasNext())
				buf.append(',');
		}
		return buf.toString();
	}

	public String getSubjectID() {
		return subjectID;
	}

	public String getExpID() {
		return expID;
	}

	public String getExpName() {
		return expName;
	}

	public String getSiteID() {
		return siteID;
	}

	public String getTimeStamp() {
		return timeStamp;
	}

	public Set<String> getUniqSiteIDs() {
		return uniqSiteIDs;
	}

	public Map<String, VisitSegAsResultValues> getVsarvMap() {
		return vsarvMap;
	}

	public List<VisitSegAsResultValues> getVsarvList() {
		return new ArrayList<VisitSegAsResultValues>(vsarvMap.values());
	}

	public String getParamJSON() {
		StringBuilder buf = new StringBuilder();
		buf.append("['subjectID=").append(subjectID).append("',");
		buf.append("'siteID=").append(siteID);
		buf.append("','expID=").append(expID).append("']");
		return buf.toString();
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("AssessmentResultSummary::[");
		sb.append("subjectID=").append(subjectID);
		sb.append(",expID=").append(expID);
		sb.append(",expName=").append(expName);
		sb.append(",siteID=").append(siteID);
		sb.append(",timeStamp=").append(timeStamp);
		for (VisitSegAsResultValues vsarv : vsarvMap.values()) {
			sb.append("\n\t").append(vsarv);
		}
		sb.append(']');
		return sb.toString();
	}
}
