package clinical.web.vo.upload;

import java.io.Serializable;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;

import clinical.utils.Assertion;
import clinical.web.CBFBIRNConstants;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: VisitInfo.java 783 2013-03-15 23:01:31Z bozyurt $
 */
public class VisitInfo implements Serializable {
	private static final long serialVersionUID = 1L;
	String name;
	String descr;
	int visitId;
	String subjectID;
	String visitType;
	int expId;
	Date visitDate;
	RGInfo rgInfo;
	String scannerType;
	String siteName;
	boolean hasAnat = false;
	boolean hasFieldMap = false;
	boolean has3DPCASL = false; // IBO 7/13/2015

	List<SegmentInfo> siList = new ArrayList<SegmentInfo>(10);
	List<AssessmentType> assessments = new ArrayList<AssessmentType>(1);
	List<String> extraFiles = new ArrayList<String>(1);
	static SimpleDateFormat sdf = new SimpleDateFormat("MM/dd/yyyy hh:mm");

	public VisitInfo(int visitId, String name, String descr, String subjectID,
			String visitType, int expId, Date visitDate, RGInfo rgInfo) {
		super();
		this.visitId = visitId;
		this.name = name;
		this.descr = descr;
		this.subjectID = subjectID;
		this.visitType = visitType;
		this.expId = expId;
		this.visitDate = visitDate;
		this.rgInfo = rgInfo;
	}

	public int getVisitId() {
		return visitId;
	}

	public String getName() {
		return name;
	}

	public String getDescr() {
		return descr;
	}

	public String getSubjectID() {
		return subjectID;
	}

	public String getVisitType() {
		return visitType;
	}

	public List<SegmentInfo> getSiList() {
		return siList;
	}

	public void addSI(SegmentInfo si) {
		siList.add(si);
	}

	public void addAssessment(AssessmentType at) {
		assessments.add(at);
	}

	public void addExtraFile(String extraFile) {
		this.extraFiles.add(extraFile);
	}

	public JSONObject toJSON() throws JSONException {
		JSONObject js = new JSONObject();
		js.put("visitId", visitId);
		js.put("name", name);
		js.put("descr", descr);
		js.put("subjectID", subjectID);
		js.put("expId", expId);
		js.put("visitType", visitType);

		js.put("visitDate", sdf.format(visitDate));
		if (rgInfo != null) {
			js.put("rgInfo", rgInfo.toJSON());
		}

		JSONArray arr = new JSONArray();
		for (SegmentInfo si : siList) {
			arr.put(si.toJSON());
		}
		js.put("siList", arr);

		arr = new JSONArray();
		for (AssessmentType at : assessments) {
			arr.put(at.toJSON());
		}
		js.put("assessments", arr);

		arr = new JSONArray();
		for (String ef : extraFiles) {
			arr.put(ef);
		}
		js.put("extraFiles", arr);

		if (scannerType != null) {
			js.put("scannerType", scannerType);
		}
		if (siteName != null) {
			js.put("siteName", siteName);
		}
		js.put("hasAnat", hasAnat);
		js.put("hasFM", hasFieldMap);
		// IBO 7/13/2015
		js.put("has3DPCASL", has3DPCASL);
		return js;
	}

	public static String getField(JSONObject js, String fieldName)
			throws JSONException {
		if (js.has(fieldName)) {
			return js.getString(fieldName);
		}
		return null;
	}

	public static VisitInfo initializeFromJSON(JSONObject js)
			throws JSONException, ParseException {
		int visitId = js.getInt("visitId");
		String name = getField(js, "name");
		String descr = getField(js, "descr");
		String subjectID = js.getString("subjectID");
		int expId = js.getInt("expId");
		String visitType = getField(js, "visitType");
		Date visitDate = sdf.parse(js.getString("visitDate"));
		RGInfo rgInfo = null;
		if (js.has("rgInfo")) {
			JSONObject jsObj = js.getJSONObject("rgInfo");
			rgInfo = RGInfo.initializeFromJSON(jsObj);
		}

		VisitInfo vi = new VisitInfo(visitId, name, descr, subjectID,
				visitType, expId, visitDate, rgInfo);
		JSONArray arr = js.getJSONArray("siList");
		for (int i = 0; i < arr.length(); i++) {
			JSONObject json = arr.getJSONObject(i);
			vi.addSI(SegmentInfo.initializeFromJSON(json));
		}
		if (js.has("assessments")) {
			arr = js.getJSONArray("assessments");
			for (int i = 0; i < arr.length(); i++) {
				JSONObject json = arr.getJSONObject(i);
				vi.addAssessment(AssessmentType.initializeFromJSON(json));
			}
		}

		if (js.has("extraFiles")) {
			arr = js.getJSONArray("extraFiles");
			for (int i = 0; i < arr.length(); i++) {
				vi.addExtraFile(arr.getString(i));
			}
		}

		vi.setSiteName(getField(js, "siteName"));
		vi.setScannerType(getField(js, "scannerType"));
		if (js.has("hasAnat")) {
			vi.hasAnat = js.getBoolean("hasAnat");
		} else {
			vi.hasAnat = false;
		}
		if (js.has("hasFM")) {
			vi.hasFieldMap = js.getBoolean("hasFM");
		} else {
			vi.hasFieldMap = false;
		}
		// IBO 7/13/2015
		if (js.has("has3DPCASL")) {
			vi.has3DPCASL = js.getBoolean("has3DPCASL");
		} else {
			vi.has3DPCASL = false;
		}
		return vi;
	}

	public void determineIfHasAnat() {
		for (SegmentInfo si : siList) {
			if (CBFBIRNConstants.ANATOMICAL.equals(si.getName())) {
				this.hasAnat = true;
				break;
			}
		}
	}

	public void determineIfHasFieldMap() {
		int fmCount = 0;
		for (SegmentInfo si : siList) {
			if (CBFBIRNConstants.FIELD_MAP.equals(si.getName())) {
				fmCount++;
			}
		}
		if (fmCount == 1 || fmCount == 2) {
			this.hasFieldMap = true;
		}
	}

	// IBO 7/13/2015
	public void determineIfIs3DPCASL() {
		this.has3DPCASL = false;
		for (SegmentInfo si : siList) {
			if (CBFBIRNConstants.D3PCASL.equals(si.getName())) {
				has3DPCASL = true;
				break;
			}
		}
	}

	public int getExpId() {
		return expId;
	}

	public RGInfo getRgInfo() {
		return rgInfo;
	}

	public Date getVisitDate() {
		return visitDate;
	}

	public List<AssessmentType> getAssessments() {
		return assessments;
	}

	public List<String> getExtraFiles() {
		return extraFiles;
	}

	public String toString() {
		StringBuilder sb = new StringBuilder();
		sb.append("VisitInfo::[");
		sb.append("subjectID:").append(subjectID);
		sb.append(",visitId:").append(visitId);
		sb.append(",visitDate:").append(visitDate);
		for (SegmentInfo si : siList) {
			sb.append("\n\t").append(si.toString());
		}
		if (!extraFiles.isEmpty()) {
			sb.append("\nExtra Files");
			for (String ef : extraFiles) {
				sb.append("\n\t").append(ef);
			}
		}
		sb.append("]");

		return sb.toString();
	}

	public String getScannerType() {
		return scannerType;
	}

	public void setScannerType(String scannerType) {
		this.scannerType = scannerType;
	}

	public String getSiteName() {
		return siteName;
	}

	public void setSiteName(String siteName) {
		this.siteName = siteName;
	}

	public boolean isHasAnat() {
		return hasAnat;
	}

	public boolean isHasFieldMap() {
		return hasFieldMap;
	}

	public void setHasFieldMap(boolean hasFieldMap) {
		this.hasFieldMap = hasFieldMap;
	}

	public static VisitInfo initializeFromJSON(JSONObject js,
			Map<String, SeriesHeaderInfo> shiMap) throws JSONException,
			ParseException {
		int visitId = js.getInt("visitId");
		String name = getField(js, "name");
		String descr = getField(js, "descr");
		String subjectID = js.getString("subjectID");
		int expId = js.getInt("expId");
		String visitType = getField(js, "visitType");
		Date visitDate = sdf.parse(js.getString("visitDate"));
		RGInfo rgInfo = null;
		if (js.has("rgInfo")) {
			JSONObject jsObj = js.getJSONObject("rgInfo");
			rgInfo = RGInfo.initializeFromJSON(jsObj);
		}

		VisitInfo vi = new VisitInfo(visitId, name, descr, subjectID,
				visitType, expId, visitDate, rgInfo);
		JSONArray arr = js.getJSONArray("siList");
		for (int i = 0; i < arr.length(); i++) {
			JSONObject json = arr.getJSONObject(i);
			String seriesName = json.getString("seriesName");
			SeriesHeaderInfo shi = shiMap.get(seriesName);
			Assertion.assertNotNull(shi);
			vi.addSI(SegmentInfo.initializeFromJSON(json, shi));
		}
		if (js.has("assessments")) {
			arr = js.getJSONArray("assessments");
			for (int i = 0; i < arr.length(); i++) {
				JSONObject json = arr.getJSONObject(i);
				vi.addAssessment(AssessmentType.initializeFromJSON(json));
			}
		}

		if (js.has("extraFiles")) {
			arr = js.getJSONArray("extraFiles");
			for (int i = 0; i < arr.length(); i++) {
				vi.addExtraFile(arr.getString(i));
			}
		}

		vi.setSiteName(getField(js, "siteName"));
		vi.setScannerType(getField(js, "scannerType"));
		if (js.has("hasAnat")) {
			vi.hasAnat = js.getBoolean("hasAnat");
		} else {
			vi.hasAnat = false;
		}
		if (js.has("hasFM")) {
			vi.hasFieldMap = js.getBoolean("hasFM");
		} else {
			vi.hasFieldMap = false;
		}
		// IBO 7/13/2015
		if (js.has("has3DPCASL")) {
			vi.has3DPCASL = js.getBoolean("has3DPCASL");
		} else {
			vi.has3DPCASL = false;
		}
		return vi;
	}

	// IBO 7/13/2015
	public boolean isHas3DPCASL() {
		return has3DPCASL;
	}

	public void setHas3DPCASL(boolean has3dpcasl) {
		has3DPCASL = has3dpcasl;
	}
}
