package clinical.web.workflow.cbf;

import java.io.File;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.List;

import org.json.JSONException;
import org.json.JSONObject;

import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.workflow.common.JSONUtils;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class AFNIBrik {
	File headerFile;
	File brikFile;
	String brikName;

	public AFNIBrik(File headerFile, File brikFile) {
		this.headerFile = headerFile;
		this.brikFile = brikFile;
		File file = (headerFile != null) ? headerFile : brikFile;
		Assertion.assertNotNull(file);
		this.brikName = file.getName().replaceFirst("\\.\\w+$", "");
	}

	public static AFNIBrik create(File file, AFNIBrik brik) {
		if (brik == null) {
			if (isBrik(file.getName())) {
				return new AFNIBrik(null, file);
			} else {
				return new AFNIBrik(file, null);
			}
		} else {
			if (isBrik(file.getName())) {
				return new AFNIBrik(brik.getHeaderFile(), file);
			} else {
				return new AFNIBrik(file, brik.getBrikFile());
			}
		}
	}

	public static AFNIBrik create(List<File> brikFiles) {
		Assertion.assertTrue(brikFiles.size() == 2);
		File firstFile = brikFiles.get(0);
		File secondFile = brikFiles.get(1);
		if (AFNIBrik.isHeader(firstFile.getName())) {
			return new AFNIBrik(firstFile, secondFile);
		} else {
			return new AFNIBrik(secondFile, firstFile);
		}
	}

	public JSONObject toJSON() throws JSONException {
		JSONObject js = new JSONObject();
		JSONUtils.addJSONField("headerFile", headerFile, js);
		JSONUtils.addJSONField("brikFile", brikFile, js);
		return js;
	}

	public static AFNIBrik initializeFromJSON(JSONObject js)
			throws JSONException, ParseException {
		File brikFile = JSONUtils.getFile("brikFile", js);
		File headerFile = JSONUtils.getFile("headerFile", js);
		return new AFNIBrik(headerFile, brikFile);

	}

	public File getHeaderFile() {
		return headerFile;
	}

	public File getBrikFile() {
		return brikFile;
	}

	public String getBrikName() {
		return brikName;
	}

	public String getSeriesName() {
		int idx = brikName.indexOf('+');
		Assertion.assertTrue(idx != -1);
		return brikName.substring(0, idx);
	}

	public static String getSeriesName(String filename) {
		int idx = filename.indexOf('+');
		Assertion.assertTrue(idx != -1);
		return filename.substring(0, idx);
	}

	public static boolean isHeader(String path) {
		return path.endsWith(".HEAD");
	}

	public static boolean isBrik(String path) {
		return path.endsWith(".BRIK");
	}

	public static boolean isAFNIBrikFile(String path) {
		return path.endsWith(".BRIK") || path.endsWith(".HEAD");
	}

	public List<File> copyTo(File destDir, boolean overwrite) throws Exception {
		File brikFile = getBrikFile();
		File headerFile = getHeaderFile();
		File brikDestFile = new File(destDir, brikFile.getName());
		File headDestFile = new File(destDir, headerFile.getName());
		if ((brikDestFile.exists() || headDestFile.exists()) && !overwrite) {
			return new ArrayList<File>(0);
		}
		FileUtils.copyFile(brikFile.getCanonicalPath(),
				brikDestFile.getCanonicalPath());
		FileUtils.copyFile(headerFile.getCanonicalPath(),
				headDestFile.getCanonicalPath());

		List<File> destFiles = new ArrayList<File>(2);
		destFiles.add(brikDestFile);
		destFiles.add(headDestFile);
		return destFiles;
	}

	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder(128);
		sb.append("AFNIBrik::[");
		sb.append("brik:").append(brikFile);
		sb.append("\n\thead:").append(headerFile);
		sb.append(']');
		return sb.toString();
	}
}