package clinical.web.workflow.cbf;

import static clinical.web.CBFBIRNConstants.ANATOMICAL;
import static clinical.web.CBFBIRNConstants.CSF;
import static clinical.web.CBFBIRNConstants.MINCON;
import static clinical.web.CBFBIRNConstants.TCAT_PATTERN;
import static clinical.web.CBFBIRNConstants.CSF_TCAT_BRIK_PREFIX;
import static clinical.web.CBFBIRNConstants.MATLAB_EXEC;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.velocity.Template;
import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;

import clinical.server.vo.Deriveddata;
import clinical.server.vo.Jobs;
import clinical.utils.Assertion;
import clinical.utils.DateTimeUtils;
import clinical.utils.Executor;
import clinical.utils.FileUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.IDerivedImageDataService;
import clinical.web.IJobManagementService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.exception.BaseException;
import clinical.web.helpers.CBFBirnHelper;
import clinical.web.scheduler.JobRecord;
import clinical.web.services.IJobProvenanceService;
import clinical.web.vo.DerivedDataInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamInfo;
import clinical.web.vo.upload.SegmentInfo;
import clinical.web.vo.upload.SeriesHeaderInfo;
import clinical.web.vo.upload.VisitInfo;

import com.pixelmed.dicom.Attribute;
import com.pixelmed.dicom.AttributeList;
import com.pixelmed.dicom.DicomFileUtilities;
import com.pixelmed.dicom.TagFromName;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id: AbstractCBFProcJobHelper.java 565 2012-03-15 00:13:42Z bozyurt
 *          $
 */
public class AbstractCBFProcJobHelper {
	protected CBFWFContext context;
	protected IDerivedImageDataService didService;
	protected IJobProvenanceService jpmService;
	protected UserInfo ui;
	protected List<BrikWrapper> brikInfoList = new ArrayList<BrikWrapper>(20);
	protected List<BrikWrapper> existingBrikInfoList = new ArrayList<BrikWrapper>(
			20);
	protected String templateDir;
	protected String envPath;
	protected Map<String, String> envMap = new HashMap<String, String>(17);
	protected static Log _log = LogFactory
			.getLog(AbstractCBFProcJobHelper.class);
	protected static final boolean TEST_MODE = false;
	private boolean canceled = false;

	protected AbstractCBFProcJobHelper(UserInfo ui, CBFWFContext context,
			String templateDir) throws BaseException {
		this.context = context;
		this.ui = ui;
		this.templateDir = templateDir;

		didService = ServiceFactory.getDerivedImageDataService(context
				.getDbID());
		jpmService = ServiceFactory.getJobProvenanceService(context.getDbID());
	}

	public static String getPrefix(File brikFile) {
		String prefix = brikFile.getName();
		prefix = prefix.replaceFirst("\\.[^\\.]+$", "");
		return prefix;
	}

	public String getEnvPath() {
		return envPath;
	}

	public void setEnvPath(String envPath) {
		this.envPath = envPath;
		System.out.println("envPath set:" + envPath);
	}

	public void addEnvironmentVar(String name, String value) {
		envMap.put(name, value);
	}

	protected void cleanupPreviousDerivedData() throws Exception {
		File rdRootFile = getRawDataRoot4Visit();
		Assertion.assertNotNull(rdRootFile);
		File derivedDir = new File(rdRootFile, CBFBIRNConstants.DERIVED);
		if (derivedDir.isDirectory()) {
			FileUtils.deleteRecursively(derivedDir);
		}
		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();
		int expId = vi.getExpId();
		int visitId = vi.getVisitId();
		// cleanup any previous derived data from the database
		didService.removeDerivedData(this.ui, subjectID, expId, visitId, null);
	}

	protected File copyBrikFilesTo(File brikFile, File afniDIR)
			throws IOException {
		File newBrikFile = new File(afniDIR, brikFile.getName());
		FileUtils.copyFile(brikFile.getAbsolutePath(),
				newBrikFile.getAbsolutePath());
		String headName = brikFile.getName();
		headName = headName.replaceFirst("\\.BRIK$", ".HEAD");
		File headFile = new File(brikFile.getParent(), headName);
		Assertion.assertTrue(headFile.exists());
		File newHeadFile = new File(afniDIR, headName);
		FileUtils.copyFile(headFile.getAbsolutePath(),
				newHeadFile.getAbsolutePath());

		return newBrikFile;
	}

	public static boolean hasAFNIExtension(String filename) {
		return (filename.endsWith(".BRIK") || filename.endsWith(".HEAD"));
	}

	public CBFProcessInfo handleEmbeddedCSFBrikSplit(CBFProcessInfo cpi,
			File matlabDir, File destDir) throws Exception {
		if (isCanceled()) {
			return null;
		}
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		String prefix = getPrefix(cpi.getCbfBrik());
		String aslBrikPath = new File(cpi.getCbfBrik().getParent(), prefix)
				.getAbsolutePath();
		ctx.put("cbf", aslBrikPath);
		ctx.put("matlabdir", matlabDir);
		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("gencsf.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(cpi.getOutputDir(), "gencsf.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		// at destination
		File csfBrikFile = null;
		File aslBrikFile = null;

		File[] files = cpi.getOutputDir().listFiles();
		for (File f : files) {
			if (!f.isFile()) {
				continue;
			}
			String name = f.getName();
			if (hasAFNIExtension(name)) {
				// "tcat+orig" "csfbriktcat+orig"
				if (name.indexOf(TCAT_PATTERN) != -1
						|| name.startsWith(CSF_TCAT_BRIK_PREFIX)) {
					File destFile = new File(destDir, f.getName());
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					if (name.startsWith(CSF_TCAT_BRIK_PREFIX)) {
						if (!AFNIBrik.isHeader(destFile.getAbsolutePath())) {
							csfBrikFile = destFile;
						}
					} else {
						if (!AFNIBrik.isHeader(destFile.getAbsolutePath())) {
							aslBrikFile = destFile;
						}
					}
				}
			}
		}
		CBFProcessInfo newCPI = new CBFProcessInfo(aslBrikFile, csfBrikFile,
				null, cpi.getCbfSI(), cpi.getCbfSI(), null, destDir);

		return newCPI;

	}

	public void handleCBFProcessingUsingResultFrom(CBFProcessInfo cpi,
			JobProvenanceInfo jpi, File matlabDir, boolean useFabber)
			throws Exception {
		File srcRootDir = new File(jpi.getDataURI());
		Assertion.assertTrue(srcRootDir.isDirectory());
		File srcResultDir = null;
		// Assumption: only one CBF is allowed per visit
		File[] files = srcRootDir.listFiles();
		for (File f : files) {
			if (f.isDirectory()) {
				srcResultDir = f;
				break;
			}
		}

		String cbfPrefix = getPrefix(cpi.getCbfBrik());
		cbfPrefix = cbfPrefix.replaceFirst("\\+orig$", "");

		// for fabber
		Set<String> excludedFromFinalOutputSet = new HashSet<String>();
		File srcInputRootDir = srcRootDir.getParentFile();
		files = srcInputRootDir.listFiles();
		for (File f : files) {
			if (f.isFile() && hasAFNIExtension(f.getName())
					&& f.getName().startsWith(cbfPrefix)) {
				File destFile = new File(cpi.getOutputDir(), f.getName());
				FileUtils.copyFile(f.getAbsolutePath(),
						destFile.getAbsolutePath());
				excludedFromFinalOutputSet.add(destFile.getAbsolutePath());
			}
		}

		Assertion.assertNotNull(srcResultDir);
		File srcIntermediateDir = new File(srcResultDir, "intermediate");
		Assertion.assertTrue(srcIntermediateDir.isDirectory());

		File processedDir = new File(cpi.getOutputDir(), "processed");
		processedDir.mkdir();
		Assertion.assertTrue(processedDir.isDirectory());
		File skullDir = new File(this.context.getCacheDir(), "skull_strip");

		File[] srcIntermediateFiles = srcIntermediateDir.listFiles();
		for (File f : srcIntermediateFiles) {
			if (!f.isFile()) {
				continue;
			}
			String name = f.getName();
			if (hasAFNIExtension(name)) {
				File destFile = new File(skullDir, f.getName());
				// FIXME name based registered anatomical BRIK detection is
				// brittle!!
				if (name.startsWith("anat+reg+orig")
						|| name.startsWith("anat_ASL+reg+orig")) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addRegAnatImage(destFile);
				} else if (name.startsWith(cbfPrefix)
						&& name.indexOf("+reg+orig") != -1) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addCbfRegImage(destFile);
				}
			}

			if (name.indexOf("+motion") != -1) {
				File destFile = new File(processedDir, f.getName());
				FileUtils.copyFile(f.getAbsolutePath(),
						destFile.getAbsolutePath());
			}

			// this file is required for fabber (9/12/2013)
			if (name.endsWith("CSFvalue.txt")) {
				File destFile = new File(cpi.getOutputDir(), f.getName());
				FileUtils.copyFile(f.getAbsolutePath(),
						destFile.getAbsolutePath());
				excludedFromFinalOutputSet.add(destFile.getAbsolutePath());
			}

		}

		// handling of intermediate files
		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		Assertion.assertTrue(intermediateDir.isDirectory());
		File derivedIntDir = createResultsDerivedDataDir(cpi, intermediateDir);

		for (File f : srcIntermediateFiles) {
			if (!f.isFile() || f.getName().endsWith(".m")) {
				continue;
			}
			File destFile = new File(derivedIntDir, f.getName());
			FileUtils.copyFile(f.getAbsolutePath(), destFile.getAbsolutePath());
		}

		File outputDir = cpi.getOutputDir();
		for (File f : srcResultDir.listFiles()) {
			if (f.isFile()) {
				String name = f.getName();
				if (hasAFNIExtension(name) || name.endsWith(".mat")) {
					File destFile = new File(outputDir, f.getName());
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			}
		}

		boolean fabberFlag = false;
		if (useFabber) {
			fabberFlag = context.isDoFabber();
		}
		handleGMCBFValueExtraction(cpi, matlabDir,
				context.isDoSkullStripping(), context.getGmThreshold(),
				context.getGaussianFilterParam(), fabberFlag);

		// handling of intermediate files

		File resultsDir = new File(this.context.getCacheDir(), "results");
		File derivedDir = createResultsDerivedDataDir(cpi, resultsDir);

		files = cpi.getOutputDir().listFiles();
		for (File f : files) {
			// skip any files used but not updated by fabber
			if (excludedFromFinalOutputSet.contains(f.getAbsolutePath())) {
				continue;
			}
			String name = f.getName();
			if (name.equals("fmri_struct.mat")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}

				cpi.setMatFile(resultFile);
			} else if (name.endsWith(".png")) {
				if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
						|| name.equals("motion.png")) {
					// image files summarizing the results
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
					cpi.addResultImage(resultFile);
				} else {
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						// just warn and copy anyway
						_log.warn("cbf2: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			} else if (name.equals("gm.txt")
					|| name.equals("process_summary.txt")
					|| name.equals("cbfbirn_path1_summary.txt")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
			} else if (name.startsWith("CBF+orig.")
					|| name.startsWith("MPPCASL_Mask+orig.")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move brik file:" + f.getName());
				}
			} else {
				// other intermediate files
				if (f.isFile() && !f.getName().endsWith(".m")) {
					if (f.getName().equals("process_summary.txt")
							|| f.getName().equals("cbfbirn_path1_summary.txt")) {
						File resultFile = new File(derivedDir, f.getName());
						FileUtils.copyFile(f.getAbsolutePath(),
								resultFile.getAbsolutePath());
						continue;
					}
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						// just warn and copy anyway
						_log.warn("cbf2: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			}
		}
		if (processedDir.isDirectory()) {
			File[] allFiles = processedDir.listFiles();
			// result files can be anywhere (seems to keep changing from version
			// to version of the MATLAB code) 5/18/2012
			for (File f : allFiles) {
				String name = f.getName();
				if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
						|| name.equals("motion.png")) {
					// image files summarizing the results
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
					cpi.addResultImage(resultFile);
				} else if (name.equals("gm.txt")
						|| name.equals("process_summary.txt")
						|| name.equals("cbfbirn_path1_summary.txt")) {
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
				}
			}

			for (File f : allFiles) {
				if (f.isFile() && !f.getName().endsWith(".m")) {
					File destFile = new File(derivedIntDir, f.getName());
					if (!destFile.exists()) {
						FileUtils.copyFile(f.getAbsolutePath(),
								destFile.getAbsolutePath());
					}
				}
			}

		}
		// also save mask files as intermediate information
		/*
		 * File[] allFiles = cpi.getBinaryMaskFile().listFiles(); for (File f :
		 * allFiles) { if (f.isFile() && f.getName().endsWith(".txt")) { File
		 * destFile = new File(derivedIntDir, f.getName());
		 * FileUtils.copyFile(f.getAbsolutePath(), destFile.getAbsolutePath());
		 * } }
		 */

		// register all intermediate files to their CBFProcessInfo object for
		// final sanctioned location move
		File[] intFiles = derivedIntDir.listFiles();

		Map<String, BrikInfo> brikInfoMap = new LinkedHashMap<String, BrikInfo>(
				17);
		for (File intFile : intFiles) {
			if (intFile.isFile()) {
				String basename = intFile.getName()
						.replaceFirst("\\.\\w+$", "");

				if (hasAFNIExtension(intFile.getName())) {
					BrikInfo bi = brikInfoMap.get(basename);
					if (bi == null) {
						bi = new BrikInfo(basename);
						brikInfoMap.put(basename, bi);
					}
					if (intFile.getName().endsWith(".HEAD")) {
						bi.setHeadFile(intFile);
					} else {
						bi.setBrikFile(intFile);
					}
				} else {
					SingleFileInfo sfi = new SingleFileInfo(intFile);
					cpi.addIntermediateFile(sfi);
				}
			}
		}
		for (BrikInfo bi : brikInfoMap.values()) {
			cpi.addIntermediateFile(bi);
		}

		// register CBF+orig, MPPCASL_Mask+orig briks and process_summary.txt
		// 5/18/2012
		File[] derivedDirFiles = derivedDir.listFiles();
		for (File derivedFile : derivedDirFiles) {
			String name = derivedFile.getName();
			if (derivedFile.isFile()
					&& (name.startsWith("CBF+orig.")
							|| name.startsWith("MPPCASL_Mask+orig.")
							|| name.equals("process_summary.txt") || name
								.equals("cbfbirn_path1_summary.txt"))) {
				SingleFileInfo sfi = new SingleFileInfo(derivedFile);
				cpi.addAdditionalResultFile(sfi);
			}
		}

		// directory cleanup
		if (!TEST_MODE) {
			FileUtils.deleteRecursively(cpi.getOutputDir());
			cpi.getOutputDir().mkdir();
		}

	}

	public void handleCBFProcessingStep1(CBFProcessInfo cpi, File matlabDir,
			boolean doPVC, int alignment, int csfMethod, boolean doFabber)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		File afniDir = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		ctx.put("brikdir", afniDir.getAbsolutePath());
		ctx.put("cbf", getPrefix(cpi.getCbfBrik()));
		ctx.put("csf", getPrefix(cpi.getCsfBrik()));
		ctx.put("mincon", getPrefix(cpi.getMinconBrik()));
		if (cpi.getAnatBrik() != null) {
			ctx.put("anat", getPrefix(cpi.getAnatBrik()));
			ctx.put("hasAnat", Boolean.TRUE);
		} else {
			ctx.put("anat", "");
			ctx.put("hasAnat", Boolean.FALSE);
		}
		ctx.put("outdir", cpi.getOutputDir());
		ctx.put("matlabdir", matlabDir);
		String pvcStr = doPVC ? "1" : "0";
		ctx.put("doPVC", pvcStr);
		ctx.put("align", alignment);
		ctx.put("csfMethod", csfMethod);
		ctx.put("fabber", doFabber);

		String siteName = "none";
		if (!context.getViList().isEmpty()) {
			siteName = context.getViList().get(0).getSiteName();
			if (siteName == null || siteName.trim().length() == 0) {
				siteName = "none";
			} else {
				siteName = siteName.toLowerCase();
			}
		}
		ctx.put("site", siteName);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("cbf1.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(cpi.getOutputDir(), "cbf1.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		// set .mat file to cpi

		File[] files = cpi.getOutputDir().listFiles();
		for (File f : files) {
			String name = f.getName();
			if (name.equals("interim.mat")) {
				cpi.setInterimMatFile(f);
			}
		}
		File processedDir = new File(cpi.getOutputDir(), "processed");
		Assertion.assertTrue(processedDir.isDirectory());
		File skullDir = new File(this.context.getCacheDir(), "skull_strip");

		String cbfPrefix = getPrefix(cpi.getCbfBrik());
		cbfPrefix = cbfPrefix.replaceFirst("\\+orig$", "");

		for (File f : processedDir.listFiles()) {
			if (!f.isFile()) {
				continue;
			}
			String name = f.getName();
			if (hasAFNIExtension(name)) {
				File destFile = new File(skullDir, f.getName());
				// FIXME name based registered anatomical BRIK detection is
				// brittle!!
				if (name.startsWith("anat+reg+orig")
						|| name.startsWith("anat_ASL+reg+orig")) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addRegAnatImage(destFile);
				} else if (name.startsWith(cbfPrefix)
						&& name.indexOf("+reg+orig") != -1) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addCbfRegImage(destFile);
				}
			}
		}

		// handling of intermediate files
		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		Assertion.assertTrue(intermediateDir.isDirectory());
		File derivedIntDir = createResultsDerivedDataDir(cpi, intermediateDir);
		Assertion.assertTrue(derivedIntDir.isDirectory());
		for (File f : processedDir.listFiles()) {
			if (!f.isFile() || f.getName().endsWith(".m")) {
				continue;
			}
			File destFile = new File(derivedIntDir, f.getName());
			if (destFile.exists()) {
				// just warn and copy anyway
				_log.warn("cbf1: intermediate file already exists:" + destFile);
			}
			FileUtils.copyFile(f.getAbsolutePath(), destFile.getAbsolutePath());

		}
	}

	public void handleCBFProcessingAutomated(CBFProcessInfo cpi,
			File matlabDir, boolean doSkullStripping, double gmThreshold,
			boolean doPartialVolumeCorrection, int alignment, int csfMethod,
			boolean useLTMethod, double smoothingParam, boolean doFabber)
			throws Exception {
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();
		File afniDir = new File(this.context.getCacheDir(),
				CBFBIRNConstants.AFNI_DIR_NAME);
		ctx.put("brikdir", afniDir.getAbsolutePath());
		ctx.put("cbf", getPrefix(cpi.getCbfBrik()));

		// IBO 07/16/2015
		if (CBFBirnHelper.is3DPCASL(cpi.getCbfSI().getProtocolId())) {
			ctx.put("csf","[]");
			ctx.put("mincon","[]");
		} else {
			ctx.put("csf", "'" + getPrefix(cpi.getCsfBrik()) + "'");
			ctx.put("mincon", "'" + getPrefix(cpi.getMinconBrik()) + "'");
		}
		if (cpi.getAnatBrik() != null) {
			ctx.put("anat", getPrefix(cpi.getAnatBrik()));
			ctx.put("hasAnat", Boolean.TRUE);
		} else {
			ctx.put("anat", "");
			ctx.put("hasAnat", Boolean.FALSE);
		}
		ctx.put("outdir", cpi.getOutputDir());
		ctx.put("matlabdir", matlabDir);

		String pvcStr = doPartialVolumeCorrection ? "1" : "0";
		String ltMethodStr = useLTMethod ? "1" : "0";
		ctx.put("pvc", pvcStr);
		ctx.put("align", alignment);
		ctx.put("csfMethod", csfMethod);
		ctx.put("ltMethod", ltMethodStr);
		ctx.put("fabber", doFabber);

		String siteName = null;
		if (!context.getViList().isEmpty()) {
			siteName = context.getViList().get(0).getSiteName();
			if (siteName == null || siteName.trim().length() == 0) {
				siteName = "none";
			} else {
				siteName = siteName.toLowerCase();
			}
		}

		ctx.put("site", siteName);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("cbf.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(cpi.getOutputDir(), "cbf.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash  < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		File processedDir = new File(cpi.getOutputDir(), "processed");
		Assertion.assertTrue(processedDir.isDirectory());
		File skullDir = new File(this.context.getCacheDir(), "skull_strip");

		String cbfPrefix = getPrefix(cpi.getCbfBrik());
		cbfPrefix = cbfPrefix.replaceFirst("\\+orig$", "");

		// make sure the necessary AFNI briks for skull stripping copied to
		// their expected locations
		for (File f : processedDir.listFiles()) {
			if (!f.isFile()) {
				continue;
			}
			String name = f.getName();
			if (hasAFNIExtension(name)) {
				File destFile = new File(skullDir, f.getName());
				if (name.startsWith("anat+reg+orig")) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addRegAnatImage(destFile);
				} else if (name.startsWith(cbfPrefix)
						&& name.indexOf("+reg+orig") != -1) {
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
					cpi.addCbfRegImage(destFile);
				}
			}
		}

		// write the GM CBF value to gm.txt file in the outputDir
		handleGMCBFValueExtraction(cpi, matlabDir, doSkullStripping,
				gmThreshold, smoothingParam, false);

		// handling of intermediate files
		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		Assertion.assertTrue(intermediateDir.isDirectory());

		File derivedIntDir = createResultsDerivedDataDir(cpi, intermediateDir);

		File resultsDir = new File(this.context.getCacheDir(), "results");
		File derivedDir = createResultsDerivedDataDir(cpi, resultsDir);

		File[] files = processedDir.listFiles();

		for (File f : files) {
			String name = f.getName();
			if (name.equals("fmri_struct.mat")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}

				cpi.setMatFile(resultFile);
			} else if (name.endsWith(".png")) {
				if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
						|| name.equals("motion.png")) {
					// image files summarizing the results
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
					cpi.addResultImage(resultFile);
				} else {
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						_log.warn("cbf: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			} else if (name.equals("gm.txt")
					|| name.equals("process_summary.txt")
					|| name.equals("cbfbirn_path1_summary.txt")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
			} else if (name.startsWith("CBF+orig.")
					|| name.startsWith("MPPCASL_Mask+orig.")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move brik file:" + f.getName());
				}
			} else {
				// other intermediate files
				if (f.isFile() && !f.getName().endsWith(".m")) {
					if (f.getName().equals("process_summary.txt")
							|| f.getName().equals("cbfbirn_path1_summary.txt")) {
						File resultFile = new File(derivedDir, f.getName());
						FileUtils.copyFile(f.getAbsolutePath(),
								resultFile.getAbsolutePath());
						continue;
					}
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						_log.warn("cbf: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			}
		}

		// result files can be anywhere (seems to keep changing from version to
		// version of the MATLAB code) 5/18/2012
		File[] outDirFiles = cpi.getOutputDir().listFiles();
		for (File f : outDirFiles) {
			String name = f.getName();
			if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
					|| name.equals("motion.png")) {
				// image files summarizing the results
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
				cpi.addResultImage(resultFile);
			} else if (name.equals("gm.txt")
					|| name.equals("process_summary.txt")
					|| name.equals("cbfbirn_path1_summary.txt")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
			}
		}

		// register all intermediate files to their CBFProcessInfo object for
		// final sanctioned location move
		File[] intFiles = derivedIntDir.listFiles();

		Map<String, BrikInfo> brikInfoMap = new LinkedHashMap<String, BrikInfo>(
				17);
		for (File intFile : intFiles) {
			if (intFile.isFile()) {
				String basename = intFile.getName()
						.replaceFirst("\\.\\w+$", "");

				if (hasAFNIExtension(intFile.getName())) {
					BrikInfo bi = brikInfoMap.get(basename);
					if (bi == null) {
						bi = new BrikInfo(basename);
						brikInfoMap.put(basename, bi);
					}
					if (intFile.getName().endsWith(".HEAD")) {
						bi.setHeadFile(intFile);
					} else {
						bi.setBrikFile(intFile);
					}
				} else {
					SingleFileInfo sfi = new SingleFileInfo(intFile);
					cpi.addIntermediateFile(sfi);
				}
			}
		}
		for (BrikInfo bi : brikInfoMap.values()) {
			cpi.addIntermediateFile(bi);
		}

		// register CBF+orig, MPPCASL_Mask+orig briks and process_summary.txt
		// 5/18/2012
		File[] derivedDirFiles = derivedDir.listFiles();
		for (File derivedFile : derivedDirFiles) {
			String name = derivedFile.getName();
			if (derivedFile.isFile()
					&& (name.startsWith("CBF+orig.")
							|| name.startsWith("MPPCASL_Mask+orig.")
							|| name.equals("process_summary.txt") || name
								.equals("cbfbirn_path1_summary.txt"))) {
				SingleFileInfo sfi = new SingleFileInfo(derivedFile);
				cpi.addAdditionalResultFile(sfi);
			}
		}

		// directory cleanup
		if (!TEST_MODE) {
			FileUtils.deleteRecursively(cpi.getOutputDir());
			cpi.getOutputDir().mkdir();
		}
	}

	public void handleCBFProcessingStep2(CBFProcessInfo cpi, File matlabDir,
			boolean doSkullStripping, double gmThreshold, boolean doPVC,
			int csfMethod, double smoothingParam, boolean doFabber)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		p.setProperty("runtime.log.logsystem.class",
				"org.apache.velocity.runtime.log.NullLogSystem");
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		ctx.put("matfile", cpi.getInterimMatFile());
		ctx.put("mask", cpi.getBinaryMaskFile());
		ctx.put("outdir", cpi.getOutputDir());
		ctx.put("matlabdir", matlabDir);
		ctx.put("sliceNoFile", cpi.getSelectedSliceFile());
		String pvcStr = doPVC ? "1" : "0";
		ctx.put("doPVC", pvcStr);
		ctx.put("csfMethod", csfMethod);
		ctx.put("fabber", doFabber);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("cbf2.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(cpi.getOutputDir(), "cbf2.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash  < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);

		// write the GM CBF value to gm.txt file in the outputDir
		handleGMCBFValueExtraction(cpi, matlabDir, doSkullStripping,
				gmThreshold, smoothingParam, false);

		// handling of intermediate files
		File intermediateDir = new File(this.context.getCacheDir(),
				"intermediate");
		Assertion.assertTrue(intermediateDir.isDirectory());

		File derivedIntDir = createResultsDerivedDataDir(cpi, intermediateDir);

		File resultsDir = new File(this.context.getCacheDir(), "results");
		File derivedDir = createResultsDerivedDataDir(cpi, resultsDir);

		File[] files = cpi.getOutputDir().listFiles();
		for (File f : files) {
			String name = f.getName();
			if (name.equals("fmri_struct.mat")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}

				cpi.setMatFile(resultFile);
			} else if (name.endsWith(".png")) {
				if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
						|| name.equals("motion.png")) {
					// image files summarizing the results
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
					cpi.addResultImage(resultFile);
				} else {
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						// just warn and copy anyway
						_log.warn("cbf2: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			} else if (name.equals("gm.txt")
					|| name.equals("process_summary.txt")
					|| name.equals("cbfbirn_path1_summary.txt")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move file:" + f.getName());
				}
			} else if (name.startsWith("CBF+orig.")
					|| name.startsWith("MPPCASL_Mask+orig.")) {
				File resultFile = new File(derivedDir, name);
				boolean ok = FileUtils.moveTo(f, resultFile);
				if (!ok) {
					throw new Exception("Cannot move brik file:" + f.getName());
				}
			} else {
				// other intermediate files
				if (f.isFile() && !f.getName().endsWith(".m")) {
					if (f.getName().equals("process_summary.txt")
							|| f.getName().equals("cbfbirn_path1_summary.txt")) {
						File resultFile = new File(derivedDir, f.getName());
						FileUtils.copyFile(f.getAbsolutePath(),
								resultFile.getAbsolutePath());
						continue;
					}
					File destFile = new File(derivedIntDir, f.getName());
					if (destFile.exists()) {
						// just warn and copy anyway
						_log.warn("cbf2: intermediate file already exists:"
								+ destFile);
					}
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			}
		}
		File processedDir = new File(cpi.getOutputDir(), "processed");
		if (processedDir.isDirectory()) {
			File[] allFiles = processedDir.listFiles();
			// result files can be anywhere (seems to keep changing from version
			// to version of the MATLAB code) 5/18/2012
			for (File f : allFiles) {
				String name = f.getName();
				if (name.equals("cbfmap.png") || name.equals("cbfhist.png")
						|| name.equals("motion.png")) {
					// image files summarizing the results
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
					cpi.addResultImage(resultFile);
				} else if (name.equals("gm.txt")
						|| name.equals("process_summary.txt")
						|| name.equals("cbfbirn_path1_summary.txt")) {
					File resultFile = new File(derivedDir, name);
					boolean ok = FileUtils.moveTo(f, resultFile);
					if (!ok) {
						throw new Exception("Cannot move file:" + f.getName());
					}
				}
			}

			for (File f : allFiles) {
				if (f.isFile() && !f.getName().endsWith(".m")) {
					File destFile = new File(derivedIntDir, f.getName());
					if (!destFile.exists()) {
						FileUtils.copyFile(f.getAbsolutePath(),
								destFile.getAbsolutePath());
					}
				}
			}

		}
		// also save mask files as intermediate information
		File[] allFiles = cpi.getBinaryMaskFile().listFiles();
		for (File f : allFiles) {
			if (f.isFile() && f.getName().endsWith(".txt")) {
				File destFile = new File(derivedIntDir, f.getName());
				FileUtils.copyFile(f.getAbsolutePath(),
						destFile.getAbsolutePath());
			}
		}

		// register all intermediate files to their CBFProcessInfo object for
		// final sanctioned location move
		File[] intFiles = derivedIntDir.listFiles();

		Map<String, BrikInfo> brikInfoMap = new LinkedHashMap<String, BrikInfo>(
				17);
		for (File intFile : intFiles) {
			if (intFile.isFile()) {
				String basename = intFile.getName()
						.replaceFirst("\\.\\w+$", "");

				if (hasAFNIExtension(intFile.getName())) {
					BrikInfo bi = brikInfoMap.get(basename);
					if (bi == null) {
						bi = new BrikInfo(basename);
						brikInfoMap.put(basename, bi);
					}
					if (intFile.getName().endsWith(".HEAD")) {
						bi.setHeadFile(intFile);
					} else {
						bi.setBrikFile(intFile);
					}
				} else {
					SingleFileInfo sfi = new SingleFileInfo(intFile);
					cpi.addIntermediateFile(sfi);
				}
			}
		}
		for (BrikInfo bi : brikInfoMap.values()) {
			cpi.addIntermediateFile(bi);
		}

		// register CBF+orig, MPPCASL_Mask+orig briks and process_summary.txt
		// 5/18/2012
		File[] derivedDirFiles = derivedDir.listFiles();
		for (File derivedFile : derivedDirFiles) {
			String name = derivedFile.getName();
			if (derivedFile.isFile()
					&& (name.startsWith("CBF+orig.")
							|| name.startsWith("MPPCASL_Mask+orig.")
							|| name.equals("process_summary.txt") || name
								.equals("cbfbirn_path1_summary.txt"))) {
				SingleFileInfo sfi = new SingleFileInfo(derivedFile);
				cpi.addAdditionalResultFile(sfi);
			}
		}

		// directory cleanup
		if (!TEST_MODE) {
			FileUtils.deleteRecursively(cpi.getOutputDir());
			cpi.getOutputDir().mkdir();
		}
	}

	protected File createResultsDerivedDataDir(CBFProcessInfo cpi,
			File resultsDir) {
		String srcPrefix = cpi.getCbfBrik().getName();
		srcPrefix = srcPrefix.replaceFirst("\\.\\w+$", "");
		File derivedDir = new File(resultsDir, srcPrefix);
		derivedDir.mkdir();
		return derivedDir;
	}

	protected File findFmriStructMatFileDir(CBFProcessInfo cpi)
			throws Exception {
		File outputDir = cpi.getOutputDir();
		File matFile = null;
		File[] files = outputDir.listFiles();
		for (File f : files) {
			String name = f.getName();
			if (name.equals("fmri_struct.mat")) {
				matFile = f;
				break;
			}
		}
		if (matFile != null) {
			return outputDir;
		}
		// try processed directory if exists
		File processedDir = new File(cpi.getOutputDir(), "processed");
		if (processedDir.isDirectory()) {
			files = processedDir.listFiles();
			for (File f : files) {
				String name = f.getName();
				if (name.equals("fmri_struct.mat")) {
					matFile = f;
					break;
				}
			}
		}
		if (matFile != null) {
			return processedDir;
		} else {
			throw new Exception("Cannot find fmri_struct.mat!");
		}

	}

	public void handleGMCBFValueExtraction(CBFProcessInfo cpi, File matlabDir,
			boolean doSkullStripping, double gmThreshold,
			double smoothingParam, boolean doFabber) throws Exception {
		if (isCanceled()) {
			return;
		}
		if (cpi.getAnatBrik() == null) {
			_log.info("needs anatomical mask for GM CBF "
					+ "calculation. Will skip GM CBF calculation.");
		}
		Properties p = new Properties();
		p.setProperty("file.resource.loader.path", templateDir);
		Velocity.init(p);
		VelocityContext ctx = new VelocityContext();

		File outputDir = findFmriStructMatFileDir(cpi);

		File[] files = outputDir.listFiles();

		File matFile = null;

		for (File f : files) {
			String name = f.getName();
			if (name.equals("fmri_struct.mat")) {
				matFile = f;
				break;
			}
		}
		if (matFile == null) {
			// should not happen
			_log.warn("handleGMCBFValueExtraction:: no fmri_struct.mat "
					+ "file under " + cpi.getOutputDir());
			return;
		}

		if (doSkullStripping) {
			// check if anat+reg and asl+reg briks are there
			if (!cpi.canDoSkullStripping()) {
				doSkullStripping = false;
				// always copy ant and cbf reg briks to the output directory
				// from the processed dir (10/04/2012)
				for (File f : cpi.getAnatRegImageFiles()) {
					String name = f.getName();
					File destFile = new File(outputDir, name);
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
				for (File f : cpi.getCbfRegImageFiles()) {
					String name = f.getName();
					File destFile = new File(outputDir, name);
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			} else {
				for (File f : cpi.getAnatRegImageFiles()) {
					String name = f.getName();
					File destFile = new File(outputDir, name);
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
				for (File f : cpi.getCbfRegImageFiles()) {
					String name = f.getName();
					File destFile = new File(outputDir, name);
					FileUtils.copyFile(f.getAbsolutePath(),
							destFile.getAbsolutePath());
				}
			}
		}

		String ssCodeStr = doSkullStripping ? "1" : "0";

		ctx.put("matfile", matFile);
		ctx.put("matlabdir", matlabDir);
		ctx.put("outdir", outputDir.getAbsolutePath());
		ctx.put("gmthresh", String.valueOf(gmThreshold));
		ctx.put("skullStripping", ssCodeStr);
		ctx.put("smoothingParam", String.valueOf(smoothingParam));
		// for fabber 9/12/2013
		ctx.put("fabber", doFabber);

		StringWriter sw = new StringWriter();
		Template template = Velocity.getTemplate("getgmcbf.vm");
		template.merge(ctx, sw);

		System.out.println(sw.toString());
		File matlabDriverScriptFile = new File(outputDir, "getgmcbf.m");
		matlabDriverScriptFile.delete();
		FileUtils.save2File(sw.toString(),
				matlabDriverScriptFile.getAbsolutePath());

		Executor executor = new Executor(MATLAB_EXEC, true);
		String cmdLine = "-nodesktop -nosplash  < "
				+ matlabDriverScriptFile.getAbsolutePath();

		runProcess(executor, cmdLine);
	}

	protected void cleanDirectory(File rootDir) {
		File[] files = rootDir.listFiles();
		for (File f : files) {
			if (f.isFile()) {
				f.delete();
			}
		}
	}

	protected void runProcess(Executor executor, String cmdLine)
			throws Exception {
		if (isCanceled()) {
			return;
		}
		executor.addEnvParam("PATH", this.envPath);
		for (String ev : this.envMap.keySet()) {
			String value = envMap.get(ev);
			executor.addEnvParam(ev, value);
		}
		setMatlabPrefDirEnvVar(executor);
		executor.showOutput(true);
		executor.execute(cmdLine);

		String output = executor.getOutput();
		System.out.println(output);
	}

	private void setMatlabPrefDirEnvVar(Executor executor) {
		File prefDir = new File(this.context.getCacheDir(), ".matlab");
		if (!prefDir.exists()) {
			prefDir.mkdirs();
		}
		Assertion.assertTrue(prefDir.exists() && prefDir.isDirectory());
		executor.addEnvParam("MATLAB_PREFDIR", prefDir.getAbsolutePath());
	}

	// multi-workflow support

	protected List<JobProvenanceInfo> getExistingWFResults() throws Exception {
		// find all with metadata (from DB) and legacy (no-metadata from FS)
		VisitInfo visitInfo = context.getViList().get(0);

		List<File> derivedDirectoriesForVisit = MultipleWFHelper
				.getWFDerivedDirectoriesForVisit(ui, context);

		if (derivedDirectoriesForVisit.isEmpty()) {
			return new ArrayList<JobProvenanceInfo>(0);
		}

		List<JobProvenanceInfo> jpiList = jpmService.findJobProvenances(ui,
				visitInfo.getExpId(), visitInfo.getSubjectID(),
				visitInfo.getVisitId());

		// in most cases corresponds to the legacy WF result
		File derivedDir = null;
		for (File ddf : derivedDirectoriesForVisit) {
			if (ddf.getName().equals(CBFBIRNConstants.DERIVED)) {
				derivedDir = ddf;
			}
		}
		Assertion.assertNotNull(derivedDir);

		if (jpiList.isEmpty()) {
			jpiList = new ArrayList<JobProvenanceInfo>(1);

			IJobManagementService jms = ServiceFactory
					.getJobManagementService(context.getDbID());

			List<JobRecord> jobsForVisit = jms.getSuccesfulJobsForVisit(
					this.ui, visitInfo.getExpId(), visitInfo.getSubjectID(),
					visitInfo.getVisitId());
			Assertion.assertTrue(!jobsForVisit.isEmpty());
			// find the latest job record to assign job provenance to
			JobRecord latestJR = null;
			for (JobRecord jr : jobsForVisit) {
				if (latestJR == null) {
					latestJR = jr;
				} else {
					if (latestJR.getDateFinished().before(jr.getDateFinished())) {
						latestJR = jr;
					}
				}
			}

			Jobs theJob = jms.getTheJob(ui, latestJR.getUser(),
					latestJR.getJobID());
			JobProvenanceInfo jpi = new JobProvenanceInfo("", "no param info",
					derivedDir.getAbsolutePath(), new Date(), theJob);

			// add Finish Date also as provenance data
			String value = DateTimeUtils.formatDate(theJob.getJobenddate());
			JobProvenanceParamInfo jppi = new JobProvenanceParamInfo(
					"Finish Date", value);
			jpi.addParam(jppi);

			jpmService.saveJobProvenance(ui, jpi);

			return jpmService.findJobProvenances(ui, visitInfo.getExpId(),
					visitInfo.getSubjectID(), visitInfo.getVisitId());
		}

		return jpiList;
	}

	/**
	 * The pattern of the derived data directories is (derived, derived.1, ....,
	 * derived.n)
	 * 
	 * @param visitJPIList
	 * @param createDir
	 *            if true create the directory also
	 * @return the next derived data directory to be used for the current WF run
	 */
	protected File getNextDerivedDir(List<JobProvenanceInfo> visitJPIList,
			boolean createDir) {
		File rdRootFile = getRawDataRoot4Visit();
		Assertion.assertNotNull(rdRootFile);
		File derivedDir = null;
		if (visitJPIList.isEmpty()) {
			derivedDir = new File(rdRootFile, CBFBIRNConstants.DERIVED);
		} else {
			derivedDir = new File(rdRootFile, CBFBIRNConstants.DERIVED + "."
					+ visitJPIList.size());
		}
		if (createDir) {
			derivedDir.mkdir();
		}
		return derivedDir;
	}

	public List<Deriveddata> saveDerivedData(VisitProcessInfo vpi,
			SegmentInfo fieldMap1SI, SegmentInfo fieldMap2SI) throws Exception {
		List<File> fmBrikFilesAvail = new ArrayList<File>(10);
		List<File> seBrikFilesAvail = new ArrayList<File>(10);
		List<DerivedDataInfo> ddiList = new ArrayList<DerivedDataInfo>();

		// orig
		/*
		 * File rdRootFile = getRawDataRoot4Visit();
		 * Assertion.assertNotNull(rdRootFile); File derivedDir = new
		 * File(rdRootFile, "derived");
		 * 
		 * derivedDir.mkdir();
		 */

		// multi-workflow support
		List<JobProvenanceInfo> jpiList = getExistingWFResults();

		File derivedDir = getNextDerivedDir(jpiList, true);

		VisitInfo vi = context.getViList().get(0);
		String subjectID = vi.getSubjectID();
		int expId = vi.getExpId();
		int visitId = vi.getVisitId();
		if (vpi.getFmList() != null) {
			for (FieldMapInfo fmi : vpi.getFmList()) {
				for (File f : fmi.getFmOutBrikFiles()) {
					File derivedFile = copy2DerivedDataLoc(f, derivedDir);
					DerivedDataInfo ddi = prepDerivedDataInfo(expId, visitId,
							subjectID, fmi.getPfileSI(), derivedFile);
					ddiList.add(ddi);
					fmBrikFilesAvail.add(derivedFile);
				}
				// FIXME duplicate check!!
				for (File f : fmi.getOutBrikFiles()) {
					File derivedFile = copy2DerivedDataLoc(f, derivedDir);
					DerivedDataInfo ddi = prepDerivedDataInfo(expId, visitId,
							subjectID, fmi.getPfileSI(), derivedFile);
					ddiList.add(ddi);
					fmBrikFilesAvail.add(derivedFile);
				}
			}
		}

		if (vpi.getSeList() != null) {
			for (SenseInfo se : vpi.getSeList()) {
				for (File f : se.getSenseOutBrikFiles()) {
					File derivedFile = copy2DerivedDataLoc(f, derivedDir);
					DerivedDataInfo ddi = prepDerivedDataInfo(expId, visitId,
							subjectID, se.getPfileSI(), derivedFile);
					ddiList.add(ddi);
					seBrikFilesAvail.add(derivedFile);
				}
			}
		}

		// also the supporting briks (if not existing) anat, mincon, csf,
		// fieldmap1 fieldmap2 briks
		boolean anatBrikHandled = false;
		boolean csfBrikHandled = false;
		boolean minconBrikHandled = false;
		if (!vpi.getCpiList().isEmpty()) {
			CBFProcessInfo cpi = vpi.getCpiList().get(0);

			if (!fmBrikFilesAvail.isEmpty()) {
				if (cpi.getAnatBrik() != null
						&& !isExistingBrik(cpi.getAnatBrik(), fmBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getAnatBrik(), cpi.getAnatSI(),
							derivedDir, subjectID, expId, visitId, ddiList);
					anatBrikHandled = true;
				}

				if (!isExistingBrik(cpi.getCsfBrik(), fmBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getCsfBrik(), cpi.getCsfSI(),
							derivedDir, subjectID, expId, visitId, ddiList);
					csfBrikHandled = true;
				}
				if (!isExistingBrik(cpi.getMinconBrik(), fmBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getMinconBrik(),
							cpi.getMinconSI(), derivedDir, subjectID, expId,
							visitId, ddiList);
					minconBrikHandled = true;
				}
			}

			if (!seBrikFilesAvail.isEmpty()) {
				if (cpi.getAnatBrik() != null
						&& !isExistingBrik(cpi.getAnatBrik(), seBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getAnatBrik(), cpi.getAnatSI(),
							derivedDir, subjectID, expId, visitId, ddiList);
					anatBrikHandled = true;
				}

				if (!isExistingBrik(cpi.getCsfBrik(), seBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getCsfBrik(), cpi.getCsfSI(),
							derivedDir, subjectID, expId, visitId, ddiList);
					csfBrikHandled = true;
				}
				if (!isExistingBrik(cpi.getMinconBrik(), seBrikFilesAvail)) {
					prepDDForSupportBrik(cpi.getMinconBrik(),
							cpi.getMinconSI(), derivedDir, subjectID, expId,
							visitId, ddiList);
					minconBrikHandled = true;
				}
			}
		}
		if (fieldMap1SI != null || fieldMap2SI != null) {
			for (BrikWrapper bi : this.brikInfoList) {
				if (bi.getSi() == fieldMap1SI || bi.getSi() == fieldMap2SI) {
					prepDDForSupportBrik(bi.getBrikFile(), bi.getSi(),
							derivedDir, subjectID, expId, visitId, ddiList);
				}
			}
		}

		// CBF processing derived data
		for (CBFProcessInfo cpi : vpi.getCpiList()) {
			String srcPrefix = cpi.getCbfBrik().getName();
			srcPrefix = srcPrefix.replaceFirst("\\.\\w+$", "");
			File cbfDerivedDir = new File(derivedDir, srcPrefix);
			cbfDerivedDir.mkdir();

			prepDDForCBFBrik(cpi, cbfDerivedDir, subjectID, expId, visitId,
					ddiList);
			if (!csfBrikHandled && !isExistingBrik(cpi.getCsfBrik(), null)) {
				prepDDForSupportBrik(cpi.getCsfBrik(), cpi.getCsfSI(),
						derivedDir, subjectID, expId, visitId, ddiList);
			}
			if (!minconBrikHandled
					&& !isExistingBrik(cpi.getMinconBrik(), null)) {
				prepDDForSupportBrik(cpi.getMinconBrik(), cpi.getMinconSI(),
						derivedDir, subjectID, expId, visitId, ddiList);
			}
			if (!anatBrikHandled && cpi.getAnatBrik() != null
					&& !isExistingBrik(cpi.getAnatBrik(), null)) {
				prepDDForSupportBrik(cpi.getAnatBrik(), cpi.getAnatSI(),
						derivedDir, subjectID, expId, visitId, ddiList);
			}
		}

		return saveDerivedDataWithJobProvenance(ddiList, derivedDir);
	}

	protected List<Deriveddata> saveDerivedDataWithJobProvenance(
			List<DerivedDataInfo> ddiList, File derivedDir) throws Exception {
		// prep JobProvenanceInfo and save with the derived data
		// FIXME name and description from the user for JobProvenanceInfo
		JobProvenanceInfo jpi = MultipleWFHelper.prepJobProvenanceInfo("", "",
				context);
		jpi.setDataURI(derivedDir.getAbsolutePath());

		Jobs theJob = this.jpmService.getTheJob(ui, context.getJobID(),
				ui.getPerceivedName());
		jpi.setJob(theJob);

		// add Finish Date also as provenance data
		String value = DateTimeUtils.formatDate(theJob.getJobenddate());
		JobProvenanceParamInfo jppi = new JobProvenanceParamInfo("Finish Date",
				value);
		jpi.addParam(jppi);

		List<Deriveddata> ddList = null;

		try {
			ddList = didService.addDerivedDataWithJobProvenance(ui, ddiList,
					jpi);

			// save provenance XML file to the derived dir
			File provenanceXml = new File(derivedDir, "provenance.xml");
			FileUtils.saveXML(jpi.toXML(), provenanceXml.getAbsolutePath());

		} catch (Throwable t) {
			if (derivedDir.isDirectory()) {
				FileUtils.deleteRecursively(derivedDir);
			}
			if (t instanceof Exception) {
				throw (Exception) t;
			} else {
				throw new Exception(t);
			}
		}
		return ddList;
	}

	protected boolean isExistingBrik(File brikFile, List<File> fmBrikFilesAvail) {
		for (BrikWrapper bi : this.existingBrikInfoList) {
			if (bi.getBrikFile().equals(brikFile)) {
				return true;
			}
		}
		if (fmBrikFilesAvail != null) {
			for (File bf : fmBrikFilesAvail) {
				if (bf.getName().equals(brikFile.getName())) {
					return true;
				}
			}
		}
		return false;
	}

	protected void prepDDForSupportBrik(File brikFile, SegmentInfo si,
			File derivedDir, String subjectID, int expId, int visitId,
			List<DerivedDataInfo> ddiList) throws IOException {
		String prefix = brikFile.getName().replaceFirst("\\.\\w+$", "");
		File[] files = brikFile.getParentFile().listFiles();
		List<File> afniFiles = new ArrayList<File>(2);
		for (File f : files) {
			if (f.getName().startsWith(prefix) && hasAFNIExtension(f.getName())) {
				afniFiles.add(f);
			}
		}
		Assertion.assertTrue(afniFiles.size() == 2);
		for (File f : afniFiles) {
			File derivedFile = copy2DerivedDataLoc(f, derivedDir);
			DerivedDataInfo ddi = prepDerivedDataInfo(expId, visitId,
					subjectID, si, derivedFile);
			ddiList.add(ddi);
		}
	}

	protected void prepDDForCBFBrik(CBFProcessInfo cpi, File derivedDir,
			String subjectID, int expId, int visitId,
			List<DerivedDataInfo> ddiList) throws IOException {
		File matFile = cpi.getMatFile();
		File derivedFile = copy2DerivedDataLoc(matFile, derivedDir);
		DerivedDataInfo ddi = prepDerivedDataInfo(expId, visitId, subjectID,
				cpi.getCbfSI(), derivedFile);
		ddiList.add(ddi);

		// handle gm.txt
		File gmFile = new File(matFile.getParent(), "gm.txt");
		if (gmFile.exists()) {
			copy2DerivedDataLoc(gmFile, derivedDir);
		}

		for (File resultImgFile : cpi.getResultImageFiles()) {
			File derivedImgFile = copy2DerivedDataLoc(resultImgFile, derivedDir);
			DerivedDataInfo iddi = prepDerivedDataInfo(expId, visitId,
					subjectID, cpi.getCbfSI(), derivedImgFile);
			ddiList.add(iddi);
		}
		// also add brik file if it is not an existing brik file as derived data
		boolean existing = false;
		for (BrikWrapper bi : this.existingBrikInfoList) {
			if (cpi.getCbfSI() == bi.getSi()) {
				existing = true;
			}
		}
		if (!existing) {
			prepDDForSupportBrik(cpi.getCbfBrik(), cpi.getCbfSI(), derivedDir,
					subjectID, expId, visitId, ddiList);
		}

		// include intermediate files generated by CBF process also (04/12/2011)
		if (!cpi.getIntermediateFiles().isEmpty()) {
			File intDir = new File(derivedDir, "intermediate");
			intDir.mkdir();
			Assertion.assertTrue(intDir.isDirectory());
			for (IFileInfo ifi : cpi.getIntermediateFiles()) {
				List<File> files = ifi.getFiles();
				for (File f : files) {
					File derivedDataFile = copy2DerivedDataLoc(f, intDir);
					DerivedDataInfo addi = prepDerivedDataInfo(expId, visitId,
							subjectID, cpi.getCbfSI(), derivedDataFile);
					ddiList.add(addi);
				}
			}
		}

		// include additional result files generated by the CBF process
		// (5/18/2012)
		if (!cpi.getAdditionalResultImageFiles().isEmpty()) {
			for (IFileInfo ifi : cpi.getAdditionalResultImageFiles()) {
				List<File> files = ifi.getFiles();
				for (File f : files) {
					File derivedDataFile = copy2DerivedDataLoc(f, derivedDir);
					DerivedDataInfo addi = prepDerivedDataInfo(expId, visitId,
							subjectID, cpi.getCbfSI(), derivedDataFile);
					ddiList.add(addi);
				}
			}
		}

	}

	public static String getSeriesName(String dicomSeriesDir) throws Exception {
		AttributeList atList = new AttributeList();
		File[] files = new File(dicomSeriesDir).listFiles();
		for (File f : files) {
			if (DicomFileUtilities.isDicomOrAcrNemaFile(f)) {
				atList.read(f.getAbsolutePath());
				break;
			}
		}
		return Attribute.getSingleStringValueOrNull(atList,
				TagFromName.SeriesDescription);
	}

	protected DerivedDataInfo prepDerivedDataInfo(int expID, int visitID,
			String subjectID, SegmentInfo si, File derivedFile) {
		DerivedDataInfo ddi = new DerivedDataInfo();
		ddi.setExperimentID(expID);
		ddi.setVisitID(visitID);
		ddi.setSegmentID(si.getSegmentId());
		ddi.setSubjectID(subjectID);
		ddi.setBad(false);
		ddi.setDataURI(derivedFile.getAbsolutePath());
		return ddi;
	}

	protected File copy2DerivedDataLoc(File srcFile, File derivedDir)
			throws IOException {
		String filename = srcFile.getName();

		File derivedFile = new File(derivedDir, filename);
		FileUtils.copyFile(srcFile.getAbsolutePath(),
				derivedFile.getAbsolutePath());
		return derivedFile;
	}

	protected File getRawDataRoot4Visit() {
		VisitInfo vi = this.context.getViList().get(0);

		List<String> imageFiles = new ArrayList<String>();
		for (SegmentInfo si : vi.getSiList()) {
			SeriesHeaderInfo shi = si.getShi();
			if (shi.getImages() != null && !shi.getImages().isEmpty()) {
				imageFiles.add(shi.getImages().get(0));
			}
		}

		String commonRoot = FileUtils.findCommonRoot(imageFiles);
		if (commonRoot == null)
			return null;
		return new File(commonRoot);
	}

	protected BrikWrapper findBrikInfo(String protocol) {
		for (BrikWrapper bi : this.brikInfoList) {
			if (bi.getSi().getProtocolId().equals(protocol)) {
				return bi;
			}
		}

		for (BrikWrapper bi : this.existingBrikInfoList) {
			if (bi.getSi().getProtocolId().equals(protocol)) {
				return bi;
			}
		}
		return null;
	}

	protected boolean hasAnatomical() {
		VisitInfo visitInfo = context.getViList().get(0);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getProtocolId().equals(ANATOMICAL)) {
				return true;
			}
		}
		return false;
	}

	public static boolean isFieldMapProcessable(SegmentInfo si) {
		return (si.getProtocolId().equals(MINCON)
				|| si.getProtocolId().equals(CSF) || CBFBirnHelper
					.isCBFProcessable(si.getProtocolId()));
	}

	protected List<SegmentInfo> getSegmentsNeedingAFNIConversion() {
		List<SegmentInfo> si4AFNIList = new ArrayList<SegmentInfo>();
		VisitInfo visitInfo = context.getViList().get(0);
		for (SegmentInfo si : visitInfo.getSiList()) {
			if (si.getProtocolId().equals(MINCON)
					|| si.getProtocolId().equals(CSF)
					|| si.getProtocolId().equals(ANATOMICAL)
					// || si.getProtocolId().equals(FIELD_MAP)
					|| CBFBirnHelper.isCBFProcessable(si.getProtocolId())) {
				if (!si.getShi().getImageType().equals(SeriesHeaderInfo.AFNI)) {
					// if P-file and B0 correction is requested
					// then do not include, since the B0-corrected briks will be
					// used instead
					if (si.getShi().getImageType()
							.equals(SeriesHeaderInfo.PFILE)
							&& this.context.isDoB0Correction()
							&& isFieldMapProcessable(si)) {
						continue;
					}

					si4AFNIList.add(si);
				} else {
					File brikFile = null;
					for (String imgFile : si.getShi().getImages()) {
						if (imgFile.endsWith(".BRIK")) {
							brikFile = new File(imgFile);
							break;
						}
					}
					Assertion.assertNotNull(brikFile,
							"Missing BRIK file for series:" + si.getName());
					existingBrikInfoList.add(new BrikWrapper(brikFile, si));
				}
			}
		}
		return si4AFNIList;
	}

	public static class BrikWrapper {
		File brikFile;
		final SegmentInfo si;

		public BrikWrapper(File brikFile, SegmentInfo si) {
			super();
			this.brikFile = brikFile;
			this.si = si;
		}

		public File getBrikFile() {
			return brikFile;
		}

		public SegmentInfo getSi() {
			return si;
		}

		public void setBrikFile(File brikFile) {
			this.brikFile = brikFile;
		}
	}// ;

	public static class AnalyzeWrapper {
		final File analyzeImgFile;
		final SegmentInfo si;

		public AnalyzeWrapper(File brikFile, SegmentInfo si) {
			super();
			this.analyzeImgFile = brikFile;
			this.si = si;
		}

		public SegmentInfo getSi() {
			return si;
		}

		public File getAnalyzeImgFile() {
			return analyzeImgFile;
		}
	}// ;

	public File convertDICOMSeries2AFNI(File srcDicomDir, File destDir,
			String prefix) throws CBFException {
		File workDir = new File(destDir, srcDicomDir.getName());
		workDir.mkdir();
		File[] dicomFiles = srcDicomDir.listFiles();
		try {
			String fileNameFirstChar = null;
			for (int i = 0; i < dicomFiles.length; i++) {
				if (dicomFiles[i].isFile()) {
					File df = new File(workDir, dicomFiles[i].getName());
					FileUtils.copyFile(dicomFiles[i].getAbsolutePath(),
							df.getAbsolutePath());
					if (fileNameFirstChar == null) {
						fileNameFirstChar = String.valueOf(dicomFiles[i]
								.getName().charAt(0));
					}
				}
			}

			// image sequencing
			Executor executor = new Executor("/usr/local/bin/imseq", true);

			executor.execute(workDir.getAbsolutePath());

			executor = new Executor("/usr/local/bin/d2afni_hid", true);
			executor.setErrorStreamReliable(false);
			StringBuilder sb = new StringBuilder(128);
			sb.append("-od ").append(workDir.getAbsolutePath());
			sb.append(" -prefix ").append(prefix).append(' ');
			sb.append(workDir.getAbsolutePath()).append("/");
			sb.append(fileNameFirstChar).append('*');
			String cmdLine = sb.toString();
			System.out.println("d2afni:" + cmdLine);

			executor.execute(cmdLine);

			File[] files = workDir.listFiles();
			List<File> afniFiles = new ArrayList<File>();
			for (File f : files) {
				String name = f.getName();
				if (f.isFile() && name.startsWith(prefix)
						&& (name.endsWith(".BRIK") || name.endsWith(".HEAD"))) {
					afniFiles.add(f);
				}
			}
			Assertion.assertTrue(afniFiles.size() == 2);

			File brikFile = null;
			for (File af : afniFiles) {
				File destFile = new File(destDir, af.getName());
				if (!FileUtils.moveTo(af, destFile)) {
					throw new Exception("Cannot move file:" + af.getName());
				}
				if (destFile.getName().endsWith(".BRIK")) {
					brikFile = destFile;
				}
			}

			Assertion.assertNotNull(brikFile);

			// time to cleanup work directory
			FileUtils.deleteRecursively(workDir);

			return brikFile;
		} catch (Throwable t) {
			throw new CBFException("convertDICOMSeries2AFNI", t);
		}
	}

	protected static void setEnvironment(CBFProcJobHelper helper) {
		helper.setEnvPath("/usr/local/bin:/bin:/usr/bin:/data/apps/fsl/bin:/data/apps/afni/bin:.");
		helper.addEnvironmentVar("FSLDIR", "/data/apps/fsl");
		helper.addEnvironmentVar("FSLOUTPUTTYPE", "ANALYZE");
		helper.addEnvironmentVar("AFNI_PLUGINPATH", "/data/apps/afni/bin");
		helper.addEnvironmentVar("FSLMULTIFILEQUIT", "TRUE");
		helper.addEnvironmentVar("FSLCONFDIR", "/data/apps/fsl/bin/config");
		helper.addEnvironmentVar("FSLMACHTYPE", "x86_64-redhat-linux-gcc4.1.2");
	}

	public AbstractCBFProcJobHelper() {
		super();
	}

	public void setCanceled(boolean canceled) {
		this.canceled = canceled;
	}

	public boolean isCanceled() {
		return canceled;
	}

}