package clinical.web.workflow.cbf;

import java.io.File;
import java.io.FilenameFilter;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import clinical.server.vo.Deriveddata;
import clinical.utils.Assertion;
import clinical.utils.FileUtils;
import clinical.web.CBFBIRNConstants;
import clinical.web.IDerivedImageDataService;
import clinical.web.ServiceFactory;
import clinical.web.common.UserInfo;
import clinical.web.vo.JobProvenanceInfo;
import clinical.web.vo.JobProvenanceInfo.JobProvenanceParamInfo;
import clinical.web.vo.upload.VisitInfo;

/**
 * 
 * @author I. Burak Ozyurt
 * @version $Id$
 */
public class MultipleWFHelper {
	public static enum PatternType {
		PREFIX, ANYWHERE
	}

	/**
	 * 
	 * @param theJPI
	 * @param availableJPIList
	 *            as generated by a call to
	 *            {@link AbstractCBFProcJobHelper#getExistingWFResults()}
	 *            includes the original derived dir.
	 * @return
	 */
	public static Map<String, WFStep> findWFSteps2Skip(
			JobProvenanceInfo theJPI, List<JobProvenanceInfo> availableJPIList) {
		Map<String, WFStep> stepMap = new HashMap<String, WFStep>(3);

		if (availableJPIList == null || availableJPIList.isEmpty()) {
			// return empty stepMap
			return stepMap;
		}

		int size = availableJPIList.size();
		List<JobProvenanceInfo> copyOfAvailJPIList = new ArrayList<JobProvenanceInfo>(
				size);

		copyOfAvailJPIList.addAll(availableJPIList);
		if (copyOfAvailJPIList.size() > 1) {
			// sort in descending number of parameters order
			Collections.sort(copyOfAvailJPIList,
					new Comparator<JobProvenanceInfo>() {
						public int compare(JobProvenanceInfo o1,
								JobProvenanceInfo o2) {
							return o2.getNumOfParams() - o1.getNumOfParams();
						}
					});
		}

		for (JobProvenanceInfo jpi : copyOfAvailJPIList) {
			if (jpi.getNumOfParams() == 0) {
				List<AFNIBrik> fmBriks = getBriksWithPattern(jpi.getDataURI(),
						CBFBIRNConstants.FIELDMAP_BRIK_PATTERN,
						PatternType.ANYWHERE);
				List<AFNIBrik> senseBriks = getBriksWithPattern(
						jpi.getDataURI(), CBFBIRNConstants.SENSE_BRIK_PATTERN,
						PatternType.PREFIX);
				Assertion.assertFalse(!fmBriks.isEmpty()
						&& !senseBriks.isEmpty());
				if (!fmBriks.isEmpty()) {
					WFStep wfStep = new WFStep(CBFBIRNConstants.KEY_FIELDMAP,
							jpi);
					for (AFNIBrik brik : fmBriks) {
						wfStep.addBrik(brik);
					}
					stepMap.put(CBFBIRNConstants.KEY_FIELDMAP, wfStep);
				} else if (!senseBriks.isEmpty()) {
					WFStep wfStep = new WFStep(CBFBIRNConstants.KEY_SENSE, jpi);
					for (AFNIBrik brik : senseBriks) {
						wfStep.addBrik(brik);
					}
					stepMap.put(CBFBIRNConstants.KEY_SENSE, wfStep);
				}

			} else {
				boolean hasFieldMap = getBoolValue(
						CBFBIRNConstants.KEY_FIELDMAP, jpi);
				boolean hasSense = getBoolValue(CBFBIRNConstants.KEY_SENSE, jpi);
				Assertion.assertFalse(hasFieldMap && hasSense);
				if (hasFieldMap) {
					List<AFNIBrik> briks = getBriksWithPattern(
							jpi.getDataURI(),
							CBFBIRNConstants.FIELDMAP_BRIK_PATTERN,
							PatternType.ANYWHERE);
					if (!briks.isEmpty()) {
						WFStep wfStep = new WFStep(
								CBFBIRNConstants.KEY_FIELDMAP, jpi);
						for (AFNIBrik brik : briks) {
							wfStep.addBrik(brik);
						}
						stepMap.put(CBFBIRNConstants.KEY_FIELDMAP, wfStep);
					}
				} else if (hasSense) {
					List<AFNIBrik> briks = getBriksWithPattern(
							jpi.getDataURI(),
							CBFBIRNConstants.SENSE_BRIK_PATTERN,
							PatternType.PREFIX);
					if (!briks.isEmpty()) {
						WFStep wfStep = new WFStep(CBFBIRNConstants.KEY_SENSE,
								jpi);
						for (AFNIBrik brik : briks) {
							wfStep.addBrik(brik);
						}
						stepMap.put(CBFBIRNConstants.KEY_SENSE, wfStep);
					}
				}

			}
		}

		// if only difference is in fabber param
		boolean hasFabberParam = hasParam(CBFBIRNConstants.KEY_FABBER, theJPI);
		if (hasFabberParam) {
			Set<String> ignoreParamSet = new HashSet<String>(7);
			// any changes in smoothing is ignored also
			ignoreParamSet.add(CBFBIRNConstants.KEY_SMOOTHING);
			ignoreParamSet.add(CBFBIRNConstants.PROV_FINISH_DATE);
			ignoreParamSet.add(CBFBIRNConstants.KEY_FABBER);
			for (JobProvenanceInfo jpi : copyOfAvailJPIList) {
				if (jpi.getNumOfParams() > 0) {
					if (matches(theJPI, jpi, ignoreParamSet)) {
						WFStep wfStep = new WFStep(CBFBIRNConstants.KEY_FABBER,
								jpi);

						stepMap.put(CBFBIRNConstants.KEY_FABBER, wfStep);
						break;
					}
				}
			}
		}

		// if only difference is in smoothing param
		boolean hasGaussianFilterParam = hasParam(
				CBFBIRNConstants.KEY_SMOOTHING, theJPI);
		if (hasGaussianFilterParam) {
			Set<String> ignoreParamSet = new HashSet<String>(7);
			ignoreParamSet.add(CBFBIRNConstants.KEY_SMOOTHING);
			ignoreParamSet.add(CBFBIRNConstants.PROV_FINISH_DATE);
			// both smoothing and fabber is handled in the last step
			ignoreParamSet.add(CBFBIRNConstants.KEY_FABBER);
			// ignoreParamSet.add("automated");
			for (JobProvenanceInfo jpi : copyOfAvailJPIList) {
				if (jpi.getNumOfParams() > 0) {
					if (matches(theJPI, jpi, ignoreParamSet)) {
						WFStep wfStep = new WFStep(
								CBFBIRNConstants.KEY_SMOOTHING, jpi);

						stepMap.put(CBFBIRNConstants.KEY_SMOOTHING, wfStep);
						break;
					}
				}
			}
		}

		return stepMap;
	}

	public static boolean paramValueMatches(JobProvenanceInfo refJPI,
			JobProvenanceInfo otherJPI, String paramName) {
		for (JobProvenanceParamInfo pi : refJPI.getParams()) {
			if (pi.getName().equals(paramName)) {
				JobProvenanceParamInfo otherPI = otherJPI.findParam(pi
						.getName());
				if (otherPI != null
						&& pi.getValue().equalsIgnoreCase(otherPI.getValue())) {
					return true;
				}
				break;
			}
		}
		return false;
	}

	public static boolean matches(JobProvenanceInfo refJPI,
			JobProvenanceInfo otherJPI, Set<String> ignoreParamSet) {
		boolean ok = true;
		for (JobProvenanceParamInfo pi : refJPI.getParams()) {
			if (ignoreParamSet != null && ignoreParamSet.contains(pi.getName())) {
				continue;
			}
			JobProvenanceParamInfo otherPI = otherJPI.findParam(pi.getName());
			// FIXME kludge for David remove in production

			/*
			 * if (otherPI == null) { // just ignore missing parameters in older
			 * runs continue; }
			 */

			if (otherPI == null
					|| (otherPI.getValue() == null && pi.getValue() != null)) {

				ok = false;
				break;
			}
			// FIXME the other is N/A and ref param is no
			if (otherPI.getValue() != null && pi.getValue() != null) {
				if (!otherPI.getValue().equals(pi.getValue())) {
					ok = false;
					break;
				}
			}
		}
		return ok;
	}

	public static boolean getBoolValue(String paramName, JobProvenanceInfo jpi) {
		JobProvenanceParamInfo pi = jpi.findParam(paramName);
		return pi != null && pi.getValue().equals("yes");
	}

	public static boolean hasParam(String paramName, JobProvenanceInfo jpi) {
		JobProvenanceParamInfo pi = jpi.findParam(paramName);
		return pi != null;
	}

	/**
	 * skips intermediate BRIKs, for CSF embedded Pfiles also includes the CSF
	 * TCAT brik
	 * 
	 * @param rootDir
	 * @param pattern
	 * @param patternType
	 * @return
	 */
	public static List<AFNIBrik> getBriksWithPattern(String rootDir,
			final String pattern, final PatternType patternType) {
		FilenameFilter filter = new FilenameFilter() {
			@Override
			public boolean accept(File dir, String name) {
				if (name.endsWith(".BRIK") || name.endsWith(".HEAD")) {
					if (dir.getAbsolutePath().indexOf("intermediate") == -1) {
						if (patternType == PatternType.PREFIX) {
							if (name.startsWith(pattern)
									|| name.startsWith(CBFBIRNConstants.CSF_TCAT_BRIK_PREFIX)) {
								return true;
							}
						} else if (name.indexOf(pattern) != -1
								|| name.startsWith(CBFBIRNConstants.CSF_TCAT_BRIK_PREFIX)) {
							return true;
						}
					}
				}
				return false;
			}
		};

		List<File> matchingFiles = FileUtils.getAllFilesMatching(new File(
				rootDir), filter);
		Map<String, List<File>> brikMap = new HashMap<String, List<File>>(7);
		for (File f : matchingFiles) {
			String seriesName = AFNIBrik.getSeriesName(f.getName());
			List<File> brikFiles = brikMap.get(seriesName);
			if (brikFiles == null) {
				brikFiles = new ArrayList<File>(2);
				brikMap.put(seriesName, brikFiles);
			}
			brikFiles.add(f);
		}

		List<AFNIBrik> matchingBriks = new ArrayList<AFNIBrik>(brikMap.size());
		for (List<File> files : brikMap.values()) {
			Assertion.assertTrue(files.size() >= 2);
			if (files.size() > 2) {
				files = selectLongestPathSet(files);
			}

			AFNIBrik brik = AFNIBrik.create(files);
			matchingBriks.add(brik);
		}

		// if embedded CSF brik, weed out original dbl brik (before splitting)

		if (matchingBriks.size() > 2) {
			AFNIBrik csfTcatBrik = null;
			for (AFNIBrik brik : matchingBriks) {
				if (brik.getBrikName().startsWith(
						CBFBIRNConstants.CSF_TCAT_BRIK_PREFIX)) {
					csfTcatBrik = brik;
					break;
				}
			}

			if (csfTcatBrik != null) {
				List<AFNIBrik> briks = new ArrayList<AFNIBrik>();
				briks.add(csfTcatBrik);
				Map<String, List<AFNIBrik>> prefix2brikListMap = new HashMap<String, List<AFNIBrik>>();
				for (AFNIBrik brik : matchingBriks) {
					if (brik == csfTcatBrik) {
						continue;
					}
					String seriesName = brik.getSeriesName();
					if (seriesName.endsWith("tcat")) {
						seriesName = seriesName.substring(0,
								seriesName.length() - 4);
					}
					List<AFNIBrik> list = prefix2brikListMap.get(seriesName);
					if (list == null) {
						list = new ArrayList<AFNIBrik>(1);
						prefix2brikListMap.put(seriesName, list);
					}
					list.add(brik);
				}
				for (List<AFNIBrik> list : prefix2brikListMap.values()) {
					if (list.size() == 1) {
						briks.add(list.get(0));
					} else {
						// choose tcat one
						for (AFNIBrik brik : list) {
							if (brik.getBrikName().indexOf(
									CBFBIRNConstants.TCAT_PATTERN) != -1) {
								briks.add(brik);
								break;
							}
						}
					}
				}
				matchingBriks = briks;
			}
		}

		return matchingBriks;
	}

	static List<File> selectLongestPathSet(List<File> files) {
		List<File> filteredFiles = new ArrayList<File>(2);
		Collections.sort(files, new Comparator<File>() {
			@Override
			public int compare(File o1, File o2) {
				return o2.getAbsolutePath().length()
						- o1.getAbsolutePath().length();
			}
		});

		String seriesName1 = AFNIBrik.getSeriesName(files.get(0).getName());
		String seriesName2 = AFNIBrik.getSeriesName(files.get(1).getName());
		Assertion.assertTrue(seriesName1.equals(seriesName2));
		filteredFiles.add(files.get(0));
		filteredFiles.add(files.get(1));

		return filteredFiles;

	}

	public static JobProvenanceInfo prepJobProvenanceInfo(String name,
			String description, CBFWFContext ctx) {
		JobProvenanceInfo jpi = new JobProvenanceInfo(name, description);

		jpi.addParam(new JobProvenanceParamInfo("fieldmap", ctx
				.isDoB0Correction() ? "yes" : "no"));

		jpi.addParam(new JobProvenanceParamInfo("registration",
				ctx.isDoReg() ? "yes" : "no"));
		jpi.addParam(new JobProvenanceParamInfo("sense",
				ctx.isDoSenseval() ? "yes" : "no"));

		jpi.addParam(new JobProvenanceParamInfo("skullStripping", ctx
				.isDoSkullStripping() ? "yes" : "no"));

		jpi.addParam(new JobProvenanceParamInfo("automated", ctx
				.isUseAutomatedSegmentation() ? "yes" : "no"));
		jpi.addParam(new JobProvenanceParamInfo("pvc", ctx
				.isDoPartialVolumeCorrection() ? "yes" : "no"));
		jpi.addParam(new JobProvenanceParamInfo("gmThresh", String.valueOf(ctx
				.getGmThreshold())));
		String value = ctx.getAlignment() == 0 ? "No Alignment" : ctx
				.getAlignment() + ". method";
		jpi.addParam(new JobProvenanceParamInfo("alignment", value));
		value = "Automatic";
		switch (ctx.getCsfMethod()) {
		case CBFWFContext.AUTOMATIC:
			value = "Automatic";
			break;
		case CBFWFContext.TOP5:
			value = "Top 5";
			break;
		default:
			value = "PVM";
		}
		jpi.addParam(new JobProvenanceParamInfo("csfMethod", value));

		jpi.addParam(new JobProvenanceParamInfo("ltMethod",
				ctx.isUseLTMethod() ? "yes" : "no"));

		jpi.addParam(new JobProvenanceParamInfo("smoothingParam", String
				.valueOf(ctx.getGaussianFilterParam())));

		jpi.addParam(new JobProvenanceParamInfo(CBFBIRNConstants.KEY_FABBER,
				ctx.isDoFabber() ? "yes" : "no"));
		return jpi;
	}

	public static List<File> getWFDerivedDirectoriesForVisit(UserInfo ui,
			CBFWFContext ctx) throws Exception {
		IDerivedImageDataService didService = ServiceFactory
				.getDerivedImageDataService(ctx.getDbID());
		VisitInfo visitInfo = ctx.getViList().get(0);
		List<Deriveddata> ddList = didService.findDerivedData(ui,
				visitInfo.getSubjectID(), visitInfo.getExpId(),
				visitInfo.getVisitId(), null);
		if (ddList.isEmpty()) {
			return new ArrayList<File>(0);
		}
		List<String> paths = new ArrayList<String>(ddList.size());
		for (Deriveddata dd : ddList) {
			paths.add(dd.getDatauri());
		}
		Pattern derivedPattern = Pattern.compile("/derived(\\.\\d+)?");
		Set<String> ddPartSet = new HashSet<String>(11);
		List<File> derivedDirList = new ArrayList<File>(5);
		for (String path : paths) {
			File f = new File(path);

			Matcher m = derivedPattern.matcher(f.getAbsolutePath());
			if (m.find()) {
				String derivedDir = path.substring(0, m.end());
				if (!ddPartSet.contains(derivedDir)) {
					ddPartSet.add(derivedDir);
					File ddf = new File(derivedDir);
					if (ddf.isDirectory()) {
						derivedDirList.add(ddf);
					}
				}
			}
		}

		return derivedDirList;
	}

}
