#!/bin/bash
##
##  Copyright 2007-2014 SRI International
##
##  http://nitrc.org/projects/lwdp/
##
##  This file is part of the Lightweight Data Pipeline (LWDP), developed
##  and distributed by the Data Integration Component of the National
##  Consortium on Alcohol and NeuroDevelopment in Adolescence, supported by
##  the U.S. National Institute on Alcohol Abuse and Alcoholism (NIAAA) under
##  Grant No. 1U01 AA021697
##
##  The Lightweight Data Pipeline (LWDP) is free software: you can
##  redistribute it and/or modify it under the terms of the GNU General Public
##  License as published by the Free Software Foundation, either version 3 of
##  the License, or (at your option) any later version.
##
##  The Lightweight Data Pipeline (LWDP) is distributed in the hope that it
##  will be useful, but WITHOUT ANY WARRANTY; without even the implied
##  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
##  GNU General Public License for more details.
##
##  You should have received a copy of the GNU General Public License along
##  with the Lightweight Data Pipeline.  If not, see <http://www.gnu.org/licenses/>.
##
##  $Revision: 22 $
##
##  $LastChangedDate: 2022-12-21 10:16:54 -0500 (Wed, 21 Dec 2022) $
##
##  $LastChangedBy: pohl $
##

# Check whether we have "lockfile" tool available and include proper script with locking functions
LWDP_HAVE_LOCKFILE=0
if which lockfile > /dev/null; then
    LWDP_HAVE_LOCKFILE=1
fi
export LWDP_HAVE_LOCKFILE

LWDP_changed_data_hook()
{
   # If needed overwrite this function
   # see also test/test_hooks 
   return
}

LWDP_error_hook()
{
   # If needed overwrite this function
   # see also test/test_hooks 
   return
}

LWDP_error_exit()
{
    LWDP_error_hook
    exit 1
}


# Check whether we need to write all checked dependencies into a file (for diagnostic graph creation)
if [ -n "${LWDP_DEP_OUTFILE}" ]; then
    if [ -z "${LWDP_DEP_OUTFILE_CREATED}" ]; then
	echo -e "digraph lwdp_deps {\n	size=\"6,6\";\n	node [color=lightblue2, style=filled];" > ${LWDP_DEP_OUTFILE}
	export LWDP_DEP_OUTFILE_CREATED="1"
    fi
fi

# List of recognized compression suffixes. Dependencies checking considers compressed files as equivalent to uncompressed ones.
LWDP_COMPRESSION_SUFFIXES="gz xz bz2 Z"

LWDP_check_file_compression()
{
    local file=$1

    local c
    for c in ${LWDP_COMPRESSION_SUFFIXES}; do
        if [ -e ${file}.${c} ]; then
            echo ${file}.${c}
	    return
	fi
    done

    echo ${file}
}

#
# Get path name for a manual override file
#
LWDP_get_override_file_path()
{
    echo $(dirname ${1})/manual/$(basename ${1})
}

#
# See if a manual override file exists for a given (input) file. An override file inserts
#  'manual/' into the file system path immediately prior to the file's base name.
#
LWDP_get_override_file()
{
    local fpath=$1
    local manual=$(LWDP_get_override_file_path ${fpath})

    manual=$(LWDP_check_file_compression ${manual})
    if [ -e ${manual} ]; then
	echo ${manual}
	return
    fi

    LWDP_check_file_compression $fpath
}

#
# Get override files, if they exist, for a list of input files
#
LWDP_get_override_files_list()
{
    local f
    for f in $*; do
	LWDP_get_override_file $f
    done
}

##
## Helper functions for file locking and dependency checking
##

# check whether file needs update due to updated dependencies
LWDP_needs_update()
{
    local target=$1

    if [ -e ${target}.lock ]; then
	echo "========================================================================"
	echo "TARGET LOCKED ${target}"
	echo "========================================================================"
	# if target is locked pipeline should end processing
	LWDP_error_exit
    fi

    local realtarget=`LWDP_check_file_compression $target`
    
    shift
    local sources="$*"

    if [ -n "${LWDP_DEP_OUTFILE}" ]; then
	for source in ${sources}; do
	    echo "  \"${source}\" -> \"${target}\"" >> ${LWDP_DEP_OUTFILE}
	done
    fi

    for source in ${sources}; do
	local realsource=`LWDP_check_file_compression ${source}`

	if [ ! -e ${realsource} ]; then
	    echo "========================================================================"
	    echo "TARGET ${target}"
	    echo "MISSING SOURCE ${source}"
	    echo "========================================================================"
	    LWDP_error_exit
	fi
	
	if [ -e ${source}.lock ]; then
	    echo "========================================================================"
	    echo "TARGET ${target}"
	    echo "SOURCE LOCKED ${source}"
	    echo "========================================================================"
	    # if source is locked pipeline should end processing
	    LWDP_error_exit 
	fi
    done
    if [ ! -e ${realtarget} ]; then
	echo "========================================================================"
	echo "CREATE ${target}"
        echo "========================================================================"
	LWDP_changed_data_hook
	return 0
    fi

    local source
    for source in ${sources}; do
	local realsource=`LWDP_check_file_compression ${source}`
	if test ${realtarget} -ot ${realsource}; then
            echo "========================================================================"
	    echo "UPDATE ${target}"
            echo "DUE TO ${source}"
            echo "========================================================================"
	    rm -rf ${realtarget}
	    LWDP_changed_data_hook
	    return 0
	fi
    done
    
    return 1
}

# NFS-safe (hopefully) file locking.
LWDP_lockfile_create()
{
    local lockfile=$1.lock
    if [ -e ${lockfile} ]; then
	# lockfile already exists, so clearly we were not the first
	return 1;
    fi
    
    mkdir -p `dirname ${lockfile}`

    if [ ${LWDP_HAVE_LOCKFILE} -gt 0 ]; then
	# Use procmail's "lockfile" tool for better NFS safety
	if lockfile -0 -r0 ${lockfile}; then
	    trap "rm -f ${lockfile}; exit" INT TERM EXIT
	    return 0
	fi
    else
	# We don't have "lockfile" tool; use our own hack as a fallback
	local hostpid=`hostname`-$$
	echo ${hostpid} >> ${lockfile}
	if [ `head -n 1 ${lockfile}` == ${hostpid} ]; then
            # first one to write PID was us
	    trap "rm -f ${lockfile}; exit" INT TERM EXIT
	    return 0;
	fi
    fi
    return 1;
}

LWDP_lockfile_delete()
{
    local file=${1}
    local lockfile=${file}.lock

    local realfile=`LWDP_check_file_compression ${file}`
    if [ -e ${realfile} ]; then
	# do not update timestamp if file is older than lock (i.e., existed before and did not get updated at all)
	if test ${lockfile} -ot ${realfile}; then
	    touch --no-create -r ${lockfile} ${realfile}
	fi
    fi
    
    rm -f ${lockfile}
}

#
# Combine dependency checking with locking of target
#
LWDP_needs_update_and_lock()
{
    local target=$1
    if LWDP_needs_update $*; then
	if LWDP_lockfile_create ${target}; then
	    return 0
	else
	    echo "========================================================================"
	    echo "TARGET LOCKED ${target}"
	    echo "========================================================================"
	    # if target is locked pipeline should end processing
	    LWDP_error_exit
	fi
    fi
    return 1
}
