/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvController.h,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#ifndef SVV_CONTROLLER_H_ /* SV */
#  define SVV_CONTROLLER_H_
// Includes svvTypes (which includes svvConfiguration).
#  include "svvMacros.h"
// Includes svvConsts (which includes svvTypes).
#  include "svvMath.h"
// from VTK Common
#  include "vtkObject.h"

class vtkDataSet;
class vtkPolyDataSource;
class vtkImplicitFunction;
class vtkRenderWindowInteractor;

class vtkProp;
class vtkProp3D;
class vtkProperty;
class vtkActor;
class vtkRenderer;
class vtkActorCollection;

class vtkPolyDataPipeline;

VTK_EXTENSIONS_NAMESPACE_BEGIN

class vtkInputSource;
class vtkInputSourceCollection;
class vtkPolyDataPipelineCollection;

SVV_NESTED_NAMESPACE_BEGIN

class svvMaterialNameMap;
class svvColorNameMap;

/** \class   svvController
 *  \brief   svvController application base class (singleton object).
 * 
 * svvController is a ...
 * 
 * \author  $Author: christianh $
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:58:35 $
 * 
 * \sa
 * vtkObject
 */
class SVV_EXPORT svvController : public vtkObject
{
public:
  /** Return pointer to the singleton instance. */
  static svvController* New (void);

  /*@{*/
  /** The vtkObject subclass methods. */
  vtkTypeRevisionMacro (svvController,vtkObject);
  /*@}*/

  /** Invoked by print to print information about object including superclasses.
   * Typically not called by the user (use Print() instead), but used in the 
   * hierarchical print process to combine the output of several classes.
   */
  void PrintSelf (std::ostream&, vtkIndent);

  /*@{*/
  /** The instance owns itself and may not be registered by other objects. */
  void Register (vtkObjectBase*) {}
  void UnRegister (vtkObjectBase*) {}
  /*@}*/

  static int    Args (int* c, char** v[], char** p[], int& i,
		     int (*cb)(int,char**,int&));
  /** Returns dereferenced argument count. */
  static int    argc (void) { return *(svvController::s_argc_ptr_); }
  /** Returns dereferenced argument string vector. */
  static char** argv (void) { return *(svvController::s_argv_ptr_); }
  /** Returns dereferenced environment string vector. */
  static char** envp (void) { return *(svvController::s_envp_ptr_); }

  /*@{*/
  /** Set/Get maximum size of actors using display lists vs. immediate mode. */
  vtkSetMacro (RenderModePointsThreshold,int);
  vtkGetMacro (RenderModePointsThreshold,int);
  vtkSetMacro (RenderModeCellsThreshold,int);
  vtkGetMacro (RenderModeCellsThreshold,int);
  /*@}*/

  /** Update render modes for collection of actors using current thresholds. */
  void UpdateRenderModes (void);

  svvMaterialNameMap*   GetMaterialNameMap (void) const;
  svvColorNameMap*      GetColorNameMap (void) const;

  /*@{*/
  /** Finds named defined colors and materials by name. */
  bool FindMaterial (const char* const&, SvvMaterial&);
  bool AddMaterial (const char* const&, const SvvMaterial&);
  bool FindColor (const char* const&, float&, float&, float&, float&);
  bool FindColor (const char* const&, float&, float&, float&);
  bool FindColor (const char* const&, float [4]);
  bool FindColor (const char* const&,
                  SvvColorByte&, SvvColorByte&, SvvColorByte&, SvvColorByte&);
  bool FindColor (const char* const&,
                  SvvColorByte&, SvvColorByte&, SvvColorByte&);
  bool FindColor (const char* const&, SvvColorByte [4]);
  bool AddColor (const char* const&,
                 const float&, const float&, const float&, const float&);
  bool AddColor (const char* const&, const float&, const float&, const float&);
  bool AddColor (const char* const&, const float [4]);
  vtkProperty* MakeProperty (SvvMaterial&, float=1, float=1, float=1);
  vtkProperty* MakeProperty (float [4], float=0.2f, float=0.8f, float=0.0f);
  vtkProperty* MakeProperty (const char* const&);
  /*@}*/

  void PrintMaterials (std::ostream& os);

  /** Return any surface data sources in the global collection. */
  vtkGetObjectMacro (InputSources,
                     VTK_EXTENSIONS_NAMESPACE_QUALIFIER
                     vtkInputSourceCollection);

  /*@{*/
  /** Add/Remove surface sources to the collection. */
  void AddSurfaceDataSource (VTK_EXTENSIONS_NAMESPACE_QUALIFIER
                             vtkInputSource*);
  void RemoveSurfaceDataSource (VTK_EXTENSIONS_NAMESPACE_QUALIFIER
                                vtkInputSource*);
  /*@}*/

  /** Given input file name, create surface source and add it to the list. */
  vtkInputSource* MakeSurfaceDataSource (const char*, int  = 0, bool = false);

  /** Return any surfaces in the global collection. */
  vtkGetObjectMacro (SurfaceActors,vtkActorCollection);

  /*@{*/
  /** Add/Remove surfaces to the collection. */
  void AddSurfaceActor (vtkActor*);
  void RemoveSurfaceActor (vtkActor*);
  /*@}*/

  /*@{*/
  /** Create new surface adding it to the list and return a pointer to it. */
  vtkActor* MakeSurfaceActor (vtkPolyDataSource*, vtkProperty* = 0, bool=false);
  vtkActor* MakeSurfaceActor (vtkPolyDataSource*, SvvMaterial&, bool = false);
  vtkActor* MakeSurfaceActor (vtkPolyDataSource*, float [4], bool = false);
  vtkActor* MakeSurfaceActor (vtkPolyDataSource*, const char*, bool = false);
  vtkActor* MakeSurfaceActor (vtkInputSource*, vtkProperty* = 0, bool = false);
  vtkActor* MakeSurfaceActor (vtkInputSource*, SvvMaterial&, bool = false);
  vtkActor* MakeSurfaceActor (vtkInputSource*, float [4], bool = false);
  vtkActor* MakeSurfaceActor (vtkInputSource*, const char*, bool = false);
  /*@}*/

  vtkActor* MakeClipSelectionActor (vtkPolyDataSource*,
                                    vtkImplicitFunction*,
                                    const char*);

  /** Return any surfaces in the global collection. */
  vtkGetObjectMacro (SurfacePipelines, vtkPolyDataPipelineCollection);

  /*@{*/
  /** Add/Remove surfaces to the collection. */
  void AddSurfacePipeline (vtkPolyDataPipeline*);
  void RemoveSurfacePipeline (vtkPolyDataPipeline*);
  /*@}*/

  /*@{*/
  /** Create new surface adding it to the list and return a pointer to it. */
  vtkPolyDataPipeline* MakeSurfacePipeline (vtkInputSource*, vtkProperty* = 0);
  vtkPolyDataPipeline* MakeSurfacePipeline (vtkInputSource*, SvvMaterial&);
  vtkPolyDataPipeline* MakeSurfacePipeline (vtkInputSource*, float rgba[4]);
  vtkPolyDataPipeline* MakeSurfacePipeline (vtkInputSource*, const char*);
  /*@}*/

  /** Set a reference to the interactor associated with this object. */
  void SetInteractor (vtkRenderWindowInteractor*);

  /** Get overall bounds of all actors and volumes in a renderer. */
  bool GetRenderedBounds (float bounds[6], vtkRenderer* renderer);

  /** Get overall bounds of all inputs to actors and volumes in a renderer. */
  bool GetInputBounds (float bounds[6], vtkRenderer* renderer);

  /** Get the axes prop. */
  vtkProp* GetAxes (vtkRenderer* renderer);

  /** Update/create the axes prop using the overall bounds of the
   * source collection. Type of axes depends on whether or not a
   * reference to an orienting camera is held.
   */
  vtkProp* UpdateAxes (vtkRenderer* renderer);

  /** Remove axes (if any) from a renderer. */
  void  RemoveAxes (vtkRenderer* aRenderer, vtkProp** removed = NULL);

  /** Toggle presense/absense of axes within a renderer. */
  bool  ToggleAxes (vtkRenderer* renderer, vtkProp** prop = NULL);

  /** Update/create the bounding box prop using the overall bounds of the
   * source collection. The reference to an orienting camera allows proper
   * depth sorting of the translucent geometry.
   */
  void  UpdateBoundingBox (vtkRenderer* = 0);

  /*@{*/
  /** Set/Get the bounding box prop. */
  void SetBoundingBox (vtkProp3D*);
  vtkProp3D* GetBoundingBox (vtkRenderer* = 0);
  /*@}*/

  /*@{*/
  /** Set/Get the camera used to orient the bounding box prop. */
  void SetBoundingBoxRenderer (vtkRenderer*);
  vtkGetObjectMacro (BoundingBoxRenderer,vtkRenderer);
  /*@}*/

  /** Update/create the bounding box prop using the overall bounds of the
   * source collection. The reference to an orienting camera allows proper
   * depth sorting of the translucent geometry.
   */
  void  UpdateMaterialsDemo (vtkRenderer* = 0);

  /*@{*/
  /** Set/Get the bounding box prop. */
  void SetMaterialsDemo (vtkProp3D*);
  vtkProp3D* GetMaterialsDemo (vtkRenderer* = 0);
  /*@}*/

  /*@{*/
  /** Set/Get the camera used to orient the bounding box prop. */
  void SetMaterialsDemoRenderer (vtkRenderer*);
  vtkGetObjectMacro (MaterialsDemoRenderer,vtkRenderer);
  /*@}*/

  /*@{*/
  /**  */
  vtkSetMacro (MaterialsDemoRadii,int);
  vtkGetMacro (MaterialsDemoRadii,int);
  vtkSetMacro (MaterialsDemoResolution,int);
  vtkGetMacro (MaterialsDemoResolution,int);
  /*@}*/

  /** Get name of environment var to hold surface and volume data paths. */
  vtkGetStringMacro (DataPathsEnvVar);

protected:
  svvController (void);
  ~svvController();

  static void ClassInitialize (void);
  static void ClassFinalize (void);

  static int*           s_argc_ptr_;
  static char***        s_argv_ptr_;
  static char***        s_envp_ptr_;
  static int            s_args_called_;

  int                   RenderModePointsThreshold;
  int                   RenderModeCellsThreshold;

  svvMaterialNameMap*   MaterialNameMap;
  svvColorNameMap*      ColorNameMap;

  void SetInputSources (vtkInputSourceCollection*);
  void SetSurfacePipelines (vtkPolyDataPipelineCollection*);
  void SetSurfaceActors (vtkActorCollection*);

  vtkInputSourceCollection*             InputSources;
  vtkPolyDataPipelineCollection*        SurfacePipelines;
  vtkActorCollection*                   SurfaceActors;

  vtkRenderWindowInteractor*            Interactor;

  vtkProp3D*            BoundingBox;
  vtkRenderer*          BoundingBoxRenderer;

  vtkProp3D*            MaterialsDemo;
  int                   MaterialsDemoRadii;
  int                   MaterialsDemoResolution;
  vtkRenderer*          MaterialsDemoRenderer;

  /** Set name of environment var to hold surface and volume data paths. */
  vtkSetStringMacro (DataPathsEnvVar);

  char*                 DataPathsEnvVar;

public:
  /** \internal
   * Nested initializer class.
   * The static initializer object is declared below.
   */
  class Init
  {
  public:
    Init (void);
    ~Init();
  private:
    static long TranslationUnits;
    friend class svvController;
  };

  friend class Init;

private:
  svvController (const svvController&);
  void operator= (const svvController&);
};

#  ifndef SVV_CONTROLLER_INITIALIZED_
#    define SVV_CONTROLLER_INITIALIZED_
/** \internal
 * Global initializer object, to ensure construction of static objects.
 */
static svvController::Init svvControllerInitInstance;
#  endif /* SVV_CONTROLLER_INITIALIZED_ */

/** Singleton instance reference. */
extern svvController& svv;

SVV_NAMESPACE_END

#endif /* SVV_CONTROLLER_H_ */
/* 
 * End of: $Id: svvController.h,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $.
 * 
 */
