/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvMaterialNameMap.cxx,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "svvMaterialNameMap.h"
// C++ forwarding ANSI C
#include <cstdlib>
// STL
#  include <string>
#  include <map>
#  include <set>
#  include <algorithm> // for find_if
// VTK Common
#include "vtkObjectFactory.h"


SVV_NAMESPACE_BEGIN

// ----------------------------------------------------------------------------
//      s v v M a t e r i a l N a m e M a p
// ----------------------------------------------------------------------------

vtkCxxRevisionMacro (svvMaterialNameMap, "$Revision: 1.1.1.1 $");
vtkStandardNewMacro (svvMaterialNameMap);


// ----------------------------------------------------------------------------
ostream&
operator<< (ostream& aTarget, SvvGLMaterial& a)
{
  aTarget << "(\tKa = { "
          << a.Ka[0] << ", " << a.Ka[1] << ", " << a.Ka[2] << ", " << a.Ka[3]
          << " }\n"
          << "\tKd = { "
          << a.Kd[0] << ", " << a.Kd[1] << ", " << a.Kd[2] << ", " << a.Kd[3]
          << " }\n"
          << "\tKs = { "
          << a.Ks[0] << ", " << a.Ks[1] << ", " << a.Ks[2] << ", " << a.Ks[3]
          << " }\n"
          << "\tKe = { "
          << a.Ke[0] << ", " << a.Ke[1] << ", " << a.Ke[2] << ", " << a.Ke[3]
          << " }\n"
          << "\tSe = " << a.Se
          << "\t)";
  
  return aTarget;
}

ostream&
operator<< (ostream& aTarget, SvvMaterial& a)
{
  aTarget << "(\tKa = { "
          << a.Ka[0] << ", " << a.Ka[1] << ", " << a.Ka[2]
          << " }\n"
          << "\tKd = { "
          << a.Kd[0] << ", " << a.Kd[1] << ", " << a.Kd[2] << ", " << a.Kd[3]
          << " }\n"
          << "\tKs = { "
          << a.Ks[0] << ", " << a.Ks[1] << ", " << a.Ks[2]
          << " }\n"
          << "\tSe = " << a.Se
          << "\t)";
  
  return aTarget;
}


/** Material name / data pair. */
typedef vtkstd::map<vtkstd::string, SvvMaterial>        _SvvMaterialNameMap;
typedef vtkstd::pair<vtkstd::string, SvvMaterial>       _SvvMaterialNamePair;
typedef vtkstd::set<vtkstd::string>                     _SvvMaterialNameSet;


/* Initial number of entries. */
static const _SvvMaterialNameMap::size_type s_default_count_ = 52;

/* 
 * much lifted from Nate Robins' lightmaterial tutorial program
 */
static SvvMaterial s_default_materials_[] =
{
  /*  0, "default" */
  { { 0.200000, 0.200000, 0.200000 },           /* Ka */
    { 0.800000, 0.800000, 0.800000, 1.000000 }, /* Kd */
    { 0.000000, 0.000000, 0.000000 },           /* Ks */
    0.000000 },                                 /* Se */
  /*  1, "brass" */
  { { 0.329412, 0.223529, 0.027451 },           /* Ka */
    { 0.780392, 0.568627, 0.113725, 1.000000 }, /* Kd */
    { 0.992157, 0.941176, 0.807843 },           /* Ks */
    27.897400 },                                /* Se */
  /*  2, "bronze" */
  { { 0.212500, 0.127500, 0.054000 },           /* Ka */
    { 0.714000, 0.428400, 0.181440, 1.000000 }, /* Kd */
    { 0.393548, 0.271906, 0.166721 },           /* Ks */
    25.600000 },                                /* Se */
  /*  3, "polished bronze" */
  { { 0.250000, 0.148000, 0.064750 },           /* Ka */
    { 0.400000, 0.236800, 0.103600, 1.000000 }, /* Kd */
    { 0.774597, 0.458561, 0.200621 },           /* Ks */
    76.800003 },                                /* Se */
  /*  4, "chrome" */
  { { 0.250000, 0.250000, 0.250000 },           /* Ka */
    { 0.400000, 0.400000, 0.400000, 1.000000 }, /* Kd */
    { 0.774597, 0.774597, 0.774597 },           /* Ks */
    76.800003 },                                /* Se */
  /*  5, "copper" */
  { { 0.191250, 0.073500, 0.022500 },           /* Ka */
    { 0.703800, 0.270480, 0.082800, 1.000000 }, /* Kd */
    { 0.256777, 0.137622, 0.086014 },           /* Ks */
    12.800000 },                                /* Se */
  /*  6, "polished copper" */
  { { 0.229500, 0.088250, 0.027500 },           /* Ka */
    { 0.550800, 0.211800, 0.066000, 1.000000 }, /* Kd */
    { 0.580594, 0.223257, 0.069570 },           /* Ks */
    51.200001 },                                /* Se */
  /*  7, "gold" */
  { { 0.247250, 0.199500, 0.074500 },           /* Ka */
    { 0.751640, 0.606480, 0.226480, 1.000000 }, /* Kd */
    { 0.628281, 0.555802, 0.366065 },           /* Ks */
    51.200001 },                                /* Se */
  /*  8, "polished gold" */
  { { 0.247250, 0.224500, 0.064500 },           /* Ka */
    { 0.346150, 0.314300, 0.090300, 1.000000 }, /* Kd */
    { 0.797357, 0.723991, 0.208006 },           /* Ks */
    83.199997 },                                /* Se */
  /*  9, "pewter" */
  { { 0.105882, 0.058824, 0.113725 },           /* Ka */
    { 0.427451, 0.470588, 0.541176, 1.000000 }, /* Kd */
    { 0.333333, 0.333333, 0.521569 },           /* Ks */
    9.846150 },                                 /* Se */
  /* 10, "silver" */
  { { 0.192250, 0.192250, 0.192250 },           /* Ka */
    { 0.507540, 0.507540, 0.507540, 1.000000 }, /* Kd */
    { 0.508273, 0.508273, 0.508273 },           /* Ks */
    51.200001 },                                /* Se */
  /* 11, "polished silver" */
  { { 0.231250, 0.231250, 0.231250 },           /* Ka */
    { 0.277500, 0.277500, 0.277500, 1.000000 }, /* Kd */
    { 0.773911, 0.773911, 0.773911 },           /* Ks */
    89.599998 },                                /* Se */
  /* 12, "emerald" */
  { { 0.021500, 0.174500, 0.021500 },           /* Ka */
    { 0.075680, 0.614240, 0.075680, 0.550000 }, /* Kd */
    { 0.633000, 0.727811, 0.633000 },           /* Ks */
    76.800003 },                                /* Se */
  /* 13, "jade" */
  { { 0.135000, 0.222500, 0.157500 },           /* Ka */
    { 0.540000, 0.890000, 0.630000, 0.950000 }, /* Kd */
    { 0.316228, 0.316228, 0.316228 },           /* Ks */
    12.800000 },                                /* Se */
  /* 14, "obsidian" */
  { { 0.053750, 0.050000, 0.066250 },           /* Ka */
    { 0.182750, 0.170000, 0.225250, 0.820000 }, /* Kd */
    { 0.332741, 0.328634, 0.346435 },           /* Ks */
    38.400002 },                                /* Se */
  /* 15, "pearl" */
  { { 0.250000, 0.207250, 0.207250 },           /* Ka */
    { 1.000000, 0.829000, 0.829000, 0.922000 }, /* Kd */
    { 0.296648, 0.296648, 0.296648 },           /* Ks */
    11.264000 },                                /* Se */
  /* 16, "ruby" */
  { { 0.174500, 0.011750, 0.011750 },           /* Ka */
    { 0.614240, 0.041360, 0.041360, 0.550000 }, /* Kd */
    { 0.727811, 0.626959, 0.626959 },           /* Ks */
    76.800003 },                                /* Se */
  /* 17, "turquoise" */
  { { 0.100000, 0.187250, 0.174500 },           /* Ka */
    { 0.396000, 0.741510, 0.691020, 0.800000 }, /* Kd */
    { 0.297254, 0.308290, 0.306678 },           /* Ks */
    12.800000 },                                /* Se */
  /* 18, "black plastic" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.010000, 0.010000, 0.010000, 1.000000 }, /* Kd */
    { 0.500000, 0.500000, 0.500000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 19, "cyan plastic" */
  { { 0.000000, 0.100000, 0.060000 },           /* Ka */
    { 0.000000, 0.509804, 0.509804, 1.000000 }, /* Kd */
    { 0.501961, 0.501961, 0.501961 },           /* Ks */
    32.000000 },                                /* Se */
  /* 20, "green plastic" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.100000, 0.350000, 0.100000, 1.000000 }, /* Kd */
    { 0.450000, 0.550000, 0.450000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 21, "red plastic" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.500000, 0.000000, 0.000000, 1.000000 }, /* Kd */
    { 0.700000, 0.600000, 0.600000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 22, "white plastic" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.550000, 0.550000, 0.550000, 1.000000 }, /* Kd */
    { 0.700000, 0.700000, 0.700000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 23, "yellow plastic" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.500000, 0.500000, 0.000000, 1.000000 }, /* Kd */
    { 0.600000, 0.600000, 0.500000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 24, "gray plastic" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.280000, 0.280000, 0.280000, 1.000000 }, /* Kd */
    { 0.600000, 0.600000, 0.600000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 25, "blue plastic" */
  { { 0.000000, 0.060000, 0.100000 },           /* Ka */
    { 0.200000, 0.250000, 0.550000, 1.000000 }, /* Kd */
    { 0.500000, 0.500000, 0.650000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 26, "dark green plastic" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.050000, 0.250000, 0.050000, 1.000000 }, /* Kd */
    { 0.350000, 0.450000, 0.350000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 27, "purple plastic" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.450000, 0.100000, 0.500000, 1.000000 }, /* Kd */
    { 0.650000, 0.500000, 0.650000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 28, "magenta plastic" */
  { { 0.100000, 0.000000, 0.050000 },           /* Ka */
    { 0.550000, 0.300000, 0.500000, 1.000000 }, /* Kd */
    { 0.700000, 0.500000, 0.600000 },           /* Ks */
    32.000000 },                                /* Se */
  /* 29, "black rubber" */
  { { 0.020000, 0.020000, 0.020000 },           /* Ka */
    { 0.010000, 0.010000, 0.010000, 1.000000 }, /* Kd */
    { 0.400000, 0.400000, 0.400000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 30, "cyan rubber" */
  { { 0.000000, 0.050000, 0.050000 },           /* Ka */
    { 0.400000, 0.500000, 0.500000, 1.000000 }, /* Kd */
    { 0.040000, 0.700000, 0.700000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 31, "green rubber" */
  { { 0.000000, 0.050000, 0.000000 },           /* Ka */
    { 0.400000, 0.500000, 0.400000, 1.000000 }, /* Kd */
    { 0.040000, 0.700000, 0.040000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 32, "red rubber" */
  { { 0.050000, 0.000000, 0.000000 },           /* Ka */
    { 0.500000, 0.400000, 0.400000, 1.000000 }, /* Kd */
    { 0.700000, 0.040000, 0.040000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 33, "white rubber" */
  { { 0.050000, 0.050000, 0.050000 },           /* Ka */
    { 0.500000, 0.500000, 0.500000, 1.000000 }, /* Kd */
    { 0.700000, 0.700000, 0.700000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 34, "yellow rubber" */
  { { 0.050000, 0.050000, 0.000000 },           /* Ka */
    { 0.500000, 0.500000, 0.400000, 1.000000 }, /* Kd */
    { 0.700000, 0.700000, 0.040000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 35, "gray rubber" */
  { { 0.035000, 0.035000, 0.035000 },           /* Ka */
    { 0.255000, 0.255000, 0.255000, 1.000000 }, /* Kd */
    { 0.550000, 0.550000, 0.550000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 36, "blue rubber" */
  { { 0.000000, 0.000000, 0.050000 },           /* Ka */
    { 0.400000, 0.400000, 0.500000, 1.000000 }, /* Kd */
    { 0.040000, 0.040000, 0.700000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 37, "magenta rubber" */
  { { 0.050000, 0.000000, 0.050000 },           /* Ka */
    { 0.500000, 0.400000, 0.500000, 1.000000 }, /* Kd */
    { 0.700000, 0.040000, 0.700000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 38, "lavender rubber" */
  { { 0.050000, 0.000000, 0.050000 },           /* Ka */
    { 0.500000, 0.450000, 0.500000, 1.000000 }, /* Kd */
    { 0.350000, 0.300000, 0.550000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 39, "sky blue rubber" */
  { { 0.000000, 0.050000, 0.100000 },           /* Ka */
    { 0.400000, 0.500000, 0.600000, 1.000000 }, /* Kd */
    { 0.000000, 0.563750, 1.000000 },           /* Ks */
    10.000000 },                                /* Se */
  /* 40, "gloss black" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.019608, 0.007843, 0.019608, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 41, "gloss blue" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.078431, 0.141176, 0.709804, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 42, "gloss brown" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.423529, 0.227451, 0.035294, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 43, "gloss dark green" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.027451, 0.521569, 0.086275, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 44, "gloss light green" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.176471, 0.882353, 0.082353, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 45, "gloss orange" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.917647, 0.474510, 0.000000, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 46, "gloss red" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.823529, 0.000000, 0.000000, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 47, "gloss tan" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.725490, 0.678431, 0.278431, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 48, "gloss teal" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.231373, 0.501961, 0.603922, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 49, "gloss violet" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.552941, 0.258824, 0.831373, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 50, "gloss white" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 1.000000, 0.968627, 0.921569, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
  /* 51, "gloss yellow" */
  { { 0.000000, 0.000000, 0.000000 },           /* Ka */
    { 0.913725, 0.925490, 0.000000, 1.000000 }, /* Kd */
    { 1.000000, 1.000000, 1.000000 },           /* Ks */
    56.000000 },                                /* Se */
};

// data
static const _SvvMaterialNamePair s_default_material_pairs_[] =
{
  _SvvMaterialNamePair( "default",            s_default_materials_[ 0] ),
  _SvvMaterialNamePair( "brass",              s_default_materials_[ 1] ),
  _SvvMaterialNamePair( "bronze",             s_default_materials_[ 2] ),
  _SvvMaterialNamePair( "polished bronze",    s_default_materials_[ 3] ),
  _SvvMaterialNamePair( "chrome",             s_default_materials_[ 4] ),
  _SvvMaterialNamePair( "copper",             s_default_materials_[ 5] ),
  _SvvMaterialNamePair( "polished copper",    s_default_materials_[ 6] ),
  _SvvMaterialNamePair( "gold",               s_default_materials_[ 7] ),
  _SvvMaterialNamePair( "polished gold",      s_default_materials_[ 8] ),
  _SvvMaterialNamePair( "pewter",             s_default_materials_[ 9] ),
  _SvvMaterialNamePair( "silver",             s_default_materials_[10] ),
  _SvvMaterialNamePair( "polished silver",    s_default_materials_[11] ),
  _SvvMaterialNamePair( "emerald",            s_default_materials_[12] ),
  _SvvMaterialNamePair( "jade",               s_default_materials_[13] ),
  _SvvMaterialNamePair( "obsidian",           s_default_materials_[14] ),
  _SvvMaterialNamePair( "pearl",              s_default_materials_[15] ),
  _SvvMaterialNamePair( "ruby",               s_default_materials_[16] ),
  _SvvMaterialNamePair( "turquoise",          s_default_materials_[17] ),
  _SvvMaterialNamePair( "black plastic",      s_default_materials_[18] ),
  _SvvMaterialNamePair( "cyan plastic",       s_default_materials_[19] ),
  _SvvMaterialNamePair( "green plastic",      s_default_materials_[20] ),
  _SvvMaterialNamePair( "red plastic",        s_default_materials_[21] ),
  _SvvMaterialNamePair( "white plastic",      s_default_materials_[22] ),
  _SvvMaterialNamePair( "yellow plastic",     s_default_materials_[23] ),
  _SvvMaterialNamePair( "gray plastic",       s_default_materials_[24] ),
  _SvvMaterialNamePair( "blue plastic",       s_default_materials_[25] ),
  _SvvMaterialNamePair( "dark green plastic", s_default_materials_[26] ),
  _SvvMaterialNamePair( "purple plastic",     s_default_materials_[27] ),
  _SvvMaterialNamePair( "magenta plastic",    s_default_materials_[28] ),
  _SvvMaterialNamePair( "black rubber",       s_default_materials_[29] ),
  _SvvMaterialNamePair( "cyan rubber",        s_default_materials_[30] ),
  _SvvMaterialNamePair( "green rubber",       s_default_materials_[31] ),
  _SvvMaterialNamePair( "red rubber",         s_default_materials_[32] ),
  _SvvMaterialNamePair( "white rubber",       s_default_materials_[33] ),
  _SvvMaterialNamePair( "yellow rubber",      s_default_materials_[34] ),
  _SvvMaterialNamePair( "gray rubber",        s_default_materials_[35] ),
  _SvvMaterialNamePair( "blue rubber",        s_default_materials_[36] ),
  _SvvMaterialNamePair( "magenta rubber",     s_default_materials_[37] ),
  _SvvMaterialNamePair( "lavender rubber",    s_default_materials_[38] ),
  _SvvMaterialNamePair( "sky blue rubber",    s_default_materials_[39] ),
  _SvvMaterialNamePair( "gloss black",        s_default_materials_[40] ),
  _SvvMaterialNamePair( "gloss blue",         s_default_materials_[41] ),
  _SvvMaterialNamePair( "gloss brown",        s_default_materials_[42] ),
  _SvvMaterialNamePair( "gloss dark green",   s_default_materials_[43] ),
  _SvvMaterialNamePair( "gloss light green",  s_default_materials_[44] ),
  _SvvMaterialNamePair( "gloss orange",       s_default_materials_[45] ),
  _SvvMaterialNamePair( "gloss red",          s_default_materials_[46] ),
  _SvvMaterialNamePair( "gloss tan",          s_default_materials_[47] ),
  _SvvMaterialNamePair( "gloss teal",         s_default_materials_[48] ),
  _SvvMaterialNamePair( "gloss violet",       s_default_materials_[49] ),
  _SvvMaterialNamePair( "gloss white",        s_default_materials_[50] ),
  _SvvMaterialNamePair( "gloss yellow",       s_default_materials_[51] ),
};


// ----------------------------------------------------------------------------
class svvMaterialNameMapInternal
{
public:
  _SvvMaterialNameMap           Map;
  _SvvMaterialNameSet           Names;
  _SvvMaterialNameMap::iterator MapIT;
  _SvvMaterialNameSet::iterator NameIT;

  vtkstd::string                Title;

  svvMaterialNameMapInternal (void)
    : Map(),
      Names(),
      MapIT(Map.end()),
      NameIT(Names.end()),
      Title("map")
    {}
};


// ----------------------------------------------------------------------------
svvMaterialNameMap::svvMaterialNameMap (void)
  : Internal(new svvMaterialNameMapInternal)
{
}

svvMaterialNameMap::~svvMaterialNameMap()
{
  delete this->Internal;
  this->Internal = NULL;
}

// ----------------------------------------------------------------------------
void
svvMaterialNameMap::SetTitle (const char* const& aName)
{
  if (aName != NULL && *aName != '\0')
    {
    this->Internal->Title.assign(aName);
    }
  else
    {
    this->Internal->Title.assign("map");
    }
}

void
svvMaterialNameMap::GetTitle (const char* & aName)
{
  aName = this->Internal->Title.c_str();
}

// ----------------------------------------------------------------------------
void
svvMaterialNameMap::InsertDefaults (void)
{
  this->Internal->Map.insert( &s_default_material_pairs_[0],
                              &s_default_material_pairs_[s_default_count_] );
}

// ----------------------------------------------------------------------------
bool
svvMaterialNameMap::AddMaterial (const char* const& aName,
                                 const SvvMaterial& aMaterial)
{
  return ( this->Internal->
           Map.insert(vtkstd::make_pair(aName, aMaterial)) ).second;
}

bool
svvMaterialNameMap::FindMaterial (const char* const& aName,
                                  SvvMaterial&       aMaterial)
{
  _SvvMaterialNameMap::iterator pos;

  if ((pos = this->Internal->Map.find(aName)) != this->Internal->Map.end())
    {
    aMaterial = pos->second;
    return true;
    }

  return false;
}

bool
svvMaterialNameMap::EraseMaterial (const char* const& aName)
{
  return (this->Internal->Map.erase(aName) > 0);
}

// ----------------------------------------------------------------------------
class MaterialEq
{
  SvvMaterial material_;

public:
  MaterialEq (const SvvMaterial& aMaterial) : material_(aMaterial) {}

  bool operator() (std::pair<const std::string, SvvMaterial> aElem)
    { return ( aElem.second.Ka[0] == this->material_.Ka[0] &&
               aElem.second.Ka[1] == this->material_.Ka[1] &&
               aElem.second.Ka[2] == this->material_.Ka[2] &&               
               aElem.second.Kd[0] == this->material_.Kd[0] &&               
               aElem.second.Kd[1] == this->material_.Kd[1] &&               
               aElem.second.Kd[2] == this->material_.Kd[2] &&               
               aElem.second.Kd[3] == this->material_.Kd[3] &&               
               aElem.second.Ks[0] == this->material_.Ks[0] &&               
               aElem.second.Ks[1] == this->material_.Ks[1] &&               
               aElem.second.Ks[2] == this->material_.Ks[2] &&               
               aElem.second.Se    == this->material_.Se ); }
};

bool
svvMaterialNameMap::FindName (const SvvMaterial& aMaterial,
                              const char* &      aName)
{
  _SvvMaterialNameMap::iterator pos = this->Internal->Map.begin();

  if (!this->Internal->Names.empty())
    {
    this->Internal->Names.clear();
    }

  while ( ( pos = vtkstd::find_if( pos, this->Internal->Map.end(),
                                   MaterialEq(aMaterial) ) )
          != this->Internal->Map.end() )
    {
    this->Internal->Names.insert(pos->first);
    }

  if ( (this->Internal->NameIT = this->Internal->Names.begin())
       != this->Internal->Names.end() )
    {
    aName = this->Internal->NameIT->c_str();
    return true;
    }

  return false;
}

bool
svvMaterialNameMap::FindNextName (const char* & aName)
{
  if (++(this->Internal->NameIT) != this->Internal->Names.end())
    {
    aName = this->Internal->NameIT->c_str();
    return true;
    }

  return false;
}

// ----------------------------------------------------------------------------
void
svvMaterialNameMap::GetSize (SvvSize& a)
{
  a = SvvSize(this->Internal->Map.size());
}

// ----------------------------------------------------------------------------
void
svvMaterialNameMap::PrintMaterial (ostream& aTarget, const char* const& aName)
{
  _SvvMaterialNameMap::iterator pos;

  if ((pos = this->Internal->Map.find(aName)) != this->Internal->Map.end())
    {
    aTarget << "\"" << aName << "\":\n" << pos->second << endl;
    }
}

void
svvMaterialNameMap::PrintAll (ostream& aTarget)
{
  aTarget << this->Internal->Title << ": "
          << this->Internal->Map.size() << " materials\n" << endl;

  _SvvMaterialNameMap::iterator pos;

  for ( pos  = this->Internal->Map.begin();
        pos != this->Internal->Map.end();
        ++pos )
    {
    this->PrintMaterial(aTarget, pos->first.c_str());
    }
}

// ----------------------------------------------------------------------------
void
svvMaterialNameMap::InitTraversal (const char* const&   aName)
{
  if      (this->Internal->Map.empty())
    {
    this->Internal->MapIT = this->Internal->Map.end();
    }
  else if (aName != NULL && *aName != '\0')
    {
    this->Internal->MapIT = this->Internal->Map.find(aName);
    }
  else
    {
    this->Internal->MapIT = this->Internal->Map.begin();
    }
}

bool
svvMaterialNameMap::TraverseForward (const char* &      aName,
                                     SvvMaterial&       aMaterial)
{
  if (this->Internal->MapIT == this->Internal->Map.end())
    {
    return false;
    }

  aName         = this->Internal->MapIT->first.c_str();
  aMaterial     = this->Internal->MapIT->second;

  ++(this->Internal->MapIT);

  return true;
}

SVV_NAMESPACE_END

/* 
 * End of: $Id: svvMaterialNameMap.cxx,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $.
 * 
 */
