/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvMaterialNameMap.h,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#ifndef SVV_MATERIAL_NAME_MAP_H_
#  define SVV_MATERIAL_NAME_MAP_H_
#  include "svvTypes.h"
// VTK Common
#  include "vtkObject.h"

SVV_NAMESPACE_BEGIN

class svvMaterialNameMapInternal;

/** \class   svvMaterialNameMap
 *  \brief   Associative array of material - name pairs.
 * 
 * \author  Sean McInerney
 * \version $Revision: 1.1.1.1 $
 * \date    $Date: 2006/12/19 22:58:35 $
 */
class svvMaterialNameMap : public vtkObject
{
public:
  static svvMaterialNameMap* New (void);
  vtkTypeRevisionMacro (svvMaterialNameMap, vtkObject);

  void  SetTitle (const char* const&            name);
  void  GetTitle (const char* &                 name);

  void  InsertDefaults (void);

  /** Add an associated \c name - \c material pair to the map.
   * 
   * \return \c true on success, \c false otherwise.
   */
  bool  AddMaterial (const char* const&         name,
                     const SvvMaterial&         material);

  /** Find \c material data associated with the given \c name.
   * 
   * \return \c true on success, \c false otherwise.
   */
  bool  FindMaterial (const char* const&        name,
                      SvvMaterial&              material);

  /** Erase the material pair associated with the given \c name.
   * 
   * \return \c true on success, \c false otherwise.
   */
  bool  EraseMaterial (const char* const&       name);

  /** Find the first \c name in the map associated with the given \c material.
   * 
   * \return \c true on success, \c false otherwise.
   */
  bool  FindName (const SvvMaterial&            material,
                  const char* &                 name);

  /**
   * Find the next \c name in the map associated with the material 
   * specified in a previous call to \c FindName.
   * 
   * \return \c true on success, \c false otherwise.
   */
  bool  FindNextName (const char* &             name);

  /**
   * Print the material data associated with \c name as a formatted 
   * string on the \c target output stream.
   */
  void  PrintMaterial (ostream&                 target,
                       const char* const&       name);

  /**
   * Print the all material pairs as a formatted string on 
   * the \c target output stream.
   */
  void  PrintAll (ostream&                      target);

  /** Get the number of entries in the map. */
  void  GetSize (SvvSize&                       n);

  /** Initiates traversal through the map.
   *
   * \param name  The name of the material at which the traversal
   *              iterator will be positioned. If \name is \c NULL or
   *              empty, traversal starts at the first pair in the map.
   */
  void  InitTraversal (const char* const&       name = NULL);

  /**
   * Get the current \c name - \c material association in the map
   * and iterate forward to the next pair.
   * 
   * \return \c false if either the map is empty, traversal was initialized
   *         with a name not associated with any material in the map, or
   *         traversal has reached the end of the map. Otherwise \c true
   *         is returned.
   */
  bool  TraverseForward (const char* &          name,
                         SvvMaterial&           material);

protected:
  svvMaterialNameMap (void);
  ~svvMaterialNameMap();

  svvMaterialNameMapInternal*   Internal;

private:
  svvMaterialNameMap (const svvMaterialNameMap&); // Not implemented.
  void operator= (const svvMaterialNameMap&); // Not implemented.
};


ostream& operator<< (ostream&, SvvGLMaterial&);
ostream& operator<< (ostream&, SvvMaterial&);


SVV_NAMESPACE_END

#endif /* SVV_MATERIAL_NAME_MAP_H_ */
/* 
 * End of: $Id: svvMaterialNameMap.h,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $.
 * 
 */
