/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvModel.h,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $
 * 
 * Copyright (c) 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#ifndef SVV_MODEL_H_
#  define SVV_MODEL_H_
#  include "svvConfiguration.h"
#  include <GL/gl.h>

SVV_NAMESPACE_BEGIN

typedef struct SvvModelVerticesRec_
{
  GLvoid* pointer; // vertices
  GLsizei count;
  // GL_V2F, GL_V3F, GL_C4UB_V2F, GL_C4UB_V3F, GL_C3F_V3F, GL_N3F_V3F, 
  //   GL_C4F_N3F_V3F, GL_T2F_V3F, GL_T4F_V4F, GL_T2F_C4UB_V3F, GL_T2F_C3F_V3F, 
  //   GL_T2F_N3F_V3F, GL_T2F_C4F_N3F_V3F, and GL_T4F_C4F_N3F_V4F
  GLenum  format;
  // Offset in bytes between each aggregate array element.
  //   If stride is 0, the aggregate elements are stored consecutively.
  GLsizei stride;
} SvvModelVertices;

typedef struct SvvModelIndicesRec_
{
  GLvoid* pointer;  // indices
  GLsizei count;
  // GL_POINTS, GL_LINE_STRIP, GL_LINE_LOOP, GL_LINES, GL_TRIANGLE_STRIP,
  //   GL_TRIANGLE_FAN, GL_TRIANGLES, GL_QUAD_STRIP, GL_QUADS, and GL_POLYGON
  GLenum  mode;
  // GL_UNSIGNED_BYTE, GL_UNSIGNED_SHORT, or GL_UNSIGNED_INT
  GLenum  type;
} SvvModelIndices;

#  define SVV_MODELDATA_NAME_SIZE 40
typedef struct SvvModelDataRec_
{
  SvvModelVertices vertices;
  SvvModelIndices  indices;
  char             name[SVV_MODELDATA_NAME_SIZE];
} SvvModelData;


// 
// svvModel class
// 
class svvModel : public SvvModelDataRec_
{
private:
  GLuint DisplayList;

public:
  // stipped down and out strlcpy()
  inline static void SetName (char* name, const char* str);
  // get the total number of elements for indices to specified rendering mode
  inline static GLsizei GetTotalElements (GLsizei count, GLenum mode);

  // constructor
  svvModel (const char* modelname,
            GLvoid*     vptr,
            GLsizei     nvertices,
            GLvoid*     iptr,
            GLsizei     nindices,
            GLenum      arrayformat   = GL_N3F_V3F,
            GLsizei     elementstride = 0,
            GLenum      rendermode    = GL_TRIANGLES,
            GLenum      indextype     = GL_UNSIGNED_SHORT)
    : DisplayList(0)
    {
      if(modelname)  svvModel::SetName(this->name,modelname);
      else           this->name[0] = '\0';

      this->vertices.pointer = vptr;
      this->vertices.count   = nvertices;
      this->vertices.format  = arrayformat;
      this->vertices.stride  = elementstride;

      this->indices.pointer  = iptr;
      this->indices.count    = nindices;
      this->indices.mode     = rendermode;
      this->indices.type     = indextype;
    }

  // get the display list (created if necessary)
  virtual GLuint GetDisplayList (void);
  // delete the display list
  virtual void   DeleteDisplayList (void);
};

// 
// svvModel pointer type
// 
typedef svvModel* svvModelPtr;


// 
// inline static methods
// 

inline void
svvModel::SetName (char* name, const char* str)
{
  register char*       d = name;
  register const char* s = str;
  register std::size_t n = SVV_MODELDATA_NAME_SIZE;

  // Copy as many bytes as will fit
  if(n && --n)
    {
    do
      {
      if ((*d++ = *s++) == 0) break;
    }
    while (--n);
    }
  // Not enough room in dst, NUL-terminate dst
  if (!n) *d = '\0';
}

inline GLsizei
svvModel::GetTotalElements (GLsizei count, GLenum mode)
{
  switch (mode)
    {
    case GL_POINTS:         return count;
    case GL_LINES:          return (count * 2);
    case GL_LINE_STRIP:     return (count + 1);
    case GL_LINE_LOOP:      return count;
    case GL_TRIANGLES:      return (count * 3);
    case GL_TRIANGLE_STRIP: return (count + 2);
    case GL_TRIANGLE_FAN:   return (count + 2);
    case GL_QUADS:          return (count * 4);
    case GL_QUAD_STRIP:     return ((count+1) * 2);
    default:                return 0;
    }
}


// 
// inline instance methods
// 

inline GLuint
svvModel::GetDisplayList (void)
{
  if (this->DisplayList != 0)
    {
    return this->DisplayList;
    }

  this->DisplayList = glGenLists(2);

  glNewList(this->DisplayList, GL_COMPILE);
    {
    glPushClientAttrib(GL_CLIENT_VERTEX_ARRAY_BIT);
      {
      glInterleavedArrays( this->vertices.format,
                           this->vertices.stride,
                           this->vertices.pointer );
      glDrawElements( this->indices.mode,
                      svvModel::GetTotalElements( this->indices.count,
                                                  this->indices.mode ),
                      this->indices.type,
                      this->indices.pointer );
      }
    glPopClientAttrib();
    }
  glEndList();

  glNewList(this->DisplayList+1, GL_COMPILE);
    {
    // first, render the back faces of the geometry
    glCullFace(GL_FRONT);
    glCallList(this->DisplayList);
    // next, render the front faces of the geometry
    glCullFace(GL_BACK);
    glCallList(this->DisplayList);
    }
  glEndList();
      
  return this->DisplayList;
}

inline void
svvModel::DeleteDisplayList (void)
{
  if (this->DisplayList != 0)
    {
    glDeleteLists(this->DisplayList,2);
    this->DisplayList = 0;
    }
}

SVV_NAMESPACE_END

#endif /* SVV_MODEL_H_ */
/*
 * End of: $Id: svvModel.h,v 1.1.1.1 2006/12/19 22:58:35 christianh Exp $.
 * 
 */
