/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: svvPalette.cxx,v 1.1.1.1 2006/12/19 22:58:36 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include "svvPalette.h"
// C++ forwarding standard C headers.
#include <cmath>        // floor(3M), rint(3M)
#include <cctype>       // isspace(3C)
#include <cstdio>       // sscanf(3C)
// STL
#include <string>       // string
#include <map>          // map, pair
#include <set>          // set
#include <sstream>      // istringstream
// VTK Common
#include "vtkSmartPointer.h"


SVV_NAMESPACE_BEGIN

// ----------------------------------------------------------------------------
//      s v v P a l e t t e
// ----------------------------------------------------------------------------
vtkCxxRevisionMacro (svvPalette, "$Revision: 1.1.1.1 $");


// ----------------------------------------------------------------------------
ostream&
operator<< (ostream& aTarget, SvvGLMaterial& a)
{
  aTarget << "(\tKa = { "
          << a.Ka[0] << ", " << a.Ka[1] << ", " << a.Ka[2] << ", " << a.Ka[3]
          << " }\n"
          << "\tKd = { "
          << a.Kd[0] << ", " << a.Kd[1] << ", " << a.Kd[2] << ", " << a.Kd[3]
          << " }\n"
          << "\tKs = { "
          << a.Ks[0] << ", " << a.Ks[1] << ", " << a.Ks[2] << ", " << a.Ks[3]
          << " }\n"
          << "\tKe = { "
          << a.Ke[0] << ", " << a.Ke[1] << ", " << a.Ke[2] << ", " << a.Ke[3]
          << " }\n"
          << "\tSe = " << a.Se
          << "\t)";
  
  return aTarget;
}

ostream&
operator<< (ostream& aTarget, SvvMaterial& a)
{
  aTarget << "(\tKa = { "
          << a.Ka[0] << ", " << a.Ka[1] << ", " << a.Ka[2]
          << " }\n"
          << "\tKd = { "
          << a.Kd[0] << ", " << a.Kd[1] << ", " << a.Kd[2] << ", " << a.Kd[3]
          << " }\n"
          << "\tKs = { "
          << a.Ks[0] << ", " << a.Ks[1] << ", " << a.Ks[2]
          << " }\n"
          << "\tSe = " << a.Se
          << "\t)";
  
  return aTarget;
}


// ----------------------------------------------------------------------------
typedef vtkstd::string                                  _SvvString;
typedef vtkSmartPointer<svvPalette>                     _SvvPaletteSPtr;
typedef vtkstd::map<_SvvString, _SvvPaletteSPtr>        _SvvPaletteMap;
typedef vtkstd::pair<_SvvString, _SvvPaletteSPtr>       _SvvPalettePair;
typedef vtkstd::set<_SvvString>                         _SvvStringSet;


// ----------------------------------------------------------------------------
class svvPaletteInternal
{
public:
  friend class svvPalette;

  svvPalette*                   Parent;

  _SvvString                    BaseName;
  _SvvString                    FullName;

  _SvvPaletteMap                Palettes;
  _SvvPaletteMap::iterator      PaletteIT;


  static bool                           ReadingFloats;

  /** \internal
   * Static search path along which to seach for file(s) of RGB-name pairs.
   */
  static _SvvStringSet                  SearchPaths;
  static _SvvStringSet::iterator        SearchPathIT;

  /** \internal
   * Static name of the file(s) of RGB-name pairs.
   */
  static _SvvStringSet                  SearchFiles;
  static _SvvStringSet::iterator        SearchFileIT;

  svvPaletteInternal (void)
    : Parent(NULL),
      BaseName("."),
      FullName(BaseName),
      Palettes(),
      PaletteIT(Palettes.end())
    {}
};


// ----------------------------------------------------------------------------
const char* const
svvPalette::PalettePathsEnvVar = "SVV_PALETTE_PATHS";

const char* const
svvPalette::PaletteFilesEnvVar = "SVV_PALETTE_FILES";

static const _SvvString s_paths_[] =
{
  "/usr/X11R6/lib/X11",
  "/usr/openwin/lib/X11",
  "/usr/lib/X11",
  "/usr/local/share/svv",
  "/usr/share/svv",
  ".",
};

static const _SvvString s_files_[] =
{
  "rgb.txt",
  "materials.xml",
  "palette.xml",
};

bool
svvPaletteInternal::ReadingFloats = false;

_SvvStringSet
svvPaletteInternal::SearchPaths;

_SvvStringSet::iterator
svvPaletteInternal::SearchPathIT =
svvPaletteInternal::SearchPaths.end();

_SvvStringSet
svvPaletteInternal::SearchFiles;

_SvvStringSet::iterator
svvPaletteInternal::SearchFileIT =
svvPaletteInternal::SearchFiles.end();


// ----------------------------------------------------------------------------
svvPalette::svvPalette (void)
  : Internal(new svvPaletteInternal)
{
}

svvPalette::~svvPalette()
{
  delete this->Internal;
  this->Internal = NULL;
}

// ----------------------------------------------------------------------------
void
svvPalette::BuildSearchLists (void)
{
  if (svvPaletteInternal::SearchPaths.empty())
    {
    svvPaletteInternal::SearchPaths.insert( &s_paths_[0],
                                            &s_paths_[1] );
    }

  if (svvPaletteInternal::SearchFiles.empty())
    {
    svvPaletteInternal::SearchFiles.insert( &s_files_[0],
                                            &s_files_[1] );
    }

  const _SvvString delims(" \t:;");
  const _SvvString dirsep("/");

  _SvvString concat, pathSub, nameSub;
  _SvvString::size_type beg, end;

  char* cptr;

  if (cptr = getenv(svvPalette::PalettePathsEnvVar))
    {
    concat = cptr;
    end    = 0;

    while ((beg = concat.find_first_not_of(delims,end)) != _SvvString::npos)
      {
      if ((end = concat.find_first_of(delims,beg)) == _SvvString::npos)
        {
        end = concat.length(); // End of word is end of line.
        }

      pathSub = concat.substr(beg,end-beg);

      svvPaletteInternal::SearchPaths.insert(pathSub);
      }
    }

  if (cptr = getenv(svvPalette::PaletteFilesEnvVar))
    {
    concat = cptr;
    end    = 0;

    while ((beg = concat.find_first_not_of(delims,end)) != _SvvString::npos)
      {
      if ((end = concat.find_first_of(delims,beg)) == _SvvString::npos)
        {
        end = concat.length(); // End of word is end of line.
        }

      nameSub = concat.substr(beg,end-beg);

      svvPaletteInternal::SearchFiles.insert(nameSub);
      }
    }
}

// ----------------------------------------------------------------------------
void
svvPalette::InitSearchPathsTraversal (const char* const& aName)
{
  if      (svvPaletteInternal::SearchPaths.empty())
    {
    svvPaletteInternal::SearchPathIT =
      svvPaletteInternal::SearchPaths.end();
    }
  else if (aName != NULL && *aName != '\0')
    {
    svvPaletteInternal::SearchPathIT =
      svvPaletteInternal::SearchPaths.find(aName);
    }
  else
    {
    svvPaletteInternal::SearchPathIT =
      svvPaletteInternal::SearchPaths.begin();
    }
}

bool
svvPalette::TraverseSearchPathsForward (const char* & aName)
{
  if ( svvPaletteInternal::SearchPathIT
       == svvPaletteInternal::SearchPaths.end() )
    {
    return false;
    }

  aName = svvPaletteInternal::SearchPathIT->c_str();

  ++(svvPaletteInternal::SearchPathIT);

  return true;
}

void
svvPalette::InitSearchFilesTraversal (const char* const& aName)
{
  if      (svvPaletteInternal::SearchFiles.empty())
    {
    svvPaletteInternal::SearchFileIT =
      svvPaletteInternal::SearchFiles.end();
    }
  else if (aName != NULL && *aName != '\0')
    {
    svvPaletteInternal::SearchFileIT =
      svvPaletteInternal::SearchFiles.find(aName);
    }
  else
    {
    svvPaletteInternal::SearchFileIT =
      svvPaletteInternal::SearchFiles.begin();
    }
}

bool
svvPalette::TraverseSearchFilesForward (const char* & aName)
{
  if ( svvPaletteInternal::SearchFileIT
       == svvPaletteInternal::SearchFiles.end() )
    {
    return false;
    }

  aName = svvPaletteInternal::SearchFileIT->c_str();

  ++(svvPaletteInternal::SearchFileIT);

  return true;
}

// ----------------------------------------------------------------------------
svvPalette*
svvPalette::GetParent (void) const
{
  return this->Internal->Parent;
}

// ----------------------------------------------------------------------------
void
svvPalette::SetBaseName (const char* const& aName)
{
  if (aName != NULL && *aName != '\0')
    {
    this->Internal->BaseName.assign(aName);
    }
  else
    {
    this->Internal->BaseName.assign(".");
    }
}

void
svvPalette::GetBaseName (const char* & aName)
{
  aName = this->Internal->BaseName.c_str();
}

void
svvPalette::GetFullName (const char* & aName)
{
  const _SvvString slash("/");
  const char* parentName = "/";

  if (this->IsRoot())
    {
    const char* parentName = NULL;

    this->Internal->Parent->GetFullName(parentName);
    this->Internal->FullName.assign(parentName);
    }

  this->Internal->FullName += slash;
  this->Internal->FullName += this->Internal->BaseName;

  aName = this->Internal->FullName.c_str();
}

// ----------------------------------------------------------------------------
void
svvPalette::GetNumberOfPalettes (SvvSize& aResult)
{
  aResult = SvvSize(this->Internal->Palettes.size());
}

// ----------------------------------------------------------------------------
void
svvPalette::InitPalettesTraversal (const char* const& aName)
{
  if      (this->Internal->Palettes.empty())
    {
    this->Internal->PaletteIT = this->Internal->Palettes.end();
    }
  else if (aName != NULL && *aName != '\0')
    {
    this->Internal->PaletteIT = this->Internal->Palettes.find(aName);
    }
  else
    {
    this->Internal->PaletteIT = this->Internal->Palettes.begin();
    }
}

bool
svvPalette::TraversePalettesForward (const char* &         aName,
                                     svvPalette* & aPalette)
{
  if (this->Internal->PaletteIT == this->Internal->Palettes.end())
    {
    return false;
    }

  aName         = this->Internal->PaletteIT->first.c_str();
  aPalette      = this->Internal->PaletteIT->second.GetPointer();

  ++(this->Internal->PaletteIT);

  return true;
}

// ----------------------------------------------------------------------------
void
svvPalette::InsertDefaultPalettes (bool aFlag)
{
  // no-op
}

// ----------------------------------------------------------------------------
void
svvPalette::InsertDefaultMaterials (bool aFlag)
{
  // no-op
}

// ----------------------------------------------------------------------------
static bool
s_get_next_color (ifstream&             aStream,
                  _SvvString&       aName,
                  SvvColorWord&         aColor)
{
  _SvvString line;
  float          r, g, b;
  SvvColorWord   color;

  do
    {
    if (!getline(aStream,line)) return false;
    }
  while (line[0]=='!' || line[0]=='#' || line[0]=='\n');

  vtkstd::istringstream ist(line);

  ist >> r >> g >> b;

  //while (ist.good())
  //  {
  //  ist >> aName;
  //  if (ist.eof()) break;
  //  aName.append(" ");
  //  }

  if (!ist.good()) return false;

  SvvColorByte rb, gb, bb;

  if      (svvPaletteInternal::ReadingFloats)
    {
    rb = SvvColorByte(rint(r*float(SVV_RGB_MAX)));
    gb = SvvColorByte(rint(g*float(SVV_RGB_MAX)));
    bb = SvvColorByte(rint(b*float(SVV_RGB_MAX)));
    aColor = SVV_MAKE_RGB( SVV_CLAMPBYTE(rb),
                           SVV_CLAMPBYTE(gb),
                           SVV_CLAMPBYTE(bb) );
    }
  else if (floor(r)!=r || floor(g)!=g || floor(b)!=b)
    {
    svvPaletteInternal::ReadingFloats = true;
    rb = SvvColorByte(rint(r*float(SVV_RGB_MAX)));
    gb = SvvColorByte(rint(g*float(SVV_RGB_MAX)));
    bb = SvvColorByte(rint(b*float(SVV_RGB_MAX)));
    aColor = SVV_MAKE_RGB( SVV_CLAMPBYTE(rb),
                           SVV_CLAMPBYTE(gb),
                           SVV_CLAMPBYTE(bb) );
    }
  else
    {
    rb = SVV_CLAMPBYTE(r);
    gb = SVV_CLAMPBYTE(g);
    bb = SVV_CLAMPBYTE(b);
    aColor = SVV_MAKE_RGB(rb,gb,bb);
    }

  ist.clear();
  aName.erase();

  const _SvvString ws(" ");
  _SvvString       tmp;

  while (ist.good())
    {
    ist >> tmp;
    aName.append(tmp);
    if (ist.eof()) break;
    aName.append(ws);
    }
  
  return ((aStream.eof() || !aStream.good()) ? false : true);
}

void
svvPalette::InsertDefaultColors (bool aFlag)
{
  svvPalette::BuildSearchLists();

  const std::string dirsep("/");

  _SvvString pathName;
  _SvvString colorName;
  SvvColorWord   colorData;

  const char* path;
  const char* file;

  for ( svvPalette::InitSearchPathsTraversal();
        svvPalette::TraverseSearchPathsForward(path); )
    {
    for ( svvPalette::InitSearchFilesTraversal();
          svvPalette::TraverseSearchFilesForward(file); )
      {
      pathName = _SvvString(path) + dirsep + _SvvString(file);

      ifstream ifs(pathName.c_str(), ios::in | ios::ate);

      if (ifs.is_open())
        {
        ifs.seekg(0, ios::beg);

        while (s_get_next_color(ifs, colorName, colorData))
          {
          if (!this->AddColor(colorName.c_str(), colorData) && aFlag)
            {
            if (this->RemoveColor(colorName.c_str()))
              {
              if (!this->AddColor(colorName.c_str(), colorData))
                {
                vtkWarningMacro(<< "Failed to add color '"
                                << colorName.c_str() << "'.");
                }
              }
            }
          }
        }
      svvPaletteInternal::ReadingFloats = false;
      }
    }
}

// ----------------------------------------------------------------------------
bool
svvPalette::AddPalette (const char* const&    aName,
                        svvPalette* & aPalette)
{
  const char* a = aName;

  if (aName[0] == '/')
    {
    _SvvPaletteMap::iterator pos;
    char* ptr;

    if ((ptr = strchr(++a,'/')) != NULL)
      {
      if (( pos = this->Internal->
            Palettes.find(_SvvString(a, _SvvString::size_type(ptr-a))) )
          != this->Internal->Palettes.end())
        {
        return pos->second->AddPalette(ptr, aPalette);
        }
      return false;
      }
    }

  return ( this->Internal->
           Palettes.insert( _SvvPalettePair( a,
                                             _SvvPaletteSPtr(aPalette) ) ) ).
    second;
}

bool
svvPalette::FindPalette (const char* const&    aName,
                         svvPalette* & aPalette)
{
  _SvvPaletteMap::iterator pos;

  if ( (pos = this->Internal->Palettes.find(aName))
       != this->Internal->Palettes.end())
    {
    aPalette = pos->second.GetPointer();
    return true;
    }

  return false;
}

bool
svvPalette::RemovePalette (const char* const& aName)
{
  return (this->Internal->Palettes.erase(aName) > 0);
}

/* ----------------------------------------------------------------------------
 *
 *  HOW TO RETURN hsl.to.rgb(h, s, l): 
 *     SELECT: 
 *        l<=0.5: PUT l*(s+1) IN m2
 *        ELSE: PUT l+s-l*s IN m2
 *     PUT l*2-m2 IN m1
 *     PUT hue.to.rgb(m1, m2, h+1/3) IN r
 *     PUT hue.to.rgb(m1, m2, h    ) IN g
 *     PUT hue.to.rgb(m1, m2, h-1/3) IN b
 *     RETURN (r, g, b)
 *
 *  HOW TO RETURN hue.to.rgb(m1, m2, h):
 *     IF h<0: PUT h+1 IN h
 *     IF h>1: PUT h-1 IN h
 *     IF h*6<1: RETURN m1+(m2-m1)*h*6
 *     IF h*2<1: RETURN m2
 *     IF h*3<2: RETURN m1+(m2-m1)*(2/3-h)*6
 *     RETURN m1
 */
static inline SvvColorByte
s_hue_to_rgb_component (float aM1, float aM2, float aHue)
{
  if (aHue<0.f) aHue += 1.f;
  if (aHue>0.f) aHue -= 1.f;

  int v;

  if      ((aHue*6.f)<1.f) v = int(rint(aM1+(aM2-aM1)*aHue*6.f));
  else if ((aHue*2.f)<1.f) v = int(rint(aM2));
  else if ((aHue*3.f)<2.f) v = int(rint(aM1+(aM2-aM1)*(2.f/3.f-aHue)*6.f));
  else                     v = int(rint(aM1));

  return SvvColorByte(v<0 ? 0 : (v>255 ? 255 : v));
}

static inline float
s_hsl_to_rgb (float         aHue,
              float         aSat,
              float         aLight,
              SvvColorByte& aRed,
              SvvColorByte& aGrn,
              SvvColorByte& aBlu)
{
  float m2 = (aLight<=0.5 ? aLight*(aSat + 1.f) : aLight + aSat - aLight*aSat);
  float m1 = aLight*2.f - m2;

  aRed = s_hue_to_rgb_component(m1, m2, aHue + 1.f/3.f);
  aGrn = s_hue_to_rgb_component(m1, m2, aHue          );
  aBlu = s_hue_to_rgb_component(m1, m2, aHue - 1.f/3.f);

  return m1;
}

/** Returns the integral value of the hexidecimal char (0-9,a-f,A-F). */
static inline SvvColorByte
s_get_hex_digit_value (char c)
{
  if (c >= '0' && c <= '9')
    return (c - '0');

  if (c >= 'a' && c <= 'f')
    return (c - 'a' + 10);

  if (c >= 'A' && c <= 'F')
    return (c - 'A' + 10);

  return 0;
}

/* 
 * SkipLeadingBlanks skips all standard white-space characters at the
 * beginning of the string and returns a pointer to the new position. 
 */
static inline const char*
s_skip_leading_blanks (const char* aPtr)
{
  while (isspace(*aPtr)) aPtr++;

  return aPtr;
}

// ----------------------------------------------------------------------------
const char*
svvPalette::StringToColor (const char*   aValue,
                           SvvColorByte& aRed,
                           SvvColorByte& aGrn,
                           SvvColorByte& aBlu,
                           SvvColorByte& aAlpha)
{
  char  colorName[128];
  int   i;
  bool  failed = true;
  const char* ptr = aValue;

  aRed   = 0;
  aGrn   = 0;
  aBlu   = 0;
  aAlpha = 255;

  if (*ptr=='#')
    {
    // 
    // We expect an hexa encoding like F0F or FF00FF.
    // 
    if (isxdigit(ptr[1]) && isxdigit(ptr[2]) && isxdigit(ptr[3]))
      {
      if (!isxdigit(ptr[4]))
        {
        // 
        // encoded on 3 digits #F0F
        // 
        aRed = s_get_hex_digit_value(ptr[1]) * 16 +
          s_get_hex_digit_value(ptr[1]);
        aGrn = s_get_hex_digit_value(ptr[2]) * 16 +
          s_get_hex_digit_value(ptr[2]);
        aBlu = s_get_hex_digit_value(ptr[3]) * 16 +
          s_get_hex_digit_value(ptr[3]);
        ptr = &ptr[4];
        failed = false;
        }
      else if (isxdigit(ptr[5]) && isxdigit(ptr[6]))
        {
        // 
        // encoded on 6 digits #FF00FF
        // 
        aRed = s_get_hex_digit_value(ptr[1]) * 16 +
          s_get_hex_digit_value(ptr[2]);
        aGrn = s_get_hex_digit_value(ptr[3]) * 16 +
          s_get_hex_digit_value(ptr[4]);
        aBlu = s_get_hex_digit_value(ptr[5]) * 16 +
          s_get_hex_digit_value(ptr[6]);
        ptr = &ptr[7];
        failed = false;
        }
      }

    }
  else if (!strncasecmp(ptr,"rgb",3))
    {
    //
    // Encoded as 'rgb(r, g, b)' or 'rgb(r%, g%, b%)'.
    //
    bool parseAlpha = false;

    ptr += 3;
    if (*ptr=='a' || *ptr=='A')
      {
      //
      // Encoded as 'rgba(r, g, b, [0..1])' or 'rgba(r%, g%, b%, [0..1])'.
      //
      parseAlpha = true;
      ptr++;
      }

    ptr  = s_skip_leading_blanks(ptr);
    if (*ptr=='(')
      {
      ptr++;
      ptr = s_skip_leading_blanks(ptr);
      failed = false;

      int r, g, b, rr, gg, bb;

      sscanf(ptr,"%d",&r);
      while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        rr = (r*255 / 100);
        aRed = SvvColorByte(rr<0 ? 0 : (rr>255 ? 255 : rr));
        while (*ptr!='\0' && *ptr!=',') ptr++;
        }
      else
        {
        aRed = SvvColorByte(r<0 ? 0 : (r>255 ? 255 : r));
        }
      ptr++;

      sscanf(ptr,"%d",&g);
      while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        gg = (g*255 / 100);
        aGrn = SvvColorByte(gg<0 ? 0 : (gg>255 ? 255 : gg));
        while (*ptr!='\0' && *ptr!=',') ptr++;
        }
      else
        {
        aGrn = SvvColorByte(g<0 ? 0 : (g>255 ? 255 : g));
        }
      ptr++;

      sscanf(ptr,"%d",&b);
      if (parseAlpha)
        while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      else
        while (*ptr!='\0' && *ptr!=')' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        bb = (b*255 / 100);
        aBlu = SvvColorByte(bb<0 ? 0 : (bb>255 ? 255 : bb));
        if (parseAlpha)
          while (*ptr!='\0' && *ptr!=',') ptr++;
        else
          while (*ptr!='\0' && *ptr!=')') ptr++;
        }
      else
        {
        aBlu = SvvColorByte(b<0 ? 0 : (b>255 ? 255 : b));
        }

      if (parseAlpha)
        {
        float a;

        sscanf(ptr,"%f",&a);
        while (*ptr!='\0' && *ptr!=')') ptr++;
        aAlpha = SvvColorByte(a<0.f ? 0 : (a>1.f ? 255 : rint(a*255)));
        }

      // Search the rgb end.
      if (*ptr==')') ptr++;
      }

    }
  else if (!strncasecmp(ptr,"hsl",3))
    {

    //
    // Encoded as 'hsl(h, s, l)' or 'hsl(h%, s%, l%)'.
    //

    bool parseAlpha = false;

    ptr += 3;
    if (*ptr=='a' || *ptr=='A')
      {
      //
      // Encoded as 'hsla(h, s, l, [0..1])' or 'hsla(h%, s%, l%, [0..1])'.
      //
      parseAlpha = true;
      ptr++;
      }

    ptr = s_skip_leading_blanks(ptr);
    if (*ptr=='(')
      {
      ptr++;
      ptr = s_skip_leading_blanks(ptr);
      failed = false;

      int h, s, l;
      float hh, ss, ll;

      sscanf(ptr,"%d",&h);
      while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        hh = float(h) / 100.f;
        while (*ptr!='\0' && *ptr!=',') ptr++;
        }
      else
        {
        hh = float(h) / 255.f;
        }
      ptr++;

      sscanf(ptr,"%d",&s);
      while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        ss = float(s) / 100.f;
        while (*ptr!='\0' && *ptr!=',') ptr++;
        }
      else
        {
        ss = float(s) / 255.f;
        }
      ptr++;

      sscanf(ptr,"%d",&l);
      if (parseAlpha)
        while (*ptr!='\0' && *ptr!=',' && *ptr!='%') ptr++;
      else
        while (*ptr!='\0' && *ptr!=')' && *ptr!='%') ptr++;
      if (*ptr=='%')
        {
        ll = float(l) / 100.f;
        if (parseAlpha)
          while (*ptr!='\0' && *ptr!=',') ptr++;
        else
          while (*ptr!='\0' && *ptr!=')') ptr++;
        }
      else
        {
        ll = float(l) / 255.f;
        }

      float v = s_hsl_to_rgb(hh,ss,ll, aRed,aGrn,aBlu);

      if (parseAlpha)
        {
        float a;

        sscanf(ptr,"%f",&a);
        while (*ptr!='\0' && *ptr!=')') ptr++;
        aAlpha = SvvColorByte(a<0.f ? 0 : (a>1.f ? 255 : rint(a*255)));
        }

      // Search the hsl end.
      if (*ptr==')') ptr++;
      }
    
    }
  else if (isalpha(*ptr) || isxdigit(*ptr))
    {
    // 
    // We expect a color name like "red". Store it in colorName.
    // 
    int   len  = (sizeof(colorName) / sizeof(char)) - 1;
    const char* sptr = ptr;

    for(i=0; i<len && isalnum(*ptr); i++)
      {
      colorName[i] = *ptr;
      ptr++;
      }
    colorName[i] = '\0';
      
    // Look for the color name in our own color name database.
    if (failed = !(this->FindColor(colorName, aRed, aGrn, aBlu, aAlpha)))
      {
      // 
      // It may be a sequence of 3 or 6 hex. digits missing the leading '#'.
      // 
      ptr = sptr;
      if (isxdigit (ptr[0]) && isxdigit (ptr[1]) && isxdigit (ptr[2]))
        {
        // We expect an hexa encoding like F0F or FF00FF.
        if (!isxdigit (ptr[3]))
          {
          // Encoded on 3 digits e.g. '#F0F'.
          aRed = s_get_hex_digit_value(ptr[0]) * 16 +
            s_get_hex_digit_value(ptr[0]);
          aGrn = s_get_hex_digit_value(ptr[1]) * 16 +
            s_get_hex_digit_value(ptr[1]);
          aBlu = s_get_hex_digit_value(ptr[2]) * 16 +
            s_get_hex_digit_value(ptr[2]);
          ptr = &ptr[3];
          failed = false;
          }
        else if (isxdigit (ptr[4]) && isxdigit (ptr[5]))
          {
          // Encoded on 6 digits e.g. '#FF00FF'.
          aRed = s_get_hex_digit_value(ptr[0]) * 16 +
            s_get_hex_digit_value(ptr[1]);
          aGrn = s_get_hex_digit_value(ptr[2]) * 16 +
            s_get_hex_digit_value(ptr[3]);
          aBlu = s_get_hex_digit_value(ptr[4]) * 16 +
            s_get_hex_digit_value(ptr[5]);
          ptr = &ptr[6];
          failed = false;
          }
        }
      }
    }

  if (failed)
    {
    return aValue;
    }
  else
    {
    return ptr;
    }
}

SVV_NAMESPACE_END

/* 
 * End of: $Id: svvPalette.cxx,v 1.1.1.1 2006/12/19 22:58:36 christianh Exp $.
 * 
 */
