/* 
 * $Id: xminisv.cxx,v 1.1.1.1 2006/12/19 22:59:33 christianh Exp $
 * 
 * g++ -g -Wno-deprecated -I/usr/local/include/vtk \
 *   -I../../../vtkExtensions/src -I../../common -I. -c \
 *   ../../common/DefinedColors.cxx ../../common/DefinedMaterials.cxx \
 *   ../../common/Sv.cxx Pipeline.cxx Annotation.cxx Usage.cxx \
 *   xminisv.cxx
 * 
 * g++ -g -L/usr/local/lib/vtk -L../../../vtkExtensions-build/lib \
 *   xminisv.o Sv.o Annotation.o Pipeline.o Usage.o DefinedMaterials.o \
 *   DefinedColors.o -o xminisv \
 *   -lvtkExtensions -lvtkHybrid -lvtkRendering -lvtkGraphics -lvtkFiltering \
 *   -lvtkImaging -lvtkIO -lvtkCommon
 */
// standard C
#include <cctype>
#include <cmath>
// SV
#include "svvController.h"
#include "vtkInstantiator.h" // provoke singleton instanciation
// VTK Common
#include "vtkVersion.h"
#include "vtkCommand.h"
#include "vtkDataSet.h"
// VTK Rendering
#include "vtkTexture.h"
#include "vtkProperty.h"
#include "vtkPolyDataMapper.h"
#include "vtkActor.h"
#include "vtkLODActor.h"
#include "vtkAssembly.h"
#include "vtkRenderer.h"
#include "vtkActorCollection.h"
#include "vtkCamera.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkImporter.h"
#include "vtkExporter.h"
// VTK Filtering
#include "vtkSphere.h"
// VTK Graphics
#include "vtkAppendPolyData.h"
#include "vtkCleanPolyData.h"
#include "vtkTransformTextureCoords.h"
// VTK Hybrid
#include "vtkDepthSortPolyData.h"
// vtkExtensions
#include "vtkRenderingContext.h"
#include "vtkGeosphereSource.h"
#include "vtkPolyDataNormals.h"
#include "vtkSpiralShellSource.h"
#include "vtkUpstream.h"
#include "vtkPolyDataPipeline.h"
#include "vtkInputSource.h"

#if VTK_MAJOR_VERSION>4 || (VTK_MAJOR_VERSION==4 && VTK_MINOR_VERSION>0)
// VTK Hybrid
#  include "vtkSphereWidget.h"
#else
// ...
#endif /* VTK Version >= 4.1 */

// SV
#include "Pipeline.H"

/* #define USE_CUBE_AXES_ACTOR_2D 1 */
/* #define DEPTH_SORT_TRANSLUCENT_POLYS 1 */
/* #define EXTRACT_FEATURE_EDGES 1 */

VTK_EXTENSIONS_NAMESPACE_USING(vtkRenderingContext);
VTK_EXTENSIONS_NAMESPACE_USING(vtkInputSource);

SVV_NAMESPACE_USING(svvController);
SVV_NAMESPACE_USING(svv);

enum TextureMapType
{
  NOTEXTURE = 0x0,
  PLANE,
  SPHERE,
  CYLINDER,
  EXISTENT
};

enum SubdivisionType
{
  NOSUBDIVISION = 0x0,
  APPROXIMATE,
  BUTTERFLY,
  LINEAR
};


#if VTK_MAJOR_VERSION>4 || (VTK_MAJOR_VERSION==4 && VTK_MINOR_VERSION>0)
// Callback for the interaction
class vtkSphereCutterCb : public vtkCommand
{
public:
  static vtkSphereCutterCb* New (void) { return new vtkSphereCutterCb; }

  virtual void Execute (vtkObject* caller, unsigned long, void*)
  {
    vtkSphereWidget* sphereWidget = static_cast<vtkSphereWidget*>(caller);
    sphereWidget->GetSphere(this->Sphere);
    this->Actor->VisibilityOn();
  }

  vtkSphereCutterCb (void) : Sphere(0), Actor(0) { }

  vtkSphere* Sphere;
  vtkActor*  Actor;
};
#endif /* VTK Version >= 4.1 */

// Callback for the interaction
class vtkDepthSortCb : public vtkCommand
{
public:
  static vtkDepthSortCb* New (void) { return new vtkDepthSortCb; }

  virtual void Execute (vtkObject* caller, unsigned long, void*)
  {
    vtkActorCollection* ac = static_cast<vtkActorCollection*>(caller);
    vtkActor* actor = 0;
    vtkPolyDataMapper* mapper = 0;

    ac->InitTraversal();
    while(actor = ac->GetNextActor())
      {
      if (mapper = vtkPolyDataMapper::SafeDownCast(actor->GetMapper()))
        {
        vtkDepthSortPolyData* sorter = vtkDepthSortPolyData::New();
        sorter->SetInput(mapper->GetInput());
        sorter->SetProp3D(actor);
        sorter->SetDepthSortModeToParametricCenter();
        sorter->SetDirectionToBackToFront();
        sorter->SetCamera(this->Camera);
        //sorter->SetVector(1.0, 1.0, 1.0);
        //sorter->SetSortScalars(1/*On*/);
        sorter->Update();
        mapper->SetInput(sorter->GetOutput());
        sorter->Delete();
        mapper->Update();
	}
      // if (actor->GetProperty()->GetOpacity() < 1.0) { }
      }
  }

  vtkDepthSortCb (void) : Camera(0) { }

  vtkCamera* Camera;
};

static int  parse_args (int argc, char** argv, int& i);

extern void PrintUsage (vtkstd::ostream&, const char*);
extern void PrintMaterials (vtkstd::ostream&);

extern vtkProp*  MakeVolumetricText (vtkActor*, vtkCamera*);

static int   show_volumetric_text = 0;
static int   show_axes            = 0;
static int   show_bounding_box    = 0;
static int   show_features        = 0;
static int   show_materials       = 0;
static int   show_hull            = 0;
static int   generate_normals     = 0;
static float target_reduction     = 0.f;
static char* composite_output     = NULL;

static vtkRenderingContext* gpRenderingContext = 0;

static short        features_stipple    = 0;
static vtkProperty* gpFeaturesProperty  = NULL;
static int          hull_steps          = 0;
static vtkProperty* gpHullProperty      = NULL;
static vtkTexture*  gpTexture           = NULL;

static TextureMapType texture_mapping   = NOTEXTURE;

static SubdivisionType subdivision_type = NOSUBDIVISION;
static int             num_subdivisions = 0;

int
main (int argc, char* argv[], char* envp[])
{
  if (argc<2)
    {
    PrintUsage(vtkstd::cerr, argv[0]);
    vtkstd::exit(0);
    }

  // call this so iostream plays nice with stdio
  ios::sync_with_stdio();

  // --------------------------------------------------------------------------
  // Create the renderers, render window, and interactor
  // --------------------------------------------------------------------------
  gpRenderingContext = vtkRenderingContext::New();
    {
    gpRenderingContext->Initialize();
    gpRenderingContext->SetName("SV");
    gpRenderingContext->SetDesiredUpdateRate(1.0); // 4 fps
    gpRenderingContext->SetStillUpdateRate(0.001);
    gpRenderingContext->SetSize(768, 768);
    }

  svv.SetInteractor(gpRenderingContext->GetInteractor());

  int i, ret;
  // consume all switches from argv.  Returns number of words eaten.
  // Returns zero on error.  'i' will either point at first word that
  // does not start with '-', at the error word, or after a '--', or at
  // argc.  If your program does not take any word arguments you can
  // report an error if i < argc.
  if (!(ret = svvController::Args(&argc, &argv, &envp, i, parse_args)))
    {
    PrintUsage(vtkstd::cerr,argv[0]);
    vtkstd::exit(0);
    }
  else
    {
    if (svv.GetDebug())
      vtkstd::clog << "Args() returned "<< ret << vtkstd::endl;
    }
  
  // --------------------------------------------------------------------------
  vtkRenderer* pRenderer = gpRenderingContext->GetCurrentRenderer();
  vtkCamera*   pCamera   = gpRenderingContext->GetCurrentCamera();
  vtkImporter* pImporter;
  vtkExporter* pExporter;
  
  // --------------------------------------------------------------------------
  // Do importing first, if specified
  // --------------------------------------------------------------------------
  if (pImporter = gpRenderingContext->GetImporter())
    {
    pImporter->Update();
    }
  
  // --------------------------------------------------------------------------
  // show all materials
  // --------------------------------------------------------------------------
  if (show_materials)
    {
    vtkProp3D* materialsProp3D;

    if ((materialsProp3D = svv.GetMaterialsDemo(pRenderer)) != NULL)
      {
      pRenderer->AddProp(materialsProp3D);
      }
    }

    {
    vtkActor* actor = NULL;
    vtkActorCollection* ac;
    vtkProp* propPtr = NULL;
    vtkPropCollection* mc;
    vtkPolyDataMapper* mapper = NULL;
    
    ac = svv.GetSurfaceActors();
    ac->InitTraversal();

    if (show_volumetric_text)
      {
      mc = vtkPropCollection::New();
      mc->InitTraversal();
      }
    while ((actor = ac->GetNextActor()) != NULL)
      {
      pRenderer->AddProp(actor);
      
      if (show_volumetric_text)
        {
        if (propPtr = MakeVolumetricText(actor,pCamera))
          {
          mc->AddItem(propPtr);
          }
        }
      }
    
    if (show_volumetric_text)
      {
      mc->InitTraversal();

      while ((propPtr = mc->GetNextProp()) != NULL)
        {
        pRenderer->AddProp(propPtr);
        }
      }
    
    }
  
  // --------------------------------------------------------------------------
  // make the bounding box if requested
  // --------------------------------------------------------------------------
  if (show_bounding_box)
    {
    vtkProp3D* boundingBox;

    if ((boundingBox = svv.GetBoundingBox(pRenderer)) != NULL)
      {
      pRenderer->AddProp(boundingBox);
      }
    }
  
  // --------------------------------------------------------------------------
  // make the axes if requested
  // --------------------------------------------------------------------------
  if (show_axes)
    {
    vtkProp* axes;

    if ((axes = svv.GetAxes(pRenderer)) != NULL)
      {
      pRenderer->AddProp(axes);
      }
    }
  
  if (svv.GetDebug())
    vtkstd::clog << "About to start the event loop ..." << vtkstd::endl;
  
  // --------------------------------------------------------------------------
  // start the event loop
  // --------------------------------------------------------------------------
  gpRenderingContext->Start();
  
  if (svv.GetDebug()) vtkstd::clog << "Event loop started ..." << vtkstd::endl;
  
  // --------------------------------------------------------------------------
  // Do exporting last, if specified
  // --------------------------------------------------------------------------
  if (pExporter = gpRenderingContext->GetExporter())
    {
    pExporter->Update();
    }
  
  if (composite_output)
    {
    vtkActor*           actor;
    vtkPolyData*        pPolyData;
    vtkActorCollection* actorCollection = pRenderer->GetActors();
    vtkAppendPolyData*  pAppendPolyData = vtkAppendPolyData::New();
    vtkUpstream*        pUpstream = vtkUpstream::New();
    
    actorCollection->InitTraversal();
    while(actor = actorCollection->GetNextActor())
      {
      pUpstream->SetSource(actor);
      if (pPolyData = pUpstream->GetOutputPolyData())
        {
        pAppendPolyData->AddInput(pPolyData);
        }
      }
    
    int fileType = 0;

    if ( ! VTK_EXTENSIONS_NAMESPACE_QUALIFIER
         vtkPolyDataPipeline::WriteData( pAppendPolyData->GetOutput(),
                                         composite_output,
                                         fileType ) )
      {
      vtkstd::clog << "Failed to write '" << composite_output << "'" << vtkstd::endl;
      }
    
    pAppendPolyData->Delete();
    pUpstream->Delete();
    delete [] composite_output;
    }

  if (svv.GetDebug())
    {
    vtkstd::clog << "\n^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^\n"
                 << vtkstd::endl;
    vtkActor* actor;
    vtkActorCollection* actorCollection = pRenderer->GetActors();
    vtkUpstream*        pUpstream = vtkUpstream::New();
    
    actorCollection->InitTraversal();
    while(actor = actorCollection->GetNextActor())
      {
      pUpstream->SetSource(actor);
      pUpstream->PrintOutputPolyDataInfo(vtkstd::clog);
      vtkstd::clog << "\n-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-:-\n"
                   << vtkstd::endl;
      }
    gpRenderingContext->GetInteractor()->Print(vtkstd::clog);
    vtkstd::clog << "\n^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^\n"
                 << vtkstd::endl;
    gpRenderingContext->GetRenderWindow()->Print(vtkstd::clog);
    vtkstd::clog << "\n^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^\n"
                 << vtkstd::endl;
    gpRenderingContext->GetCurrentRenderer()->Print(vtkstd::clog);
    vtkstd::clog << "\n^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^v^\n"
                 << vtkstd::endl;
    gpRenderingContext->GetCurrentCamera()->Print(vtkstd::clog);
    
    pUpstream->Delete();
    }

  // svv.PrintMaterials(vtkstd::clog);
  
  gpRenderingContext->Delete();
  
  return 0;
}

char*
args_cat_dup (int nargs, char** args)
{
  char*       cat = 0;
  vtkstd::size_t size = 0;

  if (nargs>1)
    {
    // concatenate
    if (svv.GetDebug())
      vtkstd::clog << "concatenating " << nargs << " args\n";

    size = (nargs + 1); // spaces btwn args + terminating nul.

    for (int n=0; n<nargs; n++) size += strlen(args[n]);

    cat = new char [size];
    cat[0] = '\0';

    for (int n=0; n<nargs; n++)
      {
      strcat(cat,args[n]);
      if (n<(nargs-1)) strcat(cat," ");
      }
    }
  else
    {
    cat = strcpy(new char [strlen(args[0]) + 1], args[0]);
    }

  if (svv.GetDebug())
    {
    vtkstd::clog << "alloc=" << size << " vs "
                 << " strlen=" << strlen(cat) << vtkstd::endl;
    }

  return cat;
}

// generate feature edges over the surface
static void
add_features (vtkSource*    aSource,
              vtkProperty*& aProperty,
              short         aStipple = 0xFFFF)
{
  vtkPolyDataSource* polyDataSource = MakeFeatureEdgesSource(aSource);

  if (aProperty == NULL)
    {
    aProperty = svv.MakeProperty("polished gold");
    }

  aProperty->SetLineStipplePattern(aStipple);
  aProperty->SetLineWidth(2.0);

  vtkActor* actor = svv.MakeSurfaceActor(polyDataSource,aProperty);

  actor->GetMapper()->SetResolveCoincidentTopologyToPolygonOffset();
  actor->GetMapper()->ScalarVisibilityOff();

  // References retained in the list, so decr ref count here.
  polyDataSource->Delete();
  actor->Delete();
}

static void
add_hull (vtkSource*    aSource,
          vtkProperty*& aProperty,
          int           steps = 4)
{
  vtkActor* actor;
  vtkPolyDataSource* polyDataSource = MakeHullSource(aSource, steps);

  if (aProperty == NULL)
    {
    aProperty = svv.MakeProperty("emerald");
    aProperty->SetOpacity(1.f/3.f);
    aProperty->SetEdgeColor(1.f, 1.f, 1.f);
    aProperty->EdgeVisibilityOn();
    }

  if (aProperty->GetOpacity() < 1.f)
    {
    vtkDepthSortPolyData* depthSortPolyData = vtkDepthSortPolyData::New();
      {
      depthSortPolyData->SetInput(polyDataSource->GetOutput());

      vtkCamera* camera;

      if ((camera = gpRenderingContext->GetCurrentCamera()) != NULL)
        {
        depthSortPolyData->SetCamera(camera);
        depthSortPolyData->SetDirectionToBackToFront();
        }
      else
        {
        depthSortPolyData->SetVector(0.0, 0.0, 1.0);
        depthSortPolyData->SetDirectionToSpecifiedVector();
        }
      }
      actor = svv.MakeSurfaceActor(depthSortPolyData, aProperty);
      depthSortPolyData->SetProp3D(actor);
      depthSortPolyData->Update();
      depthSortPolyData->Delete();
    }
  else
    {
    actor = svv.MakeSurfaceActor(polyDataSource, aProperty);
    }

  // References retained in the list, so decr ref count here.
  polyDataSource->Delete();
  actor->Delete();
}


static int
parse_args (int argc, char** argv, int& i)
{
  static vtkProperty* property = vtkProperty::New();

  // pointer to the current flag
  const char*  flag = argv[i];
  char**       args;

  bool valid = false;

  // determines how many arguments follow the current flag
  int nargs = 0;
    {
    while((i + nargs + 1) < argc && argv[i + nargs + 1][0] != '-') nargs++;
    
    if (svv.GetDebug()) vtkstd::clog << " '" << &flag[0] << "' + ( ";
    if (nargs)
      {
      args = argv + vtkstd::ptrdiff_t(i + 1);
      if (svv.GetDebug())
        for (int n=0; n<nargs; n++) vtkstd::clog<< "'"<< args[n]<< "' ";
      }
    else
      {
      args = NULL;
      if (svv.GetDebug()) vtkstd::clog << "none ";
      }
    if (svv.GetDebug()) vtkstd::clog << ")" << vtkstd::endl;
    }

    // first check for flags that do not take arguments
    if (args==0)
      {

      // show axes based on overall bounds
      if (flag[1]=='X' && !flag[2])
        { show_axes = 1; valid = true; }

      // show overall bounding box
      else if (flag[1]=='B' && !flag[2])
        { show_bounding_box = 1; valid = true; }

      // show volumetric information
      else if (flag[1]=='V' && !flag[2])
        { show_volumetric_text = 1; valid = true; }

      // force generation of normals 
      else if (flag[1]=='N' && !flag[2])
        { generate_normals = 1; valid = true; }

      // use orthographic viewing projection
      else if (flag[1]=='O' && !flag[2])
        {
	gpRenderingContext->GetCurrentCamera()->ParallelProjectionOn();
	valid = true;
        }

      // show feature edges (optional material name)
      else if (flag[1]=='F')
        {
        short st = 0xFFFF;
        show_features = 1;
        if (flag[2])
          {
          const char* p = &flag[2];
          char*       pe;
	  if ((st=short(rint(strtod(p,&pe))))==0 && pe==p) st = 0xFFFF;
          }
        features_stipple = st;
        valid = true;
        }

      // convex hull, without specifying color/material name
      else if (flag[1]=='H')
        {
        int steps = 4;
        show_hull = 1;
        if (flag[2])
          {
          const char* p = &flag[2];
          char*       pe;
	  if ((steps=int(rint(strtod(p,&pe))))==0 && pe==p) steps = 4;
          }
        hull_steps = steps;
        valid = true;
        } // '-H'

      // mesh decimation
      else if (flag[1]=='D' && !flag[2])
        { target_reduction = 0.9; valid = true; }

      // mesh decimation or mesh subdivision
      else if (flag[1]=='D' && !flag[3])
        {
        if (!flag[2])
          {
          target_reduction = 0.9;
          valid = true;
          }
        else
          {
          switch (flag[2])
            {
            case 'a': subdivision_type = APPROXIMATE; break;
            case 'b': subdivision_type = BUTTERFLY; break;
            case 'l':
            default:  subdivision_type = LINEAR; break;
            }
          num_subdivisions = 1;
          valid = true;
          }
        } // '-D'

      // display help and exit
      else if (flag[1]=='h' && !flag[2])
        {
        PrintUsage(vtkstd::cerr,(svv.argv())[0]);
        PrintMaterials(vtkstd::cerr);
        vtkstd::exit(0);
        }

      // show materials 
      else if (flag[1]=='M' && !flag[2])
        { show_materials = 1; valid = true; }

      // be verbose
      else if (flag[1]=='v' && !flag[2])
        { svv.DebugOn(); valid = true; }

      // be quiet
      else if (flag[1]=='q' && !flag[2])
        { svv.DebugOff(); valid = true; }

      }
    else
      {

      // some flags require that multiple arguments be concatenated
      if ( flag[1]=='c' || flag[1]=='m' ||
           flag[1]=='K' || flag[1]=='S' ||
           flag[1]=='F' || flag[1]=='H' ||
           flag[1]=='b' )
        {

        const char* ptr = args_cat_dup(nargs, args);
	
        int         n;
        const char* first;
        char*       last;
        float       rgba[4] = {1.f, 1.f, 1.f, 1.f};
      
        first = ptr;
        last  = 0/*NULL*/;

        // specify an RGB(A) color by name, by triplet, or by quadruplet
        if (flag[1]=='c'  && !flag[2])
          {
	
          // first attempt to parse an RGB triplet or RGBA quadruplet
          for (n=0; n<4; n++)
            {
            if ((rgba[n] = float(strtod(first,&last)))==0 && last==first) break;
            while(isspace(*last)) last++; /* chomp whitespace */
            first = last; /* point to char after last conversion */
            last  = 0; /* rezero to allow check for conversion failure */
            }
	
          if (property) property->Delete();

          if (n>2) property = svv.MakeProperty(rgba);
          else    property = svv.MakeProperty(ptr);

          if (svv.GetDebug() && property) property->Print(vtkstd::clog);
          valid = true;
          } // '-c'

        else if (flag[1]=='b' && flag[2]=='g' && !flag[3])
          {
	
          // first attempt to parse an RGB triplet or RGBA quadruplet
          for (n=0; n<4; n++)
            {
            if ((rgba[n] = float(strtod(first,&last)))==0 && last==first) break;
            while(isspace(*last)) last++; /* chomp whitespace */
            first = last; /* point to char after last conversion */
            last  = 0; /* rezero to allow check for conversion failure */
            }

          if (n<3) svv.FindColor(ptr,rgba);

          gpRenderingContext->GetCurrentRenderer()->SetBackground(rgba);

          valid = true;
          } // '-bg'

        // specify a material by name
        else if (flag[1]=='m' && !flag[2])
          {

          if (property) property->Delete();
          property = svv.MakeProperty(ptr);

          if (svv.GetDebug() && property) property->Print(vtkstd::clog);
          valid = true;
          } // '-m'

        // specify material reflectance
        else if (flag[1]=='K' && !flag[3])
          {

          // first attempt to parse an RGB triplet
          for (n=0; n<3; n++)
            {
            if ((rgba[n] = float(strtod(first,&last)))==0 && last==first) break;
            while(isspace(*last)) last++; /* chomp whitespace */
            first = last; /* point to char after last conversion */
            last  = 0; /* rezero to allow check for conversion failure */
            }
          if (n)
            {
            if (flag[2]=='a')
              {
              if (n<3) property->SetAmbient(rgba[0]);
              else property->SetAmbientColor(rgba);
              valid = true;
              }
            else if (flag[2]=='d')
              {
              if (n<3) property->SetDiffuse(rgba[0]);
              else property->SetDiffuseColor(rgba);
              valid = true;
              }
            else if (flag[2]=='s')
              {
              if (n<3) property->SetSpecular(rgba[0]);
              else property->SetSpecularColor(rgba);
              valid = true;
              }
            }
          } // '-Ka' || '-Kd' || '-Ks'

        // specify material reflectance
        else if (flag[1]=='S' && flag[2]=='e' && !flag[3])
          {
          property->SetSpecularPower(atof(args[0]));
          valid = true;
          } // '-Se'

        // generate spiral shell
        else if (flag[1]=='S' && (!flag[2] || (flag[2]=='t' && !flag[3])))
          {
          float argVec[6];
          for (n=0; n<9; n++)
            {
            if ((argVec[n] = float(strtod(first,&last)))==0&&last==first) break;
            while(isspace(*last)) last++; /* chomp whitespace */
            first = last; /* point to char after last conversion */
            last  = 0; /* rezero to allow check for conversion failure */
            }
          if (n<6) return 0;

          // vtkstd::clog << "'-S' '" << int(argVec[0]) << "' '"
          //              << int(argVec[1]) << "' '" << argVec[2] << "' '"
          //              << argVec[3] << "' '" <<  argVec[4] << "' '"
          //              << argVec[5] << "'" << vtkstd::endl;

          vtkPolyDataNormals* normals = vtkPolyDataNormals::New();
            {
            vtkSpiralShellSource* spiralShell = vtkSpiralShellSource::New();
              {
              spiralShell->DebugOn();
              spiralShell->SetThetaResolution(int(argVec[0]));
              spiralShell->SetPhiResolution(int(argVec[1]));
              spiralShell->SetNumberOfSpirals(argVec[2]);
              spiralShell->SetFinalRadius(argVec[3]);
              spiralShell->SetInnerRadius(argVec[4]);
              spiralShell->SetHeight(argVec[5]);
              if (n>=7) spiralShell->SetFlipX(int(argVec[6]));
              if (n>=8) spiralShell->SetFlipY(int(argVec[7]));
              if (n>=9) spiralShell->SetFlipZ(int(argVec[8]));
              if (flag[2]=='t') spiralShell->TriangleTessellationOn();
              //spiralShell->SetCenter(-(argVec[3]+argVec[4]),0.0,0.0);
              spiralShell->GenerateTextureMappingOn();
              //spiralShell->PreventSeamOn();
              spiralShell->Print(vtkstd::clog);
              }
            normals->SetInput(spiralShell->GetOutput());
            spiralShell->Delete();
            }
            
          vtkActor* actor = svv.MakeSurfaceActor(normals, property);
          actor->GetMapper()->SetScalarModeToUseCellData();
          
          // generate a convex hull over the surface
          if (show_hull)
            {
            add_hull(normals, gpHullProperty, hull_steps);
            if (gpHullProperty)
              {
              gpHullProperty->Delete();
              gpHullProperty = 0;
              }
            show_hull = 0;
            } // '-H' && '-S'
          
          // generate feature edges over the surface
          if (show_features)
            {
            add_features(normals, gpFeaturesProperty, features_stipple);
            if (gpFeaturesProperty)
              {
              gpFeaturesProperty->Delete();
              gpFeaturesProperty = 0;
              }
            show_features = 0;
            } // '-F' && '-S'
          
          if (gpTexture)
            {
            actor->SetTexture(gpTexture);
            gpTexture->Delete();
            gpTexture = 0;
            } // '-T' && '-S'
          
          normals->Delete();
          actor->Delete();
          
          valid = true;
          } // '-S'

        // feature edges, specifying color/material name
        else if (flag[1]=='F')
          {
          short st = 0xFFFF;
          show_features = 1;
          if (flag[2])
            {
            const char* p = &flag[2];
            char*       pe;
            if ((st=short(rint(strtod(p,&pe))))==0 && pe==p) st = 0xFFFF;
            }
          features_stipple = st;
          gpFeaturesProperty = svv.MakeProperty(ptr);
          valid = true;
          } // '-F'

        // convex hull, specifying color/material name
        else if (flag[1]=='H')
          {
          int steps = 4;
          show_hull = 1;
          if (flag[2])
            {
            const char* p = &flag[2];
            char*       pe;
            if ((steps=int(rint(strtod(p,&pe))))==0 && pe==p) steps = 4;
            }
          hull_steps = steps;
          gpHullProperty = svv.MakeProperty(ptr);
          valid = true;
          } // '-H'
        
        if (ptr)
          {
          delete [] ptr;
          ptr = 0;
          if (svv.GetDebug()) vtkstd::clog << "deallocated concatenated args\n";
          }
        
        } // '-c' || '-m' || '-K'
      
      // 
      // these flags take one or more non-concatenated arguments
      // 
      else
        {
        
        // specify input pathnames
        if (flag[1]=='i')
          {
          vtkInputSource* source;
          vtkActor*       actor;
          
          for (int n=0; n<nargs; n++)
            {
            
            // make the "standard" surface
            source = svv.MakeSurfaceDataSource(args[n]);
            actor  = svv.MakeSurfaceActor(source, property);
            
            // Mesh Subdivision.
            if (num_subdivisions>0 && subdivision_type!=NOSUBDIVISION)
              {
              switch (subdivision_type)
                {
                case BUTTERFLY:
                  InsertButterflySubdivision(actor,num_subdivisions); break;
                case LINEAR:
                  InsertLinearSubdivision(actor,num_subdivisions); break;
                case APPROXIMATE:
                default:
                  InsertLoopSubdivision(actor,num_subdivisions); break;
                }
              InsertDecimation(actor, target_reduction);
              target_reduction = 0.f;
              }
            
            // Mesh Decimation.
            if (target_reduction>0.f)
              {
              InsertDecimation(actor, target_reduction);
              target_reduction = 0.f;
              }
            
            // Force generation of normals at end of pipeline.
            if (generate_normals)
              {
              InsertNormals(actor);
              generate_normals = 0;
              }
            
            // If texture has been created, insert texture mapping.
            if (gpTexture)
              {
              
              bool needsTCoords = true;
              vtkTransformTextureCoords* texTransform;
              
              switch (texture_mapping)
                {
                case SPHERE:
                  texTransform = InsertTextureMapToSphere(actor,false); break;
                case CYLINDER:
                  texTransform = InsertTextureMapToCylinder(actor,false); break;
                case PLANE:
                  texTransform = InsertTextureMapToPlane(actor); break;
                case EXISTENT:
                default:
                  vtkUpstream* pUpstream = vtkUpstream::New();
                  pUpstream->SetSource(actor);
                  if (pUpstream->GetOutputTCoordsArray()) needsTCoords = false;
                  else texTransform = InsertTextureMapToPlane(actor);
                  pUpstream->Delete();
                  break;
                }
              // Maybe use the texture transform if created
              if (needsTCoords)
                {
                texTransform->SetFlipR(0);
                texTransform->SetFlipS(0);
                texTransform->SetFlipT(0);
                texTransform->SetScale(1.f, 1.f, 1.f);
                texTransform->Update();
                texTransform->Delete(); // reference in pipeline retained
                }
              vtkstd::clog << "\nsvv.GetDebug() = "
                           << (svv.GetDebug()?"On":"Off")<< "\n\n";
              if (svv.GetDebug())
                {
                if (needsTCoords)
                  {
                  vtkstd::clog << "Generated texture coordinates.\n";
                  }
                else
                  {
                  vtkstd::clog << "Using existent texture coordinates.\n";
                  }
                }
              // assign the texture
              actor->SetTexture(gpTexture);
              gpTexture->Delete();
              gpTexture = 0;
              }
            
            // generate a convex hull and/or feature edges over the surface
            if (show_hull || show_features)
              {
              
              vtkMapper* mapper = actor->GetMapper();
              mapper->Update();
              
              vtkSource* source = mapper->GetInput()->GetSource();
              
              //vtkstd::clog << "Mapper = " << mapper
              //             << ", Source = " << source << vtkstd::endl;
              
              // generate a convex hull over the surface
              if (show_hull)
                {
                if (gpHullProperty == NULL)
                  {
                  gpHullProperty = vtkProperty::New();
                  gpHullProperty->DeepCopy(property);
                  gpHullProperty->SetOpacity(property->GetOpacity()/2.0);
                  gpHullProperty->SetEdgeColor(1.f, 1.f, 1.f);
                  gpHullProperty->EdgeVisibilityOn();
                  }
                add_hull(source, gpHullProperty, hull_steps);
                if (gpHullProperty)
                  {
                  gpHullProperty->Delete();
                  gpHullProperty = 0;
                  }
                show_hull = 0;
                } // '-H' && '-i'
            
              // generate feature edges over the surface
              if (show_features)
                {
                add_features(source, gpFeaturesProperty, features_stipple);
                if (gpFeaturesProperty)
                  {
                  gpFeaturesProperty->Delete();
                  gpFeaturesProperty = 0;
                  }
                show_features = 0;
                } // '-F' && '-i'

              } // ('-F' || '-H') && '-i'

            // references are retained in the list so decrement ref count here
            actor->Delete();
            source->Delete();
            actor  = NULL;
            source = NULL;
            }
          valid = true;
          } // '-i'

        // specify composite geometry output pathname
        else if (flag[1]=='o' && !flag[2])
          {
          composite_output = new char[strlen(args[0])+1];
          strcpy(composite_output, args[0]);
          valid = true;
          } // '-o'

        // specify imported scene pathname
        else if (flag[1]=='I' && !flag[2])
          {
          gpRenderingContext->Import(args[0]);
          valid = true;
          } // '-I'

        // specify exported scene pathname
        else if (flag[1]=='E' && !flag[2])
          {
          gpRenderingContext->Export(args[0]);
          valid = true;
          } // '-E'

        // independently specify alpha
        else if (flag[1]=='a' && !flag[2])
          {
          if (property) property->SetOpacity(atof(args[0]));
          valid = true;
          } // '-a'

        // display materials demo
        else if (flag[1]=='M' && !flag[2])
          {
          show_materials = 1;
          svv.SetMaterialsDemoResolution(atoi(args[0]));
          valid = true;
          } // '-M'

        // Geosphere
        else if (flag[1]=='G')
          {
          vtkCleanPolyData* cleanPolyData = vtkCleanPolyData::New();
            {
            vtkGeosphereSource* pGeosphereSource = vtkGeosphereSource::New();
              {
              if (flag[2])
                {
                pGeosphereSource->SetNumberOfSubdivisions(atoi(&flag[2]));
                pGeosphereSource->GenerateTextureMappingOn();
                if (flag[3] && flag[3]=='s')
                  {
                  pGeosphereSource->PreventSeamOn();
                  }
                else if (flag[3] && flag[3]=='t')
                  {
                  pGeosphereSource->SetBaseGeometryToTetrahedron();
                  if (flag[4] && flag[4]=='s')
                    pGeosphereSource->PreventSeamOn();
                  }
                else if (flag[3] && (flag[3]=='h' || flag[3]=='c'))
                  {
                  pGeosphereSource->SetBaseGeometryToHexahedron();
                  if (flag[4] && flag[4]=='s')
                    pGeosphereSource->PreventSeamOn();
                  }
                else if (flag[3] && flag[3]=='o')
                  {
                  pGeosphereSource->SetBaseGeometryToOctahedron();
                  if (flag[4] && flag[4]=='s')
                    pGeosphereSource->PreventSeamOn();
                  }
                else if (flag[3] && flag[3]=='i')
                  {
                  pGeosphereSource->SetBaseGeometryToIcosahedron();
                  if (flag[4] && flag[4]=='s')
                    pGeosphereSource->PreventSeamOn();
                  }
                }
              pGeosphereSource->SetRadius(atof(args[0]));
              pGeosphereSource->RetainNormalsOn();
              }
            cleanPolyData->SetInput(pGeosphereSource->GetOutput());
            cleanPolyData->PointMergingOn();
            pGeosphereSource->Delete();
            }
          vtkActor* actor = svv.MakeSurfaceActor(cleanPolyData, property);
          
          // generate a convex hull over the surface
          if (show_hull)
            {
            add_hull(cleanPolyData, gpHullProperty, hull_steps);
            if (gpHullProperty)
              {
              gpHullProperty->Delete();
              gpHullProperty = 0;
              }
            show_hull = 0;
            } // '-H' && '-G*'
          
          // generate feature edges over the surface
          if (show_features)
            {
            add_features(cleanPolyData, gpFeaturesProperty, features_stipple);
            if (gpFeaturesProperty)
              {
              gpFeaturesProperty->Delete();
              gpFeaturesProperty = 0;
              }
            show_features = 0;
            } // '-F' && '-G*'
          
          if (gpTexture)
            {
            actor->SetTexture(gpTexture);
            gpTexture->Delete();
            gpTexture = 0;
            }
          
          cleanPolyData->Delete();
          actor->Delete();
          
          valid = true;
          } // '-G'
        
        // mesh decimation or mesh subdivision
        else if (flag[1]=='D')
          {
          if (!flag[2])
            {
            target_reduction = atof(args[0]);
            valid = true;
            }
          else if (!flag[3])
            {
            switch (flag[2])
              {
              case 'a': subdivision_type = APPROXIMATE; break;
              case 'b': subdivision_type = BUTTERFLY; break;
              case 'l':
              default:  subdivision_type = LINEAR; break;
              }
            num_subdivisions = atoi(args[0]);
            valid = true;
            }
          } // '-D'
        
        // specify texture data file (optionally specify mapping)
        else if (flag[1]=='T')
          {
          if (!flag[2])
            {
            texture_mapping = PLANE;
            }
          else
            {
            switch (flag[2])
              {
              case 's': texture_mapping = SPHERE; break;
              case 'c': texture_mapping = CYLINDER; break;
              case 'p': texture_mapping = PLANE; break;
              case 'e': texture_mapping = EXISTENT; break;
              }
            }
          gpTexture = gpRenderingContext->MakeTexture(args[0]);
          valid = true;
          } // '-T'
        
        // limit the size of display lists
        else if (flag[1]=='L' && !flag[2])
          {
          svv.SetRenderModePointsThreshold(atoi(args[0]));
          svv.SetRenderModeCellsThreshold(atoi(args[0]));
          valid = true;
          } // '-L'
        
        } // 'i' || 'o' || 'I' || 'E' || 'a' || || 'D' || 'T' || 'L'
      
      }
    
    if (valid)
      {
      i += (nargs+1);
      return (nargs+1);
      }
    else
      {
      return 0;
      }
}

/* 
 * End of: $Id: xminisv.cxx,v 1.1.1.1 2006/12/19 22:59:33 christianh Exp $.
 * 
 */
