//
// "$Id: svBrowser.cxx,v 1.1.1.1 2006/12/19 22:59:33 christianh Exp $"
//
// Widget type code for the Fast Light Tool Kit (FLTK).
//
// Each object described by Fluid is one of these objects.  They
// are all stored in a double-linked list.
//
// They "type" of the object is covered by the virtual functions.
// There will probably be a lot of these virtual functions.
//
// The type browser is also a list of these objects, but they
// are "factory" instances, not "real" ones.  These objects exist
// only so the "make" method can be called on them.  They are
// not in the linked list and are not written to files or
// copied or otherwise examined.
//

#include <FL/Fl.H>
#include <FL/Fl_Browser_.H>
#include <FL/fl_draw.H>
#include <ctype.h>
#include <stdlib.h>
#include <string.h>
#include <iostream.h>

#include "svBrowserObject.H"

#include <FL/Fl_Pixmap.H>
#include "lock.xpm"
#include "unlock.xpm"


static Fl_Pixmap lock_pixmap(lock_xpm);
static Fl_Pixmap unlock_pixmap(unlock_xpm);


// 
class svBrowser : public Fl_Browser_
{
  friend class svBrowserObject;

  // Returns the first item in the list.
  void* item_first() const;
  // Returns the item in the list after p. 
  void* item_next(void* p) const;
  // Returns the item in the list before p.
  void* item_prev(void* p) const;
  // Return 1 if p is selected and 0 otherwise.
  //   (supports multiple selections in the browser)
  int   item_selected(void* p) const;
  // For item p, s specifies the selection state: 0=off, 1=on. 
  void  item_select(void* p, int s);
  // Returns the width of the item p in pixels. Allows for two
  //   additional pixels for the list selection box. 
  int   item_width(void* p) const;
  // Return the height of the item p in pixels. Allows for two
  //   additional pixels for the list selection box.
  int   item_height(void* p) const;
  // Draws the item p in the area indicated by x, y, w, and h.
  void  item_draw(void* p, int x, int y, int, int) const;
  // Returns the average height of all items, to be used for scrolling.
  //   The default implementation uses the height of the first item.
  int   incr_height() const;

public:
  // handles an event within the normal widget bounding box.
  int   handle(int);
  // 
  void  callback();
  // 
  svBrowser(int, int, int, int, const char* = 0);
};

// single instance of the browser
static svBrowser* widget_browser;

// non-member constructor for static instance
Fl_Widget*
make_widget_browser(int x, int y, int w, int h)
{
  return (widget_browser = new svBrowser(x,y,w,h));
}

// 
// non-member callbacks for the static instance
// 

void
select(svBrowserObject* o, int v)
{
  widget_browser->select(o,v,1);
  // svBrowserObject::current = o;
}

void
select_only(svBrowserObject* o)
{
  widget_browser->select_only(o,1);
}

void
deselect()
{
  widget_browser->deselect();
  // svBrowserObject::current = 0; // this breaks the paste & merge functions
}

void
redraw_browser()
{
  widget_browser->redraw();
}

static void
svBrowser_callback(Fl_Widget* o, void*)
{
  (reinterpret_cast<svBrowser*>(o))->callback();
}

svBrowser::svBrowser(int x, int y, int w, int h, const char* l = 0)
  : Fl_Browser_(x,y,w,h,l)
{
  this->type(FL_MULTI_BROWSER);
  Fl_Widget::callback(svBrowser_callback);
  this->when(FL_WHEN_RELEASE);
}

void*
svBrowser::item_first() const
{
  return svBrowserObject::first;
}

void*
svBrowser::item_next(void* p) const
{
  return (reinterpret_cast<svBrowserObject*>(p))->next;
}

void*
svBrowser::item_prev(void* p) const
{
  return (reinterpret_cast<svBrowserObject*>(p))->prev;
}

int
svBrowser::item_selected(void* p) const
{
  return (reinterpret_cast<svBrowserObject*>(p))->new_selected;
}

void
svBrowser::item_select(void* p, int s)
{
  (reinterpret_cast<svBrowserObject*>(p))->new_selected = s;
}

int
svBrowser::item_height(void* p) const
{
  return ((reinterpret_cast<svBrowserObject*>(p))->visible ? this->textsize()+2 : 0);
}

int
svBrowser::incr_height() const
{
  return this->textsize()+2;
}

// static pointer for drawing specific item and for the event handler
static svBrowserObject* pushedtitle;

extern const char* subclassname(svBrowserObject*);

void
svBrowser::item_draw(void* p, int x, int y, int, int) const
{
  svBrowserObject* l = reinterpret_cast<svBrowserObject*>(p);

  x += 3 + 16 + l->level * 10;
  if(l->new_selected)
    fl_color(fl_contrast(FL_BLACK,FL_SELECTION_COLOR));
  else
    fl_color(FL_BLACK);

  if(l->is_public() == 0)
    lock_pixmap.draw(x - 16, y);
  else if(l->is_public() > 0)
    unlock_pixmap.draw(x - 16, y);

  if(l->is_parent()) {
    if(!l->next || l->next->level <= l->level) {
      if(l->open_!=(l==pushedtitle)) {
	fl_loop(x, y+7, x+5, y+12, x+10, y+7);
      } else {
	fl_loop(x+2, y+2, x+7, y+7, x+2, y+12);
      }
    } else {
      if(l->open_!=(l==pushedtitle)) {
	fl_polygon(x, y+7, x+5, y+12, x+10, y+7);
      } else {
	fl_polygon(x+2, y+2, x+7, y+7, x+2, y+12);
      }
    }
    x += 10;
  }

  if(l->is_widget() || l->is_class()) {
    const char* c = subclassname(l);
    if(!strncmp(c,"Fl_",3)) c += 3;
    fl_font(this->textfont(), this->textsize());
    fl_draw(c, x, y+13);
    x += int(fl_width(c)+fl_width('n'));
    c = l->name();
    if(c) {
      fl_font(this->textfont()|FL_BOLD, this->textsize());
      fl_draw(c, x, y+13);
    } else if((c=l->label())) {
      char  buf[50];
      char* p = buf;
      *p++ = '"';
      for(int i=20; i--; ) {
	if(!(*c & -32)) break;
	*p++ = *c++;
      }
      if(*c) {
        strcpy(p,"...");
        p+=3;
      }
      *p++ = '"';
      *p = 0;
      fl_draw(buf, x, y+13);
    }
  } else {
    const char* c = l->title();
    char        buf[60];
    char*       p = buf;
    for(int i=55; i--; ) {
      if(!(*c & -32)) break;
      *p++ = *c++;
    }
    if(*c) {
      strcpy(p,"...");
      p+=3;
    }
    *p = 0;
    fl_font( this->textfont() |
             ( l->is_code_block() &&
               (l->level==0 || l->parent->is_class()) ? 0 : FL_BOLD),
             this->textsize() );
    fl_draw(buf, x, y+13);
  }
}

int
svBrowser::item_width(void* p) const
{
  svBrowserObject* l = reinterpret_cast<svBrowserObject*>(p);

  if(!l->visible) return 0;

  int w = 3 + 16 + l->level * 10;
  if(l->is_parent()) w += 10;

  if(l->is_widget() || l->is_class()) {
    const char* c = l->type_name();
    if(!strncmp(c,"Fl_",3)) c += 3;
    fl_font(this->textfont(), this->textsize());
    w += int(fl_width(c) + fl_width('n'));
    c = l->name();
    if(c) {
      fl_font(this->textfont()|FL_BOLD, this->textsize());
      w += int(fl_width(c));
    } else if((c=l->label())) {
      char  buf[50];
      char* p = buf;
      *p++ = '"';
      for(int i = 20; i--;) {
	if(! (*c & -32)) break;
	*p++ = *c++;
      }
      if(*c) {strcpy(p,"..."); p+=3;}
      *p++ = '"';
      *p = 0;
      w += int(fl_width(buf));
    }
  } else {
    const char* c = l->title();
    char        buf[60];
    char*       p = buf;
    for(int i = 55; i--;) {
      if(! (*c & -32)) break;
      *p++ = *c++;
    }
    if(*c) {strcpy(p,"..."); p+=3;}
    *p = 0;
    fl_font( this->textfont() |
             ( l->is_code_block() &&
               (l->level==0 || l->parent->is_class()) ? 0 : FL_BOLD ),
             this->textsize() );
    w += int(fl_width(buf));
  }

  return w;
}

void
svBrowser::callback()
{
  selection_changed(reinterpret_cast<svBrowserObject*>(this->selection()));
}

int
svBrowser::handle(int event)
{
  static svBrowserObject* title;
  svBrowserObject* l;
  int X, Y, W, H;

  this->bbox(X,Y,W,H);

  switch (event)
    {
      
    case FL_PUSH:
      if(!Fl::event_inside(X,Y,W,H)) break;
      l = reinterpret_cast<svBrowserObject*>(find_item(Fl::event_y()));
      if(l) {
        X += 10 * l->level + 16;
        if(l->is_parent() && Fl::event_x()>X && Fl::event_x()<X+13) {
          title = pushedtitle = l;
          this->redraw_line(l);
          return 1;
        }
      }
      break;

    case FL_DRAG:
      if(!title) break;
      l = reinterpret_cast<svBrowserObject*>(find_item(Fl::event_y()));
      if(l) {
        X += 10*l->level;
        if(l->is_parent() && Fl::event_x()>X && Fl::event_x()<X+13) ;
        else l = 0;
      }
      if(l != pushedtitle) {
        if(pushedtitle) this->redraw_line(pushedtitle);
        if(l) this->redraw_line(l);
        pushedtitle = l;
      }
      return 1;

    case FL_RELEASE:
      if(!title) {
        l = reinterpret_cast<svBrowserObject*>(find_item(Fl::event_y()));
        if(l && l->new_selected && ( Fl::event_clicks() ||
                                     Fl::event_state(FL_CTRL) )) l->open();
        break;
      }
      l = pushedtitle;
      title = pushedtitle = 0;
      if(l) {
        if(l->open_) {
          l->open_ = 0;
          for(svBrowserObject* k=l->next; k&&k->level>l->level; k=k->next)
            k->visible = 0;
        } else {
          l->open_ = 1;
          for(svBrowserObject* k=l->next; k&&k->level>l->level; ) {
            k->visible = 1;
            if(k->is_parent() && !k->open_) {
              svBrowserObject* j;
              for(j=k->next; j && j->level>k->level; j=j->next);
              k = j;
            } else
              k = k->next;
          }
        }
        this->redraw();
      }
      return 1;

    } // switch (event)

  return Fl_Browser_::handle(event);
}

//
// End of "$Id: svBrowser.cxx,v 1.1.1.1 2006/12/19 22:59:33 christianh Exp $".
//
