// 
// $Id: svFacePropertiesView.H,v 1.1.1.1 2006/12/19 22:59:34 christianh Exp $
// 
#ifndef SV_FACEPROPERTIESVIEW_H_
#  define SV_FACEPROPERTIESVIEW_H_

#  include <FL/gl.h>
#  include <FL/Fl_Gl_Window.H>
#  include <cmath>

#  include "svFaceProperties.H"
#  include "svTrackball.H"
#  include "svModels.H"


class svFacePropertiesView : public Fl_Gl_Window
{
  svFaceProperties* FaceProperties;
  // trackball interface stuff
  svTrackball*  Trackball;
  GLboolean     spinmode_;
  GLboolean     spinning_;
  GLfloat       scale_;
  GLfloat       translate_[3];
  friend void   animate(void* p);
  // display list head
  GLuint        TestGeometry;
  GLuint        ContrastGeometry;
  // renderer primitives parameters
  GLboolean     PointSmoothing;
  GLboolean     LineSmoothing;
  GLboolean     PolygonSmoothing;
  // renderer lighting/model parameters
  GLboolean     TwoSidedLighting;
  GLboolean     PositionalLighting;
  GLfloat       LightFocalPoint[3];
  GLfloat       LightPosition[3];
  GLfloat       LightColor[3];
  GLfloat       LightIntensity;
  GLfloat       LightAttenuation[3];
  GLfloat       LightSpotCutoff;
  GLfloat       LightSpotExponent;
  // light model ambient lighting
  GLfloat       AmbientLighting[4];
  // contrast materials with BW or color object
  GLboolean     MonoBackground;
  // 
  GLboolean     ConfiguredSmoothing;
  GLboolean     ConfiguredLight;
  GLboolean     ConfiguredLightPosition;
  GLboolean     ConfiguredLightModel;
  GLboolean     ConfiguredModel;
  // debugging flags
  GLbitfield    Debug;
  // 
  static svModels ModelMap;
  svModel*        Model;

protected:
  virtual void draw();
  virtual int  handle(int event);
  virtual void ConfigureSmoothing();
  virtual void ConfigureLight();
  virtual void ConfigureLightPosition();
  virtual void ConfigureLightModel();
  virtual void ConfigureModel();
  void         ReconfigureLightModel();
  void         ReconfigureSmoothing();
  void         ReconfigureLight();
  void         ReconfigureLightPosition();
  void         ReconfigureModel();
  virtual void init();
  virtual void reshape();

public:
  // constructor / destructor methods
  svFacePropertiesView(int x,int y,int w,int h,const char* l=0);
  virtual ~svFacePropertiesView();
  // 
  void       SetDebug(GLbitfield arg);
  GLbitfield GetDebug() const;
  void       DebugOn();
  void       DebugOff();
  // 
  const svFaceProperties* GetFaceProperties() const;
  // 
  svProperty* GetFront();
  svProperty* GetBack();
  //
  void      SetModel(const char* name);
  // 
  void      DrawLightCone(void);
  // 
  void      SetPointSmoothing(GLboolean arg);
  GLboolean GetPointSmoothing() const;
  void      PointSmoothingOn();
  void      PointSmoothingOff();
  // 
  void      SetLineSmoothing(GLboolean arg);
  GLboolean GetLineSmoothing() const;
  void      LineSmoothingOn();
  void      LineSmoothingOff();
  // 
  void      SetPolygonSmoothing(GLboolean arg);
  GLboolean GetPolygonSmoothing() const;
  void      PolygonSmoothingOn();
  void      PolygonSmoothingOff();
  // 
  void      SetDistinctFrontAndBack(GLboolean arg);
  GLboolean GetDistinctFrontAndBack() const;
  void      DistinctFrontAndBackOn();
  void      DistinctFrontAndBackOff();
  // 
  void SetTwoSidedLighting(GLboolean arg);
  GLboolean GetTwoSidedLighting() const;
  void TwoSidedLightingOn();
  void TwoSidedLightingOff();
  // 
  void SetPositionalLighting(GLboolean arg);
  GLboolean GetPositionalLighting() const;
  void PositionalLightingOn();
  void PositionalLightingOff();
  // Set/Get the scene light focal point.
  virtual void SetLightFocalPoint(GLfloat x, GLfloat y, GLfloat z);
  virtual void SetLightFocalPoint(GLfloat v[3]);
  virtual const GLfloat* GetLightFocalPoint() const;
  virtual void GetLightFocalPoint(GLfloat &x, GLfloat &y, GLfloat &z);
  virtual void GetLightFocalPoint(GLfloat v[3]);
  virtual void SetLightFocalPointX(GLfloat r);
  virtual GLfloat GetLightFocalPointX() const;
  virtual void SetLightFocalPointY(GLfloat g);
  virtual GLfloat GetLightFocalPointY() const;
  virtual void SetLightFocalPointZ(GLfloat b);
  virtual GLfloat GetLightFocalPointZ() const;
  // Set/Get the scene light position.
  virtual void SetLightPosition(GLfloat x, GLfloat y, GLfloat z);
  virtual void SetLightPosition(GLfloat v[3]);
  virtual const GLfloat* GetLightPosition() const;
  virtual void GetLightPosition(GLfloat &x, GLfloat &y, GLfloat &z);
  virtual void GetLightPosition(GLfloat v[3]);
  virtual void SetLightPositionX(GLfloat r);
  virtual GLfloat GetLightPositionX() const;
  virtual void SetLightPositionY(GLfloat g);
  virtual GLfloat GetLightPositionY() const;
  virtual void SetLightPositionZ(GLfloat b);
  virtual GLfloat GetLightPositionZ() const;
  // Set/Get the scene light color.
  virtual void SetLightColor(GLfloat r, GLfloat g, GLfloat b);
  virtual void SetLightColor(GLfloat v[3]);
  virtual const GLfloat* GetLightColor() const;
  virtual void GetLightColor(GLfloat &r, GLfloat &g, GLfloat &b);
  virtual void GetLightColor(GLfloat v[3]);
  virtual void SetLightRed(GLfloat r);
  virtual GLfloat GetLightRed() const;
  virtual void SetLightGreen(GLfloat g);
  virtual GLfloat GetLightGreen() const;
  virtual void SetLightBlue(GLfloat b);
  virtual GLfloat GetLightBlue() const;
  // Set/Get the intensity of the scene light
  void SetLightIntensity(GLfloat arg);
  GLfloat GetLightIntensity() const;
  // Set/Get quadratic attenuation constants,
  //   specified as constant, linear, and quadratic, in that order.
  virtual void SetLightAttenuation(GLfloat c, GLfloat l, GLfloat q);
  virtual void SetLightAttenuation(GLfloat v[3]);
  virtual const GLfloat* GetLightAttenuation() const;
  virtual void GetLightAttenuation(GLfloat &c, GLfloat &l, GLfloat &q);
  virtual void GetLightAttenuation(GLfloat v[3]);
  // Set/Get the intensity of the scene light
  void SetLightSpotCutoff(GLfloat arg);
  GLfloat GetLightSpotCutoff() const;
  // Set/Get the intensity of the scene light
  void SetLightSpotExponent(GLfloat arg);
  GLfloat GetLightSpotExponent() const;
  // Set the position and focal point of the light based on elevation and 
  // azimuth. The light is moved so it is shining from the given angle. Angles 
  // are given in degrees. If light is positional, it is now made directional.
  void SetLightDirectionAngle(GLfloat elevation, GLfloat azimuth);
  void SetLightDirectionAngle(GLfloat ang[2]);
  // Set/Get the scene's ambient lighting color
  void SetAmbientLighting(GLfloat r, GLfloat g, GLfloat b, GLfloat a);
  void SetAmbientLighting(GLfloat v[4]);
  const GLfloat* GetAmbientLighting() const;
  void GetAmbientLighting(GLfloat &r, GLfloat &g, GLfloat &b, GLfloat &a);
  void GetAmbientLighting(GLfloat v[4]);
  void SetAmbientRed(GLfloat r);
  GLfloat GetAmbientRed() const;
  void SetAmbientGreen(GLfloat g);
  GLfloat GetAmbientGreen() const;
  void SetAmbientBlue(GLfloat b);
  GLfloat GetAmbientBlue() const;
  // 
  void MonoBackgroundOn();
  void MonoBackgroundOff();
};

// 
inline void
svFacePropertiesView::SetDebug(GLbitfield arg)
{
  this->Debug = arg;
}

inline GLbitfield
svFacePropertiesView::GetDebug() const
{
  return this->Debug;
}

inline void
svFacePropertiesView::DebugOn()
{
  this->Debug = 1;
}

inline void
svFacePropertiesView::DebugOff()
{
  this->Debug = 0;
}

// 
inline const svFaceProperties*
svFacePropertiesView::GetFaceProperties() const
{
  return this->FaceProperties;
}

// 
inline svProperty*
svFacePropertiesView::GetFront()
{
  return this->FaceProperties->GetFront();
}

inline svProperty*
svFacePropertiesView::GetBack()
{
  return this->FaceProperties->GetBack();
}

inline void
svFacePropertiesView::SetModel(const char* name)
{
  svModel* tmp = 0;
  if(this->Model) tmp = this->Model;

  this->Model = svFacePropertiesView::ModelMap[name];
  if(this->Model) {
    if(tmp) tmp->DeleteDisplayList();
    this->TestGeometry = this->Model->GetDisplayList();
  }
  else {
    if(tmp) this->Model = tmp;
    this->TestGeometry = this->Model->GetDisplayList();
  }
}

inline void
svFacePropertiesView::ReconfigureSmoothing()
{
  this->ConfiguredSmoothing = GL_FALSE;
  this->redraw();
}

inline void
svFacePropertiesView::ReconfigureLight()
{
  this->ConfiguredLight = GL_FALSE;
  this->redraw();
}

inline void
svFacePropertiesView::ReconfigureLightPosition()
{
  this->ConfiguredLightPosition = GL_FALSE;
  this->ConfiguredModel = GL_FALSE;
  this->redraw();
}

inline void
svFacePropertiesView::ReconfigureLightModel()
{
  this->ConfiguredLightModel = GL_FALSE;
  this->redraw();
}

inline void
svFacePropertiesView::ReconfigureModel()
{
  this->ConfiguredModel = GL_FALSE;
  this->redraw();
}

// 
inline void
svFacePropertiesView::SetPointSmoothing(GLboolean arg)
{
  this->PointSmoothing = arg;
  this->ReconfigureSmoothing();
}

inline GLboolean
svFacePropertiesView::GetPointSmoothing() const
{
  return this->PointSmoothing;
}

inline void
svFacePropertiesView::PointSmoothingOn()
{
  this->SetPointSmoothing(GL_TRUE);
}

inline void
svFacePropertiesView::PointSmoothingOff()
{
  this->SetPointSmoothing(GL_FALSE);
}

// 
inline void
svFacePropertiesView::SetLineSmoothing(GLboolean arg)
{
  this->LineSmoothing = arg;
  this->ReconfigureSmoothing();
}

inline GLboolean
svFacePropertiesView::GetLineSmoothing() const
{
  return this->LineSmoothing;
}

inline void
svFacePropertiesView::LineSmoothingOn()
{
  this->SetLineSmoothing(GL_TRUE);
}

inline void
svFacePropertiesView::LineSmoothingOff()
{
  this->SetLineSmoothing(GL_FALSE);
}

// 
inline void
svFacePropertiesView::SetPolygonSmoothing(GLboolean arg)
{
  this->PolygonSmoothing = arg;
  this->ReconfigureSmoothing();
}

inline GLboolean
svFacePropertiesView::GetPolygonSmoothing() const
{
  return this->PolygonSmoothing;
}

inline void
svFacePropertiesView::PolygonSmoothingOn()
{
  this->SetPolygonSmoothing(GL_TRUE);
}

inline void
svFacePropertiesView::PolygonSmoothingOff()
{
  this->SetPolygonSmoothing(GL_FALSE);
}

// 
inline void
svFacePropertiesView::SetDistinctFrontAndBack(GLboolean arg)
{
  this->FaceProperties->SetDistinctFrontAndBack(arg);
  this->ReconfigureLightModel();
}

inline GLboolean
svFacePropertiesView::GetDistinctFrontAndBack() const
{
  return this->FaceProperties->GetDistinctFrontAndBack();
}

inline void
svFacePropertiesView::DistinctFrontAndBackOn()
{
  this->FaceProperties->DistinctFrontAndBackOn();
  this->ReconfigureLightModel();
}

inline void
svFacePropertiesView::DistinctFrontAndBackOff()
{
  this->FaceProperties->DistinctFrontAndBackOff();
  this->ReconfigureLightModel();
}

// 
inline void
svFacePropertiesView::SetTwoSidedLighting(GLboolean arg)
{
  this->TwoSidedLighting = arg;
  this->ReconfigureLightModel();
}

inline GLboolean
svFacePropertiesView::GetTwoSidedLighting() const
{
  return this->TwoSidedLighting;
}

inline void
svFacePropertiesView::TwoSidedLightingOn()
{
  this->SetTwoSidedLighting(GL_TRUE);
}

inline void
svFacePropertiesView::TwoSidedLightingOff()
{
  this->SetTwoSidedLighting(GL_FALSE);
}

// 
inline void
svFacePropertiesView::SetPositionalLighting(GLboolean arg)
{
  this->PositionalLighting = arg;
  this->ReconfigureLight();
  this->ReconfigureLightPosition();
}

inline GLboolean
svFacePropertiesView::GetPositionalLighting() const
{
  return this->PositionalLighting;
}

inline void
svFacePropertiesView::PositionalLightingOn()
{
  this->SetPositionalLighting(GL_TRUE);
}

inline void
svFacePropertiesView::PositionalLightingOff()
{
  this->SetPositionalLighting(GL_FALSE);
}

// 
inline void
svFacePropertiesView::SetLightFocalPoint(GLfloat x, GLfloat y, GLfloat z)
{
  this->LightFocalPoint[0] = x;
  this->LightFocalPoint[1] = y;
  this->LightFocalPoint[2] = z;
  this->ReconfigureLightPosition();
}

inline void
svFacePropertiesView::SetLightFocalPoint(GLfloat v[3])
{
  this->SetLightFocalPoint(v[0], v[1], v[2]);
}

inline const GLfloat*
svFacePropertiesView::GetLightFocalPoint() const
{
  return this->LightFocalPoint;
}

inline void
svFacePropertiesView::GetLightFocalPoint(GLfloat &x, GLfloat &y, GLfloat &z)
{
  x = this->LightFocalPoint[0];
  y = this->LightFocalPoint[1];
  z = this->LightFocalPoint[2];
}

inline void
svFacePropertiesView::GetLightFocalPoint(GLfloat v[3])
{
  this->GetLightFocalPoint(v[0], v[1], v[2]);
}

inline void
svFacePropertiesView::SetLightFocalPointX(GLfloat x)
{
  this->LightFocalPoint[0] = x;
  this->ReconfigureLightPosition();
}

inline GLfloat
svFacePropertiesView::GetLightFocalPointX() const
{
  return this->LightFocalPoint[0];
}

inline void
svFacePropertiesView::SetLightFocalPointY(GLfloat y)
{
  this->LightFocalPoint[1] = y;
  this->ReconfigureLightPosition();
}

inline GLfloat
svFacePropertiesView::GetLightFocalPointY() const
{
  return this->LightFocalPoint[1];
}

inline void
svFacePropertiesView::SetLightFocalPointZ(GLfloat z)
{
  this->LightFocalPoint[2] = z;
  this->ReconfigureLightPosition();
}

inline GLfloat
svFacePropertiesView::GetLightFocalPointZ() const
{
  return this->LightFocalPoint[2];
}

// 
inline void
svFacePropertiesView::SetLightPosition(GLfloat x, GLfloat y, GLfloat z)
{
  this->LightPosition[0] = x;
  this->LightPosition[1] = y;
  this->LightPosition[2] = z;
  this->ReconfigureLightPosition();
}

inline void
svFacePropertiesView::SetLightPosition(GLfloat v[3])
{
  this->SetLightPosition(v[0], v[1], v[2]);
}

inline const GLfloat*
svFacePropertiesView::GetLightPosition() const
{
  return this->LightPosition;
}

inline void
svFacePropertiesView::GetLightPosition(GLfloat &x, GLfloat &y, GLfloat &z)
{
  x = this->LightPosition[0];
  y = this->LightPosition[1];
  z = this->LightPosition[2];
}

inline void
svFacePropertiesView::GetLightPosition(GLfloat v[3])
{
  this->GetLightPosition(v[0], v[1], v[2]);
}

inline void
svFacePropertiesView::SetLightPositionX(GLfloat x)
{
  this->LightPosition[0] = x;
  this->ReconfigureLightPosition();
}

inline GLfloat
svFacePropertiesView::GetLightPositionX() const
{
  return this->LightPosition[0];
}

inline void
svFacePropertiesView::SetLightPositionY(GLfloat y)
{
  this->LightPosition[1] = y;
  this->ReconfigureLightPosition();
}

inline GLfloat
svFacePropertiesView::GetLightPositionY() const
{
  return this->LightPosition[1];
}

inline void
svFacePropertiesView::SetLightPositionZ(GLfloat z)
{
  this->LightPosition[2] = z;
  this->ReconfigureLightPosition();
}

inline GLfloat
svFacePropertiesView::GetLightPositionZ() const
{
  return this->LightPosition[2];
}

// 
inline void
svFacePropertiesView::SetLightColor(GLfloat r,GLfloat g,GLfloat b)
{
  this->LightColor[0] = r;
  this->LightColor[1] = g;
  this->LightColor[2] = b;
  this->ReconfigureLight();
}

inline void
svFacePropertiesView::SetLightColor(GLfloat v[3])
{
  this->SetLightColor(v[0], v[1], v[2]);
}

inline const GLfloat*
svFacePropertiesView::GetLightColor() const
{
  return this->LightColor;
}

inline void
svFacePropertiesView::GetLightColor(GLfloat &r, GLfloat &g, GLfloat &b)
{
  r = this->LightColor[0];
  g = this->LightColor[1];
  b = this->LightColor[2];
}

inline void
svFacePropertiesView::GetLightColor(GLfloat v[3])
{
  this->GetLightColor(v[0], v[1], v[2]);
}

inline void
svFacePropertiesView::SetLightRed(GLfloat r)
{
  this->LightColor[0] = r;
  this->ReconfigureLight();
}

inline GLfloat
svFacePropertiesView::GetLightRed() const
{
  return this->LightColor[0];
}

inline void
svFacePropertiesView::SetLightGreen(GLfloat g)
{
  this->LightColor[1] = g;
  this->ReconfigureLight();
}

inline GLfloat
svFacePropertiesView::GetLightGreen() const
{
  return this->LightColor[1];
}

inline void
svFacePropertiesView::SetLightBlue(GLfloat b)
{
  this->LightColor[2] = b;
  this->ReconfigureLight();
}

inline GLfloat
svFacePropertiesView::GetLightBlue() const
{
  return this->LightColor[2];
}

inline void
svFacePropertiesView::SetLightIntensity(GLfloat arg)
{
  this->LightIntensity = arg;
  this->ReconfigureLight();
}

inline GLfloat
svFacePropertiesView::GetLightIntensity() const
{
  return this->LightIntensity;
}

// 
inline void
svFacePropertiesView::SetLightAttenuation(GLfloat c, GLfloat l, GLfloat q)
{
  this->LightAttenuation[0] = c;
  this->LightAttenuation[1] = l;
  this->LightAttenuation[2] = q;
  this->ReconfigureLight();
}

inline void
svFacePropertiesView::SetLightAttenuation(GLfloat v[3])
{
  this->SetLightAttenuation(v[0], v[1], v[2]);
}

inline const GLfloat*
svFacePropertiesView::GetLightAttenuation() const
{
  return this->LightAttenuation;
}

inline void
svFacePropertiesView::GetLightAttenuation(GLfloat &c, GLfloat &l, GLfloat &q)
{
  c = this->LightAttenuation[0];
  l = this->LightAttenuation[1];
  q = this->LightAttenuation[2];
}

inline void
svFacePropertiesView::GetLightAttenuation(GLfloat v[3])
{
  this->GetLightAttenuation(v[0], v[1], v[2]);
}

inline void
svFacePropertiesView::SetLightSpotCutoff(GLfloat arg)
{
  this->LightSpotCutoff = arg;
  this->ReconfigureLight();
}

inline GLfloat
svFacePropertiesView::GetLightSpotCutoff() const
{
  return this->LightSpotCutoff;
}

inline void
svFacePropertiesView::SetLightSpotExponent(GLfloat arg)
{
  this->LightSpotExponent = arg;
  this->ReconfigureLight();
}

inline GLfloat
svFacePropertiesView::GetLightSpotExponent() const
{
  return this->LightSpotExponent;
}

// Set the position and focal point of the light based on elevation and 
// azimuth. The light is moved so it is shining from the given angle. Angles 
// are given in degrees. If light is positional, it is now made directional.
inline void
svFacePropertiesView::SetLightDirectionAngle(GLfloat elevation, GLfloat azimuth)
{
  elevation *= 0.017453292; // multiply by Pi/180 to convert to radians
  azimuth   *= 0.017453292; // multiply by Pi/180 to convert to radians

  this->SetLightPosition( cos(elevation) * sin(azimuth),
			  sin(elevation),
			  cos(elevation) * cos(azimuth) );

  this->SetLightFocalPoint(0.f, 0.f, 0.f);
  this->PositionalLightingOff();
}

inline void
svFacePropertiesView::SetLightDirectionAngle(GLfloat ang[2])
{
  this->SetLightDirectionAngle(ang[0], ang[1]);
}

// 
inline void
svFacePropertiesView::SetAmbientLighting(GLfloat r, GLfloat g, GLfloat b, GLfloat a)
{
  this->AmbientLighting[0] = r;
  this->AmbientLighting[1] = g;
  this->AmbientLighting[2] = b;
  this->AmbientLighting[3] = a;
  this->ReconfigureLightModel();
}

inline void
svFacePropertiesView::SetAmbientLighting(GLfloat v[4])
{
  this->SetAmbientLighting(v[0], v[1], v[2], v[3]);
}

inline const GLfloat*
svFacePropertiesView::GetAmbientLighting() const
{
  return this->AmbientLighting;
}

inline void
svFacePropertiesView::GetAmbientLighting(GLfloat &r, GLfloat &g, GLfloat &b, GLfloat &a)
{
  r = this->AmbientLighting[0];
  g = this->AmbientLighting[1];
  b = this->AmbientLighting[2];
  a = this->AmbientLighting[3];
}

inline void
svFacePropertiesView::GetAmbientLighting(GLfloat v[4])
{
  this->GetAmbientLighting(v[0], v[1], v[2], v[3]);
}

inline void
svFacePropertiesView::SetAmbientRed(GLfloat r)
{
  this->AmbientLighting[0] = r;
  this->ReconfigureLightModel();
}

inline GLfloat
svFacePropertiesView::GetAmbientRed() const
{
  return this->AmbientLighting[0];
}

inline void
svFacePropertiesView::SetAmbientGreen(GLfloat g)
{
  this->AmbientLighting[1] = g;
  this->ReconfigureLightModel();
}

inline GLfloat
svFacePropertiesView::GetAmbientGreen() const
{
  return this->AmbientLighting[1];
}

inline void
svFacePropertiesView::SetAmbientBlue(GLfloat b)
{
  this->AmbientLighting[2] = b;
  this->ReconfigureLightModel();
}

inline GLfloat
svFacePropertiesView::GetAmbientBlue() const
{
  return this->AmbientLighting[2];
}

// 
inline void
svFacePropertiesView::MonoBackgroundOn()
{
  this->MonoBackground = GL_TRUE;
  this->redraw();
}

inline void
svFacePropertiesView::MonoBackgroundOff()
{
  this->MonoBackground = GL_FALSE;
  this->redraw();
}

#endif /* SV_FACEPROPERTIESVIEW_H_ */
// 
// End of: $Id: svFacePropertiesView.H,v 1.1.1.1 2006/12/19 22:59:34 christianh Exp $.
// 
