// 
// $Id: svProperty.H,v 1.1.1.1 2006/12/19 22:59:34 christianh Exp $
// 
#ifndef SV_PROPERTY_H_
#  define SV_PROPERTY_H_

#  include <GL/gl.h>
#  include <iostream>


class svProperty
{
  GLfloat    Color[3];
  GLfloat    AmbientColor[3];
  GLfloat    DiffuseColor[3];
  GLfloat    SpecularColor[3];
  GLfloat    Ambient;
  GLfloat    Diffuse;
  GLfloat    Specular;
  GLfloat    Shininess;
  GLfloat    Opacity;
  GLfloat    PointSize;
  GLfloat    LineWidth;
  GLenum     ShadeModel;
  GLenum     PolygonMode;
  GLbitfield Debug;

public:
  // Constructor
  svProperty()
    {
      this->AmbientColor[0] =this->AmbientColor[1] =this->AmbientColor[2] =1.f;
      this->DiffuseColor[0] =this->DiffuseColor[1] =this->DiffuseColor[2] =1.f;
      this->SpecularColor[0]=this->SpecularColor[1]=this->SpecularColor[2]=0.f;
      // these should match OpenGL default material properties
      this->Ambient     = 0.2f;
      this->Diffuse     = 0.8f;
      this->Specular    = 0.f;
      this->Shininess   = 0.f;
      this->Opacity     = 1.f;
      this->PointSize   = 1.f;
      this->LineWidth   = 1.f;
      this->ShadeModel  = GL_SMOOTH;
      this->PolygonMode = GL_FILL;
      this->Debug = 0;
    }
  // Set/Get debugging flag(s)
  void SetDebug(GLbitfield arg);
  GLbitfield GetDebug() const;
  void DebugOn();
  void DebugOff();
  // Update the composite RGB color of the object (ambient+diffuse+specular).
  void UpdateColor();
  // Set the composite RGB color of the object.
  void SetColor(GLfloat r,GLfloat g,GLfloat b);
  // Set the composite RGB color of the object from the provided tuple.
  void SetColor(GLfloat v[3]);
  // return ptr to composite color vector of object (ambient+diffuse+specular)
  const GLfloat* GetColor();
  // Assign composite color component values to provided reference parameters.
  void GetColor(GLfloat &r, GLfloat &g, GLfloat &b);
  // Copy composite color of object (ambient+diffuse+specular) into array.
  void GetColor(float v[3]);
  // Set/Get individual components of composite color
  void SetRed(GLfloat arg);
  void SetGreen(GLfloat arg);
  void SetBlue(GLfloat arg);
  GLfloat GetRed();
  GLfloat GetGreen();
  GLfloat GetBlue();
  // Set/Get the intensity of material reflectance parameters: [-1..1]
  void SetAmbient(GLfloat arg);
  void SetDiffuse(GLfloat arg);
  void SetSpecular(GLfloat arg);
  GLfloat GetAmbient() const;
  GLfloat GetDiffuse() const;
  GLfloat GetSpecular() const;
  // Set/Get the specular exponent material parameter: [0..128]
  void SetShininess(GLfloat arg);
  GLfloat GetShininess() const;
  // Set/Get the object transparancy
  void    SetOpacity(GLfloat arg);
  GLfloat GetOpacity() const;
  // Set/Get the object point size
  void    SetPointSize(GLfloat arg);
  GLfloat GetPointSize() const;
  // Set/Get the object transparancy
  void    SetLineWidth(GLfloat arg);
  GLfloat GetLineWidth() const;
  // Set/Get material ambient reflectance color.
  virtual void SetAmbientColor(GLfloat r, GLfloat g, GLfloat b);
  virtual void SetAmbientColor(GLfloat v[3]);
  virtual const GLfloat* GetAmbientColor() const;
  virtual void GetAmbientColor(GLfloat &r, GLfloat &g, GLfloat &b);
  virtual void GetAmbientColor(GLfloat v[3]);
  virtual void SetAmbientRed(GLfloat r);
  virtual void SetAmbientGreen(GLfloat g);
  virtual void SetAmbientBlue(GLfloat b);
  virtual GLfloat GetAmbientRed();
  virtual GLfloat GetAmbientGreen();
  virtual GLfloat GetAmbientBlue();
  // Set/Get material diffuse reflectance color.
  virtual void SetDiffuseColor(GLfloat r, GLfloat g, GLfloat b);
  virtual void SetDiffuseColor(GLfloat v[3]);
  virtual const GLfloat* GetDiffuseColor() const;
  virtual void GetDiffuseColor(GLfloat &r, GLfloat &g, GLfloat &b);
  virtual void GetDiffuseColor(GLfloat v[3]);
  virtual void SetDiffuseRed(GLfloat r);
  virtual void SetDiffuseGreen(GLfloat g);
  virtual void SetDiffuseBlue(GLfloat b);
  virtual GLfloat GetDiffuseRed();
  virtual GLfloat GetDiffuseGreen();
  virtual GLfloat GetDiffuseBlue();
  // Set/Get material specular reflectance color.
  virtual void SetSpecularColor(GLfloat r, GLfloat g, GLfloat b);
  virtual void SetSpecularColor(GLfloat v[3]);
  virtual const GLfloat* GetSpecularColor() const;
  virtual void GetSpecularColor(GLfloat &r, GLfloat &g, GLfloat &b);
  virtual void GetSpecularColor(GLfloat v[3]);
  virtual void SetSpecularRed(GLfloat r);
  virtual void SetSpecularGreen(GLfloat g);
  virtual void SetSpecularBlue(GLfloat b);
  virtual GLfloat GetSpecularRed();
  virtual GLfloat GetSpecularGreen();
  virtual GLfloat GetSpecularBlue();
  // Set/Get color interpolation mode.
  void   SetShadeModel(GLenum arg);
  GLenum GetShadeModel() const;
  void   SetShadeModelToFlat();
  void   SetShadeModelToSmooth();
  // Set/Get polygon rasterization mode.
  void   SetPolygonMode(GLenum arg);
  GLenum GetPolygonMode() const;
  void   SetPolygonModeToFill();
  void   SetPolygonModeToLine();
  void   SetPolygonModeToPoint();
  // Get some GL state as strings
  const char* GetFaceAsString(GLenum face);
  const char* GetShadeModelAsString(GLenum model);
  const char* GetShadeModelAsString();
  const char* GetPolygonModeAsString(GLenum mode);
  const char* GetPolygonModeAsString();
  const char* GetColorMaterialAsString();
  const char* GetColorMaterialFaceAsString();
  const char* GetColorMaterialParameterAsString();
  // 
  void PrintMaterial(ostream& os, GLenum face, GLenum pname);
  static void PrintMaterials(ostream& os);
  void PrintLight(ostream& os, GLenum light, GLenum pname);
  static void PrintLighting(ostream& os);
  // Apply the material properties to the current GL state.
  void Apply(GLenum face = GL_FRONT_AND_BACK);
  void ApplyOpaque(GLenum face = GL_FRONT_AND_BACK);
  //
  void PrintSelf(ostream& os);
};

// 
// inline functions
// 

// 
inline void
svProperty::SetDebug(GLbitfield arg)
{
  this->Debug = arg;
}

inline GLbitfield
svProperty::GetDebug() const
{
  return this->Debug;
}

inline void
svProperty::DebugOn()
{
  this->Debug = 1;
}

inline void
svProperty::DebugOff()
{
  this->Debug = 0;
}

// Update the composite RGB color of the object (ambient + diffuse + specular).
inline void
svProperty::UpdateColor()
{
  GLfloat norm;
  
  if((this->Ambient + this->Diffuse + this->Specular) > 0.f)
    norm = 1.f / (this->Ambient + this->Diffuse + this->Specular);
  else
    norm = 0.f;
  
  for(int i=0; i<3; i++) {
    this->Color[i] =  this->AmbientColor[i]  * this->Ambient  * norm;
    this->Color[i] += this->DiffuseColor[i]  * this->Diffuse  * norm;
    this->Color[i] += this->SpecularColor[i] * this->Specular * norm;
  }
}

// Set the composite RGB color of the object.
inline void
svProperty::SetColor(GLfloat r,GLfloat g,GLfloat b)
{
  this->SetAmbientColor(r,g,b);
  this->SetDiffuseColor(r,g,b);
  this->SetSpecularColor(r,g,b);
  this->UpdateColor();
}

// Set the composite RGB color of the object from the provided tuple.
inline void
svProperty::SetColor(GLfloat v[3])
{
  this->SetColor(v[0], v[1], v[2]);
}

// return pointer to composite color vector of object (ambient + diffuse + specular)
inline const GLfloat*
svProperty::GetColor()
{
  this->UpdateColor();
  return this->Color;
}

// assign composite color component values to the provided reference parameters
inline void
svProperty::GetColor(GLfloat &r, GLfloat &g, GLfloat &b)
{
  this->UpdateColor();
  r = this->Color[0];
  g = this->Color[1];
  b = this->Color[2];
}

// Copy composite color of object (ambient + diffuse + specular) into array provided.
inline void
svProperty::GetColor(float v[3])
{
  this->UpdateColor();
  v[0] = this->Color[0];
  v[1] = this->Color[1];
  v[2] = this->Color[2];
}

// 
inline void
svProperty::SetRed(GLfloat arg)
{
  this->UpdateColor();
  this->SetColor(arg, this->Color[1], this->Color[2]);
}

inline void
svProperty::SetGreen(GLfloat arg)
{
  this->UpdateColor();
  this->SetColor(this->Color[0], arg, this->Color[2]);
}

inline void
svProperty::SetBlue(GLfloat arg)
{
  this->UpdateColor();
  this->SetColor(this->Color[0], this->Color[1], arg);
}

inline GLfloat
svProperty::GetRed()
{
  this->UpdateColor();
  return this->Color[0];
}

inline GLfloat
svProperty::GetGreen()
{
  this->UpdateColor();
  return this->Color[1];
}

inline GLfloat
svProperty::GetBlue()
{
  this->UpdateColor();
  return this->Color[2];
}

// 
inline void
svProperty::SetAmbient(GLfloat arg)
{
  this->Ambient = (arg<-1.f ? -1.f : (arg>1.f ? 1.f : arg));
}

inline void
svProperty::SetDiffuse(GLfloat arg)
{
  this->Diffuse = (arg<-1.f ? -1.f : (arg>1.f ? 1.f : arg));
}

inline void
svProperty::SetSpecular(GLfloat arg)
{
  this->Specular = (arg<-1.f ? -1.f : (arg>1.f ? 1.f : arg));
}

inline void
svProperty::SetShininess(GLfloat arg)
{
  this->Shininess = (arg<0.f ? 0.f : (arg>128.f ? 128.f : arg));
}

inline GLfloat
svProperty::GetAmbient() const
{
  return this->Ambient;
}

inline GLfloat
svProperty::GetDiffuse() const
{
  return this->Diffuse;
}

inline GLfloat
svProperty::GetSpecular() const
{
  return this->Specular;
}

inline GLfloat
svProperty::GetShininess() const
{
  return this->Shininess;
}

// 
inline void
svProperty::SetOpacity(GLfloat arg)
{
  this->Opacity  = (arg<0.f ? 0.f : (arg>1.f ? 1.f : arg));
}

inline GLfloat
svProperty::GetOpacity() const
{
  return this->Opacity;
}

// 
inline void
svProperty::SetPointSize(GLfloat arg)
{
  this->PointSize  = (arg<0.f ? 0.f : (arg>1.0e+38F ? 1.0e+38F : arg));
}

inline GLfloat
svProperty::GetPointSize() const
{
  return this->PointSize;
}

// 
inline void
svProperty::SetLineWidth(GLfloat arg)
{
  this->LineWidth  = (arg<0.f ? 0.f : (arg>1.0e+38F ? 1.0e+38F : arg));
}

inline GLfloat
svProperty::GetLineWidth() const
{
  return this->LineWidth;
}

// 
inline void
svProperty::SetAmbientColor(GLfloat r, GLfloat g, GLfloat b)
{
  this->AmbientColor[0] = r;
  this->AmbientColor[1] = g;
  this->AmbientColor[2] = b;
}

inline void
svProperty::SetAmbientColor(GLfloat v[3])
{
  this->SetAmbientColor(v[0], v[1], v[2]);
}

inline const GLfloat*
svProperty::GetAmbientColor() const
{
  return this->AmbientColor;
}

inline void
svProperty::GetAmbientColor(GLfloat &r, GLfloat &g, GLfloat &b)
{
  r = this->AmbientColor[0];
  g = this->AmbientColor[1];
  b = this->AmbientColor[2];
}

inline void
svProperty::GetAmbientColor(GLfloat v[3])
{
  this->GetAmbientColor(v[0], v[1], v[2]);
}

inline void
svProperty::SetAmbientRed(GLfloat r)
{
  this->AmbientColor[0] = r;
}

inline void
svProperty::SetAmbientGreen(GLfloat g)
{
  this->AmbientColor[1] = g;
}

inline void
svProperty::SetAmbientBlue(GLfloat b)
{
  this->AmbientColor[2] = b;
}

inline GLfloat
svProperty::GetAmbientRed()
{
  return this->AmbientColor[0];
}

inline GLfloat
svProperty::GetAmbientGreen()
{
  return this->AmbientColor[1];
}

inline GLfloat
svProperty::GetAmbientBlue()
{
  return this->AmbientColor[2];
}

// 
inline void
svProperty::SetDiffuseColor(GLfloat r, GLfloat g, GLfloat b)
{
  this->DiffuseColor[0] = r;
  this->DiffuseColor[1] = g;
  this->DiffuseColor[2] = b;
}

inline void
svProperty::SetDiffuseColor(GLfloat v[3])
{
  this->SetDiffuseColor(v[0], v[1], v[2]);
}

inline const GLfloat*
svProperty::GetDiffuseColor() const
{
  return this->DiffuseColor;
}

inline void
svProperty::GetDiffuseColor(GLfloat &r, GLfloat &g, GLfloat &b)
{
  r = this->DiffuseColor[0];
  g = this->DiffuseColor[1];
  b = this->DiffuseColor[2];
}

inline void
svProperty::GetDiffuseColor(GLfloat v[3])
{
  this->GetDiffuseColor(v[0], v[1], v[2]);
}

inline void
svProperty::SetDiffuseRed(GLfloat r)
{
  this->DiffuseColor[0] = r;
}

inline void
svProperty::SetDiffuseGreen(GLfloat g)
{
  this->DiffuseColor[1] = g;
}

inline void
svProperty::SetDiffuseBlue(GLfloat b)
{
  this->DiffuseColor[2] = b;
}

inline GLfloat
svProperty::GetDiffuseRed()
{
  return this->DiffuseColor[0];
}

inline GLfloat
svProperty::GetDiffuseGreen()
{
  return this->DiffuseColor[1];
}

inline GLfloat
svProperty::GetDiffuseBlue()
{
  return this->DiffuseColor[2];
}

// 
inline void
svProperty::SetSpecularColor(GLfloat r, GLfloat g, GLfloat b)
{
  this->SpecularColor[0] = r;
  this->SpecularColor[1] = g;
  this->SpecularColor[2] = b;
}

inline void
svProperty::SetSpecularColor(GLfloat v[3])
{
  this->SetSpecularColor(v[0], v[1], v[2]);
}

inline const GLfloat*
svProperty::GetSpecularColor() const
{
  return this->SpecularColor;
}

inline void
svProperty::GetSpecularColor(GLfloat &r, GLfloat &g, GLfloat &b)
{
  r = this->SpecularColor[0];
  g = this->SpecularColor[1];
  b = this->SpecularColor[2];
}

inline void
svProperty::GetSpecularColor(GLfloat v[3])
{
  this->GetSpecularColor(v[0], v[1], v[2]);
}

inline void
svProperty::SetSpecularRed(GLfloat r)
{
  this->SpecularColor[0] = r;
}

inline void
svProperty::SetSpecularGreen(GLfloat g)
{
  this->SpecularColor[1] = g;
}

inline void
svProperty::SetSpecularBlue(GLfloat b)
{
  this->SpecularColor[2] = b;
}

inline GLfloat
svProperty::GetSpecularRed()
{
  return this->SpecularColor[0];
}

inline GLfloat
svProperty::GetSpecularGreen()
{
  return this->SpecularColor[1];
}

inline GLfloat
svProperty::GetSpecularBlue()
{
  return this->SpecularColor[2];
}

// 
inline void
svProperty::SetShadeModel(GLenum arg)
{
  this->ShadeModel = arg;
}

inline GLenum
svProperty::GetShadeModel() const
{
  return this->ShadeModel;
}

inline void
svProperty::SetShadeModelToFlat()
{
  this->ShadeModel = GL_FLAT;
}

inline void
svProperty::SetShadeModelToSmooth()
{
  this->ShadeModel = GL_SMOOTH;
}

// 
inline void
svProperty::SetPolygonMode(GLenum arg)
{
  this->PolygonMode = arg;
}

inline GLenum
svProperty::GetPolygonMode() const
{
  return this->PolygonMode;
}

inline void
svProperty::SetPolygonModeToFill()
{
  this->PolygonMode = GL_FILL;
}

inline void
svProperty::SetPolygonModeToLine()
{
  this->PolygonMode = GL_LINE;
}

inline void
svProperty::SetPolygonModeToPoint()
{
  this->PolygonMode = GL_POINT;
}

inline const char*
svProperty::GetFaceAsString(GLenum face)
{
  switch (face) {
  case GL_FRONT:          return "GL_FRONT";
  case GL_BACK:           return "GL_BACK";
  case GL_FRONT_AND_BACK: return "GL_FRONT_AND_BACK";
  default:                return "INVALID";
  }
}

inline const char*
svProperty::GetShadeModelAsString(GLenum model)
{
  switch (model) {
  case GL_SMOOTH: return "GL_SMOOTH";
  case GL_FLAT:   return "GL_FLAT";
  default:        return "INVALID";
  }
}

inline const char*
svProperty::GetShadeModelAsString()
{
  GLint model;
  glGetIntegerv(GL_SHADE_MODEL, &model);
  return this->GetShadeModelAsString(GLenum(model));
}

inline const char*
svProperty::GetPolygonModeAsString(GLenum mode)
{
  switch (mode) {
  case GL_FILL:  return "GL_FILL";
  case GL_LINE:  return "GL_LINE";
  case GL_POINT: return "GL_POINT";
  default:       return "INVALID";
  }
}

inline const char*
svProperty::GetPolygonModeAsString()
{
  GLint mode;
  glGetIntegerv(GL_POLYGON_MODE, &mode);
  return this->GetPolygonModeAsString(GLenum(mode));
}

inline const char*
svProperty::GetColorMaterialAsString()
{
  GLint state;
  glGetIntegerv(GL_COLOR_MATERIAL, &state);
  switch (state) {
  case GL_TRUE:  return "GL_TRUE";
  case GL_FALSE: return "GL_FALSE";
  default:       return "INVALID";
  }
}

inline const char*
svProperty::GetColorMaterialFaceAsString()
{
  GLint face;
  glGetIntegerv(GL_COLOR_MATERIAL_FACE, &face);
  switch (face) {
  case GL_FRONT:          return "GL_FRONT";
  case GL_BACK:           return "GL_BACK";
  case GL_FRONT_AND_BACK: return "GL_FRONT_AND_BACK";
  default:                return "INVALID";
  }
}

inline const char*
svProperty::GetColorMaterialParameterAsString()
{
  GLint parameter;
  glGetIntegerv(GL_COLOR_MATERIAL_PARAMETER, &parameter);
  switch (parameter) {
  case GL_EMISSION:            return "GL_EMISSION";
  case GL_AMBIENT:             return "GL_AMBIENT";
  case GL_DIFFUSE:             return "GL_DIFFUSE";
  case GL_AMBIENT_AND_DIFFUSE: return "GL_AMBIENT_AND_DIFFUSE";
  case GL_SPECULAR:            return "GL_SPECULAR";
  default:                     return "INVALID";
  }
}

// 
inline void
svProperty::Apply(GLenum face = GL_FRONT_AND_BACK)
{
  GLfloat c[4];

  c[3] = this->Opacity;

  for(int n=0; n<3; n++) c[n] = this->Ambient  * this->AmbientColor[n];
  glMaterialfv(face, GL_AMBIENT, c);
  for(int n=0; n<3; n++) c[n] = this->Diffuse  * this->DiffuseColor[n];
  glMaterialfv(face, GL_DIFFUSE, c);
  for(int n=0; n<3; n++) c[n] = this->Specular * this->SpecularColor[n];
  glMaterialfv(face, GL_SPECULAR, c);
  glMaterialf(face, GL_SHININESS, this->Shininess);

  if(face == GL_FRONT || face == GL_FRONT_AND_BACK) {
    // set interpolation method
    glShadeModel(this->ShadeModel);
    // The material properties set above are used if shading is enabled. The 
    // color set here is used if shading is disabled.
    this->GetColor(c);
    c[3] = 1.f;
    glColor4fv(c);

    // Set the PointSize
    glPointSize(this->PointSize);
    // Set the LineWidth
    glLineWidth(this->LineWidth);
  }

  glPolygonMode(face, this->PolygonMode);

  if(this->Debug) {
    cerr << "svProperty(" << this << ")::Apply("
	 << this->GetFaceAsString(face) << ")\n";
    this->PrintSelf(cerr);
  } // Debug
} // Apply()

inline void
svProperty::ApplyOpaque(GLenum face = GL_FRONT_AND_BACK)
{
  GLfloat tmp = this->Opacity;
  this->Opacity = 1.f;
  this->Apply(face);
  this->Opacity = tmp;
}

#endif /* SV_PROPERTY_H_ */
// 
// End of: $Id: svProperty.H,v 1.1.1.1 2006/12/19 22:59:34 christianh Exp $.
// 
