//
// "$Id: svType.H,v 1.1.1.1 2006/12/19 22:59:35 christianh Exp $"
//
// Widget type header file for the Fast Light Tool Kit (FLTK).
//
// Each object described by Fluid is one of these objects.  They
// are all stored in a double-linked list.
//
// There is also a single "factory" instance of each type of this.
// The method "make()" is called on this factory to create a new
// instance of this object.  It could also have a "copy()" function,
// but it was easier to implement this by using the file read/write
// that is needed to save the setup anyways.

#include <FL/Fl_Widget.H>
#include <FL/Fl_Menu.H>
#include "Fluid_Image.h"

class svBrowserObject
{
  friend class svBrowser;
  virtual void setlabel(const char*); // virtual part of label(char*)

protected:
  // protected constructor
  svBrowserObject();

  const char* name_;  // instance name
  const char* label_; // 
  const char* callback_;
  const char* user_data_;
  const char* user_data_type_;

public:	// things that should not be public:

  svBrowserObject* parent; // parent, which is previous in list
  char new_selected;       // browser highlight
  char selected;           // copied here by selection_changed()
  char open_;              // state of triangle in browser
  char visible;            // true if all parents are open
  char rtti;               // hack, this is 0 for base class
  int  level;              // number of parents over this

  // linked list of all objects
  static svBrowserObject* first;
  static svBrowserObject* last;
  svBrowserObject*        next;
  svBrowserObject*        prev;

  svBrowserObject*     factory;

public:
  // destructor
  virtual ~svBrowserObject();

  // pure virtual "virtual" constructor method
  virtual svBrowserObject* make() = 0;

  void    add(svBrowserObject* parent);  // add as new child
  void    insert(svBrowserObject* n);    // insert into list before n
  svBrowserObject* remove();             // remove from list
  void    move_before(svBrowserObject*); // move before a sibling

  virtual const char* title();         // string for browser
  virtual const char* type_name() = 0; // type for code output

  // Set/Get data
  const char* name() const {return name_;}
  void        name(const char*);
  const char* label() const {return label_;}
  void        label(const char*);
  const char* callback() const {return callback_;}
  void        callback(const char*);
  const char* user_data() const {return user_data_;}
  void        user_data(const char*);
  const char* user_data_type() const {return user_data_type_;}
  void        user_data_type(const char*);

  // 
  virtual svBrowserObject* click_test(int, int);
  virtual void add_child(svBrowserObject*, svBrowserObject* beforethis);
  virtual void move_child(svBrowserObject*, svBrowserObject* beforethis);
  virtual void remove_child(svBrowserObject*);

  // most recently picked object
  static svBrowserObject* current;

  // what happens when you double-click
  virtual void open();
  // read and write data to a saved file:
  void write();
  // get message number for I18N
  int msgnum();

  // fake rtti:
  virtual int is_parent() const {return 0;}
  virtual int is_widget() const {return 0;}
  virtual int is_group() const {return 0;}
  virtual int is_class() const {return 0;}
  virtual int is_public() const {return 1;}

  const char* class_name(const int need_nest) const;
};

class svBrowserRenderer : public svBrowserObject
{
  char public_, constructor, havewidgets;

public:
  // 
  svBrowserObject* make();
  // 
  void open();
  int  ismain() {return name_ == 0;}
  virtual const char* type_name() {return "renderer";}
  virtual const char* title()
    {return (this->name() ? this->name() : "main()");}
  int         is_parent() const {return 1;}
  virtual int is_public() const {return this->public_;}
};


class svClassType : public svBrowserObject
{
  const char* subclass_of;
  char        public_;

public:
  // state variables for output:
  svClassType* parent_class; // save class if nested
  //
  svBrowserObject* make();
  void open();
  virtual const char* type_name() {return "class";}
  int  is_parent() const {return 1;}
  int  is_class() const  {return 1;}
  virtual int is_public() const;
};

#define NUM_EXTRA_CODE 4

class svWidgetType : public svBrowserObject
{
  virtual Fl_Widget* widget(int,int,int,int) = 0;
  virtual svWidgetType* _make() = 0; // virtual constructor
  virtual void setlabel(const char*);

  const char* extra_code_[NUM_EXTRA_CODE];
  const char* subclass_;
  const char* tooltip_;
  const char* image_name_;
  const char* inactive_name_;
  uchar hotspot_;

public:
  const char* xclass; // junk string, used for shortcut
  Fl_Widget* o;
  int public_;

  Fluid_Image* image;
  void setimage(Fluid_Image* );
  Fluid_Image* inactive;
  void setinactive(Fluid_Image* );

  svWidgetType();
  svBrowserObject* make();
  void open();

  const char* extra_code(int n) const {return extra_code_[n];}
  void extra_code(int n,const char* );
  const char* subclass() const {return subclass_;}
  void subclass(const char* );
  const char* tooltip() const {return tooltip_;}
  void tooltip(const char* );
  const char* image_name() const {return image_name_;}
  void image_name(const char* );
  const char* inactive_name() const {return inactive_name_;}
  void inactive_name(const char* );

  virtual int textstuff(int what, Fl_Font &, int &, Fl_Color &);
  virtual Fl_Menu_Item* subtypes();

  virtual int is_widget() const {return 1;}
  virtual int is_public() const {return this->public_;}

  ~svWidgetType();
  void redraw();
};


class igroup : public Fl_Group
{
public:
  // Override group's resize behavior to do nothing to children:
  void resize(int X, int Y, int W, int H)
    {Fl_Widget::resize(X,Y,W,H); this->redraw();}
  igroup(int x,int y,int w,int h) : Fl_Group(x,y,w,h) {Fl_Group::current(0);}
};


class svGroupType : public svWidgetType
{
public:
  virtual const char* type_name() {return "Fl_Group";}
  Fl_Widget* widget(int x,int y,int w,int h)
    {
      igroup* g = new igroup(x,y,w,h);
      Fl_Group::current(0);
      return g;
    }
  svWidgetType* _make() {return new svGroupType();}
  svBrowserObject* make();
  void add_child(svBrowserObject*, svBrowserObject*);
  void move_child(svBrowserObject*, svBrowserObject*);
  void remove_child(svBrowserObject*);
  int  is_parent() const {return 1;}
  int  is_group() const {return 1;}
};


// object list operations:
Fl_Widget* make_widget_browser(int x,int y,int w,int h);
extern int modflag;
void delete_all(int selected_only=0);
void selection_changed(svBrowserObject* new_current);

// file operations:
void write_word(const char* );
void write_string(const char*, ...);
int  write_file(const char*, int selected_only = 0);
int  write_code(const char* cfile, const char* hfile);
int  write_strings(const char* sfile);

int  write_declare(const char*, ...);
int  is_id(char);
const char* unique_id(void* o, const char*, const char*, const char*);
void write_c(const char*, ...);
void write_h(const char*, ...);
void write_cstring(const char* );
void write_cstring(const char* ,int length);
void write_indent(int n);
void write_open(int);
void write_close(int n);
extern int write_number;
extern int indentation;
extern const char* indent();

int  read_file(const char* , int merge);
const char* read_word(int wantbrace = 0);
void read_error(const char* format, ...);

extern int include_H_from_C;

struct Fl_Menu_Item;
class Fluid_Image;

void* const LOAD = (void *)9831;
extern svWidgetType *current_widget; // one of the selected ones

//
// End of "$Id: svType.H,v 1.1.1.1 2006/12/19 22:59:35 christianh Exp $".
//
