/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*-
 * 
 * $Id: Contour.cxx,v 1.1.1.1 2006/12/19 22:59:37 christianh Exp $
 * 
 * Copyright (c) 2002, 2003 Sean McInerney 
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 *  * Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 * 
 *  * Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 *  * Neither the name of Sean McInerney nor the names of any contributors may
 *    be used to endorse or promote products derived from this software without
 *    specific prior written permission.
 * 
 *  * Modified source versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <cmath>
#include <cstdlib>
#include <cstring>
#include <iostream>
#include <vector>
#include <string>

#include "vtkPiecewiseFunction.h"
#include "vtkColorTransferFunction.h"
#include "vtkWindowLevelLookupTable.h"
#include "vtkTransform.h"

// illustrates the use of vtkFLTK classes.
#include "vtkImageData.h"
#include "vtkStructuredPoints.h"

// here we have all the usual VTK stuff that we need for our pipeline
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkDataSet.h"
#include "vtkImageData.h"

#include "vtkContourFilter.h"
#include "vtkImageThreshold.h"
#include "vtkImageFlip.h"
#include "vtkImageClip.h"
#include "vtkImageShrink3D.h"
#include "vtkImageResample.h"
#include "vtkImageConstantPad.h"
#include "vtkImageToStructuredPoints.h"
#include "vtkOutlineCornerFilter.h"
#include "vtkPolyDataNormals.h"
#include "vtkPolyDataConnectivityFilter.h"
#include "vtkTransformPolyDataFilter.h"

#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkActor.h"
#include "vtkLODActor.h"

#include "vtkCellPicker.h"
#include "vtkTextProperty.h"
#include "vtkImagePlaneWidget.h"

#include "vtkPNGWriter.h"
#include "vtkRendererSource.h"

#include "vtkImageReader.h"
#include "vtkWindowToImageFilter.h"
#include "vtkPNGWriter.h"

// and of course some fltk stuff
#include <FL/Fl.H>
#include <FL/x.H>
#include <FL/gl.h>

#include "ContourUI.h"
#include "svvController.h"
#include "vtkInputSource.h"

SVV_NAMESPACE_USING(svvController);
SVV_NAMESPACE_USING(svv);
VTK_EXTENSIONS_NAMESPACE_USING(vtkInputSource);

typedef std::vector<vtkImagePlaneWidget*> ImagePlaneWidgetVec;


static float gResolution = 1.f;
static float gMin    = VTK_LARGE_FLOAT;
static float gMax    = -VTK_LARGE_FLOAT;
static float gWindow = 1.f;
static float gHigh   = 255.f;
static float gLow    = 0.f;

static vtkImageData* gAnatomicalData = 0;
static vtkImageData* gFunctionalData = 0;

static vtkTransform* gTransform = 0;

static vtkContourFilter* gpContourFilter = 0;

// Create a transfer function mapping scalar value to opacity
static vtkPiecewiseFunction*     gpOpacityTF;
// Create a transfer function mapping scalar value to color (grey)
static vtkPiecewiseFunction*     gpGrayTF;
// Create a transfer function mapping scalar value to color (color)
static vtkColorTransferFunction* gpColorTF;
// Create a transfer function mapping magnitude of gradient to opacity
static vtkPiecewiseFunction*     gpGradientTF;

static vtkLookupTable*     gpLUT = 0;
static vtkScalarsToColors* gpMapperLUT = 0;

// ----------------------------------------------------------------------------
void
ContourVisibilityCb(Fl_Button* button, void* ptr)
{
  if(!button) return;

  ContourUI* pGUI;

  if(!(pGUI = reinterpret_cast<ContourUI*>(ptr))) return;

  vtkProp3D* pContour;

  if(pContour = pGUI->GetContour()) {
  Fl::check();
  pContour->SetVisibility(button->value());

  pGUI->UpdateView();
  }
}

// ----------------------------------------------------------------------------
void
ContourValueCb(Fl_Value_Input* input, void* ptr)
{
  if(!input) return;

  ContourUI* pGUI;

  if(gpContourFilter && (pGUI = reinterpret_cast<ContourUI*>(ptr))) {
  Fl::check();
  gpContourFilter->SetValue(0, float(input->value()));
  pGUI->UpdateView();
  }
}

// ----------------------------------------------------------------------------
void
ContouringProgressCb(void* ptr)
{
  ContourUI* pGUI;

  if(gpContourFilter && (pGUI = reinterpret_cast<ContourUI*>(ptr))) {
  float progress = gpContourFilter->GetProgress();
  pGUI->SetProgress(progress);
  Fl::check();
  std::cerr << "progress: " << progress*100 << "%" << std::endl;
  }
}

// ----------------------------------------------------------------------------
void
SaveImageCb(Fl_Button*, void* ptr)
{
  ContourUI*      pGUI;
  Fl_VTK_Window* pView;

  if(!(pGUI = reinterpret_cast<ContourUI*>(ptr))) return;
  if(!(pView = pGUI->GetView())) return;

  vtkPNGWriter* pPNGWriter = vtkPNGWriter::New();
    {
    vtkWindowToImageFilter* pWindowToImage = vtkWindowToImageFilter::New();
      {
      pWindowToImage->SetInput(pView->GetRenderWindow());
      }
      pPNGWriter->SetInput(pWindowToImage->GetOutput());
      pPNGWriter->SetFileName("Contour.png");
      pPNGWriter->Write();
      pWindowToImage->Delete();
    }
    pPNGWriter->Delete();
}

// ----------------------------------------------------------------------------
void
ConnectivityCb(void* ptr)
{
  vtkPolyDataConnectivityFilter* pConnectivity;

  if(pConnectivity = reinterpret_cast<vtkPolyDataConnectivityFilter*>(ptr)) {
  float range[2];
  int   numRegions;

  numRegions = pConnectivity->GetNumberOfExtractedRegions();
  pConnectivity->GetScalarRange(range);

  std::cerr << "Connectivity regions: ( " << numRegions << " ), range: [ "
            << range[0] << " .. " << range[1] << " ]" << std::endl;
  }
}

// ----------------------------------------------------------------------------
static int  ParseArgs(int argc, char** argv, int& i);
static void PrintUsage(ostream&, const char*);

vtkImageData* GetImageData(const char*);
vtkImageData* GetFunctionalData(const char*);
vtkActor*     MakeContourActor(vtkImageData*);
vtkActor*     MakeBoundingActor(vtkImageData*);

ImagePlaneWidgetVec
MakeImagePlaneWidgets(vtkRenderWindowInteractor*,vtkImageData*);


int
main(int argc, char* argv[], char* envp[])
{
  if(argc<2) {
  PrintUsage(cerr, argv[0]);
  std::exit(0);
  }

  // call this so iostream plays nice with stdio
  ios::sync_with_stdio();

  // create the interface
  ContourUI*                 pGUI        = new ContourUI;
  vtkRenderWindowInteractor* pInteractor = pGUI->GetView()->GetInteractor();

  svv.SetInteractor(pInteractor);

  int i, ret;
  // consume all switches from argv.  Returns number of words eaten.
  // Returns zero on error.  'i' will either point at first word that
  // does not start with '-', at the error word, or after a '--', or at
  // argc.  If your program does not take any word arguments you can
  // report an error if i < argc.
  if(!(ret = svvController::Args(&argc, &argv, &envp, i, ParseArgs))) {
  PrintUsage(cerr,argv[0]);
  std::exit(0);
  }
  else {
  if(svv.GetDebug()) cerr << "Args() returned "<< ret << endl;
  }

  vtkRenderWindow* pRenderWindow = pGUI->GetView()->GetRenderWindow();
  vtkRenderer*     pRenderer     = pGUI->GetView()->GetDefaultRenderer();
  vtkCamera*       pCamera       = pGUI->GetView()->GetDefaultCamera();

    {
    vtkActor* pActor = 0;
    vtkActorCollection* pActorCollection;
  
    pActorCollection = svv.GetSurfaceActors();
    pActorCollection->InitTraversal();
    while(pActor = pActorCollection->GetNextActor()) {
    pRenderer->AddProp(pActor);
    }
    }

    vtkProp3D* pContour = 0;
    vtkProp3D* pBoundingBox = 0;
    ImagePlaneWidgetVec* widgetVecPtr = 0;

    if(gAnatomicalData) {
    // create the volume and add it to the renderer
    pBoundingBox = MakeBoundingActor(gAnatomicalData);
    pGUI->SetBoundingBox(pBoundingBox);
    // create isosurface(s) and add it to the renderer
    pContour = MakeContourActor(gAnatomicalData);
    pGUI->SetContour(pContour);
    pContour->SetVisibility(0);
    // Set up the image planes.
    ImagePlaneWidgetVec widgetVec =
      MakeImagePlaneWidgets(pGUI->GetView()->GetInteractor(),gAnatomicalData);
    widgetVecPtr = &widgetVec;
    }
    else if(gFunctionalData) {
    // create the volume and add it to the renderer
    pBoundingBox = MakeBoundingActor(gFunctionalData);
    pGUI->SetBoundingBox(pBoundingBox);
    // create isosurface(s) and add it to the renderer
    pContour = MakeContourActor(gFunctionalData);
    pGUI->SetContour(pContour);
    pContour->SetVisibility(0);
    // Set up the image planes.
    ImagePlaneWidgetVec widgetVec =
      MakeImagePlaneWidgets(pGUI->GetView()->GetInteractor(),gFunctionalData);
    widgetVecPtr = &widgetVec;
    }

    if(gpContourFilter) {
    gpContourFilter->SetProgressMethod(ContouringProgressCb, pGUI);
    }

    pRenderer->ResetCamera();
    pInteractor->SetDesiredUpdateRate(2.0);
    pRenderWindow->SetDesiredUpdateRate(2.0);
    pRenderWindow->LineSmoothingOn();
   
    // show() the main Fl_Window
    pGUI->Show();

    // this is the standard way of "starting" a fltk application
    int fl_ret = Fl::run();

    delete pGUI;

    if(gAnatomicalData || gFunctionalData) {
    // We can now delete all our references to the VTK pipeline as the objects
    // themselves will stick around until we dereference the Fl_VTK_Window.
    //pBoundingBox->Delete();
    //pContour->Delete();
    //(*widgetVecPtr)[0]->Delete();
    //(*widgetVecPtr)[1]->Delete();
    //(*widgetVecPtr)[2]->Delete();
    //gAnatomicalData->Delete();
    }

    return fl_ret;
}

// ----------------------------------------------------------------------------
static void
BuildTransferFunctions()
{
  float gWindow = gHigh - gLow;

  // Create transfer mapping scalar value to opacity
  gpOpacityTF = vtkPiecewiseFunction::New();
    {
    gpOpacityTF->AddSegment(gLow, 0.0, gLow + gWindow * 0.10, 0.5);
    gpOpacityTF->AddSegment(gLow + gWindow * 0.30, 1.0, gHigh, 1.0);
    }

    // Create transfer mapping scalar value to grayscale
    gpGrayTF = vtkPiecewiseFunction::New();
      {
      gpGrayTF->AddSegment(gLow, 0.0, gHigh, 1.0);
      }

      // Create transfer mapping scalar value to color
      gpColorTF = vtkColorTransferFunction::New();
        {
        gpColorTF->AddRGBPoint(gLow + gWindow * 0.20, 1.0, 0.0, 0.0);
        gpColorTF->AddRGBPoint(gLow + gWindow * 0.40, 0.0, 0.0, 1.0);
        gpColorTF->AddRGBPoint(gLow + gWindow * 0.60, 0.0, 1.0, 0.0);
        gpColorTF->AddRGBPoint(gLow + gWindow * 0.80, 0.4, 0.8, 0.2);
        }

        // Create a transfer function mapping magnitude of gradient to opacity
        gpGradientTF = vtkPiecewiseFunction::New();
          {
          gpGradientTF->AddPoint(gLow + gWindow * 0.0, 0.0);
          gpGradientTF->AddPoint(gLow + gWindow * 0.5, 0.5);
          gpGradientTF->AddPoint(gLow + gWindow * 1.0, 0.5);
          }
}

// ----------------------------------------------------------------------------
// Internal function to read in a line up to 'size' characters.
// Returns zero if there was an error, else number of char read.
static std::size_t
ReadLine(ifstream& ifs, char* buffer, std::size_t bufsize)
{
  ifs.getline(buffer,bufsize);

  std::size_t length = ifs.gcount();

  if(ifs.fail())
    {
    if(ifs.eof())
      {
      return 0;
      }
    if(length == (bufsize-1))
      {
      // Read 'bufsize' chars; ignoring the rest of the line.
      ifs.clear();
      ifs.ignore(0xffffffff, '\n');
      }
    }

  return length;
}

struct AnalysisRec
{
  AnalysisRec(void) : path(0), format(0) { }
  AnalysisRec(const char* fileName) : path(0), format(0) {this->Read(fileName);}
  ~AnalysisRec(void)
  {
    if(this->path) delete [] this->path;
    if(this->format) delete [] this->format;
  }

  int Read(const char* fileName)
  {
    struct stat fs;
  
    // Open the file containing the resolutions and rotation matrix.
    if(stat(fileName, &fs) != 0)
      {
      std::cerr << "Unable to open file: "<< fileName << std::endl;
      return 0;
      }
    // Opens the file and positions the stream pointer at EOF ...
    ifstream ifs(fileName, ios::in|ios::ate);
    if(ifs.fail())
      {
      std::cerr << "Unable to open file: "<< fileName << std::endl;
      return 0;
      }
    // The read position at EOF is the file size.
    std::streampos size = ifs.tellg();
    // Rewind the stream.
    ifs.seekg(0, ios::beg);

    char buffer[1024];
    std::size_t length;

    if(length = ReadLine(ifs, buffer, sizeof(buffer)))
      {
      this->path = new char[length+1];
      strncpy(this->path, buffer, length);
      this->path[length] = '\0';
      }

    if(length = ReadLine(ifs, buffer, sizeof(buffer)))
      {
      this->format = new char[length+1];
      strncpy(this->format, buffer, length);
      this->format[length] = '\0';
      }

    ifs >> nslices >> nimages >> nrows >> ncols;
  }

  char* path;
  char* format;
  int   nslices, nimages, nrows, ncols;
};

struct RegistrationRec
{
  RegistrationRec(void) : subject(0) { }
  RegistrationRec(const char* fileName) : subject(0) {this->Read(fileName);}
  ~RegistrationRec(void) { if(this->subject) delete [] this->subject; }

  int Read(const char* fileName)
  {
    struct stat fs;
  
    // Open the file containing the resolutions and rotation matrix.
    if(stat(fileName, &fs) != 0)
      {
      std::cerr << "Unable to open file: "<< fileName << std::endl;
      return 0;
      }
    // Opens the file and positions the stream pointer at EOF ...
    ifstream ifs(fileName, ios::in|ios::ate);
    if(ifs.fail())
      {
      std::cerr << "Unable to open file: "<< fileName << std::endl;
      return 0;
      }
    // The read position at EOF is the file size.
    std::streampos size = ifs.tellg();
    // Rewind the stream.
    ifs.seekg(0, ios::beg);

    char buffer[1024];
    std::size_t length;

    if(length = ReadLine(ifs, buffer, sizeof(buffer)))
      {
      this->subject = new char[length+1];
      strncpy(this->subject, buffer, length);
      this->subject[length] = '\0';
      }

    ifs >> this->ipr >> this->tpr >> this->scaling
        >> this->m[0] >> this->m[1] >> this->m[2] >> this->m[3]
        >> this->m[4] >> this->m[5] >> this->m[6] >> this->m[7]
        >> this->m[8] >> this->m[9] >> this->m[10] >> this->m[11]
        >> this->m[12] >> this->m[13] >> this->m[14] >> this->m[15];
  }

  char*  subject;
  double ipr, tpr;
  double scaling;
  double m[16];
};

struct ScanInfoRec
{
  ScanInfoRec(void) { }
  ScanInfoRec(const char* fileName) {this->Read(fileName);}
  
  int Read(const char* fileName)
  {
    struct stat fs;
  
    // Open the file containing the resolutions and rotation matrix.
    if(stat(fileName, &fs) != 0)
      {
      std::cerr << "Unable to open file: "<< fileName << std::endl;
      return 0;
      }
    // Opens the file and positions the stream pointer at EOF ...
    ifstream ifs(fileName, ios::in|ios::ate);
    if(ifs.fail())
      {
      std::cerr << "Unable to open file: "<< fileName << std::endl;
      return 0;
      }
    // The read position at EOF is the file size.
    std::streampos size = ifs.tellg();
    // Rewind the stream.
    ifs.seekg(0, ios::beg);

    std::string tag;

    ifs >> tag >> this->imnr0;
    ifs >> tag >> this->imnr1;
    ifs >> tag >> this->ptype;
    ifs >> tag >> this->x;
    ifs >> tag >> this->y;
    ifs >> tag >> this->fov;
    ifs >> tag >> this->thick;
    ifs >> tag >> this->psiz;
    ifs >> tag >> this->locatn;
    ifs >> tag >> this->strtx;
    ifs >> tag >> this->endx;
    ifs >> tag >> this->strty;
    ifs >> tag >> this->endy;
    ifs >> tag >> this->strtz;
    ifs >> tag >> this->endz;
    ifs >> tag >> this->tr;
    ifs >> tag >> this->te;
    ifs >> tag >> this->ti;
    ifs >> tag >> this->ras_good_flag;
    ifs >> tag >> this->x_ras[0] >> this->x_ras[1] >> this->x_ras[2];
    ifs >> tag >> this->y_ras[0] >> this->y_ras[1] >> this->y_ras[2];
    ifs >> tag >> this->z_ras[0] >> this->z_ras[1] >> this->z_ras[2];
    ifs >> tag >> this->c_ras[0] >> this->c_ras[1] >> this->c_ras[2];
  }


  int    imnr0;
  int    imnr1;
  int    ptype;
  int    x, y;
  double fov;
  double thick;
  double psiz;
  double locatn;
  double strtx, endx;
  double strty, endy;
  double strtz, endz;
  double tr, te, ti;
  int    ras_good_flag;
  double x_ras[3];
  double y_ras[3];
  double z_ras[3];
  double c_ras[3];
};

// ----------------------------------------------------------------------------
static int
ReadRegistration(const char* fileName,
                 double res[3], double& scale, double m[16])
{
  struct stat fs;
  
  // Open the file containing the resolutions and rotation matrix.
  if(stat(fileName, &fs) != 0)
    {
    std::cerr << "Unable to open file: "<< fileName << std::endl;
    return 0;
    }
  // Opens the file and positions the stream pointer at EOF ...
  ifstream ifs(fileName, ios::in|ios::ate);
  if(ifs.fail())
    {
    std::cerr << "Unable to open file: "<< fileName << std::endl;
    return 0;
    }
  // The read position at EOF is the file size.
  std::streampos size = ifs.tellg();
  // Rewind the stream.
  ifs.seekg(0, ios::beg);

  char pathName[1024];

  if(!ReadLine(ifs, pathName, sizeof(pathName)))
    {
    return 0;
    }

  ifs >> res[0] >> res[2] >> scale
      >> m[0] >> m[1] >> m[2] >> m[3]
      >> m[4] >> m[5] >> m[6] >> m[7]
      >> m[8] >> m[9] >> m[10] >> m[11]
      >> m[12] >> m[13] >> m[14] >> m[15];
  res[1] = res[0];
}

// ----------------------------------------------------------------------------
vtkImageData*
GetImageData(const char* pathPrefix)
{
  std::string infoName = pathPrefix;
  infoName += ".info";

  ScanInfoRec* info = new ScanInfoRec(infoName.c_str());

  int   extent[6];
  float spacing[3];
  float origin[3];

  extent[0] = 0; extent[1] = info->x - 1;
  extent[2] = 0; extent[3] = info->y - 1;
  extent[4] = 0; extent[5] = info->imnr1 - info->imnr0;

  spacing[0] = info->psiz * 1000.f;
  spacing[1] = info->psiz * 1000.f;
  spacing[2] = info->thick * 1000.f;

  origin[0] = info->strtx * 1000.f;
  origin[1] = info->strty * 1000.f;
  origin[2] = info->strtz * 1000.f;

  vtkImageFlip* pImageFlipY = vtkImageFlip::New();
    {
    vtkImageReader* pImageReader = vtkImageReader::New();
      {
      pImageReader->SetFilePrefix(pathPrefix);
      pImageReader->SetFilePattern("%s%03d");
      pImageReader->SetHeaderSize(0);
      pImageReader->SetDataByteOrderToBigEndian();
      pImageReader->SetDataScalarType(VTK_UNSIGNED_SHORT);
      pImageReader->SetNumberOfScalarComponents(1);
      pImageReader->SetFileDimensionality(2);
      pImageReader->SetDataExtent(extent);
      pImageReader->SetDataSpacing(spacing);
      pImageReader->SetDataOrigin(origin);
      pImageReader->SetFileNameSliceOffset(info->imnr0);
      pImageReader->FileLowerLeftOff();
      //pImageReader->SetDataVOI( extent[0] + info->x / 4,
      //  			extent[1] - info->x / 4,
      //  			extent[2] + info->y / 4,
      //  			extent[3] - info->y / 4,
      //  			extent[4] + (1 + info->imnr1-info->imnr0)/4,
      //  			extent[5] - (1 + info->imnr1-info->imnr0)/4 );
      }
      pImageFlipY->SetInput(pImageReader->GetOutput());
      //pImageFlipY->SetFilteredAxis(1);
      pImageReader->Delete();
    }

    float range[2];
  
    pImageFlipY->Update();
    pImageFlipY->GetOutput()->GetScalarRange(range);

    std::cerr << "scalar range: [ " << range[0] << " .. "
              << range[1] << " ]" << std::endl;

    gMin    = range[0];
    gMax    = range[1];
    gLow    = gMin;
    gHigh   = gMax;
    gWindow = gHigh - gLow;

    BuildTransferFunctions();

    delete info;

    return pImageFlipY->GetOutput();
}

// ----------------------------------------------------------------------------
vtkImageData*
GetFunctionalData(const char* pathName)
{
  std::string analyseName = pathName;
  analyseName += "/analyse.dat";

  AnalysisRec* analyse = new AnalysisRec(analyseName.c_str());

  std::string pattern = "%s/";
  pattern += analyse->path;
  pattern += "/";
  pattern += analyse->format;

  int   extent[6];

  extent[0] = 0;
  extent[1] = analyse->ncols - 1;
  extent[2] = 0;
  extent[3] = analyse->nrows - 1;
  extent[4] = 0;
  extent[5] = analyse->nslices - 1;

  std::string regName = pathName;
  regName += "/register.dat";

  RegistrationRec* reg = new RegistrationRec(regName.c_str());

  float spacing[3];

  float origin[3];

  spacing[0] = reg->ipr;
  spacing[1] = reg->ipr;
  spacing[2] = reg->tpr;

  float maxres = ( spacing[0]<spacing[1] ?
		   (spacing[0]<spacing[2] ? spacing[0] : spacing[2]) :
		   (spacing[1]<spacing[2] ? spacing[1] : spacing[2]) );

  origin[0] = -(float(analyse->ncols) * reg->ipr / 2.f);
  origin[1] = -(float(analyse->nrows) * reg->ipr / 2.f);
  origin[2] = -(float(analyse->nslices) * reg->tpr / 2.f);

  vtkImageResample* pResample = vtkImageResample::New();
    {
    vtkImageReader* pImageReader = vtkImageReader::New();
      {
      pImageReader->DebugOn();
      pImageReader->SetFilePrefix(pathName);
      pImageReader->SetFilePattern(pattern.c_str());
      pImageReader->SetHeaderSize(0);
      pImageReader->SetDataByteOrderToBigEndian();
      pImageReader->SetDataScalarType(VTK_UNSIGNED_SHORT);
      pImageReader->SetNumberOfScalarComponents(1);
      pImageReader->SetFileDimensionality(2);
      pImageReader->SetDataExtent(extent);
      pImageReader->SetDataSpacing(spacing);
      pImageReader->SetDataOrigin(origin);
      pImageReader->SetFileNameSliceOffset(0);
      pImageReader->FileLowerLeftOff();
      //pImageReader->SetDataVOI(gExtent);
      }
      // Transform
      vtkTransform* pTransform = vtkTransform::New();
        {
        double inv[16];
        vtkMatrix4x4::Invert(reg->m, inv);
        pTransform->SetMatrix(inv);
        }
        pResample->SetInput(pImageReader->GetOutput());
        pResample->SetResliceAxesDirectionCosines( -1.0, 0.0, 0.0,
                                                   0.0, 0.0, -1.0,
                                                   0.0, 1.0, 0.0 );
        pResample->SetResliceAxesOrigin(origin[0],origin[1],origin[2]);
        pResample->SetResliceTransform(pTransform);
        pResample->SetInterpolationMode(VTK_RESLICE_CUBIC);
        pResample->SetDimensionality(3);
        pResample->SetAxisOutputSpacing(0, maxres);
        pResample->SetAxisOutputSpacing(1, maxres);
        pResample->SetAxisOutputSpacing(2, maxres);
        pResample->AutoCropOutputOn();
        pResample->MirrorOn();
        pResample->SetOutputOrigin(origin);
        pImageReader->Delete();
        pTransform->Delete();
    }

    float range[2];
  
    pResample->Update();
    pResample->GetOutput()->GetScalarRange(range);

    std::cerr << "scalar range: [ " << range[0] << " .. "
              << range[1] << " ]" << std::endl;

    gMin    = range[0];
    gMax    = range[1];
    gLow    = gMin;
    gHigh   = gMax;
    gWindow = gHigh - gLow;

    BuildTransferFunctions();

    delete analyse;
    delete reg;

    return pResample->GetOutput();
}

// ----------------------------------------------------------------------------
vtkActor*
MakeBoundingActor(vtkImageData* pImageData)
{
  vtkActor* pActor = vtkActor::New();
    {
    vtkPolyDataMapper* pPolyDataMapper = vtkPolyDataMapper::New();
      {
      vtkOutlineCornerFilter* pOutline = vtkOutlineCornerFilter::New();
        {
        pImageData->Update();
        pOutline->SetInput(pImageData);
        pOutline->SetCornerFactor(0.15);
        }
        pPolyDataMapper->SetInput(pOutline->GetOutput());
        pPolyDataMapper->ScalarVisibilityOff();
        pOutline->Delete();
      }
      pActor->SetMapper(pPolyDataMapper);
      pPolyDataMapper->Delete();
    }
    (pActor->GetProperty())->SetColor(0.5, 0.0, 0.0);
    (pActor->GetProperty())->SetSpecular(0.5);
    (pActor->GetProperty())->SetSpecularPower(50);

    return pActor;
} // MakeBoundingActor()

// ----------------------------------------------------------------------------
// NOTE: the LOD stuff does NOT seem to be doing its thing herein!!!
vtkActor*
MakeContourActor(vtkImageData* pImageData)
{
  vtkLODActor* pActor = vtkLODActor::New();
    {
    vtkPolyDataMapper* pPolyDataMapper = vtkPolyDataMapper::New();
      {
      vtkPolyDataNormals* pPolyDataNormals = vtkPolyDataNormals::New();
        {
        vtkPolyDataConnectivityFilter* pConnectivity =
          vtkPolyDataConnectivityFilter::New();
          {
          gpContourFilter = vtkContourFilter::New();
            {
            vtkImageThreshold* pImageThreshold = vtkImageThreshold::New();
              {
              vtkImageToStructuredPoints* pStructuredPointsSource =
                vtkImageToStructuredPoints::New();
                {
		pStructuredPointsSource->SetInput(pImageData);
                }
                pImageThreshold->SetInput(pStructuredPointsSource->GetOutput());
                pImageThreshold->ThresholdByLower(gLow*0.9);
                pImageThreshold->ReplaceInOn();
                pImageThreshold->SetInValue(0.0);
                pImageThreshold->SetOutputScalarTypeToFloat();
                pStructuredPointsSource->Delete();
              }
              gpContourFilter->SetInput(pImageThreshold->GetOutput());
              pImageThreshold->Delete();
              gpContourFilter->SetValue(0, 128.0);
              gpContourFilter->ComputeNormalsOn();
              gpContourFilter->ComputeGradientsOn();
              gpContourFilter->ComputeScalarsOn();
              gpContourFilter->UseScalarTreeOn();
              gpContourFilter->CreateDefaultLocator();
            }
            pConnectivity->SetInput(gpContourFilter->GetOutput());
            pConnectivity->SetExtractionModeToAllRegions();
            pConnectivity->SetExtractionModeToLargestRegion();
            //pConnectivity->ColorRegionsOn();
            pConnectivity->SetEndMethod(ConnectivityCb,pConnectivity);
            gpContourFilter->Delete();
          }
          pPolyDataNormals->SetInput(pConnectivity->GetOutput());
          pPolyDataNormals->SplittingOff();
          pPolyDataNormals->ConsistencyOn();
          pPolyDataNormals->NonManifoldTraversalOff();
          pConnectivity->Delete();
        }
        pPolyDataMapper->SetInput(pPolyDataNormals->GetOutput());
        pPolyDataMapper->SetLookupTable(gpColorTF);
        gpMapperLUT = pPolyDataMapper->GetLookupTable();
        //pPolyDataMapper->SetLookupTable(gpLUT);
        //pPolyDataMapper->CreateDefaultLookupTable();
        //gpLUT = vtkLookupTable::SafeDownCast(pPolyDataMapper->GetLookupTable());
        pPolyDataMapper->ScalarVisibilityOn();
        pPolyDataMapper->SetColorModeToMapScalars();
        pPolyDataMapper->ImmediateModeRenderingOn();
        pPolyDataNormals->Delete();
      }
      pActor->SetMapper(pPolyDataMapper);
      pPolyDataMapper->Delete();
    }

    (pActor->GetProperty())->SetSpecular(0.25);
    (pActor->GetProperty())->SetSpecularPower(10);
    vtkProperty* pBackproperty = vtkProperty::New();
    pBackproperty->DeepCopy(pActor->GetProperty());
    pActor->SetBackfaceProperty(pBackproperty);
    pBackproperty->Delete();

    return pActor;
} // MakeContourActor()

// ----------------------------------------------------------------------------
ImagePlaneWidgetVec
MakeImagePlaneWidgets(vtkRenderWindowInteractor* pInteractor,
                      vtkImageData* pImageData)
{
  vtkImagePlaneWidget* pPlaneWidgets[3];
  int dimensions[3];
  float spacing[3];

  pImageData->Update();
  pImageData->GetDimensions(dimensions);
  pImageData->GetSpacing(spacing);

  // VTK_NEAREST_RESLICE | VTK_LINEAR_RESLICE | VTK_CUBIC_RESLICE
  int yzInterpolation = VTK_LINEAR_RESLICE;
  if((spacing[1]!=spacing[0]) || (spacing[2]!=spacing[1]))
    {
    yzInterpolation = VTK_CUBIC_RESLICE;
    }

  // Define the internal picker. The set of three orthogonal planes can 
  // share the same picker so that picking is performed correctly.
  vtkCellPicker* pCellPicker = vtkCellPicker::New();
    {
    pCellPicker->SetTolerance(0.005);
    }
  // Define the shared property for the display text.
  vtkTextProperty* pTextProperty = vtkTextProperty::New();
    {
    pTextProperty->SetColor(1,1,0.9);
    pTextProperty->ShadowOn();
    pTextProperty->AntiAliasingOn();
    }

  pPlaneWidgets[0] = vtkImagePlaneWidget::New();
    {
    pPlaneWidgets[0]->SetInteractor(pInteractor);
    pPlaneWidgets[0]->SetKeyPressActivationValue('x');
    pPlaneWidgets[0]->SetPicker(pCellPicker);
    pPlaneWidgets[0]->SetInput(pImageData);
    pPlaneWidgets[0]->GetPlaneProperty()->SetColor(1,0,0);
    pPlaneWidgets[0]->SetCursorProperty(pPlaneWidgets[0]->GetPlaneProperty());
    pPlaneWidgets[0]->DisplayTextOn();
    pPlaneWidgets[0]->SetTextProperty(pTextProperty);
    pPlaneWidgets[0]->TextureInterpolateOn();
    pPlaneWidgets[0]->SetResliceInterpolate(VTK_LINEAR_RESLICE);
    pPlaneWidgets[0]->SetPlaneOrientationToXAxes();
    pPlaneWidgets[0]->SetSlicePosition(dimensions[0]*0.16667);
    // Set the internal lut to that of the first plane. In this way, the set 
    // of three orthogonal planes can share the same lut so that 
    // window-levelling is performed uniformly among planes.
    gpLUT = pPlaneWidgets[0]->GetLookupTable();
    
    vtkIdType m = gpLUT->GetNumberOfTableValues();
    float rgba[4], value = gMin, incr = (gMax-gMin)/float(m);
    
    for(vtkIdType n=0; n<m; n++)
      {
      gpMapperLUT->GetColor(value,rgba);
      rgba[3] = gpOpacityTF->GetValue(value);
      gpLUT->SetTableValue(n,rgba);
      value += incr;
      }
    
    pPlaneWidgets[0]->On();
    }

  pPlaneWidgets[1] = vtkImagePlaneWidget::New();
    {
    pPlaneWidgets[1]->SetInteractor(pInteractor);
    pPlaneWidgets[1]->SetKeyPressActivationValue('y');
    pPlaneWidgets[1]->SetPicker(pCellPicker);
    pPlaneWidgets[1]->SetInput(pImageData);
    pPlaneWidgets[1]->GetPlaneProperty()->SetColor(1,1,0);
    pPlaneWidgets[1]->SetCursorProperty(pPlaneWidgets[1]->GetPlaneProperty());
    pPlaneWidgets[1]->DisplayTextOn();
    pPlaneWidgets[1]->SetTextProperty(pTextProperty);
    pPlaneWidgets[1]->TextureInterpolateOn();
    pPlaneWidgets[1]->SetResliceInterpolate(yzInterpolation);
    pPlaneWidgets[1]->SetPlaneOrientationToYAxes();
    pPlaneWidgets[1]->SetSlicePosition(dimensions[1]*0.6667);
    pPlaneWidgets[1]->SetLookupTable(pPlaneWidgets[0]->GetLookupTable());
    pPlaneWidgets[1]->On();
    }

  pPlaneWidgets[2] = vtkImagePlaneWidget::New();
    {
    pPlaneWidgets[2]->SetInteractor(pInteractor);
    pPlaneWidgets[2]->SetKeyPressActivationValue('z');
    pPlaneWidgets[2]->SetPicker(pCellPicker);
    pPlaneWidgets[2]->SetInput(pImageData);
    pPlaneWidgets[2]->GetPlaneProperty()->SetColor(0,0,1);
    pPlaneWidgets[2]->SetCursorProperty(pPlaneWidgets[2]->GetPlaneProperty());
    pPlaneWidgets[2]->DisplayTextOn();
    pPlaneWidgets[2]->SetTextProperty(pTextProperty);
    pPlaneWidgets[2]->TextureInterpolateOn();
    pPlaneWidgets[2]->SetResliceInterpolate(yzInterpolation);
    pPlaneWidgets[2]->SetPlaneOrientationToZAxes();
    pPlaneWidgets[2]->SetSlicePosition(dimensions[2]*0.333);
    pPlaneWidgets[2]->SetLookupTable(pPlaneWidgets[0]->GetLookupTable());
    pPlaneWidgets[2]->On();
    }

  return ImagePlaneWidgetVec(&pPlaneWidgets[0], &pPlaneWidgets[3]);
}

// ----------------------------------------------------------------------------
static char*
args_cat_dup(int nargs, char** args)
{
  char*       cat = 0;
  std::size_t size = 0;

  if(nargs>1)
    { // concatenate
    if(svv.GetDebug())
      {
      std::cerr << "concatenating " << nargs << " args" << std::endl;
      }
    size = (nargs + 1); // spaces btwn args + terminating nul.
    for(int n=0; n<nargs; n++) size += strlen(args[n]);

    cat = new char[size];
    cat[0] = '\0';

    for(int n=0; n<nargs; n++)
      {
      strcat(cat,args[n]);
      if(n<(nargs-1)) strcat(cat," ");
      }
    }
  else {
  size = strlen(args[0]) + 1;
  cat = new char[size];
  strcpy(cat, args[0]);
  }

  if(svv.GetDebug())
    {
    std::cerr << "alloc=" << size << " vs " << " strlen=" << strlen(cat)
              << std::endl;
    }

  return cat;
}

// ----------------------------------------------------------------------------
static int
ParseArgs(int argc, char** argv, int& i)
{
  static vtkProperty* pProperty = vtkProperty::New();

  // pointer to the current flag
  const char*  flag = argv[i];
  char**       args;

  bool valid = false;

  // determines how many arguments follow the current flag
  int nargs = 0;
    {
    while((i+nargs+1) < argc && argv[i+nargs+1][0]!='-') nargs++;
    
    if(svv.GetDebug())
      {
      std::cerr << " '" << &flag[0] << "' + ( ";
      }
    if(nargs)
      {
      args = argv + std::ptrdiff_t(i+1);
      if(svv.GetDebug())
        {
        for(int n=0; n<nargs; n++)
          {
          std::cerr<< "'"<< args[n]<< "' ";
          }
        }
      }
    else
      {
      args = 0;
      if(svv.GetDebug())
        {
        std::cerr << "none ";
        }
      }
    if(svv.GetDebug())
      {
      std::cerr << ")" << endl;
      }
    }
    
    // first check for flags that do not take arguments
    if(args==0)
      {
      ;
      }
    else
      {
      // some flags require that multiple arguments be concatenated
      if( flag[1]=='c' || flag[1]=='m' ||
          flag[1]=='K' || flag[1]=='S' )
        {
        
        const char* ptr = args_cat_dup(nargs, args);
	
        int         n;
        const char* first;
        char*       last;
        float       rgba[4] = { 1.f, 1.f, 1.f, 1.f };
        
        first = ptr;
        last  = 0/*NULL*/;
        
        // specify an RGB(A) color by name, by triplet, or by quadruplet
        if(flag[1]=='c'  && !flag[2])
          {
          
          // first attempt to parse an RGB triplet or RGBA quadruplet
          for(n=0; n<4; n++)
            {
            if((rgba[n] = float(strtod(first, &last)))==0 && last==first) break;
            while(isspace(*last)) last++; /* chomp whitespace */
            first = last; /* point to char after last conversion */
            last  = 0; /* rezero to allow check for conversion failure */
            }
          
          if(pProperty) pProperty->Delete();
          
          if(n>2) pProperty = svv.MakeProperty(rgba);
          else    pProperty = svv.MakeProperty(ptr);
          
          if(svv.GetDebug() && pProperty) pProperty->Print(std::cerr);
          valid = true;
          } // '-c'
        
        // specify a material by name
        else if(flag[1]=='m' && !flag[2])
          {
          
          if(pProperty) pProperty->Delete();
          pProperty = svv.MakeProperty(ptr);
          
          if(svv.GetDebug() && pProperty) pProperty->Print(std::cerr);
          valid = true;
          } // '-m'
        
        // specify material reflectance
        else if(flag[1]=='K' && !flag[3])
          {
          
          // first attempt to parse an RGB triplet
          for(n=0; n<3; n++)
            {
            if((rgba[n] = float(strtod(first, &last)))==0 && last==first) break;
            while(isspace(*last)) last++; /* chomp whitespace */
            first = last; /* point to char after last conversion */
            last  = 0; /* rezero to allow check for conversion failure */
            }
          if(n)
            {
            if(flag[2]=='a')
              {
              if(n<3) pProperty->SetAmbient(rgba[0]);
              else pProperty->SetAmbientColor(rgba);
              valid = true;
              }
            else if(flag[2]=='d')
              {
              if(n<3) pProperty->SetDiffuse(rgba[0]);
              else pProperty->SetDiffuseColor(rgba);
              valid = true;
              }
            else if(flag[2]=='s')
              {
              if(n<3) pProperty->SetSpecular(rgba[0]);
              else pProperty->SetSpecularColor(rgba);
              valid = true;
              }
            }
          } // '-Ka' || '-Kd' || '-Ks'
        
        // specify material reflectance
        else if(flag[1]=='S' && flag[2]=='e' && !flag[3])
          {
          pProperty->SetSpecularPower(atof(args[0]));
          valid = true;
          } // '-Se'
        
        }
      else
        {
        // specify input pathnames
        if(flag[1]=='i')
          {
          if(nargs > 1)
            {
            std::cerr << "'-i' flag require one (and only one) anatomical data"
                      << " input path/prefix" << std::endl;
            }
          else
            {
            gAnatomicalData = GetImageData(args[0]);
            valid = true;
            }
          } // '-i'
        
        // specify input pathnames
        if(flag[1]=='f')
          {
          if(nargs > 1)
            {
            std::cerr << "'-f' flag require one (and only one) functional data"
                      << " input path" << std::endl;
            }
          else
            {
            gFunctionalData = GetFunctionalData(args[0]);
            valid = true;
            }
          } // '-f'
        
        // specify input surface pathnames
        if(flag[1]=='s')
          {
          vtkInputSource* inputSource;
          vtkActor*       actor;
          
          for(int n=0; n<nargs; n++)
            {
            // make the "standard" surface
            inputSource = svv.MakeSurfaceDataSource(args[n]);
            
            vtkPolyDataNormals* pNormals = vtkPolyDataNormals::New();
              {
              vtkTransformPolyDataFilter* pTransformFilter =
                vtkTransformPolyDataFilter::New();
                {
                pTransformFilter->SetInput(inputSource->GetPolyDataOutput());
                vtkTransform* pTransform = vtkTransform::New();
                  {
                  pTransform->Scale(1.0, -1.0, 1.0);
                  pTransform->RotateX(90.0);
                  }
                pTransformFilter->SetTransform(pTransform);
                pTransform->Delete();
                }
              pNormals->SetInput(pTransformFilter->GetOutput());
              pTransformFilter->Delete();
              }
            actor = svv.MakeSurfaceActor(pNormals, pProperty);
            pNormals->Delete();
            }
          // references are retained in the list so decrement ref count here
          actor->Delete();
          inputSource->Delete();
          actor       = NULL;
          inputSource = NULL;
          valid       = true;
          } // '-s'
        
        if(flag[1]=='r')
          {
          double res[3], m[16], inv[16], trp[16], scale;
          
          int ret = ReadRegistration(args[0], res, scale, m);
          
          std::cerr << "res: (" << res[0] << "," << res[1] << "," << res[2]
                    << ")\n scale: " << scale << "\n"
                    << "m:\t( " << m[0] << ", " << m[1] << ", " << m[2]
                    << ", " << m[3] << ",\n\t  " << m[4] << ", " << m[5] << ", "
                    << m[6] << ", " << m[7] << ",\n\t  " << m[8] << ", " << m[9]
                    << ", " << m[10] << ", " << m[11] << ",\n\t  " << m[12]
                    << ", " << m[13] << ", " << m[14] << ", " << m[15] << " )\n"
                    << std::endl;
          
          vtkMatrix4x4::Transpose(m,trp);
          
          std::cerr << "trp:\t( " << trp[0] << ", " << trp[1] << ", "
                    << trp[2] << ", " << trp[3] << ",\n\t  "
                    << trp[4] << ", " << trp[5] << ", " << trp[6]
                    << ", " << trp[7] << ",\n\t  " << trp[8] << ", "
                    << trp[9] << ", " << trp[10] << ", " << trp[11]
                    << ",\n\t  " << trp[12] << ", " << trp[13] << ", "
                    << trp[14] << ", " << trp[15] << " )\n" << std::endl;
          
          vtkMatrix4x4::Invert(trp,inv);
          
          std::cerr << "inv:\t( " << inv[0] << ", " << inv[1] << ", "
                    << inv[2] << ", " << inv[3] << ",\n\t  "
                    << inv[4] << ", " << inv[5] << ", " << inv[6]
                    << ", " << inv[7] << ",\n\t  " << inv[8] << ", "
                    << inv[9] << ", " << inv[10] << ", " << inv[11]
                    << ",\n\t  " << inv[12] << ", " << inv[13] << ", "
                    << inv[14] << ", " << inv[15] << " )\n" << std::endl;
          
          if(gTransform) gTransform->Delete();
          gTransform = vtkTransform::New();
          gTransform->SetMatrix(m);
          } // '-r'
        
        valid = true;
        }
      
      }
    
    if(valid)
      {
      i += (nargs+1);
      return (nargs+1);
      }
    else
      {
      return 0;
      }
}

// ----------------------------------------------------------------------------
static void
PrintUsage(std::ostream& aTarget, const char* aProgName)
{
  aTarget << "Usage: " << aProgName << " -i surface.vtk ...\n" << std::endl;
}

/* 
 * End of: $Id: Contour.cxx,v 1.1.1.1 2006/12/19 22:59:37 christianh Exp $.
 *  
 */
