// IOStreams
#include <iostream>
// STL
#include <string>
#include <fstream>
#include <vector>
// FLTK
#include <FL/Fl_File_Chooser.H> /* for fl_file_chooser() */
// VTK Common
#include "vtkMatrix4x4.h"
#include "vtkTransform.h"
#include "vtkImageData.h"
#include "vtkStructuredPoints.h"
#include "vtkPolyData.h"
// VTK Graphics
#include "vtkPolyDataNormals.h"
#include "vtkCleanPolyData.h"
#include "vtkContourFilter.h"
#include "vtkTriangleFilter.h"
#include "vtkCurvatures.h"
#include "vtkWindowedSincPolyDataFilter.h"
// VTK Imaging
#include "vtkImageResample.h"
#include "vtkImageThreshold.h"
#include "vtkImageDilateErode3D.h"
#include "vtkImageToStructuredPoints.h"
// VTK Rendering
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkCamera.h"
#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkActor.h"
#include "vtkLODActor.h"
// VTK IO
#include "vtkImageReader.h"
#include "vtkDataSetWriter.h"
#include "vtkXMLDataSetWriter.h"
#include "vtkZLibDataCompressor.h"
#include "vtkPolyDataReader.h"
#include "vtkXMLPolyDataReader.h"

#include "vtkImagePlaneWidget.h"
#include "vtkCellPicker.h"
#include "vtkTextProperty.h"

#include "vtkLookupTable.h"
#include "vtkScalarsToColors.h"
#include "vtkPiecewiseFunction.h"
#include "vtkColorTransferFunction.h"

#include "ContourBucharUI.h"


// ----------------------------------------------------------------------------
typedef std::vector<vtkImagePlaneWidget*> ImagePlaneWidgetVec;

// ----------------------------------------------------------------------------
vtkStructuredPoints* LoadImageData(const char* const);
vtkPolyData* CreateContourData(vtkStructuredPoints*);
vtkActor* CreateContourActor(vtkPolyData*);
vtkImagePlaneWidget**
MakeImagePlaneWidgets(vtkRenderWindowInteractor*,vtkImageData*);

// ----------------------------------------------------------------------------
static const char* gInputName = NULL;
static const char* gReferenceName = NULL;
 
static float gRefBounds[6] = { 0, 0, 0, 0, 0, 0 };
static float gRefCenter[3] = { 0, 0, 0 };
static int   gRefWholeExtent[6] = { 0, 0, 0, 0, 0, 0 };
 
static float gOutputBounds[6] = { 0, 0, 0, 0, 0, 0 };
static float gOutputCenter[3] = { 0, 0, 0 };
static int   gOutputWholeExtent[6] = { 0, 0, 0, 0, 0, 0 };

// ----------------------------------------------------------------------------
vtkStructuredPoints* gImageData = NULL;
vtkPolyData* gContourData = NULL;

// ----------------------------------------------------------------------------
vtkImageReader* gImageReader = NULL;
vtkTransform* gTransform = NULL;
vtkImageThreshold* gInputThreshold = NULL;
vtkImageResample* gImageResample = NULL;
vtkImageThreshold* gBinaryThreshold = NULL;
vtkImageDilateErode3D* gImageDilateErode3D = NULL;
vtkImageToStructuredPoints* gInputStructuredPoints = NULL;
vtkImageToStructuredPoints* gFilteredStructuredPoints = NULL;
vtkWindowedSincPolyDataFilter* gSmoother = NULL;
vtkPolyDataNormals* gNormals = NULL;
vtkTriangleFilter* gTriangleFilter = NULL;
vtkCleanPolyData* gCleaner = NULL;
vtkActor* gContourActor = NULL;
vtkImagePlaneWidget** gImagePlaneWidgets = NULL;

// ----------------------------------------------------------------------------
float gInputScalarRange[2] = { VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT };

int gSmoothing = 40;
float gLabel = 0.f;
int gKernelSize = 3;

// Create a transfer function mapping scalar value to opacity
static vtkPiecewiseFunction*     gOpacityTF;
// Create a transfer function mapping scalar value to color (grey)
static vtkPiecewiseFunction*     gGrayTF;
// Create a transfer function mapping scalar value to color (color)
static vtkColorTransferFunction* gColorTF;
// Create a transfer function mapping magnitude of gradient to opacity
static vtkPiecewiseFunction*     gGradientTF;

static vtkLookupTable*     gLUT = 0;
static vtkScalarsToColors* gMapperLUT = 0;

// ----------------------------------------------------------------------------
static double gTransformScale[3] = {1.0, 1.0, 1.0};
static double gTransformTranslate[3] = {0.0, 0.0, 0.0};
static double gTransformRotateX = 0.0;
static double gTransformRotateY = 0.0;
static double gTransformRotateZ = 0.0;
static double gDirectionCosines[9] = {-1.0,0.0,0.0,0.0,0.0,1.0,0.0,-1.0,0.0};
static float  gInputSpacing[3]  = { 1.5f, 1.5f, 3.0f };
static float  gOutputSpacing[3] = { 1.5f, 1.5f, 1.5f };

static int    gInputDimensions[3];
static int    gExtent[6];
static float  gOrigin[3];
static float  gResampleScale = 1.f;

// ----------------------------------------------------------------------------
int
main(int argc, char* argv[])
{
  if (argc < 2)
    {
    std::cerr << "\nUsage: " << argv[0] << " <in.buchar>"
              << " [<out.vtk>|<out.vtp>]\n" << std::endl;
    return -1;
    }

  gInputName = argv[1];

  if (argc > 2)
    {
    gReferenceName = argv[2];
    }

  // create the interface
  ContourBucharUI* GUI = new ContourBucharUI;
  vtkRenderWindowInteractor* interactor = GUI->GetView()->GetInteractor();
  vtkRenderWindow* renderWindow = GUI->GetView()->GetRenderWindow();
  vtkRenderer* renderer = GUI->GetView()->GetDefaultRenderer();
  vtkCamera* camera = GUI->GetView()->GetDefaultCamera();

  interactor->SetDesiredUpdateRate(4.0);
  renderWindow->SetDesiredUpdateRate(4.0);
  renderWindow->LineSmoothingOff();
   
  // show() the main Fl_Window
  GUI->Show();

  // this is the standard way of "starting" a fltk application
  int fl_ret = Fl::run();

  // Clean up
  if (gFilteredStructuredPoints) gFilteredStructuredPoints->Delete();
  if (gContourData) gContourData->GetSource()->Delete();
  if (gContourActor) gContourActor->Delete();
  gImagePlaneWidgets[0]->Delete();
  gImagePlaneWidgets[1]->Delete();
  gImagePlaneWidgets[2]->Delete();
  delete [] gImagePlaneWidgets;
  delete GUI;

  return fl_ret;
}

// ----------------------------------------------------------------------------
static void
BuildTransferFunctions(float min, float max)
{
  float span = max - min;

  // Create transfer mapping scalar value to opacity
  if (gOpacityTF) gOpacityTF->RemoveAllPoints();
  else gOpacityTF = vtkPiecewiseFunction::New();

  gOpacityTF->AddSegment(0,                0.0, 0,                0.0);
  gOpacityTF->AddSegment(1,                1.0, min + span * 0.2, 1.0);
  gOpacityTF->AddSegment(min + span * 0.2, 0.0, min + span * 0.8, 0.0);
  gOpacityTF->AddSegment(min + span * 0.8, 1.0, min + span * 1.0, 1.0);
  
  // Create transfer mapping scalar value to grayscale
  if (gGrayTF) gGrayTF->RemoveAllPoints();
  else gGrayTF = vtkPiecewiseFunction::New();

  gGrayTF->AddSegment(min, 0.0, max, 1.0);

  // Create transfer mapping scalar value to color
  if (gColorTF) gColorTF->RemoveAllPoints();
  else gColorTF = vtkColorTransferFunction::New();

  gColorTF->AddRGBPoint(1,                1.0, 0.0, 0.0);
  gColorTF->AddRGBPoint(min + span * 0.1, 0.0, 0.0, 1.0);
  gColorTF->AddRGBPoint(min + span * 0.2, 0.0, 1.0, 0.0);
  gColorTF->AddRGBPoint(min + span * 0.5, 0.0, 0.0, 0.0);
  gColorTF->AddRGBPoint(min + span * 0.8, 1.0, 0.0, 0.0);
  gColorTF->AddRGBPoint(min + span * 0.9, 0.0, 0.0, 1.0);
  gColorTF->AddRGBPoint(min + span * 1.0, 0.0, 1.0, 0.0);

  // Create a transfer function mapping magnitude of gradient to opacity
  if (gGradientTF) gGradientTF->RemoveAllPoints();
  else gGradientTF = vtkPiecewiseFunction::New();

  gGradientTF->AddPoint(min + span * 0.0, 0.0);
  gGradientTF->AddPoint(min + span * 0.5, 0.5);
  gGradientTF->AddPoint(min + span * 1.0, 0.5);
}

#if 0
// ----------------------------------------------------------------------------
vtkStructuredPoints*
LoadImageData(const char* const aFileName)
{
  if (!aFileName || !*aFileName)
    {
    std::cerr << "LoadImageData received empty filename!" << std::endl;
    return NULL;
    }

  std::string fileName(aFileName);
  std::string::size_type idx = fileName.rfind('.');
  std::string suffix(idx==std::string::npos ? "" : fileName.substr(idx+1));
  const std::string hdr(".hdr");
  std::string hdrName = fileName.substr(0,fileName.find_last_of('.')) + hdr;

  std::ifstream hdrStream;

  // Opens the file and positions the stream pointer at EOF ...
  hdrStream.open(hdrName.c_str(), std::ios::in | std::ios::ate);

  if (!hdrStream.is_open())
    {
    std::cerr << "Failed to open header file '" << hdrName << "'" << std::endl;
    return NULL;
    }

  // Rewind the stream ... and start reading the items.
  hdrStream.seekg(0, std::ios::beg);
  // Read the file dimensions from the header (rows, cols, slices, 0).
  hdrStream >> gInputDimensions[1]
            >> gInputDimensions[0]
            >> gInputDimensions[2];
  hdrStream.close();

  // Derive the extent from the dimensions.
  gExtent[0] = 0;
  gExtent[1] = gInputDimensions[0] - 1;
  gExtent[2] = 0;
  gExtent[3] = gInputDimensions[1] - 1;
  gExtent[4] = 0;
  gExtent[5] = gInputDimensions[2] - 1;

  // Always use cubic voxels.
  float min = ( gInputSpacing[0]<gInputSpacing[1] ? 
                ( gInputSpacing[0]<gInputSpacing[2] ?
                  gInputSpacing[0] : gInputSpacing[2] ) :
                ( gInputSpacing[1]<gInputSpacing[2] ?
                  gInputSpacing[1] : gInputSpacing[2] ) );

  gOutputSpacing[0] = min / gResampleScale;
  gOutputSpacing[1] = min / gResampleScale;
  gOutputSpacing[2] = min / gResampleScale;

  int scalarType = 0;

  if (suffix == "buchar")
    {
    scalarType = VTK_UNSIGNED_CHAR;
    }
  else if (suffix == "bushort")
    {
    scalarType = VTK_UNSIGNED_SHORT;
    }
  else if (suffix == "bshort")
    {
    scalarType = VTK_SHORT;
    }
  else if (suffix == "bfloat")
    {
    scalarType = VTK_FLOAT;
    }
  else
    {
    return NULL;
    }
  
  gOrigin[0] = -(float(gInputDimensions[0]) * gInputSpacing[0] / 2.f);
  gOrigin[1] = -(float(gInputDimensions[1]) * gInputSpacing[1] / 2.f);
  gOrigin[2] = -(float(gInputDimensions[2]) * gInputSpacing[2] / 2.f);

  gBinaryThreshold = vtkImageThreshold::New();
    {
    gImageResample = vtkImageResample::New();
      {
      gInputThreshold = vtkImageThreshold::New();
        {
        gImageReader = vtkImageReader::New();
          {
          gImageReader->SetFileName(fileName.c_str());
          gImageReader->SetHeaderSize(0);
          gImageReader->SetDataByteOrderToBigEndian();
          gImageReader->SetDataScalarType(scalarType);
          gImageReader->SetNumberOfScalarComponents(1);
          gImageReader->SetFileDimensionality(gInputDimensions[2]>1 ? 3 : 2);
          gImageReader->SetDataExtent(gExtent);
          gImageReader->SetDataSpacing(gInputSpacing);
          gImageReader->SetDataOrigin(gOrigin);
          gImageReader->SetFileNameSliceOffset(1/*count from 1*/);
          gImageReader->FileLowerLeftOff();
          }
        gInputThreshold->SetInput(gImageReader->GetOutput());
        gInputThreshold->SetOutputScalarTypeToFloat();
        if (gLabel == 0.f) gInputThreshold->ThresholdByUpper(1.f);
        else               gInputThreshold->ThresholdBetween(gLabel,gLabel);
        gInputThreshold->ThresholdByUpper(1.f);
        gInputThreshold->SetInValue(1.f);
        gInputThreshold->SetOutValue(0.f);
        gInputThreshold->ReplaceInOn();
        gInputThreshold->ReplaceOutOn();
        gImageReader->Delete();
        }
      if (!gTransform)
        {
        gTransform = vtkTransform::New();
        gTransform->Identity();
        gTransform->Scale(gTransformScale);
        gTransform->RotateX(gTransformRotateX);
        gTransform->RotateY(gTransformRotateY);
        gTransform->RotateZ(gTransformRotateZ);
        }
      }
    gImageResample->SetInput(gInputThreshold->GetOutput());
    gImageResample->SetResliceTransform(gTransform);
    //gImageResample->SetResliceAxesDirectionCosines( -1.0, 0.0, 0.0,
    //                                                0.0, 0.0, -1.0,
    //                                                0.0, 1.0, 0.0 );
    //gImageResample->SetResliceAxesOrigin(gOrigin[0],gOrigin[1],gOrigin[2]);
    gImageResample->SetInterpolationMode(VTK_RESLICE_CUBIC);
    gImageResample->SetDimensionality(3);
    gImageResample->SetAxisOutputSpacing(0, gOutputSpacing[0]);
    gImageResample->SetAxisOutputSpacing(1, gOutputSpacing[1]);
    gImageResample->SetAxisOutputSpacing(2, gOutputSpacing[2]);
    gImageResample->AutoCropOutputOn();
    gImageResample->MirrorOn();
    //gImageResample->SetOutputOrigin(gOrigin);
    gInputThreshold->Delete();
    gTransform->Delete();
    }
  gBinaryThreshold->SetInput(gImageResample->GetOutput());
  gBinaryThreshold->SetOutputScalarTypeToUnsignedChar();
  gBinaryThreshold->ThresholdByUpper(0.5f);
  gBinaryThreshold->SetInValue(1);
  gBinaryThreshold->SetOutValue(0);
  gBinaryThreshold->ReplaceInOn();
  gBinaryThreshold->ReplaceOutOn();
  gImageResample->Delete();


  gFilteredStructuredPoints = vtkImageToStructuredPoints::New();
    {
    gImageDilateErode3D = vtkImageDilateErode3D::New();
      {
      gImageDilateErode3D->SetInput(gBinaryThreshold->GetOutput());
      gImageDilateErode3D->SetDilateValue(1);
      gImageDilateErode3D->SetErodeValue(0);
      gImageDilateErode3D->SetKernelSize(gKernelSize, gKernelSize, gKernelSize);
      gBinaryThreshold->Delete();
      }
    gFilteredStructuredPoints->SetInput(gImageDilateErode3D->GetOutput());
    }

  gImageReader->Update();
  gImageReader->GetOutput()->GetScalarRange(gInputScalarRange);

  std::cerr << "scalar range: [ " << gInputScalarRange[0] << " .. "
            << gInputScalarRange[1] << " ]" << std::endl;

  BuildTransferFunctions(gInputScalarRange[0],gInputScalarRange[1]);

  return gFilteredStructuredPoints->GetOutput();
}
#endif /* 0 */

// ----------------------------------------------------------------------------
vtkPolyData*
CreateContourData(vtkStructuredPoints* aImageData)
{
  vtkCurvatures* curvatures = vtkCurvatures::New();
    {
    gNormals = vtkPolyDataNormals::New();
      {
      gTriangleFilter = vtkTriangleFilter::New();
        {
        gCleaner = vtkCleanPolyData::New();
          {
          gSmoother = vtkWindowedSincPolyDataFilter::New();
            {
            vtkContourFilter* contourFilter = vtkContourFilter::New();
              {
              contourFilter->SetInput(aImageData);
              contourFilter->SetValue(0, 0.5);
              contourFilter->ComputeNormalsOff();
              }
            gSmoother->SetInput(contourFilter->GetOutput());
            gSmoother->SetNumberOfIterations(20);        // default = 20
            gSmoother->SetPassBand(0.1);                 // default = 0.1
            gSmoother->SetFeatureAngle(45.0);            // default = 45.0
            gSmoother->SetEdgeAngle(15.0);               // default = 15.0
            gSmoother->SetFeatureEdgeSmoothing(0);       // default = 0 (Off)
            gSmoother->SetBoundarySmoothing(1);          // default = 1 (On)
            gSmoother->SetNonManifoldSmoothing(0);       // default = 0 (Off)
            gSmoother->SetGenerateErrorScalars(0);       // default = 0 (Off)
            gSmoother->SetGenerateErrorVectors(0);       // default = 0 (Off)
            contourFilter->Delete();
            }
          gCleaner->SetInput(gSmoother->GetOutput());
          //gCleaner->SetTolerance(0.005);
          gCleaner->ConvertLinesToPointsOn();
          gCleaner->ConvertPolysToLinesOn();
          gCleaner->ConvertStripsToPolysOn();
          gCleaner->PointMergingOn();
          gSmoother->Delete();
          }
        gTriangleFilter->SetInput(gCleaner->GetOutput());
        gTriangleFilter->PassLinesOff();
        gTriangleFilter->PassVertsOff();
        gCleaner->Delete();
        }
      gNormals->SetInput(gTriangleFilter->GetOutput());
      gNormals->SplittingOff();
      gNormals->ConsistencyOn();
      gNormals->NonManifoldTraversalOff();
      gTriangleFilter->Delete();
      }
    curvatures->SetInput(gNormals->GetOutput());
    curvatures->SetCurvatureTypeToGaussian();
    gNormals->Delete();
    }

  return curvatures->GetOutput();
}

// ----------------------------------------------------------------------------
vtkActor*
CreateContourActor(vtkPolyData* aPolyData)
{
  vtkActor* actor = vtkActor::New();
    {
    vtkPolyDataMapper* polyDataMapper = vtkPolyDataMapper::New();
      {
      //vtkLookupTable* lut = vtkLookupTable::New();
      //  {
      //  lut->SetNumberOfColors(256);
      //  lut->SetHueRange(0.15, 1.0);
      //  lut->SetSaturationRange(1.0, 1.0);
      //  lut->SetValueRange(1.0, 1.0);
      //  lut->SetAlphaRange(1.0, 1.0);
      //  lut->SetRange(-20, 20);
      //  }
      polyDataMapper->SetInput(aPolyData);
      //polyDataMapper->SetLookupTable(lut);
      polyDataMapper->SetUseLookupTableScalarRange(1);
      polyDataMapper->ImmediateModeRenderingOn();
      //lut->Delete();
      }
    actor->SetMapper(polyDataMapper);
    actor->GetProperty()->SetSpecular(0.4);
    actor->GetProperty()->SetSpecularPower(24);
    actor->GetProperty()->SetRepresentationToWireframe();
    polyDataMapper->Delete();
    }
  return actor;
}

// ----------------------------------------------------------------------------
vtkImagePlaneWidget**
MakeImagePlaneWidgets(vtkRenderWindowInteractor* aInteractor,
                      vtkImageData* aImageData)
{
  vtkImagePlaneWidget** planeWidgets = new vtkImagePlaneWidget* [3];

  int dimensions[3];
  float spacing[3];

  gInputStructuredPoints = vtkImageToStructuredPoints::New();
  gInputStructuredPoints->SetInput(aImageData);

  gInputStructuredPoints->Update();
  gInputStructuredPoints->GetOutput()->GetDimensions(dimensions);
  gInputStructuredPoints->GetOutput()->GetSpacing(spacing);

  // Define the internal picker. The set of three orthogonal planes can 
  // share the same picker so that picking is performed correctly.
  vtkCellPicker* cellPicker = vtkCellPicker::New();
    {
    cellPicker->SetTolerance(0.005);
    }
  // Define the shared property for the display text.
  vtkTextProperty* textProperty = vtkTextProperty::New();
    {
    textProperty->SetColor(1,1,0.9);
    textProperty->ShadowOn();
    textProperty->AntiAliasingOn();
    }

  planeWidgets[0] = vtkImagePlaneWidget::New();
    {
    planeWidgets[0]->SetInteractor(aInteractor);
    planeWidgets[0]->SetKeyPressActivationValue('x');
    planeWidgets[0]->SetPicker(cellPicker);
    planeWidgets[0]->SetInput(gInputStructuredPoints->GetOutput());
    planeWidgets[0]->GetPlaneProperty()->SetColor(1,0,0);
    planeWidgets[0]->SetCursorProperty(planeWidgets[0]->GetPlaneProperty());
    planeWidgets[0]->DisplayTextOn();
    planeWidgets[0]->SetTextProperty(textProperty);
    planeWidgets[0]->TextureInterpolateOff();
    planeWidgets[0]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
    planeWidgets[0]->SetPlaneOrientationToXAxes();
    planeWidgets[0]->SetSlicePosition(dimensions[0]*0.16667);
    // Set the internal lut to that of the first plane. In this way, the set 
    // of three orthogonal planes can share the same lut so that 
    // window-levelling is performed uniformly among planes.
    gLUT = planeWidgets[0]->GetLookupTable();
#if 1
    vtkIdType nValues = gLUT->GetNumberOfTableValues();
    float rgba[4];
    float value = gInputScalarRange[0];
    float incr  = (gInputScalarRange[1] - gInputScalarRange[0]) / float(nValues);

    for(vtkIdType n=0; n<nValues; n++)
      {
      gColorTF->GetColor(value,rgba);
      rgba[3] = gOpacityTF->GetValue(value);
      gLUT->SetTableValue(n,rgba);
      value += incr;
      }
#endif
    planeWidgets[0]->On();
    }

  planeWidgets[1] = vtkImagePlaneWidget::New();
    {
    planeWidgets[1]->SetInteractor(aInteractor);
    planeWidgets[1]->SetKeyPressActivationValue('y');
    planeWidgets[1]->SetPicker(cellPicker);
    planeWidgets[1]->SetInput(gInputStructuredPoints->GetOutput());
    planeWidgets[1]->GetPlaneProperty()->SetColor(1,1,0);
    planeWidgets[1]->SetCursorProperty(planeWidgets[1]->GetPlaneProperty());
    planeWidgets[1]->DisplayTextOn();
    planeWidgets[1]->SetTextProperty(textProperty);
    planeWidgets[1]->TextureInterpolateOff();
    planeWidgets[1]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
    planeWidgets[1]->SetPlaneOrientationToYAxes();
    planeWidgets[1]->SetSlicePosition(dimensions[1]*0.6667);
    planeWidgets[1]->SetLookupTable(planeWidgets[0]->GetLookupTable());
    planeWidgets[1]->On();
    }

  planeWidgets[2] = vtkImagePlaneWidget::New();
    {
    planeWidgets[2]->SetInteractor(aInteractor);
    planeWidgets[2]->SetKeyPressActivationValue('z');
    planeWidgets[2]->SetPicker(cellPicker);
    planeWidgets[2]->SetInput(gInputStructuredPoints->GetOutput());
    planeWidgets[2]->GetPlaneProperty()->SetColor(0,0,1);
    planeWidgets[2]->SetCursorProperty(planeWidgets[2]->GetPlaneProperty());
    planeWidgets[2]->DisplayTextOn();
    planeWidgets[2]->SetTextProperty(textProperty);
    planeWidgets[2]->TextureInterpolateOff();
    planeWidgets[2]->SetResliceInterpolate(VTK_NEAREST_RESLICE);
    planeWidgets[2]->SetPlaneOrientationToZAxes();
    planeWidgets[2]->SetSlicePosition(dimensions[2]*0.333);
    planeWidgets[2]->SetLookupTable(planeWidgets[0]->GetLookupTable());
    planeWidgets[2]->On();
    }

  gInputStructuredPoints->Delete();

  return planeWidgets;
}

// ----------------------------------------------------------------------------
void
UpdateButtonCb(Fl_Return_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  ContourBucharUI* GUI;

  if(!(GUI = reinterpret_cast<ContourBucharUI*>(aPtr))) return;

  // --------------------------------------------------------------------------
  gInputSpacing[0] = GUI->mSpacingX->value();
  gInputSpacing[1] = GUI->mSpacingY->value();
  gInputSpacing[2] = GUI->mSpacingZ->value();

  float minVoxelLength = ( gInputSpacing[0]<gInputSpacing[1] ? 
                           ( gInputSpacing[0]<gInputSpacing[2] ?
                             gInputSpacing[0] : gInputSpacing[2] ) :
                           ( gInputSpacing[1]<gInputSpacing[2] ?
                             gInputSpacing[1] : gInputSpacing[2] ) );
  
  gResampleScale = GUI->ScaleInput->value();

  gOutputSpacing[0] = minVoxelLength / gResampleScale;
  gOutputSpacing[1] = minVoxelLength / gResampleScale;
  gOutputSpacing[2] = minVoxelLength / gResampleScale;

  gLabel = float(GUI->LabelNumberInput->value());

  gKernelSize = int(GUI->KernelSizeInput->value());

  gDirectionCosines[0] = GUI->mCosX0->value();
  gDirectionCosines[1] = GUI->mCosX1->value();
  gDirectionCosines[2] = GUI->mCosX2->value();
  gDirectionCosines[3] = GUI->mCosY0->value();
  gDirectionCosines[4] = GUI->mCosY1->value();
  gDirectionCosines[5] = GUI->mCosY2->value();
  gDirectionCosines[6] = GUI->mCosZ0->value();
  gDirectionCosines[7] = GUI->mCosZ1->value();
  gDirectionCosines[8] = GUI->mCosZ2->value();
      
  // --------------------------------------------------------------------------
  double scale[3], translate[3], rotateX, rotateY, rotateZ;

  scale[0] = GUI->mScaleX->value();
  scale[1] = GUI->mScaleY->value();
  scale[2] = GUI->mScaleZ->value();
  rotateX  = GUI->mRotateX->value();
  rotateY  = GUI->mRotateY->value();
  rotateZ  = GUI->mRotateZ->value();
  translate[0] = GUI->mTranslateX->value();
  translate[1] = GUI->mTranslateY->value();
  translate[2] = GUI->mTranslateZ->value();

  if ( (!gTransform) ||
       (gTransformScale[0] != scale[0]) ||
       (gTransformScale[1] != scale[1]) ||
       (gTransformScale[2] != scale[2]) ||
       (gTransformRotateX != rotateX) ||
       (gTransformRotateY != rotateY) ||
       (gTransformRotateZ != rotateZ) ||
       (gTransformTranslate[0] != translate[0]) ||
       (gTransformTranslate[1] != translate[1]) ||
       (gTransformTranslate[2] != translate[2]) )
    {
    gTransformScale[0] = scale[0];
    gTransformScale[1] = scale[1];
    gTransformScale[2] = scale[2];
    gTransformRotateX  = rotateX;
    gTransformRotateY  = rotateY;
    gTransformRotateZ  = rotateZ;
    gTransformTranslate[0] = translate[0];
    gTransformTranslate[1] = translate[1];
    gTransformTranslate[2] = translate[2];
    
    double inScale[3], inOrientation[3];
    double outScale[3], outOrientation[3];
    
    if (!gTransform) gTransform = vtkTransform::New();

    gTransform->Identity();
    gTransform->Scale(gTransformScale);
    gTransform->RotateX(gTransformRotateX);
    gTransform->RotateY(gTransformRotateY);
    gTransform->RotateZ(gTransformRotateZ);
    gTransform->Translate(gTransformTranslate);

    if (gImageReader) gImageReader->Modified();
    
    gTransform->Print(std::cerr);
    }

  // --------------------------------------------------------------------------
  if (gReferenceName)
    {
    std::string fileName(gReferenceName);
    std::string::size_type idx = fileName.rfind('.');
    std::string suffix(idx==std::string::npos ? "vtk" : fileName.substr(idx+1));

    vtkActor* actor = NULL;
    vtkPolyDataMapper* mapper = NULL;

    if (suffix == "vtk")
      {
      vtkPolyDataReader* reader = vtkPolyDataReader::New();
      reader->SetFileName(gReferenceName);
      //if (reader->IsFileValid(gReferenceName))
      //  {
        mapper = vtkPolyDataMapper::New();
        mapper->SetInput(reader->GetOutput());
        actor = vtkActor::New();
        actor->SetMapper(mapper);
        mapper->Delete();
        GUI->GetView()->GetDefaultRenderer()->AddProp(actor);
        actor->Delete();
        //  }
      reader->Delete();
      }
    else if (suffix == "vtp" || suffix == "xml")
      {
      vtkXMLPolyDataReader* reader = vtkXMLPolyDataReader::New();
      reader->SetFileName(gReferenceName);
      //if (reader->CanReadFile(gReferenceName))
      //  {
        mapper = vtkPolyDataMapper::New();
        mapper->SetInput(reader->GetOutput());
        actor = vtkActor::New();
        actor->SetMapper(mapper);
        mapper->Delete();
        GUI->GetView()->GetDefaultRenderer()->AddProp(actor);
        actor->Delete();
        //  }
      reader->Delete();
      }

    if (mapper)
      {
      vtkPolyData* polyData = mapper->GetInput();
      polyData->Update();
      polyData->GetBounds(gRefBounds);
      polyData->GetCenter(gRefCenter);
      polyData->GetWholeExtent(gRefWholeExtent);
      }
    }

  // --------------------------------------------------------------------------
  if (!gImageReader)
    {
    if (!gInputName || !*gInputName)
      {
      std::cerr << "Input filename is empty!" << std::endl;
      return;
      }

    std::string fileName(gInputName);
    std::string::size_type idx = fileName.rfind('.');
    std::string suffix(idx==std::string::npos ? "" : fileName.substr(idx+1));
    const std::string hdr(".hdr");
    std::string hdrName = fileName.substr(0,fileName.find_last_of('.')) + hdr;
    
    std::ifstream hdrStream;

    // Opens the file and positions the stream pointer at EOF ...
    hdrStream.open(hdrName.c_str(), std::ios::in | std::ios::ate);

    if (!hdrStream.is_open())
      {
      std::cerr << "Failed to open header file '" << hdrName << "'"
                << std::endl;
      return;
      }

    // Rewind the stream ... and start reading the items.
    hdrStream.seekg(0, std::ios::beg);
    // Read the file dimensions from the header (rows, cols, slices, 0).
    hdrStream >> gInputDimensions[1]
              >> gInputDimensions[0]
              >> gInputDimensions[2];
    hdrStream.close();

    // Derive the extent from the dimensions.
    gExtent[0] = 0;
    gExtent[1] = gInputDimensions[0] - 1;
    gExtent[2] = 0;
    gExtent[3] = gInputDimensions[1] - 1;
    gExtent[4] = 0;
    gExtent[5] = gInputDimensions[2] - 1;

    int scalarType = 0;

    if      (suffix == "buchar")  scalarType = VTK_UNSIGNED_CHAR;
    else if (suffix == "bushort") scalarType = VTK_UNSIGNED_SHORT;
    else if (suffix == "bshort")  scalarType = VTK_SHORT;
    else if (suffix == "bfloat")  scalarType = VTK_FLOAT;
    else return;
  
    gImageReader = vtkImageReader::New();
    gImageReader->SetFileName(fileName.c_str());
    //gImageReader->SetTransform(gTransform);
    gImageReader->SetHeaderSize(0);
    gImageReader->SetDataByteOrderToBigEndian();
    gImageReader->SetDataScalarType(scalarType);
    gImageReader->SetNumberOfScalarComponents(1);
    gImageReader->SetFileDimensionality(gInputDimensions[2]>1 ? 3 : 2);
    gImageReader->SetDataExtent(gExtent);
    //gImageReader->SetFileNameSliceOffset(1/*count from 1*/);
    gImageReader->FileLowerLeftOff();
    //gTransform->Delete();
    }
  //else if (gImageReader->GetTransform() != gTransform)
  //  {
  //  gImageReader->SetTransform(gTransform);
  //  gTransform->Delete();
  //  }

  gOrigin[0] = -((float(gInputDimensions[0]) * gInputSpacing[0]) / 2.f);
  gOrigin[1] = -((float(gInputDimensions[1]) * gInputSpacing[1]) / 2.f);
  gOrigin[2] = -((float(gInputDimensions[2]) * gInputSpacing[2]) / 2.f);

  gImageReader->SetDataSpacing(gInputSpacing);
  gImageReader->SetDataOrigin(gOrigin);

  // --------------------------------------------------------------------------
  gImageReader->Update();
  gImageReader->GetOutput()->GetScalarRange(gInputScalarRange);

  std::cerr << "scalar range: [ " << gInputScalarRange[0] << " .. "
            << gInputScalarRange[1] << " ]" << std::endl;

  BuildTransferFunctions(gInputScalarRange[0],gInputScalarRange[1]);

  // --------------------------------------------------------------------------
  if (!gImagePlaneWidgets)
    {
    gImagePlaneWidgets = MakeImagePlaneWidgets( GUI->GetView()->GetInteractor(),
                                                gImageReader->GetOutput() );
    GUI->GetView()->GetRenderWindow()->Render();
    GUI->GetView()->GetDefaultRenderer()->ResetCamera();
    }
  else if (gInputStructuredPoints)
    {
    gInputStructuredPoints->SetInput(gImageReader->GetOutput());
    }
  else
    {
    std::cerr << "Failed to get input structured points!" << std::endl;
    return;
    }

  // --------------------------------------------------------------------------
  if (!gInputThreshold)
    {
    gInputThreshold = vtkImageThreshold::New();
    gInputThreshold->SetInput(gImageReader->GetOutput());
    gInputThreshold->SetOutputScalarTypeToFloat();
    gInputThreshold->ThresholdByUpper(1.f);
    gInputThreshold->SetInValue(1.f);
    gInputThreshold->SetOutValue(0.f);
    gInputThreshold->ReplaceInOn();
    gInputThreshold->ReplaceOutOn();
    gImageReader->Delete();
    }
  else if (gInputThreshold->GetInput() != gImageReader->GetOutput())
    {
    gInputThreshold->SetInput(gImageReader->GetOutput());
    gImageReader->Delete();
    }

  if (gLabel == 0.f) gInputThreshold->ThresholdByUpper(1.f);
  else               gInputThreshold->ThresholdBetween(gLabel,gLabel);

  // --------------------------------------------------------------------------
  if (!gImageResample)
    {
    gImageResample = vtkImageResample::New();
    gImageResample->SetInput(gInputThreshold->GetOutput());
    gImageResample->SetResliceAxesDirectionCosines(gDirectionCosines);
    //gImageResample->SetResliceAxesOrigin(gOrigin[0],gOrigin[1],gOrigin[2]);
    gImageResample->SetResliceTransform(gTransform);
    gImageResample->SetInterpolationMode(VTK_RESLICE_NEAREST);
    gImageResample->SetDimensionality(3);
    gImageResample->AutoCropOutputOn();
    //gImageResample->MirrorOn();
    //gImageResample->SetOutputOrigin(gOrigin);
    gInputThreshold->Delete();
    gTransform->Delete();
    }
  else if (gImageResample->GetInput() != gInputThreshold->GetOutput())
    {
    gImageResample->SetInput(gInputThreshold->GetOutput());
    gInputThreshold->Delete();
    if (gImageResample->GetResliceTransform() != gTransform)
      {
      gImageResample->SetResliceTransform(gTransform);
      gTransform->Delete();
      }
    }

  gImageResample->SetAxisOutputSpacing(0, gOutputSpacing[0]);
  gImageResample->SetAxisOutputSpacing(1, gOutputSpacing[1]);
  gImageResample->SetAxisOutputSpacing(2, gOutputSpacing[2]);

  // --------------------------------------------------------------------------
  if (!gBinaryThreshold)
    {
    gBinaryThreshold = vtkImageThreshold::New();
    gBinaryThreshold->SetInput(gImageResample->GetOutput());
    gBinaryThreshold->SetOutputScalarTypeToUnsignedChar();
    gBinaryThreshold->ThresholdByUpper(0.5f);
    gBinaryThreshold->SetInValue(1);
    gBinaryThreshold->SetOutValue(0);
    gBinaryThreshold->ReplaceInOn();
    gBinaryThreshold->ReplaceOutOn();
    gImageResample->Delete();
    }
  else if (gBinaryThreshold->GetInput() != gImageResample->GetOutput())
    {
    gBinaryThreshold->SetInput(gImageResample->GetOutput());
    gImageResample->Delete();
    }

  // --------------------------------------------------------------------------
  if (gKernelSize > 1)
    {
    if (!gImageDilateErode3D)
      {
      gImageDilateErode3D = vtkImageDilateErode3D::New();
      gImageDilateErode3D->SetInput(gBinaryThreshold->GetOutput());
      gImageDilateErode3D->SetDilateValue(1);
      gImageDilateErode3D->SetErodeValue(0);
      gBinaryThreshold->Delete();
      }
    else if (gImageDilateErode3D->GetInput() != gBinaryThreshold->GetOutput())
      {
      gImageDilateErode3D->SetInput(gBinaryThreshold->GetOutput());
      gBinaryThreshold->Delete();
      }

    gImageDilateErode3D->SetKernelSize(gKernelSize, gKernelSize, gKernelSize);

    if (!gFilteredStructuredPoints)
      {
      gFilteredStructuredPoints = vtkImageToStructuredPoints::New();
      gFilteredStructuredPoints->SetInput(gImageDilateErode3D->GetOutput());
      gImageDilateErode3D->Delete();
      }
    else if ( gFilteredStructuredPoints->GetInput() !=
              gImageDilateErode3D->GetOutput() )
      {
      gFilteredStructuredPoints->SetInput(gImageDilateErode3D->GetOutput());
      gImageDilateErode3D->Delete();
      }
    }
  else
    {
    if (gImageDilateErode3D)
      {
      gImageDilateErode3D->GetInput()->GetSource()->Register(NULL);
      if (gFilteredStructuredPoints) gFilteredStructuredPoints->SetInput(NULL);
      gImageDilateErode3D = NULL;
      }

    if (!gFilteredStructuredPoints)
      {
      gFilteredStructuredPoints = vtkImageToStructuredPoints::New();
      gFilteredStructuredPoints->SetInput(gBinaryThreshold->GetOutput());
      gBinaryThreshold->Delete();
      }
    else if ( gFilteredStructuredPoints->GetInput() !=
              gBinaryThreshold->GetOutput() )
      {
      gFilteredStructuredPoints->SetInput(gBinaryThreshold->GetOutput());
      gBinaryThreshold->Delete();
      }
    }

  // --------------------------------------------------------------------------
  vtkRenderer* renderer = GUI->GetView()->GetDefaultRenderer();

  // --------------------------------------------------------------------------
  if (!gContourData)
    {
    gContourData = CreateContourData(gFilteredStructuredPoints->GetOutput());
    if (gContourActor)
      {
      renderer->RemoveProp(gContourActor);
      gContourActor = NULL;
      }
    }

  if (gSmoother)
    {
    gSmoother->
      SetNumberOfIterations(int(GUI->SmoothingIterationsInput->value()));
    gSmoother->
      SetPassBand(float(GUI->SmoothingPassBandInput->value()));
    gSmoother->
      SetFeatureAngle(float(GUI->SmoothingFeatureAngleInput->value()));
    gSmoother->
      SetEdgeAngle(float(GUI->SmoothingEdgeAngleInput->value()));
    gSmoother->
      SetFeatureEdgeSmoothing(int(GUI->FeatureEdgeSmoothingToggle->value()));
    gSmoother->
      SetBoundarySmoothing(int(GUI->BoundarySmoothingToggle->value()));
    gSmoother->
      SetNonManifoldSmoothing(int(GUI->NonManifoldSmoothingToggle->value()));
    }

  if (gContourData)
    {
    gContourData->Update();
    gContourData->GetBounds(gOutputBounds);
    gContourData->GetCenter(gOutputCenter);
    gContourData->GetWholeExtent(gOutputWholeExtent);
    }
  
  // --------------------------------------------------------------------------
  if (!gContourActor)
    {
    gContourActor = CreateContourActor(gContourData);
    GUI->SetContour(gContourActor);
    gContourActor->Delete();
    renderer->ResetCamera();
    }

  if (gContourActor)
    {
    gContourActor->GetMapper()->Update();
    if (gContourData)
      {
      gContourData->GetBounds(gOutputBounds);
      gContourData->GetCenter(gOutputCenter);
      gContourData->GetWholeExtent(gOutputWholeExtent);
      }

    std::cerr << "\n"
              << "Ref Bounds: { " << gRefBounds[0] << ", "
              << gRefBounds[1] << ", " << gRefBounds[2] << ", "
              << gRefBounds[3] << ", " << gRefBounds[4] << ", "
              << gRefBounds[5] << " }\n"
              << "    Center: { " << gRefCenter[0] << ", "
              << gRefCenter[1] << ", " << gRefCenter[2] << " }\n"
              << "    Extent: { " << gRefWholeExtent[0] << ", "
              << gRefWholeExtent[1] << ", " << gRefWholeExtent[2] << ", "
              << gRefWholeExtent[3] << ", " << gRefWholeExtent[4] << ", "
              << gRefWholeExtent[5] << " }\n"
              << "Out Bounds: { " << gOutputBounds[0] << ", "
              << gOutputBounds[1] << ", " << gOutputBounds[2] << ", "
              << gOutputBounds[3] << ", " << gOutputBounds[4] << ", "
              << gOutputBounds[5] << " }\n"
              << "    Center: { " << gOutputCenter[0] << ", "
              << gOutputCenter[1] << ", " << gOutputCenter[2] << " }\n"
              << "    Extent: { " << gOutputWholeExtent[0] << ", "
              << gOutputWholeExtent[1] << ", " << gOutputWholeExtent[2] << ", "
              << gOutputWholeExtent[3] << ", " << gOutputWholeExtent[4] << ", "
              << gOutputWholeExtent[5] << " }\n"
              << std::endl;
    }

  GUI->UpdateView();
}

// ----------------------------------------------------------------------------
void
ContourVisibilityCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  ContourBucharUI* GUI;

  if(!(GUI = reinterpret_cast<ContourBucharUI*>(aPtr))) return;

  vtkProp3D* contour;

  if(contour = GUI->GetContour())
    {
    Fl::check();
    contour->SetVisibility(aButton->value());
    GUI->UpdateView();
    }
}

// ----------------------------------------------------------------------------
void
SaveContourCb(Fl_Button* aButton, void* aPtr)
{
  if(!aButton) return;

  ContourBucharUI* GUI;

  if(!(GUI = reinterpret_cast<ContourBucharUI*>(aPtr))) return;

  char* fileName = 0;

  if ( fileName = fl_file_chooser(
         "Write Contour Polygonal Data?",
         "PolyData Files (*.{vtk,vtp})",
         0/*NULL ... defaults to last filename*/) )
    {
    std::string outName(fileName);
    std::string::size_type idx = outName.rfind('.');
    std::string suffix;

    if (idx==std::string::npos)
      {
      suffix = "vtp";
      outName.append(".vtp");
      }
    else
      {
      suffix = outName.substr(idx+1);
      if (suffix!="vtp" && suffix!="vtk")
        {
        suffix = "vtp";
        outName.replace(idx,outName.size(),".vtp");
        }
      }

    if (!outName.empty())
      {
      if (suffix == "vtp")
        {
        vtkXMLDataSetWriter* writer = vtkXMLDataSetWriter::New();
          {
          writer->SetInput(gNormals->GetOutput());
          writer->SetByteOrderToBigEndian();
          writer->SetFileName(outName.c_str());
          vtkZLibDataCompressor* compressor = vtkZLibDataCompressor::New();
          writer->SetCompressor(compressor);
          compressor->Delete();
          //writer->SetDataModeToAscii();
          //writer->SetDataModeToBinary();
          writer->SetDataModeToAppended();
          writer->EncodeAppendedDataOn();
          writer->Write();
          }
        writer->Delete();
        }
      else if (suffix == "vtk")
        {
        vtkDataSetWriter* writer = vtkDataSetWriter::New();
          {
          writer->SetInput(gNormals->GetOutput());
          writer->SetFileName(outName.c_str());
          writer->Write();
          }
        writer->Delete();          
        }
      }
    }
}
