/* 
 * 
 * $Id: Converter.cxx,v 1.1.1.1 2006/12/19 22:59:36 christianh Exp $
 *
 */ 
#include <string>
// vtkExtensions
#include "vtkInputSource.h"
// VTK Common
#include "vtkInstantiator.h" // provoke singleton instanciation
#include "vtkPolyData.h"
#include "vtkStructuredPoints.h"
#include "vtkImageData.h"
#include "vtkStructuredGrid.h"
#include "vtkRectilinearGrid.h"
#include "vtkUnstructuredGrid.h"
// VTK Graphics
#include "vtkCleanPolyData.h"
#include "vtkDecimatePro.h"
#include "vtkWindowedSincPolyDataFilter.h"
#include "vtkPolyDataNormals.h"
// VTK IO
#include "vtkXMLDataSetWriter.h"
#include "vtkZLibDataCompressor.h"

VTK_EXTENSIONS_NAMESPACE_USING(vtkInputSource);

int
main(int argc, char* argv[])
{
  if(argc<5)
    {
    std::cerr << "Usage: " << argv[0]
              << " <smoothing> <decimation> <normals> <filename> [<outputname>]"
              << std::endl;
    return -1;
    }

  int   smoothing   = int(atoi(argv[1]));
  float dfactor     = atof(argv[2]);
  int   normals     = int(atoi(argv[3]));

  float  center[3], bounds[6], length;
  double width, height, depth;

  vtkInputSource* reader = vtkInputSource::New();
    {
    reader->DebugOn();

    reader->SetFileName(argv[4]);

    reader->GetOutput()->Update();
    reader->GetOutput()->ComputeBounds();
    reader->GetOutput()->GetCenter(center);
    reader->GetOutput()->GetBounds(bounds);
    length = reader->GetOutput()->GetLength();

    width  = bounds[1] - bounds[0];
    height = bounds[3] - bounds[2];
    depth  = bounds[5] - bounds[4];

    std::cerr << "\n In:\tcenter = (" << center[0] << ", " << center[1] << ", "
              << center[2] << ")\n\t width = " << width << "\n\theight = "
              << height << "\n\t depth = " << depth << std::endl;
    }

  vtkDataSet* outputDataSet = NULL;

  switch (reader->GetOutputType())
    {

    case VTK_POLY_DATA:
      {
      //reader->EnsureAreTrianglesOn();
      //reader->EnsureHasNormalsOn();

      vtkPolyData* polyData = reader->GetPolyDataOutput();

      if (smoothing)
        {
        vtkWindowedSincPolyDataFilter* smoother =
          vtkWindowedSincPolyDataFilter::New();
          {
          smoother->SetInput(polyData);
          smoother->SetNumberOfIterations(smoothing); // default = 20
          smoother->SetPassBand(0.2/(smoothing/20));  // default = 0.1
          smoother->SetFeatureAngle(90.3);            // default = 45.0
          smoother->SetEdgeAngle(30.1);               // default = 15.0
          smoother->SetFeatureEdgeSmoothing(1);       // default = 0 (Off)
          smoother->SetBoundarySmoothing(1);          // default = 1 (On)
          smoother->SetNonManifoldSmoothing(1);       // default = 0 (Off)
          smoother->SetGenerateErrorScalars(0);       // default = 0 (Off)
          smoother->SetGenerateErrorVectors(0);       // default = 0 (Off)
          polyData->GetSource()->Delete();
          }
        polyData = smoother->GetOutput();
        }

      if (dfactor)
        {
        vtkDecimatePro* decimator = vtkDecimatePro::New();
          {
          decimator->SetInput(polyData);
          decimator->SetTargetReduction(dfactor);
          decimator->SetFeatureAngle(15.0);        // default = 15.0
          decimator->SetPreserveTopology(1);       // default = 0 (Off)
          decimator->SetSplitAngle(75.0);          // default = 75.0
          decimator->SetSplitting(0);              // default = 1 (On)
          decimator->SetDegree(25);                // default = 25
          decimator->SetBoundaryVertexDeletion(1); // default = 1 (On)
          polyData->GetSource()->Delete();
          }
        polyData = decimator->GetOutput();
        }

      vtkCleanPolyData* cleaner = vtkCleanPolyData::New();
        {
        cleaner->SetInput(polyData);
        cleaner->ConvertLinesToPointsOn();
        cleaner->ConvertPolysToLinesOn();
        cleaner->ConvertStripsToPolysOn();
        cleaner->PointMergingOn();
        polyData->GetSource()->Delete();
        }
      polyData = cleaner->GetOutput();

      if (normals)
        {
        vtkPolyDataNormals* normalsFilter = vtkPolyDataNormals::New();
          {
          normalsFilter->SetInput(polyData);
          normalsFilter->SetFeatureAngle(45.0);      // default = 30.0
          normalsFilter->SetSplitting(0);            // default = 1 (On)
          normalsFilter->SetConsistency(1);          // default = 1 (On)
          normalsFilter->SetNonManifoldTraversal(0); // default = 1 (On)
          polyData->GetSource()->Delete();
          }
        polyData = normalsFilter->GetOutput();
        }

      polyData->Update();
      polyData->ComputeBounds();
      polyData->GetCenter(center);
      polyData->GetBounds(bounds);
      length = polyData->GetLength();

      width  = bounds[1] - bounds[0];
      height = bounds[3] - bounds[2];
      depth  = bounds[5] - bounds[4];

      std::cerr << "Out:\tcenter = (" << center[0] << ", " << center[1] << ", "
                << center[2] << ")\n\t width = " << width << "\n\theight = "
                << height << "\n\t depth = " << depth << std::endl;

      outputDataSet = polyData;
      } break; // VTK_POLY_DATA

    case VTK_STRUCTURED_POINTS:
    case VTK_IMAGE_DATA:
      {
      vtkStructuredPoints* structuredPoints =
        reader->GetStructuredPointsOutput();
      outputDataSet = structuredPoints;
      } break; // VTK_STRUCTURED_POINTS

    case VTK_STRUCTURED_GRID:
      {
      vtkStructuredGrid* structuredGrid = reader->GetStructuredGridOutput();
      outputDataSet = structuredGrid;
      } break; // VTK_STRUCTURED_GRID

    case VTK_RECTILINEAR_GRID:
      {
      vtkRectilinearGrid* rectilinearGrid = reader->GetRectilinearGridOutput();
      outputDataSet = rectilinearGrid;
      } break; // VTK_RECTILINEAR_GRID

    case VTK_UNSTRUCTURED_GRID:
      {
      vtkUnstructuredGrid* unstructuredGrid =
        reader->GetUnstructuredGridOutput();
      outputDataSet = unstructuredGrid;
      } break; // VTK_UNSTRUCTURED_GRID

    default:
      std::cerr << "\nBAD data object type: "
                << reader->GetOutputType() << ".\n" << std::endl;
      return -1;

    } // switch (reader->GetOutputType())

  if (argc>5 && outputDataSet!=NULL)
    {
    std::string outName(argv[5]);
    std::string::size_type idx = outName.rfind('.');
    std::string suffix(idx==std::string::npos ? "" : outName.substr(idx+1));

    switch (reader->GetOutputType())
      {
      case VTK_POLY_DATA:
        if (idx==std::string::npos) outName.append(".vtp");
        else if (suffix!="vtp") outName.replace(idx,outName.size(),".vtp");
        break;
      case VTK_STRUCTURED_POINTS:
      case VTK_IMAGE_DATA:
        if (idx==std::string::npos) outName.append(".vti");
        else if (suffix!="vti") outName.replace(idx,outName.size(),".vti");
        break;
      case VTK_STRUCTURED_GRID:
        if (idx==std::string::npos) outName.append(".vts");
        else if (suffix!="vts") outName.replace(idx,outName.size(),".vts");
        break;
      case VTK_RECTILINEAR_GRID:
        if (idx==std::string::npos) outName.append(".vtr");
        else if (suffix!="vtr") outName.replace(idx,outName.size(),".vtr");
        break;
      case VTK_UNSTRUCTURED_GRID:
        if (idx==std::string::npos) outName.append(".vtu");
        else if (suffix!="vtu") outName.replace(idx,outName.size(),".vtu");
        break;
      case VTK_PIECEWISE_FUNCTION:
      case VTK_DATA_OBJECT:
      case VTK_DATA_SET:
      case VTK_POINT_SET:
      default:
        outName.clear();
        std::cerr << "\nCannot write data object type: "
                  << reader->GetOutputType() << ".\n" << std::endl;
        break;
      } // switch (OutputType)

    if (!outName.empty())
      {
      std::cerr << "\noutName: \"" << outName << "\"\n" << std::endl;

      vtkXMLDataSetWriter* writer = vtkXMLDataSetWriter::New();
        {
        writer->SetInput(outputDataSet);
        writer->SetByteOrderToBigEndian();
        writer->SetFileName(outName.c_str());
        vtkZLibDataCompressor* compressor = vtkZLibDataCompressor::New();
        writer->SetCompressor(compressor);
        compressor->Delete();
        //writer->SetDataModeToAscii();
        //writer->SetDataModeToBinary();
        writer->SetDataModeToAppended();
        writer->EncodeAppendedDataOn();
        writer->Write();
        }
      writer->Delete();
      }
    }

  return 0;
}

/* 
 * End of: $Id: Converter.cxx,v 1.1.1.1 2006/12/19 22:59:36 christianh Exp $.
 * 
 */ 
