// 
// $Id: MakeModelCode.cxx,v 1.1.1.1 2006/12/19 22:59:36 christianh Exp $
// 

// g++ -g -I/usr/local/include/vtk MakeModelCode.cxx -L/usr/local/lib/vtk \
//   -o MakeModelCode -lvtkGraphics -lvtkIO -lvtkCommon 

#include <cstdio>
#include <cstdlib>
#include <iostream>
#include <cstring>
#include <cctype>

#include "vtkPolyData.h"
#include "vtkPoints.h"
#include "vtkDataArray.h"
#include "vtkCellArray.h"
#include "vtkFloatArray.h"
#include "vtkPointData.h"
#include "vtkTransform.h"

#include "vtkTriangleFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkDecimatePro.h"
#include "vtkReverseSense.h"
#include "vtkTransformPolyDataFilter.h"
#include "vtkSmoothPolyDataFilter.h"
#include "vtkWindowedSincPolyDataFilter.h"
#include "vtkPolyDataNormals.h"
#include "vtkAppendPolyData.h"
#include "vtkGeometryFilter.h"
#include "vtkDataSetSurfaceFilter.h"
#include "vtkPolyDataWriter.h"
#include "vtkInputSource.h"

#include "svvController.h"

#define VERTICES_PER_FACE       3

SVV_NAMESPACE_USING(svv);

int
main(int argc, char* argv[])
{
  ios::sync_with_stdio(); // call this so iostream plays nice with stdio

  if(argc!=6) {
    (void)fprintf(stderr,"Usage: %s <decimation> <smoothing> <reverse>"
                  " <input> <output.vtk>\n",argv[0]);
    return -1;
  }

  float dfactor     = atof(argv[1]);
  int   smoothing   = int(atoi(argv[2]));
  int   reverseflag = int(atoi(argv[3]));
  char  inFileName[2048];
  char  outFileName[2048];
  strcpy(inFileName,argv[4]);
  strcpy(outFileName,argv[5]);

  char* sfx;

  if(sfx = strrchr(outFileName,'.')) {
    if(strcmp(++sfx,"vtk")!=0) {
      strcpy(sfx,"vtk");
      sfx[3] = '\0';
    }
  }
  else {
    strcat(outFileName,".vtk");
  }

  vtkTriangleFilter* triangles = vtkTriangleFilter::New();
  triangles->SetInput( (svv.MakeSurfaceDataSource(inFileName))->
                       GetPolyDataOutput() );
  (triangles->GetInput())->Update();
  (triangles->GetInput())->ComputeBounds();
  float center[3], bounds[6];
  (triangles->GetInput())->GetCenter(center);
  (triangles->GetInput())->GetBounds(bounds);

  double w = bounds[1] - bounds[0];
  double h = bounds[3] - bounds[2];
  double d = bounds[5] - bounds[4];

  cerr << " In:\tcenter = (" << center[0] << ", " << center[1] << ", "
       << center[2] << ")\n\t width = " << w << "\n\theight = " << h
       << "\n\t depth = " << d << "\n";

  double scale = 3.5 / (triangles->GetInput())->GetLength();

  vtkTransform* transform1 = vtkTransform::New();
  transform1->Translate(-center[0], -center[1], -center[2]);
  vtkTransform* transform2 = vtkTransform::New();
  transform2->Scale(scale, scale, scale);

  vtkTransformPolyDataFilter* transformFilter1 =
    vtkTransformPolyDataFilter::New();
  vtkTransformPolyDataFilter* transformFilter2 =
    vtkTransformPolyDataFilter::New();

  if(reverseflag!=0) {
    vtkReverseSense* reverser = vtkReverseSense::New();
    reverser->SetInput(triangles->GetOutput());
    transformFilter1->SetInput(reverser->GetOutput());
  }
  else {
    transformFilter1->SetInput(triangles->GetOutput());
  }

  transformFilter2->SetInput(transformFilter1->GetOutput());

  transformFilter1->SetTransform(transform1);
  transformFilter2->SetTransform(transform2);

  vtkCleanPolyData* cleaner = vtkCleanPolyData::New();
  cleaner->SetInput(transformFilter2->GetOutput());

  vtkSmoothPolyDataFilter* smoother = 0;
  if(smoothing!=0) {
    smoother = vtkSmoothPolyDataFilter::New();
    smoother->SetInput(cleaner->GetOutput());
    smoother->SetNumberOfIterations(smoothing);  // default = 20
    smoother->SetRelaxationFactor(0.008);  // default = 0.01
    smoother->SetEdgeAngle(15.0);          // default = 15.0
    smoother->SetFeatureEdgeSmoothing(1);  // default = 1 (On)
    smoother->SetBoundarySmoothing(0);     // default = 0 (Off)
  }

  vtkPolyDataNormals* normalsFilter = vtkPolyDataNormals::New();

  if(dfactor!=0.0) {
    vtkDecimatePro* decimator = vtkDecimatePro::New();
    if(smoothing!=0) decimator->SetInput(smoother->GetOutput());
    else             decimator->SetInput(cleaner->GetOutput());
    decimator->SetTargetReduction(dfactor);
    decimator->SetFeatureAngle(25.0);        // default = 15.0
    decimator->SetPreserveTopology(0);       // default = 0 (Off)
    decimator->SetSplitAngle(100.0);         // default = 75.0
    decimator->SetSplitting(1);              // default = 1 (On)
    decimator->SetDegree(25);                // default = 25
    decimator->SetBoundaryVertexDeletion(1); // default = 1 (On)
    normalsFilter->SetInput(decimator->GetOutput());
  }
  else {
    if(smoothing!=0) normalsFilter->SetInput(smoother->GetOutput());
    else             normalsFilter->SetInput(cleaner->GetOutput());
  }

  normalsFilter->SetFeatureAngle(45.0);      // default = 30.0
  normalsFilter->SetSplitting(0);            // default = 1 (On)
  normalsFilter->SetConsistency(1);          // default = 1 (On)
  normalsFilter->SetNonManifoldTraversal(1); // default = 1 (On)

  normalsFilter->Update();

  vtkPolyData*  input     = normalsFilter->GetOutput();
  {
    input->ComputeBounds();
    float out_center[3], out_bounds[6];
    input->GetBounds(out_bounds);
    input->GetCenter(out_center);
    double out_w = out_bounds[1] - out_bounds[0];
    double out_h = out_bounds[3] - out_bounds[2];
    double out_d = out_bounds[5] - out_bounds[4];
    
    cerr << "Out:\tcenter = (" << out_center[0] << ", " << out_center[1]
         << ", " << out_center[2] << ")\n\t width = " << out_w
         << "\n\theight = " << out_h << "\n\t depth = " << out_d << "\n";
  }

  vtkPoints*    points    = input->GetPoints();
  int           npts      = points->GetNumberOfPoints();

  vtkDataArray* vertices  = points->GetData();
  float*        vptr      = (vtkFloatArray::SafeDownCast(vertices))->GetPointer(0);
  vtkPointData* pointdata = input->GetPointData();
  vtkDataArray* normals   = pointdata->GetNormals();
  float*        nptr      = (vtkFloatArray::SafeDownCast(normals))->GetPointer(0);
  int           ntuples   = normals->GetNumberOfTuples();

  if(normals && ntuples>0) {

    int ncells = 0;
    int nedges = 0;

    vtkCellArray* cells = input->GetPolys();
    
    if(cells) {
      ncells = cells->GetNumberOfCells();

      if(ncells > 1) {
        nedges = cells->GetNumberOfConnectivityEntries();

        cerr << "Writing: " << npts << " vertices, " << ncells << " facets\n";
#if 0
        (void)fprintf(stdout,"// \n// $Id: MakeModelCode.cxx,v 1.1.1.1 2006/12/19 22:59:36 christianh Exp $\n// \n#include <GL/gl.h>\n\n");
        (void)fprintf(stdout,"static GLsizei nVertices = %d;\n",npts);
        (void)fprintf(stdout,"static GLsizei nFacets   = %d;\n",ncells);
        (void)fprintf(stdout,"static GLsizei nEdges    = %d;\n\n",nedges);
        (void)fprintf(stdout,"static GLfloat vertices[][6] =\n{\n");

        int j;
        for(j=0; j<npts; j++,vptr+=3,nptr+=3) {
          (void)fprintf(stdout,"  {%13.6e, %13.6e, %13.6e,"
                        " %13.6e, %13.6e, %13.6e},\n",
                        nptr[0],nptr[1],nptr[2],
                        vptr[0],vptr[1],vptr[2]);
        }
        (void)fprintf(stdout,"};\n\n");

        vtkIdType* pts;
        vtkIdType  npts;
        (void)fprintf(stdout,"static GLushort facets[][3] =\n{\n");
        for(cells->InitTraversal(); cells->GetNextCell(npts,pts); ) {
          // ASSUMES 3 points per polygon
          (void)fprintf(stdout,"  {%4d, %4d, %4d},\n",
                        int(pts[0]),int(pts[1]),int(pts[2]));
        }
        (void)fprintf(stdout,"};\n\n");

        (void)fprintf(stdout,"GLuint\n"
                      "svGetObject(void)\n"
                      "{\n"
                      "  GLuint listhead = glGenLists(1);\n"
                      "\n"
                      "  glNewList(listhead, GL_COMPILE);\n"
                      "  {\n"
                      "    glPushClientAttrib(GL_CLIENT_VERTEX_ARRAY_BIT);\n"
                      "    {\n"
                      "      glInterleavedArrays(GL_N3F_V3F, GLsizei(0),"
                      " vertices);\n"
                      "      glDrawElements(GL_TRIANGLES, nFacets*3,"
                      " GL_UNSIGNED_SHORT, facets[0]);\n"
                      "    }\n"
                      "    glPopClientAttrib();\n"
                      "  }\n"
                      "  glEndList();\n"
                      "\n"
                      "  return listhead;\n"
                      "}\n"
                      "\n"
                      "// \n"
                      "// End of: $Id: MakeModelCode.cxx,v 1.1.1.1 2006/12/19 22:59:36 christianh Exp $\n"
                      "// \n");
#else
	vtkPolyDataWriter* writer = vtkPolyDataWriter::New();
	writer->SetFileName(outFileName);
	writer->SetFileTypeToASCII();
	writer->SetInput(normalsFilter->GetOutput());
	writer->Write();
#endif
      }
    }
  }

  return 0;
}


static char*
fgetl(char* s, int n, FILE* fp)
{
  char* cp;

  do
  {
    cp = fgets(s, n, fp) ;
    if(!cp) return 0;

    while(isspace(int(*cp))) cp++;

  } while(((*cp) == '#') || ((*cp) == '\n') || ((*cp) == 0));

  for(char* cp2=cp; *cp2; cp2++) {
    if(*cp2 == '#') *cp2 = '\0';
  }

  int len = strlen(cp);
  if(cp[len-1] == '\n') cp[len-1] = '\0'; // strip newline

  return cp;
}

// 
// End of: $Id: MakeModelCode.cxx,v 1.1.1.1 2006/12/19 22:59:36 christianh Exp $.
// 
