/* 
 * g++ -ggdb -O2 -Wno-deprecated -I/usr/local/include/vtk -c 3DMorph2.cxx
 * 
 * g++ -ggdb -O2 -L/usr/local/lib/vtk 3DMorph2.o -o 3DMorph2 \
 *   -lvtkHybrid -lvtkRendering -lvtkGraphics -lvtkImaging -lvtkFiltering -lvtkCommon
 *
 */
#include <cstdlib>
#include <cstring>
#include <iostream>
// VTK Common
#include "vtkMath.h"
#include "vtkTransform.h"
#include "vtkPointLocator.h"
#include "vtkImplicitDataSet.h"
#include "vtkIdTypeArray.h"
#include "vtkProcessStatistics.h"
#include "vtkStructuredPoints.h"
#include "vtkDataSet.h"
// VTK Rendering
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
#include "vtkPolyDataMapper.h"
#include "vtkAssembly.h"
#include "vtkProperty.h"
#include "vtkActor.h"
// VTK Hybrid
#include "vtkImplicitModeller.h"
#include "vtkCubeAxesActor2D.h"
// VTK Filtering
#include "vtkImplicitDataSet.h"
// VTK Graphics
#include "vtkStructuredPointsSource.h"
#include "vtkSphereSource.h"
#include "vtkSuperquadricSource.h"
#include "vtkTransformPolyDataFilter.h"
#include "vtkAppendPolyData.h"
#include "vtkGaussianSplatter.h"
#include "vtkContourFilter.h"
#include "vtkPolyDataNormals.h"
#include "vtkPolyDataConnectivityFilter.h"
#include "vtkProbeFilter.h"
#include "vtkWindowedSincPolyDataFilter.h"
#include "vtkTriangleFilter.h"
#include "vtkMassProperties.h"
#include "vtkAxes.h"
#include "vtkTubeFilter.h"
#include "vtkCleanPolyData.h"
#include "vtkOutlineCornerSource.h"


// gloss light green
float Ka1[3] = { 0.000000, 0.000000, 0.000000 };
float Kd1[4] = { 0.176471, 0.882353, 0.082353, 1.000000 };
float Ks1[3] = { 1.000000, 1.000000, 1.000000 };
float Se1    = 56.000000;
// gloss red
float Ka2[3] = { 0.000000, 0.000000, 0.000000 };
float Kd2[4] = { 0.823529, 0.000000, 0.000000, 1.000000 };
float Ks2[3] = { 1.000000, 1.000000, 1.000000 };
float Se2    = 56.000000;
// pewter
float Ka3[3] = { 0.105882, 0.058824, 0.113725 };
float Kd3[4] = { 0.427451, 0.470588, 0.541176, 1.000000 };
float Ks3[3] = { 0.333333, 0.333333, 0.521569 };
float Se3    = 9.846150;

static vtkProcessStatistics* gpProcessStatistics = 0;

static void
PrintProcessInfo(ostream& os)
{
  if(!gpProcessStatistics) gpProcessStatistics = vtkProcessStatistics::New();

  os << std::endl
     << "( process size: "
     << (gpProcessStatistics->GetProcessSizeInBytes())/(1024*1024) << " Mb, "
     << "CPU time: "
     << gpProcessStatistics->GetProcessCPUTimeInMilliseconds()/1000 << " s. )"
     << std::endl << std::endl;
}

static vtkPolyDataSource*
MakeTestPolyDataSource(int n = 4)
{
  int thetaRes, phiRes;
  float thetaRnd, phiRnd, scale[3], center[3];

  vtkAppendPolyData* pAppendPolyData = vtkAppendPolyData::New();
  {
    for(int i=0; i<n; i++) {
      thetaRnd  = vtkMath::Random(0.2f, 2.f);
      phiRnd    = vtkMath::Random(0.2f, 2.f);
      scale[0]  = vtkMath::Random(5.f, 20.f);
      scale[1]  = vtkMath::Random(5.f, 20.f);
      scale[2]  = vtkMath::Random(5.f, 20.f);
      thetaRes  = int(rint((scale[0]+scale[1])/2.0));
      phiRes    = int(rint((scale[1]+scale[2])/2.0));
      //thetaRes  = int(rint(vtkMath::Random(8.f, 32.f)));
      //phiRes    = int(rint(vtkMath::Random(8.f, 32.f)));
      center[0] = vtkMath::Random(-20.f, 20.f);
      center[1] = vtkMath::Random(-40.f, 40.f);
      center[2] = vtkMath::Random(-30.f, 30.f);
        
      if(vtkMath::Random()<0.5) {
        vtkSuperquadricSource* pSuperquadricSource =
          vtkSuperquadricSource::New();
        {
          pSuperquadricSource->ToroidalOff();
          pSuperquadricSource->SetThetaResolution(thetaRes);
          pSuperquadricSource->SetPhiResolution(phiRes);
          pSuperquadricSource->SetThetaRoundness(thetaRnd);
          pSuperquadricSource->SetPhiRoundness(phiRnd);
          pSuperquadricSource->SetScale(scale);
          pSuperquadricSource->SetCenter(center);
        }
        pAppendPolyData->AddInput(pSuperquadricSource->GetOutput());
        pSuperquadricSource->Delete();
      }
      else {
        vtkTransformPolyDataFilter* pTransformFilter =
          vtkTransformPolyDataFilter::New();
        {
          vtkSphereSource* pSphereSource = vtkSphereSource::New();
          {
            pSphereSource->LatLongTessellationOff();
            pSphereSource->SetThetaResolution(thetaRes);
            pSphereSource->SetPhiResolution(phiRes);
            pSphereSource->SetCenter(0.f, 0.f, 0.f);
            pSphereSource->SetRadius(0.5f);
          }
          vtkTransform* pTransform = vtkTransform::New();
          {
            pTransform->Translate(center);
            pTransform->Scale(scale);
          }
          pTransformFilter->SetInput(pSphereSource->GetOutput());
          pTransformFilter->SetTransform(pTransform);
          pSphereSource->Delete();
          pTransform->Delete();
        }
        pAppendPolyData->AddInput(pTransformFilter->GetOutput());
        pTransformFilter->Delete();
      }
    }
  }

  return pAppendPolyData;
}

static void
PrintDataSetInfo(std::ostream& os, vtkDataSet* pDataSet, const char* name)
{
  float bounds[6];
  pDataSet->GetBounds(bounds);

  os << name << " bounds = {" << bounds[0] << "," << bounds[1]
     << "," << bounds[2] << "," << bounds[3]
     << "," << bounds[4] << "," << bounds[5] << "}, "
     << "length = " << pDataSet->GetLength() << std::endl;

  os << "\t" << pDataSet->GetNumberOfPoints() << " points, "
     << pDataSet->GetNumberOfCells() << " cells" << std::endl;
}

static void
PrintSourceInfo(std::ostream& os, vtkSource* o, const char* name, bool mp=false)
{
#if 1
  vtkDataSet* pDataSet;

  o->Update();

  if(vtkPolyDataSource::SafeDownCast(o)) {
    pDataSet = vtkPolyDataSource::SafeDownCast(o)->GetOutput();
  }
  else if(vtkStructuredPointsSource::SafeDownCast(o)) {
    pDataSet = vtkStructuredPointsSource::SafeDownCast(o)->GetOutput();
  }
  else if(vtkDataSetSource::SafeDownCast(o)) {
    pDataSet = vtkDataSetSource::SafeDownCast(o)->GetOutput();
  }
  else {
    return;
  }

  PrintDataSetInfo(os, pDataSet, name);

  vtkPolyData* pPolyData;
  if(mp && (pPolyData = vtkPolyData::SafeDownCast(pDataSet))) {
    vtkMassProperties* pMassProperties = vtkMassProperties::New();
    {
      vtkTriangleFilter* pTriangleFilter = vtkTriangleFilter::New();
      {
        vtkCleanPolyData* pCleanPolyData = vtkCleanPolyData::New();
        {
          pCleanPolyData->SetInput(pPolyData);
          pCleanPolyData->ConvertLinesToPointsOn();
          pCleanPolyData->ConvertPolysToLinesOn();
          pCleanPolyData->ConvertStripsToPolysOn();
          pCleanPolyData->PointMergingOn();
          pCleanPolyData->SetTolerance(0.0);
        }
        pTriangleFilter->SetInput(pCleanPolyData->GetOutput());
        pTriangleFilter->PassVertsOff();
        pTriangleFilter->PassLinesOff();
        pCleanPolyData->Delete();
      }
      pMassProperties->SetInput(pTriangleFilter->GetOutput());
      pTriangleFilter->Delete();

      os << "\tvolume: " << pMassProperties->GetVolume() << ", surface area: "
         << pMassProperties->GetSurfaceArea() << ", normalized shape index: "
         << pMassProperties->GetNormalizedShapeIndex() << std::endl;
    }
    pMassProperties->Delete();
  }

  PrintProcessInfo(os);
#endif /* 0 */
}

void
GetExtents(vtkDataSet* pDataSet,
           float length, float bounds[6], float center[6], float closest[3])
{
  vtkIdType centralPoint = -1;

  length = pDataSet->GetLength();
  pDataSet->GetBounds(bounds);
  pDataSet->GetCenter(center);
  
  vtkPolyData* pPolyData;
  if( (pPolyData = vtkPolyData::SafeDownCast(pDataSet)) &&
      ((centralPoint = pPolyData->FindPoint(center)) < 0) ) {
    pPolyData->GetPoint(centralPoint, closest);
    cout << "central point (" << centralPoint << "): {" << closest[0]
         << "," << closest[1] << "," << closest[2] << "}" << std::endl;
  }
  else {
    pDataSet->GetPoint(0, closest);
  }
}

void
GetInflatedExtents(float originalBounds[6], float d,
                   float inflationBounds[6], int inflationRange[3])
{
  for(int i=0; i<3; i++) {
    inflationBounds[i*2+0] = originalBounds[i*2+0] - d;
    inflationBounds[i*2+1] = originalBounds[i*2+1] + d;
    inflationRange[i] =
      int(ceil((inflationBounds[i*2+1] - inflationBounds[i*2+0])/2.0));
  }

  cout << "inflation sample range: { " << inflationRange[0] << ", "
       << inflationRange[1] << ", " << inflationRange[2] << " }" << std::endl;
}

static vtkPolyDataSource*
CreateSplatter(vtkDataSet* pDataSet, float contour = 0.125)
{
  vtkContourFilter* pSplatterSource = vtkContourFilter::New();
  {
    vtkGaussianSplatter* pGaussianSplatter = vtkGaussianSplatter::New();
    {
      pGaussianSplatter->SetInput(pDataSet);
      pGaussianSplatter->SetSampleDimensions(50,50,50);
      pGaussianSplatter->SetRadius(0.05);
      pGaussianSplatter->ScalarWarpingOff();
    }
    pSplatterSource->SetInput(pGaussianSplatter->GetOutput());
    pSplatterSource->SetValue(0,0.01);
    pGaussianSplatter->Delete();
  }

  return pSplatterSource;
}

static vtkPolyDataSource*
CreateMergedContour(vtkPolyDataSource* pOriginalSource, float contour = 0.125)
{
  float originalLength, originalBounds[6], originalCenter[3];
  float closest[3];
  vtkIdType centralPoint = -1;

  pOriginalSource->Update();

  GetExtents(pOriginalSource->GetOutput(),
             originalLength, originalBounds, originalCenter, closest);

  PrintSourceInfo(cout,pOriginalSource,"original", true);

  float value = contour * originalLength;
  float dmax  = value + originalLength * 0.125;
  float inflationBounds[6];
  int   inflationRange[3];

  GetInflatedExtents(originalBounds, dmax, inflationBounds, inflationRange);

  // create implicit model of the input (original) PolyData
  vtkImplicitModeller* pInflationModel = vtkImplicitModeller::New();
  {
    pInflationModel->SetInput(pOriginalSource->GetOutput());
    pInflationModel->SetSampleDimensions(inflationRange);
    pInflationModel->SetMaximumDistance(dmax);
    pInflationModel->SetModelBounds(inflationBounds);
    pInflationModel->SetAdjustBounds(0/*Off*/);  // default: On
    pInflationModel->SetCapping(1/*On*/);        // default: On
    pInflationModel->SetProcessModeToPerVoxel(); // default: per cell
    pInflationModel->SetNumberOfThreads(1); /* should be # of processors */
    
    PrintSourceInfo(cout,pInflationModel,"inflation model");
  }

  vtkContourFilter* pInflatedContour = vtkContourFilter::New();
  {
    pInflatedContour->SetInput(pInflationModel->GetOutput());
    pInflatedContour->ComputeScalarsOn();
    pInflatedContour->UseScalarTreeOn();
    pInflatedContour->SetValue(1, value);

    PrintSourceInfo(cout,pInflatedContour,"inflated contour");
  }

  float deflationBounds[6];
  int   deflationRange[3];

  pInflatedContour->GetOutput()->GetBounds(deflationBounds);
  if( deflationBounds[0] == VTK_LARGE_FLOAT &&
      deflationBounds[1] == -VTK_LARGE_FLOAT &&
      deflationBounds[2] == VTK_LARGE_FLOAT &&
      deflationBounds[3] == -VTK_LARGE_FLOAT &&
      deflationBounds[4] == VTK_LARGE_FLOAT &&
      deflationBounds[5] == -VTK_LARGE_FLOAT ) {
    cerr << "Something is terribly wrong! Inflated Contour Empty!" << std::endl;
    pInflationModel->Delete();
    pInflatedContour->Delete();
    pOriginalSource->Register(0);
    return pOriginalSource;
  }

  for(int i=0; i<3; i++) {
    deflationRange[i] =
      int(ceil((deflationBounds[i*2+1] - deflationBounds[i*2+0])/2.0));
  }

  cout << "deflation sample range: { " << deflationRange[0] << ", "
       << deflationRange[1] << ", " << deflationRange[2] << " }" << std::endl;

  // create implicit model of the inflated contour PolyData
  vtkImplicitModeller* pDeflationModel = vtkImplicitModeller::New();
  {
    pDeflationModel->SetInput(pInflatedContour->GetOutput());
    pDeflationModel->SetSampleDimensions(deflationRange);
    pDeflationModel->SetMaximumDistance(dmax);
    pDeflationModel->SetModelBounds(deflationBounds);
    pDeflationModel->SetAdjustBounds(0/*Off*/);  // default: On
    pDeflationModel->SetCapping(1/*Off*/);       // default: On
    pDeflationModel->SetProcessModeToPerVoxel();
    pDeflationModel->SetNumberOfThreads(1); /* should be # of processors */
    
    PrintSourceInfo(cout,pDeflationModel,"deflation model");
  }

  vtkContourFilter* pDeflatedContour = vtkContourFilter::New();
  {
    pDeflatedContour->SetInput(pDeflationModel->GetOutput());
    pDeflatedContour->ComputeScalarsOn();
    pDeflatedContour->UseScalarTreeOn();
    pDeflatedContour->SetValue(0, value);

    PrintSourceInfo(cout,pDeflatedContour,"deflated contour");
  }

  vtkIdType numIds = 0;
  vtkIdTypeArray* pValidPointIds;
  
  vtkProbeFilter* pProbeFilter = vtkProbeFilter::New();
  {
    pProbeFilter->SetInput(pDeflatedContour->GetOutput());
    pProbeFilter->SetSource(pOriginalSource->GetOutput());
    pProbeFilter->Update();
    pValidPointIds = pProbeFilter->GetValidPoints();

    PrintSourceInfo(cout,pProbeFilter,"deflated probe");
    
    if(pValidPointIds) {
      numIds = pValidPointIds->GetNumberOfTuples();
      cout << "\t" << numIds << " valid probe points." << std::endl;
    }
  }

  vtkWindowedSincPolyDataFilter* pSmoother =
    vtkWindowedSincPolyDataFilter::New();
  {
    vtkPolyDataConnectivityFilter* pConnectivityFilter =
      vtkPolyDataConnectivityFilter::New();
    {
      vtkTriangleFilter* pTriangleFilter = vtkTriangleFilter::New();
      {
        vtkCleanPolyData* pCleanPolyData = vtkCleanPolyData::New();
        {
          pCleanPolyData->SetInput(pDeflatedContour->GetOutput());
          pCleanPolyData->ConvertLinesToPointsOn();
          pCleanPolyData->ConvertPolysToLinesOn();
          pCleanPolyData->ConvertStripsToPolysOn();
          pCleanPolyData->PointMergingOn();
          pCleanPolyData->SetTolerance(0.0);
        }
        pTriangleFilter->SetInput(pCleanPolyData->GetOutput());
        pTriangleFilter->PassVertsOff();
        pTriangleFilter->PassLinesOff();
        pCleanPolyData->Delete();
      }
      pConnectivityFilter->SetInput(pTriangleFilter->GetOutput());
      pTriangleFilter->Delete();
      // get the interior surface
      if(numIds>0) {
        pConnectivityFilter->InitializeSeedList();
        for(vtkIdType id=0; id<numIds; id++) {
          pConnectivityFilter->AddSeed(pValidPointIds->GetValue(id));
        }
        pConnectivityFilter->SetExtractionModeToPointSeededRegions();
      }
      else {
        pConnectivityFilter->SetExtractionModeToClosestPointRegion();
        pConnectivityFilter->SetClosestPoint(closest);
      }
      
      PrintSourceInfo(cout,pConnectivityFilter,"deflated filtered");
    }
    pSmoother->SetInput(pConnectivityFilter->GetOutput());
    pConnectivityFilter->Delete();
    
    PrintSourceInfo(cout,pSmoother,"deflated smoothed",true);
  }

  pProbeFilter->Delete();
  pInflationModel->Delete();
  pDeflationModel->Delete();
  pInflatedContour->Delete();
  pDeflatedContour->Delete();

  return pSmoother;
}

void
AddAxes(vtkRenderer* pRenderer)
{
  if(!pRenderer) return;

  vtkActor* pActor;
  vtkActorCollection* pActorCollection = pRenderer->GetActors();

  float tmp[6], bounds[6] = { VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                              VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                              VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT };

  pActorCollection->InitTraversal();
  while(pActor = pActorCollection->GetNextActor()) {
    pActor->GetBounds(tmp);
    if(tmp[0] < bounds[0]) bounds[0] = tmp[0];
    if(tmp[1] > bounds[1]) bounds[1] = tmp[1];
    if(tmp[2] < bounds[2]) bounds[2] = tmp[2];
    if(tmp[3] > bounds[3]) bounds[3] = tmp[3];
    if(tmp[4] < bounds[4]) bounds[4] = tmp[4];
    if(tmp[5] > bounds[5]) bounds[5] = tmp[5];
  }

  float center[3], range[3], length;

  center[0] = (bounds[0]+bounds[1])/2.f;
  center[1] = (bounds[2]+bounds[3])/2.f;
  center[2] = (bounds[4]+bounds[5])/2.f;
  range[0]  = bounds[1] - bounds[0];
  range[1]  = bounds[3] - bounds[2];
  range[2]  = bounds[5] - bounds[4];
  length    = float(sqrt( range[0]*range[0] +
                          range[1]*range[1] +
                          range[2]*range[2] ));

  vtkAssembly* pAssembly = vtkAssembly::New();
  {
    // axes part
    vtkActor* pAxesActor = vtkActor::New();
    {
      vtkPolyDataMapper* pAxesMapper = vtkPolyDataMapper::New();
      {
        vtkAxes* pAxes = vtkAxes::New();
        {
          pAxes->SymmetricOn();
          pAxes->SetOrigin(center);
          pAxes->SetScaleFactor(length/2.f);
          pAxes->ComputeNormalsOn();
        }
        pAxesMapper->SetInput(pAxes->GetOutput());
        pAxes->Delete();
      }
      pAxesActor->SetMapper(pAxesMapper);
      pAxesActor->SetScale(range[0]/length,range[1]/length,range[2]/length);
      pAxesMapper->Delete();
    }
    pAssembly->AddPart(pAxesActor);
    pAxesActor->Delete();
    // outline part
    vtkActor* pOutlineActor = vtkActor::New();
    {
      vtkPolyDataMapper* pOutlineMapper = vtkPolyDataMapper::New();
      {
        vtkOutlineCornerSource* pOutline = vtkOutlineCornerSource::New();
        {
          pOutline->SetBounds(bounds);
          //pOutline->SetCornerFactor(float);
        }
        pOutlineMapper->SetInput(pOutline->GetOutput());
        pOutline->Delete();
      }
      pOutlineActor->SetMapper(pOutlineMapper);
      pOutlineMapper->Delete();
    }
    pAssembly->AddPart(pOutlineActor);
    pOutlineActor->Delete();
  }
  pRenderer->AddProp(pAssembly);
  pAssembly->Delete();

  vtkCubeAxesActor2D* pCubeAxesActor2D = vtkCubeAxesActor2D::New();
  {
    pCubeAxesActor2D->SetBounds(bounds);
    pCubeAxesActor2D->SetCamera(pRenderer->GetActiveCamera());
    pCubeAxesActor2D->SetFlyModeToOuterEdges();
    pCubeAxesActor2D->SetScaling(0/*Off*/);
    pCubeAxesActor2D->SetShadow(1/*On*/);
    //pCubeAxesActor2D->SetInertia(int);
  }
  pRenderer->AddProp(pCubeAxesActor2D);
  pCubeAxesActor2D->Delete();

  pRenderer->GetRenderWindow()->Render();
}


int
main(int argc, char* argv[])
{
  // 
  // Create the RenderWindow, Renderer and both Actors
  // 
  vtkRenderer* pRenderer = vtkRenderer::New();
  vtkRenderWindow* pRenderWindow = vtkRenderWindow::New();
  pRenderWindow->AddRenderer(pRenderer);
  pRenderWindow->SetWindowName(argv[0]);
  pRenderWindow->LineSmoothingOn();
  vtkRenderWindowInteractor* pInteractor = vtkRenderWindowInteractor::New();
  pInteractor->SetRenderWindow(pRenderWindow);

  // now  make a renderer and tell it about lights and actors
  pRenderWindow->SetSize(384,384);
  pRenderer->SetBackground(0.01, 0.02, 0.03);
  pRenderer->TwoSidedLightingOn();
  
  int n = 4;
  float value = 0.15;
  if(argc>1) {
    n = atoi(argv[1]);
    if(n<2) n = 2;
    if(argc>2) {
      value = atof(argv[2]);
    }
    if(value<0.f) value = 0.125;
  }

  PrintProcessInfo(cout);

  cout << "Creating " << n << " objects; using contour inflation value: "
       << value << std::endl;

  float bounds[6];

  vtkPolyDataSource* pOriginalSource = MakeTestPolyDataSource(n);
  pOriginalSource->DebugOn();

  // the original picture
  vtkActor* pOriginalActor = vtkActor::New();
  {
    vtkPolyDataMapper* originalMapper = vtkPolyDataMapper::New();
    {
      vtkPolyDataNormals* originalNormals = vtkPolyDataNormals::New();
      {
        originalNormals->SetInput(pOriginalSource->GetOutput());
      }
      originalMapper->SetInput(originalNormals->GetOutput());
      originalMapper->ScalarVisibilityOff();
      originalNormals->Delete();
    }
    pOriginalActor->SetMapper(originalMapper);
    pOriginalActor->GetProperty()->SetAmbient(1.0);
    pOriginalActor->GetProperty()->SetDiffuse(1.0);
    pOriginalActor->GetProperty()->SetSpecular(1.0);
    pOriginalActor->GetProperty()->SetAmbientColor(Ka1);
    pOriginalActor->GetProperty()->SetDiffuseColor(Kd1);
    pOriginalActor->GetProperty()->SetSpecularColor(Ks1);
    pOriginalActor->GetProperty()->SetSpecularPower(Se1);
    pOriginalActor->GetProperty()->SetOpacity(Kd1[3]);
    originalMapper->Delete();
  }

  pRenderer->AddActor(pOriginalActor);
  pOriginalActor->GetBounds(bounds);
  pOriginalActor->AddPosition(-(bounds[1]-bounds[0])/2.f, 0.f, 0.f);
  pRenderer->ResetCamera();
  pRenderWindow->Render();

  vtkPolyDataSource* pMergedSource = CreateMergedContour(pOriginalSource,value);

  // the merged picture
  vtkActor* pMergedActor = vtkActor::New();
  {
    vtkPolyDataMapper* mergedMapper = vtkPolyDataMapper::New();
    {
      vtkPolyDataNormals* mergedNormals = vtkPolyDataNormals::New();
      {
        mergedNormals->SetInput(pMergedSource->GetOutput());
        pMergedSource->Delete();
      }
      mergedMapper->SetInput(mergedNormals->GetOutput());
      mergedMapper->ScalarVisibilityOff();
      mergedNormals->Delete();
    }
    pMergedActor->SetMapper(mergedMapper);
    pMergedActor->GetProperty()->SetAmbient(1.0);
    pMergedActor->GetProperty()->SetDiffuse(1.0);
    pMergedActor->GetProperty()->SetSpecular(1.0);
    pMergedActor->GetProperty()->SetAmbientColor(Ka2);
    pMergedActor->GetProperty()->SetDiffuseColor(Kd2);
    pMergedActor->GetProperty()->SetSpecularColor(Ks2);
    pMergedActor->GetProperty()->SetSpecularPower(Se2);
    pMergedActor->GetProperty()->SetOpacity(Kd2[3]);
    mergedMapper->Delete();
  }

  pRenderer->AddActor(pMergedActor);
  pMergedActor->AddPosition(+(bounds[1]-bounds[0])/2.f, 0.f, 0.f);
  pMergedActor->Delete();
  pRenderer->ResetCamera();
  pRenderWindow->Render();

  AddAxes(pRenderer);

  // render the image
  pInteractor->Start();

  PrintProcessInfo(cout);

  if(pOriginalSource) pOriginalSource->Delete();
  if(pOriginalActor) pOriginalActor->Delete();
  pRenderer->Delete();
  pRenderWindow->Delete();
  pInteractor->Delete();
  if(gpProcessStatistics) gpProcessStatistics->Delete();
}
