/* 
 * $Id: SurfaceGlom.cxx,v 1.1.1.1 2006/12/19 22:59:39 christianh Exp $
 *
 */
#include <cstdlib>
#include <cstring>
#include <iostream>
// VTK Common
#include "vtkMath.h"
#include "vtkTransform.h"
// VTK Rendering
#include "vtkPolyDataMapper.h"
#include "vtkProperty.h"
#include "vtkTextProperty.h"
#include "vtkActor.h"
#include "vtkAxisActor2D.h"
#include "vtkActorCollection.h"
#include "vtkAssembly.h"
#include "vtkRenderer.h"
#include "vtkRenderWindow.h"
#include "vtkRenderWindowInteractor.h"
// VTK Hybrid
#include "vtkCubeAxesActor2D.h"
// VTK Filtering
#include "vtkPolyDataSource.h"
#include "vtkSphereSource.h"
#include "vtkSuperquadricSource.h"
#include "vtkOutlineCornerSource.h"
// VTK Graphics
#include "vtkAppendPolyData.h"
#include "vtkTransformPolyDataFilter.h"
#include "vtkAxes.h"
// VTK IO
#include "vtkPolyDataWriter.h"
// vtkExtensions
#include "vtkInputSource.h"
// SV
#include "svvController.h"

SVV_NAMESPACE_USING(svvController);
SVV_NAMESPACE_USING(svv);
VTK_EXTENSIONS_NAMESPACE_USING(vtkInputSource);


static vtkAppendPolyData*    gInputAppender     = NULL;
static char*                 gOutputName        = NULL;
static bool                  gInteractiveFlag   = true;
static bool                  gAbsoluteFlag      = false;

static float  gDistance   = 0.05;
static float  gPadding    = 0.075;
static float  gScale      = 1.0;
static int    gIterations = 20;
static float  gPassBand   = 0.1;

extern vtkPolyDataSource*
CreateGlomContour (vtkPolyData*,
                   float = 0.05, float = 0.075, float = 0.5,
                   int = 20, float = 0.1, bool = false);

static vtkPolyDataSource*       MakeTestPolyDataSource (int = 8);
static void                     AddAxes (vtkRenderer*);
static void                     PrintUsage (ostream&, const char*);
static int                      ParseArgs (int argc, char** argv, int& i);


int
main (int argc, char* argv[], char* envp[])
{
  if (argc<2)
    {
    PrintUsage (std::cerr, argv[0]);
    std::exit (EXIT_FAILURE);
    }

  // call this so iostream plays nice with stdio
  ios::sync_with_stdio();

  vtkRenderer*               renderer     = NULL;
  vtkRenderWindow*           renderWindow = NULL;
  vtkRenderWindowInteractor* interactor   = NULL;

  if (gInteractiveFlag)
    {
    // 
    // Create the RenderWindow, Renderer and both Actors
    // 
    renderer     = vtkRenderer::New();
    renderWindow = vtkRenderWindow::New();
    renderWindow->AddRenderer (renderer);
    interactor   = vtkRenderWindowInteractor::New();
    interactor->SetRenderWindow (renderWindow);

    svv.SetInteractor (interactor);

    renderWindow->SetWindowName ("SurfaceGlom");
    renderWindow->SetSize (640, 640);
    renderer->SetBackground (0.3, 0.25, 0.2);
    }

  int i, ret;
  // consume all switches from argv.  Returns number of words eaten.
  // Returns zero on error.  'i' will either point at first word that
  // does not start with '-', at the error word, or after a '--', or at
  // argc.  If your program does not take any word arguments you can
  // report an error if i < argc.
  if (!(ret = svvController::Args (&argc, &argv, &envp, i, ParseArgs)))
    {
    PrintUsage (std::cerr,argv[0]);
    std::exit (EXIT_FAILURE);
    }
  else if (!gInputAppender || gInputAppender->GetNumberOfInputs() < 1)
    {
    std::cerr << "No input specified!\n" << std::endl;
    PrintUsage (std::cerr,argv[0]);
    std::exit (EXIT_FAILURE);
    }
  else
    {
    if (svv.GetDebug()) std::cerr << "Args() returned "<< ret << endl;
    }

  float bounds[6];

  if (gInteractiveFlag)
    {
    // Create representation of the original data.
    vtkProperty* property = svv.MakeProperty ("gloss light green");
    vtkActor*    actor    = svv.MakeSurfaceActor (gInputAppender, property);
    
    property->Delete();
    renderer->AddProp (actor);
    
    actor->GetBounds (bounds);
    
    actor->AddPosition (-(bounds[1]-bounds[0])/2.f, 0.f, 0.f);
    renderer->ResetCamera();
    renderWindow->Render();
    }

  // 
  // Create glommed contour source.
  // 
  vtkPolyDataSource* glomSource =
    CreateGlomContour ( gInputAppender->GetOutput(),
                        gDistance, gPadding, gScale, gIterations, gPassBand,
                        gAbsoluteFlag );

  if (gInteractiveFlag)
    {
    vtkProperty* glomProperty = svv.MakeProperty ("gloss orange");
    vtkActor*    glomActor    = svv.MakeSurfaceActor (glomSource, glomProperty);
    
    glomProperty->Delete();
    renderer->AddProp (glomActor);

    glomActor->AddPosition (+(bounds[1]-bounds[0])/2.f, 0.f, 0.f);
    renderer->ResetCamera();
    renderWindow->Render();

    AddAxes(renderer);

    // render the image
    interactor->Start();
    }

  vtkPolyDataWriter* glomWriter = vtkPolyDataWriter::New();
    {
    glomWriter->SetFileName(gOutputName);
    //glomWriter->SetFileTypeToASCII();
    glomWriter->SetFileTypeToBinary();
    glomWriter->SetInput(glomSource->GetOutput());
    glomWriter->Write();
    }
  glomWriter->Delete();

  if (gInteractiveFlag)
    {
    // clean up
    renderer->Delete();
    renderWindow->Delete();
    interactor->Delete();
    }

  std::exit (EXIT_SUCCESS);
}

// ----------------------------------------------------------------------------
void
PrintUsage (std::ostream& os, const char* progname)
{
  os << "\n"
     << "Usage: " << progname << " -i <input> -o <output> [options]\n"
     << "\t-D <distance>\t: distance moved as % bounding length\n"
     << "\t-P <padding>\t: resampling volume padding as % bounding length\n"
     << "\t-S <scale>\t: 3D range scale factor\n"
     << "\t-nw\t\t: non-interactive (windowless) execution\n"
     << std::endl;
}

// ----------------------------------------------------------------------------
static int
ParseArgs (int argc, char** argv, int& i)
{
  // pointer to the current flag
  const char*  flag = argv[i];
  char**       args;

  bool valid = false;

  // determines how many arguments follow the current flag
  int nargs = 0;
  {
    while ((i+nargs+1) < argc && argv[i+nargs+1][0]!='-') nargs++;
    
    if (svv.GetDebug()) std::cerr << " '" << &flag[0] << "' + ( ";

    if (nargs)
      {
      args = argv + std::ptrdiff_t(i+1);
      if (svv.GetDebug())
        {
        for (int n=0; n<nargs; n++) std::cerr<< "'"<< args[n]<< "' ";
        }
      }
    else
      {
      args = 0;
      if (svv.GetDebug()) std::cerr << "none ";
      }

    if (svv.GetDebug()) std::cerr << ")" << endl;
  }

  // first check for flags that do not take arguments
  if (args==0)
    {
    // display help and exit
    if (flag[1]=='h' && !flag[2])
      { PrintUsage(std::cerr,(svv.argv())[0]); std::exit(0); }

    // be verbose
    else if (flag[1]=='v' && !flag[2])
      { svv.DebugOn(); valid = true; }

    // be quiet
    else if (flag[1]=='q' && !flag[2])
      { svv.DebugOff(); valid = true; }

    // specify non-interactive (windowless) execution.
    else if (flag[1]=='n' && flag[2]=='w' && !flag[3])
      { gInteractiveFlag = false; valid = true; } // '-nw'

    // specify distances as absolute.
    else if (flag[1]=='A' && !flag[2])
      { gAbsoluteFlag = true; valid = true; } // '-A'

    // specify distances as relative.
    else if (flag[1]=='R' && !flag[2])
      { gAbsoluteFlag = false; valid = true; } // '-R'
    }
  else
    {
    // specify input pathnames
    if (flag[1]=='i' && !flag[2])
      {
      vtkInputSource* source;
      vtkPolyData* polyData;

      if (!gInputAppender) gInputAppender = vtkAppendPolyData::New();

      for (int n=0; n<nargs; n++)
        {
        // make the "standard" surface
        if (source = svv.MakeSurfaceDataSource(args[n]))
          {
          if (polyData = source->GetPolyDataOutput())
            {
            gInputAppender->AddInput(polyData);
            }
          source->Delete();
          }
        }
      valid = true;
      } // '-i'

    // specify composite geometry output pathname
    else if (flag[1]=='o' && !flag[2])
      {
      gOutputName = new char[strlen(args[0])+5];
      strcpy(gOutputName, args[0]);

      char* sfx;

      if (sfx = strrchr(gOutputName,'.'))
        {
	if (strcmp(++sfx,"vtk"))
          {
	  strcpy(sfx,"vtk");
	  sfx[3] = '\0';
          }
        }
      else
        {
	strcat(gOutputName,".vtk");
        }

      valid = true;
      } // '-o'

    // specify dmax as fraction of bounding box length.
    else if (flag[1]=='D' && !flag[2])
      {
      gDistance = float(atof(args[0]));
      std::cerr << "Distance    = " << gDistance << std::endl;
      valid = true;
      } // '-D'

    // specify padding as fraction of bounding box length.
    else if (flag[1]=='P' && !flag[2])
      {
      gPadding = float(atof(args[0]));
      std::cerr << "Padding     = " << gPadding << std::endl;
      valid = true;
      } // '-P'

    // specify scaling of 3D range.
    else if (flag[1]=='S' && !flag[2])
      {
      gScale = float(atof(args[0]));
      std::cerr << "Scaling     = " << gScale << std::endl;
      valid = true;
      } // '-S'

    // specify number of wsinc smoothing iterations.
    else if (flag[1]=='I' && !flag[2])
      {
      gIterations = int(atoi(args[0]));
      std::cerr << "Iterations  = " << gIterations << std::endl;
      valid = true;
      } // '-I'

    // specify pass band of wsinc smoothing.
    else if (flag[1]=='B' && !flag[2])
      {
      gPassBand = float(atof(args[0]));
      std::cerr << "PassBand    = " << gPassBand << std::endl;
      valid = true;
      } // '-I'
    }

  if (valid)
    {
    i += (nargs+1);
    return (nargs+1);
    }
  else
    {
    return 0;
    }
}

// ----------------------------------------------------------------------------
static vtkPolyDataSource*
MakeTestPolyDataSource(int n)
{
  int thetaRes, phiRes;
  float thetaRnd, phiRnd, scale[3], center[3];

  vtkAppendPolyData* pAppendPolyData = vtkAppendPolyData::New();
    {
    for (int i=0; i<n; i++)
      {
      thetaRnd  = vtkMath::Random(0.2f, 2.f);
      phiRnd    = vtkMath::Random(0.2f, 2.f);
      scale[0]  = vtkMath::Random(5.f, 20.f);
      scale[1]  = vtkMath::Random(5.f, 20.f);
      scale[2]  = vtkMath::Random(5.f, 20.f);
      thetaRes  = int(rint((scale[0]+scale[1])/2.0));
      phiRes    = int(rint((scale[1]+scale[2])/2.0));
      //thetaRes  = int(rint(vtkMath::Random(8.f, 32.f)));
      //phiRes    = int(rint(vtkMath::Random(8.f, 32.f)));
      center[0] = vtkMath::Random(-20.f, 20.f);
      center[1] = vtkMath::Random(-40.f, 40.f);
      center[2] = vtkMath::Random(-30.f, 30.f);
        
      if (vtkMath::Random()<0.5)
        {
        vtkSuperquadricSource* pSuperquadricSource =
          vtkSuperquadricSource::New();
          {
          pSuperquadricSource->ToroidalOff();
          pSuperquadricSource->SetThetaResolution(thetaRes);
          pSuperquadricSource->SetPhiResolution(phiRes);
          pSuperquadricSource->SetThetaRoundness(thetaRnd);
          pSuperquadricSource->SetPhiRoundness(phiRnd);
          pSuperquadricSource->SetScale(scale);
          pSuperquadricSource->SetCenter(center);
          }
        pAppendPolyData->AddInput(pSuperquadricSource->GetOutput());
        pSuperquadricSource->Delete();
        }
      else
        {
        vtkTransformPolyDataFilter* pTransformFilter =
          vtkTransformPolyDataFilter::New();
          {
          vtkSphereSource* pSphereSource = vtkSphereSource::New();
            {
            pSphereSource->LatLongTessellationOff();
            pSphereSource->SetThetaResolution(thetaRes);
            pSphereSource->SetPhiResolution(phiRes);
            pSphereSource->SetCenter(0.f, 0.f, 0.f);
            pSphereSource->SetRadius(0.5f);
            }
          vtkTransform* pTransform = vtkTransform::New();
            {
            pTransform->Translate(center);
            pTransform->Scale(scale);
            }
          pTransformFilter->SetInput(pSphereSource->GetOutput());
          pTransformFilter->SetTransform(pTransform);
          pSphereSource->Delete();
          pTransform->Delete();
          }
        pAppendPolyData->AddInput(pTransformFilter->GetOutput());
        pTransformFilter->Delete();
        }
      }
    }

  return pAppendPolyData;
}

// ----------------------------------------------------------------------------
static void
AddAxes(vtkRenderer* aRenderer)
{
  if (!aRenderer) return;

  float tmp[6], bounds[6] = { VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                              VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT,
                              VTK_LARGE_FLOAT, -VTK_LARGE_FLOAT };

  vtkActor* actor;
  vtkActorCollection* actorCollection = aRenderer->GetActors();

  actorCollection->InitTraversal();
  while (actor = actorCollection->GetNextActor())
    {
    actor->GetBounds(tmp);
    if (tmp[0] < bounds[0]) bounds[0] = tmp[0];
    if (tmp[1] > bounds[1]) bounds[1] = tmp[1];
    if (tmp[2] < bounds[2]) bounds[2] = tmp[2];
    if (tmp[3] > bounds[3]) bounds[3] = tmp[3];
    if (tmp[4] < bounds[4]) bounds[4] = tmp[4];
    if (tmp[5] > bounds[5]) bounds[5] = tmp[5];
    }

  float center[3], range[3], length;

  center[0] = (bounds[0]+bounds[1])/2.f;
  center[1] = (bounds[2]+bounds[3])/2.f;
  center[2] = (bounds[4]+bounds[5])/2.f;
  range[0]  = bounds[1] - bounds[0];
  range[1]  = bounds[3] - bounds[2];
  range[2]  = bounds[5] - bounds[4];
  length    = float(sqrt( range[0]*range[0] +
                          range[1]*range[1] +
                          range[2]*range[2] ));

  vtkAssembly* assembly = vtkAssembly::New();
    {
    // axes part
    vtkActor* axesActor = vtkActor::New();
      {
      vtkPolyDataMapper* axesMapper = vtkPolyDataMapper::New();
        {
        vtkAxes* axes = vtkAxes::New();
          {
          axes->SymmetricOn();
          axes->SetOrigin(center);
          axes->SetScaleFactor(length/2.f);
          axes->ComputeNormalsOn();
          }
        axesMapper->SetInput(axes->GetOutput());
        axes->Delete();
        }
      axesActor->SetMapper(axesMapper);
      axesActor->SetScale(range[0]/length,range[1]/length,range[2]/length);
      axesMapper->Delete();
      }
    assembly->AddPart(axesActor);
    axesActor->Delete();

    // outline part
    vtkActor* outlineActor = vtkActor::New();
      {
      vtkPolyDataMapper* outlineMapper = vtkPolyDataMapper::New();
        {
        vtkOutlineCornerSource* outline = vtkOutlineCornerSource::New();
          {
          outline->SetBounds(bounds);
          //outline->SetCornerFactor(float);
          }
        outlineMapper->SetInput(outline->GetOutput());
        outline->Delete();
        }
      outlineActor->SetMapper(outlineMapper);
      outlineMapper->Delete();
      }
    assembly->AddPart(outlineActor);
    outlineActor->Delete();
    }
  aRenderer->AddProp(assembly);
  assembly->Delete();

  // cube axes actor part
  vtkCubeAxesActor2D* cubeAxesActor2D = vtkCubeAxesActor2D::New();
    {
    cubeAxesActor2D->SetBounds(bounds);
    cubeAxesActor2D->SetCamera(aRenderer->GetActiveCamera());
    cubeAxesActor2D->SetFlyModeToOuterEdges();
    cubeAxesActor2D->ScalingOff();
    cubeAxesActor2D->SetCornerOffset(0.1);
    cubeAxesActor2D->SetInertia(3);
    cubeAxesActor2D->SetLabelFormat("%.1f");
    cubeAxesActor2D->SetFontFactor(0.8);
    cubeAxesActor2D->GetAxisTitleTextProperty()->ShadowOn();
    cubeAxesActor2D->GetAxisLabelTextProperty()->ShadowOn();
    cubeAxesActor2D->GetAxisTitleTextProperty()->AntiAliasingOn();
    cubeAxesActor2D->GetAxisLabelTextProperty()->AntiAliasingOn();
    cubeAxesActor2D->GetXAxisActor2D()->GetTitleTextProperty()->SetColor(1,0,0);
    cubeAxesActor2D->GetYAxisActor2D()->GetTitleTextProperty()->SetColor(1,1,0);
    cubeAxesActor2D->GetZAxisActor2D()->GetTitleTextProperty()->SetColor(0,0,1);
    }
  aRenderer->AddProp(cubeAxesActor2D);
  cubeAxesActor2D->Delete();

  aRenderer->GetRenderWindow()->Render();
}

/* 
 * End of: $Id: SurfaceGlom.cxx,v 1.1.1.1 2006/12/19 22:59:39 christianh Exp $.
 * 
 */
