// This example demonstrates how to write a task parallel application
// with VTK. It creates two different pipelines and assigns each to
// one processor. These pipelines are:
// 1. rtSource -> contour            -> probe               .-> append
//             \                     /                  port
//              -> gradient magnitude                  /
// 2. rtSource -> gradient -> shrink -> glyph3D -> port
// See SurfaceGlommer1.cxx and SurfaceGlommer2.cxx for the pipelines.
#include <iostream>
//
#include "SurfaceGlomParallel.H"
// SV
#include "svController.H"


static StringList inputArgs;

static void PrintUsage(ostream&, const char*);
static int  ParseArgs(int argc, char** argv, int& i);

// This function sets up properties common to both processes
// and executes the task corresponding to the current process
void
process(vtkMultiProcessController* pController, void* userData)
{
  TaskFunction task;

  // Chose the appropriate task
  switch (pController->GetLocalProcessId()) {
  case 0: task = SurfaceGlommer0; break;
  case 1: task = SurfaceGlommer1; break;
  case 2: task = SurfaceGlommer2; break;
  }

  // Run the tasks
  (*task)(*(reinterpret_cast<StringListPtr>(userData)));
}


int
main(int argc, char* argv[], char* envp[])
{
  // Note that this will create a vtkMPIController if MPI
  // is configured, vtkThreadedController otherwise.
  vtkMultiProcessController* pController = vtkMultiProcessController::New();
  pController->Initialize(&argc, &argv);

  // When using MPI, the number of processes is determined
  // by the external program which launches this application.
  // However, when using threads, we need to set it ourselves.
  if(pController->IsA("vtkThreadedController")) {
    // Set the number of processes to 3 for this example.
    pController->SetNumberOfProcesses(3);
  }

  if(pController->GetNumberOfProcesses() != 3) {
    std::cerr << "SurfaceGlomParallel requires three processes." << std::endl;
    pController->Finalize();
    pController->Delete();
    return 1;
  }

  // call this so iostream plays nice with stdio
  ios::sync_with_stdio();

  int i, ret;
  // consume all switches from argv.  Returns number of words eaten.
  // Returns zero on error.  'i' will either point at first word that
  // does not start with '-', at the error word, or after a '--', or at
  // argc.  If your program does not take any word arguments you can
  // report an error if i < argc.
  if(!(ret = Sv::Args(&argc, &argv, &envp, i, ParseArgs))) {
    PrintUsage(cerr,argv[0]);
    std::exit(0);
  }
  else {
    if(sv.GetDebug()) cerr << "Args() returned "<< ret << endl;
  }

  // Execute the function named "process" on both processes
  pController->SetSingleMethod(process, &inputArgs);
  pController->SingleMethodExecute();
  
  // Clean-up and exit
  pController->Finalize();
  pController->Delete();
  
  return 0;
}
// ----------------------------------------------------------------------------
static int
ParseArgs(int argc, char** argv, int& i)
{
  // pointer to the current flag
  const char* flag = argv[i];
  char**      args;

  bool valid = false;

  // determines how many arguments follow the current flag
  int nargs = 0;
  {
    while((i+nargs+1) < argc && argv[i+nargs+1][0]!='-') nargs++;
    
    if(sv.GetDebug()) cerr << " '" << &flag[0] << "' + ( ";

    if(nargs) {
      args = argv + std::ptrdiff_t(i+1);
      if(sv.GetDebug()) {
        for(int n=0; n<nargs; n++) cerr<< "'"<< args[n]<< "' ";
      }
    }
    else {
      args = 0;
      if(sv.GetDebug()) cerr << "none ";
    }

    if(sv.GetDebug()) cerr << ")" << endl;
  }

  // first check for flags that do not take arguments
  if(args==0) {

    // display help and exit
    if(flag[1]=='h' && !flag[2])
      { PrintUsage(cerr,(sv.argv())[0]); std::exit(0); }

    // be verbose
    else if(flag[1]=='v' && !flag[2])
      { sv.DebugOn(); inputArgs.push_back(flag); valid = true; }

    // be quiet
    else if(flag[1]=='q' && !flag[2])
      { sv.DebugOff(); inputArgs.push_back(flag); valid = true; }

  }
  else {

    // specify input pathnames
    if(flag[1]=='i' && !flag[2]) {
      inputArgs.push_back(flag);
      for(int n=0; n<nargs; n++) {
	inputArgs.push_back(args[n]);
      }
      valid = true;
    } // '-i'

    // specify composite geometry output pathname
    else if(flag[1]=='o' && !flag[2]) {
      inputArgs.push_back(flag);

      std::string            name = args[0];
      const std::string      sfx  = ".vtk";
      std::string::size_type pos  = name.rfind(sfx);

      if(pos == std::string::npos) {
	name += sfx;
      }
      else if(pos != (name.length() - sfx.length())) {
	name.replace(pos, (name.length() - pos), sfx);
      }

      inputArgs.push_back(name);

      valid = true;
    } // '-o'

    // specify dmax as fraction of bounding box length.
    else if(flag[1]=='D' && !flag[2]) {
      inputArgs.push_back(flag);
      inputArgs.push_back(args[0]);
      valid = true;
    } // '-D'

    // specify padding as fraction of bounding box length.
    else if(flag[1]=='P' && !flag[2]) {
      inputArgs.push_back(flag);
      inputArgs.push_back(args[0]);
      valid = true;
    } // '-P'

    // specify scaling of 3D range.
    else if(flag[1]=='S' && !flag[2]) {
      inputArgs.push_back(flag);
      inputArgs.push_back(args[0]);
      valid = true;
    } // '-P'
  }

  if(valid) {
    i += (nargs+1);
    return (nargs+1);
  }
  else {
    return 0;
  }
}

// ----------------------------------------------------------------------------
static void
PrintUsage(std::ostream& os, const char* progname)
{
  os << "Usage: " << progname << " -i <input> -o <output>"
     << " [-D <distance>] [-P <pad>] [-S <scale>]\n"
     << std::endl;
}

/* 
 * End of: $Id: SurfaceGlomParallel.cxx,v 1.1.1.1 2006/12/19 22:59:39 christianh Exp $.
 * 
 */
