// This example demonstrates how to write a task parallel application
// with VTK. It creates two different pipelines and assigns each to
// one processor. These pipelines are:
// 1. rtSource -> contour            -> probe               .-> append
//             \                     /                  port
//              -> gradient magnitude                  /
// 2. rtSource -> gradient -> shrink -> glyph3D -> port
// See Task1.cxx and Task2.cxx for the pipelines.

#include "TestParallel.H"

// This function sets up properties common to both processes
// and executes the task corresponding to the current process
void
process(vtkMultiProcessController* pController, void* vtkNotUsed(arg))
{
  TaskFunction task;
  int myId = pController->GetLocalProcessId();

  // Chose the appropriate task (see task3.cxx and task4.cxx)
  if(myId == 0) {
    task = Task1;
  }
  else {
    task = Task2;
  }

  // Run the tasks (see Task1.cxx and Task2.cxx)
  (*task)(EXTENT);
}


int
main(int argc, char* argv[], char* envp[])
{
  // Note that this will create a vtkMPIController if MPI
  // is configured, vtkThreadedController otherwise.
  vtkMultiProcessController* pController = vtkMultiProcessController::New();
  pController->Initialize(&argc, &argv);

  // When using MPI, the number of processes is determined
  // by the external program which launches this application.
  // However, when using threads, we need to set it ourselves.
  if(pController->IsA("vtkThreadedController")) {
    // Set the number of processes to 2 for this example.
    pController->SetNumberOfProcesses(2);
  }

  int numProcs = pController->GetNumberOfProcesses();

  if(numProcs != 2) {
    std::cerr << "TestParallel requires two processes." << std::endl;
    pController->Finalize();
    pController->Delete();
    return 1;
  }

  // Execute the function named "process" on both processes
  pController->SetSingleMethod(process, 0);
  pController->SingleMethodExecute();
  
  // Clean-up and exit
  pController->Finalize();
  pController->Delete();
  
  return 0;
}









